/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DFont.java,v 1.3 2008/05/23 14:11:47 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.core.internal;


/**
 * A DFont defines the appearance of a font. 
 * A DFont is defined by three strings: the font name (ie "Arial"), the family 
 * name (ie "Arial") and the scripts name (ie "latin-1). 
 * The family name and the script name can be set to NULL. 
 * Other parameters can be modified such as the font size in pixels or the style 
 * (NORMAL | ITALIC | BOLD | UNDERLINE | DASHED ...).
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DFont implements IDFont 
{
   private String family = null;
   private String script = null;
   private int size = - 1;
   private int style = NORMAL;
   
   
   /**
    * Checks if the style has a property.
    * @return true is _style have all _mask bit set.
    * For example IsStyle( BOLD|UNDERLINE, UNDERLINE) return false.
    * For example IsStyle( BOLD|UNDERLINE, UNDERLINE|BOLD) return true.
    */
   public static boolean IsStyle( int _style, int _mask )
   {
     return (_style&_mask)==_mask;
   }
   /**
    * Checks if the styke has one or more property.
    * @return true is _style have one (or more) _mask bit set.
    * For example HaveStyle( BOLD|UNDERLINE, UNDERLINE) return true.
    */
   public static boolean HaveStyle( int _style, int _mask )
   {
     return (_style&_mask)!=0;
   }
   
   
   /**
    * Creates a DFont. 
    * @param f font family name (can be NULL)
    * @param s font script name (can be NULL) 
    * @param si font size in pixels
    * @param st font style (NORMAL | ITALIC | BOLD | UNDERLINE | DASHED). The use of 
    * the binary operator '|' allows to combine different types. 
    */
   public DFont(String family, String script, int size, int style) 
   {
      this.family = family;
      this.script = script;
      this.size = size;
      this.style = style;
   }
   
   /**
    * Creates a DFont with a null familly a null script a non valid size and a NORMAL style.
    */
   public DFont() 
   {    
   }
   
    /**
    * Creates a DFont copying values from given parameter. 
    * 
    * @param font the font to copy. 
    *  
    */
   public DFont( IDFont font ) 
   {
     if( font !=null )
     {
       family = font.getFamily();
       script = font.getScript();
       size   = font.getSize();
       style  = font.getStyle();
     }
   }
   
   /**
    * Returns true if the DFont of the object passed in parameters equals the DFont.
    * Two fonts are equal if family, size, scrpt and style are equals.
    */
   public boolean equals(Object o) 
   {
     if( o==this ) return true;
   	 if(!(o instanceof IDFont)) return false;

     IDFont ft = (IDFont)o;
     return (ft.getStyle()  == style )
         && (ft.getFamily() == family)
 	 	 && (ft.getScript() == script)
	     && (ft.getSize()   == size  );
   }
   
   /**
    * Checks if the font has the same style.
    * @return true if font is exactly the given style.
    */
   public boolean isStyle( int _style_mask ) 
   {
    return (style & _style_mask)==_style_mask;
   }
   
   /**
    * Checks if the font has one or more property style with a mask.
    * @return true if font have one (or more) style from given mask.
    */
   public boolean haveStyle( int _style_mask ) 
   {
    return ((style & _style_mask)!=0);    
   }
   
   /**
    * Gets the family name of the DFont.
    */
   public String getFamily() 
   {
    return family;    
   }
   
   /**
    * Gets the script name of the DFont.
    */
   public String getScript() 
   {
    return script;    
   }
   
   /**
    * Gets the size of the DFont.

    */
   public int getSize() 
   {
    return size;    
   }
   
   /**
	* Gets the style of the DFont.
	*/
   public int getStyle() 
   {
	return style;    
   }
   
   /**
    * Adds the given style to font's style.
    * @param _style_mask the styles (for example DFont.BOLD|DFont.UNDERLINE)
    */
   public void addStyle( int _style_mask ) 
   {
   	  style |= _style_mask;
   }
   
   /**
    * Removes the given style from font's style.
    * @param _style_mask the styles (for example DFont.BOLD|DFont.UNDERLINE)
    */
   public void removeStyle( int _style_mask ) 
   {
      style &= ~_style_mask;
   }

   /**
    * Sets the font's style.
    * @param _style_mask the styles (for example DFont.BOLD|DFont.UNDERLINE)
    */
   public void setStyle( int _style_mask ) 
   {
      style = _style_mask;
   }
   
   /**
    * Sets the Family name.
    */
   public void setFamily(String f) 
   {
	  family = f;    
   }
   
   /**
    * Sets the script name of the DFont.
    */
   public void setScript(String script) 
   {
      this.script = script;    
   }
   
   /**
    * Sets the size of the DFont. 
    */
   public void setSize(int size) 
   {
	 this.size = size;    
   }
   
   /**
    * Sets the style of the font with a string value. The string has the following 
    * syntax:<br>
    * BOLD|ITALIC|UNDERLINE
    */
   public void setStyle(String s) 
   {
   	  int style = NORMAL;
   	  
   	  String[] list = s.split("\\|");
      
	  for (int i = 0; i < list.length; i++ )
	  {
	     String st = list[i];
	     if ( st.compareTo("BOLD"     ) == 0 ) style |= BOLD; //$NON-NLS-1$
	     if ( st.compareTo("DASHED"   ) == 0 ) style |= DASHED; //$NON-NLS-1$
	     if ( st.compareTo("ITALIC"   ) == 0 ) style |= ITALIC; //$NON-NLS-1$
	     if ( st.compareTo("UNDERLINE") == 0 ) style |= UNDERLINE; //$NON-NLS-1$
         if ( st.compareTo("OVERLINE" ) == 0 ) style |= OVERLINE; //$NON-NLS-1$
	     if ( st.compareTo("STRIKE"   ) == 0 ) style |= STRIKE;//$NON-NLS-1$
	  }
      setStyle(style);    
   }
   
   /**
    * Returns the style value of the DFont in an equivalent string value.
    * syntax:<br>
    * BOLD|ITALIC|UNDERLINE
    * 
    * @see #setStyle(String)
    */
   public String getStringStyle() 
   {
     return GetStringStyle( style );
   }

   /**
    * Returns the style value in an equivalent string value.
    * syntax:<br>
    * BOLD|ITALIC|UNDERLINE
    * 
    * @see #getStyle
    */
   public static String GetStringStyle( int style )
   {
   	  int val = style;
   	  
   	  String s=""; //$NON-NLS-1$
   	  if(HaveStyle(style, BOLD))
   	  {
   	  	s += "BOLD"; //$NON-NLS-1$
		val ^= BOLD;
		if (val!=0) s += "|"; //$NON-NLS-1$
   	  }
      if(HaveStyle(style, ITALIC))
   	  {
   	  	s += "ITALIC"; //$NON-NLS-1$
   	  	val ^= ITALIC;
		if (val!=0) s += "|"; //$NON-NLS-1$
   	  }
      if(HaveStyle(style, UNDERLINE))
	  {
		s += "UNDERLINE"; //$NON-NLS-1$
		val ^= UNDERLINE;
		if (val!=0) s += "|"; //$NON-NLS-1$
	  }
      if(HaveStyle(style, OVERLINE))
      {
        s += "OVERLINE"; //$NON-NLS-1$
        val ^= OVERLINE;
        if (val!=0) s += "|"; //$NON-NLS-1$
      }
      if(HaveStyle(style, DASHED))
	  {
	    s += "DASHED"; //$NON-NLS-1$
	    val ^= DASHED;
		if (val!=0) s += "|"; //$NON-NLS-1$
	  }
      if(HaveStyle(style, STRIKE))
	  {
	    s += "STRIKE"; //$NON-NLS-1$
	    val ^= STRIKE;
		if (val!=0) s += "|"; //$NON-NLS-1$
	  }
	  
	  if (s=="") //$NON-NLS-1$
	     s += "NORMAL"; //$NON-NLS-1$
	     
      return s;    
   }
   
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.core.internal.IDFont#getID()
   */
  public String getID() 
  {
    return Integer.toString( this.hashCode(), 16 );
  }
   
}
