/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DStyle.java,v 1.3 2008/05/23 14:11:47 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.core.internal;


/**
 * DStyle describes the way how a paragraph is presented in a document: it is composed 
 * of a name, a font, a foreground color, a background color and a format like 
 * BULLETS, NUMBERS, ROMAN LETTERS, LETTERS, ARROWS, FINGERS or NOTHING. 
 * 
 * @see DStyleRegistry
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DStyle implements IDStyle 
{
   private String name = null;
   
   /**
    * Style property: NUMBER displays  incremented number in front of each title and list item.<br>
    * 1 Introduction<br>
    * 2 Chapter One<br>
    * 3 Chapter Two<br>
    *    3.1 Paragraph One<br>
    *    3.2 Paragraph Two<br>
    * 4 Chapter Three<br>
    * ....
    */
   public static final int NUMBER = 1;
   
   /**
    * Style property: ROMAN_NUMBER displays an incremented roman number in front of 
    * each title and list item.<br>
    * I  Introduction<br>
    * II Chapter One<br>
    * III Chapter Two<br>
    *    III-I Paragraph One<br>
    *    III-II Paragraph Two<br>
    * IV Chapter Three<br>
    * ....
    */
   public static final int ROMAN_NUMBER = 2;
   public static final String T_ROMAN_NUMBER = "T_LETTER.DStyle.core.jscrib"; //$NON-NLS-1$
   
   /**
    * Style property: LETTER displays  incremented letter in front of each title and list item.<br>
    * A Introduction<br>
    * B Chapter One<br>
    * C Chapter Two<br>
    *    C.a Paragraph One<br>
    *    C.b Paragraph Two<br>
    * D Chapter Three<br>
    * ....
    */
   public static final int LETTER = 3;
   public static final String T_LETTER = "T_LETTER.DStyle.core.jscrib"; //$NON-NLS-1$
   
   /**
    * Style property: NOTHING displays nothing in front of a paragraph title and list item.<br>
    * Introduction<br>
    * Chapter One<br>
    * Chapter Two<br>
    *    Paragraph One<br>
    *    Paragraph Two<br>
    * Chapter Three<br>
    * ....<br>
    */
   public static final int NOTHING = 4;
   public static final String T_NOTHING = "T_NOTHING.DStyle.core.jscrib"; //$NON-NLS-1$
   
   /**
    * Style property: GLYPH displays an arrow in front of each title and list item.<br>
    * - Introduction<br>
    * - Chapter One<br>
    * - Chapter Two<br>
    *    - Paragraph One<br>
    *    - Paragraph Two<br>
    * - Chapter Three<br>
    * ....
    */
   public static final int GLYPH = 5;
   public static final String T_GLYPH = "T_GLYPH.DStyle.core.jscrib"; //$NON-NLS-1$
   
   private int numberingFormat = 0;
   private IDFont font ;
   private IDColor foreColor;
   private IDColor backColor;
   private IDWallpaper wallpaper;
   
   /**
    * Creates a DStyle named 'n' and gets the information background color, 
    * foreground color and title format from the style 's'.<br>
    * Take care: this is not a deep copy, new style and 's' will share font, fore ground
    * and background objects.
    * 
    * @param n name of the style
    * @param s style withe specified foreground color, background color and title 
    * format.
    */
   public DStyle(String n, IDStyle s) 
   {
      name = n==null ? "unamed" : n;
      if( s!=null )
      {
        font      = s.getFont();
        foreColor = s.getForeColor();
        backColor = s.getBackColor();
        numberingFormat = s.getNumberingFormat();
      }
   }
   
   /**
    * Creates a DStyle named 'n' with font 'f', foreground color 'fore' and 
    * background color 'back'.
    * 
    * @param n name of the style 
    * @param f font of the style
    * @param fore foregroundcolor of the style
    * @param back backgroundcolor of the style
    */
   public DStyle(String n, IDFont f, IDColor fore, IDColor back) 
   {
      name = n;
      font = f;
      foreColor = fore;
      backColor = back;    
   }
   
   /**
    * Creates a DStyle. By default this style is named 'unamed' with a null font,
    * a null background and foreground color, a null wallpaper and a numberingFormat
    * equal to 0
    */
   public DStyle() 
   {
     name="unamed";
   }
   
   /**
    * Creates a DStyle,Setting only a name with a null font,
    * a null background and foreground color, a null wallpaper and a numberingFormat
    * equal to 0
    */
   public DStyle( String n ) 
   {
     name=n;
   }
   
   /**
    * Access method for the name property.
    * 
    * @return   the current value of the name property
    */
   public String getName() 
   {
    return name;    
   }
   
   /**
    * Sets the value of the name property. If aName is null the style is named
    * automatically 'unamed'
    * 
    * @param aName the new value of the name property
    */
   public void setName(String aName) 
   {
      name = aName==null ? "unamed" : aName ;    
   }
   
   /**
    * Access method for the numberingFormat property.
    * 
    * @return   the current value of the numberingFormat property
    */
   public int getNumberingFormat() 
   {
    return numberingFormat;    
   }
   
   /**
    * Sets the value of the numberingFormat property.
    * 
    * @param aNumberingFormat the new value of the numberingFormat property
    */
   public void setNumberingFormat(int aNumberingFormat) 
   {
      numberingFormat = aNumberingFormat;    
   }
   
   /**
    * Sets the font of the style.
    */
   public void setFont(IDFont f) 
   {
      font = f;    
   }
   
   /**
    * Gets the font of the style.
    */
   public IDFont getFont() 
   {
    return font;    
   }
   
   /**
    * Sets the foreground color of the style.
    */
   public void setForeColor(IDColor c) 
   {
      foreColor = c;    
   }
   
   /**
    * Sets the background color of the style.
    */
   public void setBackColor(IDColor c) 
   {
      backColor = c;    
   }
   
   /**
    * Gets the background color of the style.
    */
   public IDColor getBackColor() 
   {
    return backColor;    
   }
   
   /**
    * Retruns true if the integer 'f' passed in parameter is a valid 
    * format. 
    */
   public boolean isValidNumberingFormat(int f) 
   {
   	  return ((f == GLYPH)
   	      || (f == LETTER)
   	      || (f == NOTHING)
   	      || (f == NUMBER)
   	      || (f == ROMAN_NUMBER));    
   }
   
   /**
    * Gets the foreground color of the style.
    */
   public IDColor getForeColor() 
   {
    return foreColor;    
   }
   
   /**
    * Compares two styles.
    * Returns true if the font, the foreground color, the background color and the 
    * title format are the same.
    * 
    * @param o object to be compared
    */
   public boolean equals(Object o) 
   {
     if( o == this ) return true;
     if( !(o instanceof IDStyle)) return false;
     IDStyle s = (IDStyle)o;
     IDFont  sfn = s.getFont();
     IDColor sfr = s.getForeColor();
     IDColor sbk = s.getBackColor();

   	 return ( font==sfn || (font!=null&& font.equals(sfn) ) )
         && ( foreColor==sfr || (foreColor!=null && foreColor.equals(sfr) ) )
         && ( backColor==sbk || (backColor!=null && backColor.equals(sbk) ) )
   	     && (s.getNumberingFormat() == getNumberingFormat()) ;
   }
   
   /**
    * Returns the string numbering format name for this integer value.
    * 
    * @param i valid numbering format integer value.
    * 
    * @see #stringToNumberingFormat
    * @see #isValidNumberingFormat
    */
   public String numberingFormatToString(int i) 
   {
   	  switch (i)
   	  {
   	  	case GLYPH:
   	  	     return "GLYPH";
   	  	case LETTER:
             return "LETTER";
   	  	case NUMBER:
   	  	     return "NUMBER";
   	  	case ROMAN_NUMBER:
   	  	     return "ROMAN_NUMBER";
   	  	default:
   	  	     return "NOTHING";
   	  }    
   }
   
   /**
    * Returns the integer numbering format value for this string name.
    * 
    * @param s valid numbering format name.
    * 
    * @see #stringToNumberingFormat
    * @see #isValidNumberingFormat
    */
   public int stringToNumberingFormat(String s) 
   {
     if( s==null ) return NOTHING;
	 if (s.equalsIgnoreCase("GLYPH")) return GLYPH; //$NON-NLS-1$
	 if (s.equalsIgnoreCase("LETTER"))	return LETTER; //$NON-NLS-1$
	 if (s.equalsIgnoreCase("NUMBER"))	return NUMBER; //$NON-NLS-1$
	 if (s.equalsIgnoreCase("ROMAN_NUMBER"))return ROMAN_NUMBER; //$NON-NLS-1$
	 return NOTHING;    
   }
   
   /**
    * Returns the wall paper of the style. The wall paper is used by
    * the UI driver to draw the background of DDocument, DCell and DGraphic.
    */
   public IDWallpaper getWallpaper()
   {
     return wallpaper;
   }
   
   /**
    * Changes (replaces) current wallpaper.
    * Please note: stored by a style, all item seems to have a wallpaper, by finally this decision
    * is made by driver which render document (this apply to other styles too).
    */
   public void setWallpaper( IDWallpaper wallpaper )
   {
     this.wallpaper = wallpaper;
   }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.core.internal.IDStyle#getID()
   */
  public String getID() 
  {
    return Integer.toString( this.hashCode(), 16 );
  }
}
