/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Generator.java,v 1.3 2005/06/10 01:42:58 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.report.core.provisional;

import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DI18N;



/**
 * Provides an entry point to build and render a chart for a given model. It is implemented as a singleton
 * and does not maintain any state information hence allowing multi-threaded requests for a single
 * generator instance.
 */
public final class Generator
{

    /**
     * The internal singleton Generator reference created lazily.
     */
    private static Generator g = null;

    /**
     * A private constructor.
     */
    private Generator()
    {

    }

    /**
     * Returns a singleton instance of the chart generator.
     * 
     * @return A singleton instance for the chart generator.
     */
    public static synchronized final Generator instance()
    {
        if (g == null)
        {
            g = new Generator();
        }
        return g;
    }

    /**
     * Builds and renders chart components using the provided renderer.
     *  
     * @param renderer A renderer used to build the chart.
     * @param model  The chart model (bound to a dataset).
     * 
     * 
     * @throws GenerationException
     */
    public final void run(IRenderer renderer, IChart model) throws RendererException
    {
		renderer.render(model);
    }
	
    /**
     * Builds and renders chart components using the provided renderer.  Runtime informatoin can also
     * be passed to the renderer.
     *  
     * @param renderer A renderer used to build the chart.
     * @param model  The chart model (bound to a dataset).
     * @param context Provides runtime information to the generator.
     * 
     * 
     * @throws GenerationException
     */
    public final void run(IRenderer renderer, IChart model, IRuntimeContext context) throws RendererException
    {
		run(renderer, model, context, null);
    }
    /**
     * Builds and renders chart components using the provided renderer.  The chart bounds 
     * (width, height) can be specified during generation time.
     *  
     * @param renderer A renderer used to build the chart.
     * @param model The chart model (bound to a dataset).
     * @param bounds Specifies the bounds of the chart.
     * 
     * @throws GenerationException
     */
    public final void run(IRenderer renderer, IChart model, IBounds bounds) throws RendererException
    {
		run(renderer, model, null, bounds);
    }
	
    /**
     * Builds and renders chart components using the provided renderer.  Runtime informatoin can also
     * be passed to the renderer.  The chart bounds (width, height) can be specified during 
     * generation time.
     *  
     * @param renderer A renderer used to build the chart.
     * @param model The chart model (bound to a dataset).
     * @param context Contains runtime configuration data.
     * @param bounds  Specifies the bounds of the chart.
     * 
     * @throws GenerationException
     */
    public final void run(IRenderer renderer, IChart model, IRuntimeContext context, IBounds bounds) throws RendererException
    {
    	DGraphic chart = null;
		if (model instanceof DDocument) {
			chart = (DGraphic)(((DDocument)model).getChildOfClass(DGraphic.class));
			
		}
		else if (model instanceof DGraphic){
			chart = (DGraphic)model;
		}
		else 
			throw new RendererException("The model object is not supported");
		if (chart != null){
			if (context != null){
				DI18N i18N = null;
				i18N = (DI18N)chart.getChildOfClass(DI18N.class);
				if (i18N == null){
					i18N = new DI18N();
					chart.addChild(i18N);
				}
				if (context.getLocale() != null){
					i18N.setCountry(context.getLocale().getCountry());
					i18N.setLanguage(context.getLocale().getLanguage());
				}
				if (context.getTimeZone() != null){
					i18N.setTimeZone(context.getTimeZone().getID());
				}
				if (context.getResourceBundle() != null){
					i18N.setResourceBundle(context.getResourceBundle());
				}
			}
			if (bounds != null){
			    chart.getProperties().store(DGraphic.P_MAX_WIDTH, bounds.getWidth());
			    chart.getProperties().store(DGraphic.P_MAX_HEIGHT, bounds.getHeight());
			}
		}
		renderer.render(model);
    }
	
	
}
