/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGRenderer.java,v 1.6 2005/06/16 20:06:02 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.report.core.provisional;

import java.io.OutputStream;

import org.eclipse.tptp.platform.report.chart.svg.internal.SVGGenerator;
import org.eclipse.tptp.platform.report.chart.svg.internal.SVGGeneratorException;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.cim.SVGChartImpl;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.w3c.dom.Document;

/**
 * SVGRenderer is a concrete class that takes in an object model that
 * represents a graphic object such as a chart and generates an SVG
 * representation of the graphic object to an output stream.  The output 
 * stream object is specified by using the setProperty method.
 *<p>
 * To render a chart in SVG, you must create an XML input file or XML DOM that 
 * represents the chart you want to render. The XML file should contain 
 * the raw data and configuration data needed by the chart service to 
 * create the chart. The following is a fragment of code that will 
 * allow you to generate a SVG chart. The line numbers are shown to 
 * further explain the code fragment. 
 * <pre>
 * 
 *    1 FileInputStream inputStream = new FileInputStream(new File("chart.xml")); 
 *    2 IChart chart = SerializerImpl.instance().read(inputStream); 
 *    3 IRenderer renderer = new SVGRenderer(); 
 *    4 FileOutputStream outputStream = new FileOutputStream(new File("chart.svg"));
 *    5 renderer.setProperty(IRenderer.OUTPUTSTREAM_IDENTIFIER, outputStream); 
 *    6 Generator.instance().run(renderer, chart);
 *    
 * Line 1 - In this line an input stream from an XML file is created. 
 *          The XML file is formatted based on the input schema and contains 
 *          the raw data and configuration data for the chart.
 * 
 * Line 2 - The SerializerImpl class provides a read method that will allow 
 *          one to take the XML stream and construct an internal representation 
 *          of the chart object model. It should be noted that the serializer 
 *          interface can also consume a XML document object model instead of 
 *          an input stream. Therefore, a developer can construct the XML 
 *          document object model in memory and pass in this DOM to the serializer. 
 * 
 * Line 3 - Construct the SVG device renderer.
 * 
 * Line 4 - An output stream is created that will generate an svg file 
 *          called chart.svg.
 * 
 * Line 5 - We use the setProperty method on the renderer interface to 
 *          pass in configuration information to the renderer. Here we set the 
 *          outputstream that the SVG renderer will use to write the generated 
 *          SVG chart.
 * 
 * Line 6 - This line generates and writes the chart to the output stream, 
 *          created on line 4, based on the chart model which was parsed in Line 2. 
 * </pre>
 * 
 */
public class SVGRenderer implements IRenderer {

	protected SVGGenerator generator;

	protected OutputStream outputStream;
	/**
	 * Constructor. 
	 */
	public SVGRenderer() {
		generator = new SVGGenerator();
	}

	/**
	 * This method provides the ability to set renderer specific parameters.  SVGRender
	 * supports IRenderer.OUTPUTSTREAM_IDENTIFIER parameter to specify the output stream.
	 * 
	 * @param key
	 *            The parameter name.
	 * @param value
	 *            The value of the parameter.
	 * @throws RendererException
	 *             Throws an exception if the parameter is not supported by
	 *             the renderer.
	 */		
	public void setProperty(String key, Object value) throws RendererException{
        if (key.equals(IRenderer.OUTPUTSTREAM_IDENTIFIER))
        {
            outputStream = (OutputStream)value;
        }
        else
			throw new RendererException("Property key: " +key +" not supported by this renderer.");
	}

	public void render(org.eclipse.tptp.platform.report.core.provisional.IChart chart) throws RendererException{
		if (chart instanceof DDocument) {
			chart = (IChart)((DDocument)chart).getChildOfClass(DGraphic.class);

		}
		if (chart instanceof DGraphic) {
			Chart input = new SVGChartImpl((DGraphic) chart);
			try {
				Document svgoutput = generator.generateGraphicDocument(input);
				// write document to file
				generator.writeDocumentToOutputStream(svgoutput, outputStream);
			} catch (SVGGeneratorException e) {
				e.printStackTrace();
				throw new RendererException(e.getMessage(), e);
			}
		}
	}


}
