/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SWTStaticImageRenderer.java,v 1.7 2008/05/23 14:12:11 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.report.core.provisional;

import java.io.OutputStream;

import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTError;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.render.internal.RenderableSnapShot;
import org.eclipse.tptp.platform.report.tools.internal.DChartTranslator;
import org.eclipse.tptp.platform.report.tools.internal.SWTIImageProvider;

/**
 * SWTStaticImageRenderer is a concrete class that takes in an object model that
 * represents a chart and generates an image representation of the chart to
 * an output stream.   This class will create a static image based on the SWT
 * chart look and feel.  This class only supports transcoding the object model to 
 * JPEG format.  The output stream object is specified by using the setProperty method.
 * <p>
 * To create a chart and render the chart in JPEG format, you must create the XML input 
 * file or XML DOM that represents the chart you want to render. The XML file should contain the 
 * raw data and configuration data needed by the chart service to create the chart. 
 * The following is a fragment of code that will allow you to generate a jpeg image of the chart.
 *<pre>
 *   1 FileInputStream inputStream = new FileInputStream(new File("chart.xml")); 
 *   2 IChart chart = SerializerImpl.instance().read(inputStream); 
 *   3 IRenderer renderer = new SWTStaticImageRenderer();
 *   4 FileOutputStream outputStream = new FileOutputStream(new File("chart.jpg"));
 *   5 renderer.setProperty(IRenderer.OUTPUTSTREAM_IDENTIFIER, outputStream);
 *   6 Generator.instance().run(renderer, chart);
 *   
 * Line 1 - In this line an input stream from an XML file is created. The XML file is formatted
 *          based on the input schema and contains the raw data and configuration data for the chart.
 * 
 * Line 2 - The SerializerImpl class provides a read method that will allow one to take the XML
 *          stream and construct an internal representation of the chart object model. It should
 *          be noted that the serializer interface can also consume a XML document object model 
 *          instead of an input stream. Therefore, a developer can construct the XML document object
 *          model in memory and pass in this DOM to the serializer. 
 * 
 * Line 3 - Construct a static image renderer. This renderer provides the ability to create static 
 *          images such as jpeg, png and pdf chart images.
 * 
 * Line 4 - An output stream is created that will generate a jpeg file called chart.jpg.
 * 
 * Line 5 - We use the setProperty method on the renderer interface to pass in configuration information 
 *          to the renderer. Here we set the outputstream that the static image renderer will use to
 *          write the generated jpeg chart.
 * 
 * Line 6 - This line generates and writes the chart to the output stream, created on line 4, based on 
 *          the chart model which was parsed in Line 2. 
 * </pre>   
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class SWTStaticImageRenderer extends StaticImageRenderer {
	/**
	 * Constructor. 
	 */	
	public SWTStaticImageRenderer() {
		this.transcode_type = StaticImageRenderer.JPEG;
	}
	
	/**
	 * This method provides the ability to set renderer specific parameters.  SWTSTaticImageRenderer
	 * supports IRenderer.OUTPUTSTREAM_IDENTIFIER parameter to specify the output stream.
	 * 
	 * @param key
	 *            The parameter name.
	 * @param value
	 *            The value of the parameter.
	 * @throws RendererException
	 *             Throws an exception if the parameter is not supported by
	 *             the renderer.
	 */		
	public void setProperty(String key, Object value) throws RendererException {
        if (key.equals(IRenderer.OUTPUTSTREAM_IDENTIFIER))
        {
            outputStream = (OutputStream)value;
        }
        else
			throw new RendererException("Property key: " +key +" not supported by this renderer.");
	}
	

	public void render(org.eclipse.tptp.platform.report.core.provisional.IChart chart) throws RendererException
	{
		if (chart instanceof DDocument) {
			chart = (IChart)((DDocument)chart).getChildOfClass(DGraphic.class);

		}		
		DGraphic graphic = null;
		if (chart instanceof DGraphic) {
	        DChartTranslator tc = new DChartTranslator();
	        graphic = tc.translate((DGraphic)chart);
		}
		Device dev = Display.getDefault();
		int w = 500, h = w;
		w *= graphic.getWidthScale();
		h *= graphic.getHeightScale();
		Rect r = new Rect(0, 0, w, h);
		SWTIImageProvider iimage_provider = new SWTIImageProvider( dev );

		Image img = RenderableSnapShot.Process(dev, r, graphic, 1.0f, iimage_provider );
		ImageLoader loader = new ImageLoader();
		if (loader.data == null) {
			loader.data = new ImageData[1];
		}
		loader.data[0] = img.getImageData();
		try {
			loader.save(outputStream, SWT.IMAGE_JPEG);
		} catch (SWTException _e) {
			System.err.println("Error saving image to output stream:" +  _e);
		} catch (SWTError _r) {
			System.err.println("Error saving image to output stream:"+ _r);
		}
		img.dispose();	
	}
}
