/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SerializerImpl.java,v 1.6 2005/06/14 20:57:47 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.report.core.provisional;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlDocumentReader;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlDocumentWriter;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlError;
import org.eclipse.tptp.platform.report.drivers.xml.internal.XSDValidate;
import org.eclipse.tptp.platform.report.drivers.xml.internal.XSDXmlParserAdapter;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * Concrete class that implements the ISerializer interface.
 * 
 * @see org.eclipse.tptp.platform.report.core.provisional.ISerializer
 */
public class SerializerImpl implements ISerializer
{

    private static ISerializer sz = null;

    /**
     * Cannot invoke constructor; use instance() instead
     */
    private SerializerImpl()
    {

    }

    /**
     * 
     * @return A singleton instance of the chart serializer
     */
    public static synchronized final ISerializer instance()
    {
        if (sz == null)
        {
            sz = new SerializerImpl();
        }
        return sz;
    }

    /*
     * (non-Javadoc)
     * 
     * @see  org.eclipse.tptp.platform.report.core.provisional.ISerialization#write(org.eclipse.tptp.platform.report.core.provisional.IChart, java.io.OutputStream)
     */
    public void write(IChart cModel, OutputStream os) throws IOException
    {
		DXmlDocumentWriter dxcw = new DXmlDocumentWriter();
		try {
			dxcw.write(os, (IDObject)cModel);
		} catch (Exception e) {
			throw new IOException(e.getMessage());
		}
    }

     /*
     * (non-Javadoc)
     * 
     * @see  org.eclipse.tptp.platform.report.core.provisional.ISerialization#toDOM(java.io.InputStream)
     */
    public Document toDOM(InputStream is) throws IOException
    {
        
		DXmlDocumentReader dxcr = new DXmlDocumentReader( new XSDXmlParserAdapter(new XSDValidate()));
        try {
		  return dxcr.readDOM(is);
        }
        catch (DXmlError e)
        {
            throw new IOException(e.getMessage());
        }
		
    }    
     /*
     * (non-Javadoc)
     * 
     * @see  org.eclipse.tptp.platform.report.core.provisional.ISerialization#read(java.io.InputStream)
     */
    public IChart read(InputStream is) throws IOException
    {
        
		DXmlDocumentReader dxcr = new DXmlDocumentReader( new XSDXmlParserAdapter(new XSDValidate()));
        try {
		  IDObject[] objects =  dxcr.read(is);
          return (IChart)objects[0];
        }
        catch (DXmlError e)
        {
            throw new IOException(e.getMessage());
        }
		
    }
	
    /*
     * (non-Javadoc)
     * 
     * @see  org.eclipse.tptp.platform.report.core.provisional.ISerialization#read(org.w3c.dom.Document)
     */
	public IChart read(Document dom) throws IOException {
		
		DXmlDocumentReader dxcr = new DXmlDocumentReader( new XSDXmlParserAdapter(new XSDValidate()));
		//need to convert dom to string to validate dom
		String strDOM = serializeGeneratedDocumentToString(dom);
		try {
		  IDObject[] objects =  dxcr.read(new InputSource(new StringReader(strDOM)));
		  return (IChart)objects[0];
        }
        catch (DXmlError e)
        {
          throw new IOException(e.getMessage());
        }
	}

	private String serializeGeneratedDocumentToString(Document generatedDocument) {
		if (generatedDocument == null) {
			return null;
		}
		OutputStreamWriter writer = null;
		ByteArrayOutputStream stream = new ByteArrayOutputStream();
		try {
			writer = new OutputStreamWriter(stream, "UTF-8");
			DOMSource source = new DOMSource(generatedDocument);
			StreamResult result = new StreamResult(writer);

			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();

			transformer.transform(source, result);
		} catch (Exception e) {
			// should never be here because UTF-8 is supported
			Utilities.assertion(false);
		}

		return stream.toString();

	}    	
  
}