/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StaticImageRenderer.java,v 1.5 2005/06/16 18:34:46 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.report.core.provisional;

import java.io.OutputStream;

import org.eclipse.tptp.platform.report.chart.svg.internal.SVGGenerator;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.cim.SVGChartImpl;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGTranscode;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.w3c.dom.Document;

/**
 * StaticImageRenderer is a concrete class that takes in an object model that
 * represents a chart and generates an image representation of the chart.  The
 * generated image is written to an output stream.  The image formats supported
 * are: JPEG, PNG and PDF.  The output stream object is specified by using the 
 * setProperty method.
 * <p>
 * To create a chart and render the chart in JPEG format, you must create the XML input 
 * file or XML DOM that represents the chart you want to render. The XML file should contain the 
 * raw data and configuration data needed by the chart service to create the chart. 
 * The following is a fragment of code that will allow you to generate a jpeg image of the chart.
 *<pre>
 *   1 FileInputStream inputStream = new FileInputStream(new File("chart.xml")); 
 *   2 IChart chart = SerializerImpl.instance().read(inputStream); 
 *   3 IRenderer renderer = new StaticImageRenderer();
 *   4 FileOutputStream outputStream = new FileOutputStream(new File("chart.jpg"));
 *   5 renderer.setProperty(IRenderer.OUTPUTSTREAM_IDENTIFIER, outputStream);
 *   6 renderer.setProperty(StaticImageRenderer.STATIC_IMAGE_TYPE, StaticImageRenderer.JPEG);
 *   7 Generator.instance().run(renderer, chart);
 *   
 * Line 1 - In this line an input stream from an XML file is created. The XML file is formatted
 *          based on the input schema and contains the raw data and configuration data for the chart.
 * 
 * Line 2 - The SerializerImpl class provides a read method that will allow one to take the XML
 *          stream and construct an internal representation of the chart object model. It should
 *          be noted that the serializer interface can also consume a XML document object model 
 *          instead of an input stream. Therefore, a developer can construct the XML document object
 *          model in memory and pass in this DOM to the serializer. 
 * 
 * Line 3 - Construct a static image renderer. This renderer provides the ability to create static 
 *          images such as jpeg, png and pdf chart images.
 * 
 * Line 4 - An output stream is created that will generate a jpeg file called chart.jpg.
 * 
 * Line 5 - We use the setProperty method on the renderer interface to pass in configuration information 
 *          to the renderer. Here we set the outputstream that the static image renderer will use to
 *          write the generated jpeg chart.
 * 
 * Line 6 - We set the type of static image we want to generate. In this case we want to create a jpeg 
 *          file so we set the STATIC_IMAGE_TYPE parameter to StaticImageRenderer.JPEG.
 * 
 * Line 7 - This line generates and writes the chart to the output stream, created on line 4, based on 
 *          the chart model which was parsed in Line 2. 
 * </pre>   
 */
public class StaticImageRenderer implements IRenderer {

	protected SVGGenerator generator; 
	protected OutputStream outputStream;

	/**
	 * Static image type that tells the renderer to generate a JPEG image. 
	 */
	public static final String JPEG = "Transcode.JPEG";
	/**
	 * Static image type that tells the renderer to generate a PNG image. 
	 */
	public static final String PNG = "Transcode.PNG";
	/**
	 * Static image type that tells the renderer to generate a PDF image. 
	 */
	public static final String PDF = "Transcode.PDF";
	
	/**
	 * Property key that represents the static image type. 
	 */
	public static final String STATIC_IMAGE_TYPE="tptp.transcode.type";
	
	protected String transcode_type;
	
	/**
	 * Constructor. 
	 */
	public StaticImageRenderer() {
		super();
		generator = new SVGGenerator();
	}
	
	/**
	 * This method provides the ability to set renderer specific parameters.  StaticImageRenderer supports 
	 * IRenderer.OUTPUTSTREAM_IDENTIFIER parameter to specify the output stream.  In addition, StaticImageRenderer
	 * supports the StaticImageRenderer.STATIC_IMAGE_TYPE parameter to specify the static
	 * image type.
	 * 
	 * @param key
	 *            The parameter name.
	 * @param value
	 *            The value of the parameter.
	 * @throws RendererException
	 *             Throws an exception if the parameter is not supported by
	 *             the renderer.
	 */		
	public void setProperty(String key, Object value) throws RendererException {
        if (key.equals(IRenderer.OUTPUTSTREAM_IDENTIFIER))
        {
            outputStream = (OutputStream)value;
        }
        else if (key.equals(STATIC_IMAGE_TYPE)){
			transcode_type = (String)value;			
        }
        else
			throw new RendererException("Property key: " +key +" not supported by this renderer.");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.core.provisional.IRenderer#render(org.eclipse.tptp.platform.report.core.provisional.IChart)
	 */
	public void render(org.eclipse.tptp.platform.report.core.provisional.IChart chart) throws RendererException{
		if (chart instanceof DDocument) {
			chart = (IChart)((DDocument)chart).getChildOfClass(DGraphic.class);

		}
		// Convert chart object to DGraphic object
		if (chart instanceof DGraphic) {
			Chart input = new SVGChartImpl((DGraphic) chart);
		try {
			Document svgoutput = generator.generateGraphicDocument(input);
			String SVGGraphic = generator.serializeGeneratedDocumentToString(svgoutput);
			if (PDF.equals(transcode_type)){
				 SVGTranscode.transcode2PDF(SVGGraphic, outputStream);
			}
			else if (PNG.equals(transcode_type)){
				 SVGTranscode.transcode2PNG(SVGGraphic, outputStream);
			}
			else if (JPEG.equals(transcode_type)){
				 SVGTranscode.transcode2JPEG(SVGGraphic, outputStream);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		}
	}

}
