/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DrawUtilIGC.java,v 1.2 2008/05/23 14:11:56 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.drawutil.internal;


import org.eclipse.tptp.platform.report.core.internal.DWallpaper;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IImage;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.Size;



/**
 * Utility for drawing using IGC.
 * Can :
 * <LI> draw text inside a rectangle using alignment and or vertical layout
 * <LI> draw vertical text.
 * <LI> return bounding box for text using aligment and/or vertical layout
 * <LI> truncate text to fit in a rectangle
 * <LI> convert IDColor to SWT Color, IDFont to SWT Font.
 * <LI> return lighter/darker color from an existing one.
 * <LI> detect if a point is near line or segment, inside an oval or a polygon
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DrawUtilIGC
{
    
  public static final int MASK=IDAlignment.VERTICAL|IDAlignment.ROTCCW90|IDAlignment.ROTCW90;

  /** @return true is _s==null or _s.length()==0 */
  public static boolean isEmpty( String _s )
  {
    if( _s==null ) return true;
    if( _s.length()==0 ) return true;
    return false;
  }
  
  /**
   * Truncate text fit given available size, take care of VERTICAL/RTOCCW90/ROTCW90 alignment).
   * Font must be set in gc_ before calling this.
   */
  public static String truncateText( IGC _gc, String _text, int _available_width, int _available_height, int _alignment )
  {
    if( HaveAlignment( _alignment, IDAlignment.VERTICAL ))
    {
      return truncateTextV( _gc, _text, _available_height );
    }
    else if( HaveAlignment( _alignment, IDAlignment.ROTCCW90) || HaveAlignment( _alignment, IDAlignment.ROTCW90 ) )
    {
      return truncateTextH( _gc, _text, _available_height );
    }
    else
    {
      return truncateTextH( _gc, _text, _available_width );
    }
  }
  
  /**
   * Truncate text to fit it inside available width, append "..."
   * Font must be setup in gc_.
   * @return truncated text. or null if nothing can be fitted
   */
  public static String truncateTextH( IGC _gc, String _text, int _available_width )
  {
     String txt = _text;
     if( _text == null ) return null;
     if( _available_width <= 0 ) return null;

     ISize size = _gc.textExtent( _text );
     if( size.getW() > _available_width )
     {
       for( ;size.getW()>_available_width; )
       {
         if ( txt.length() == 0 ) return null;
         txt = txt.substring( 0, txt.length()-1 );         
         size = _gc.textExtent( txt+"..." );
       }
       txt = txt+"...";
     }

     return txt;
    }
    
  /** trunctate VERTICAL aligned text to fit in available height */
  public static String truncateTextV( IGC _gc, String _text, int _available_height )
  {
    if( DrawUtilIGC.isEmpty(_text )) return null;
    if( _available_height <= 0 ) return null;
    String txt = _text;
    ISize size = verticalTextExtent( _gc, _text );
    if( size.getH() > _available_height )
    {
      for( ; size.getH()>_available_height; )
      {
        if( txt.length()==0 ) return null;
        txt = txt.substring( 0, txt.length()-1);
        size = verticalTextExtent( _gc, txt ); //"..." have any sens here.          
      }
    }
    return txt;
  }

  /** @return extends of the text _s displayed vertically (for VERTICAL alignment) */    
  public static ISize verticalTextExtent( IGC _gc, String _s )
  {
	if( DrawUtilIGC.isEmpty(_s)) return  new Size(0,0);
	int width=0, height=0;
	for( int i=0; i<_s.length(); ++i )
	{
	  ISize size = _gc.textExtent(_s.substring(i,i+1));
      int w = size.getW();     
	  if( w > width ) width = w;
	  height += size.getH();
	}
	return new Size( width, height );
  }
    
  /** 
   * @return text extends for string _s taking care of VERTICAL style (or horizontal)
   */
  public static ISize textExtent( IGC _gc, String _s, int _style )
  {
    if( _s==null) return new Size(0,0);
    if( HaveAlignment(_style,IDAlignment.VERTICAL) )
    { 
      return verticalTextExtent( _gc, _s );
    } 
    else if( HaveAlignment(_style,IDAlignment.ROTCW90))
    {
      return rotCW90TextExtent( _gc, _s );
    }
    else if (HaveAlignment(_style,IDAlignment.ROTCCW90))
    {
      return rotCCW90TextExtent( _gc, _s );
    } else {
      return _gc.textExtent( _s );
    }
  }
  
  /** @return text extend for a text displayed using ROTCW90 style. */
  public static ISize rotCW90TextExtent( IGC _gc, String _s )
  {
    ISize ext = _gc.textExtent( _s, Radian._3PI2 );
    return ext;
  }
  /** @return text extend for a text displayed using ROTCCW90 style. */
  public static ISize rotCCW90TextExtent( IGC _gc, String _s )
  {
    return _gc.textExtent( _s, Radian._PI2 );
  }
    
  /** 
   * Draw text with an alignment in the rectangle.
   * _align is IDAlignment/RIGHT/TOP/BOTTOM/CENTER / VERTICAL
   */
  public static final void drawText( IGC _gc, IRect _r, int _align, String _txt )
  {
    drawText( _gc, _r.getX(), _r.getY(), _r.getW(), _r.getH(), _align, _txt );
  }

  public static boolean HaveAlignment( int _alignment, int _mask)
  {
    return (_alignment&_mask)==_mask;
  }
  
  /** 
   * Draw text with an alignment in the rectangle.
   * _align a combined value from IDAligment.
   */
  public static void drawText( IGC _gc, int _x, int _y, int _w, int _h, int _align, String _txt )
  {
    if( _txt==null ) return ;

    ISize size = textExtent( _gc, _txt, _align );
    if( size==null )
    {
      //IGC doesn't return text size (perhaps it can't know how to do)
      _gc.drawText( _txt, _x,_y );
      return;
    }
    
    int x=_x, y=_y;
    
    if( HaveAlignment(_align, IDAlignment.LEFT) )
    {
      x=_x;
    }
    else if ( HaveAlignment(_align,IDAlignment.RIGHT) )
    {
      x=_x+_w-size.getW();
    }
    else //center
    {
      x=_x+(_w-size.getW())/2;
    }  
    if( HaveAlignment(_align,IDAlignment.TOP) )
    {
      y=_y;
    }
    else if ( HaveAlignment(_align,IDAlignment.BOTTOM) )
    {
      y=_y+_h-size.getH();
    }
    else //center
    {
      y=_y+(_h-size.getH())/2;
    }
    
    if ( HaveAlignment(_align, IDAlignment.VERTICAL ) )
    {
      int H=size.getW(); //it's size of vertical text
      int xctr = _x+_w/2;
      for( int i=0; i<_txt.length(); ++i )
      {
        String c = _txt.substring(i,i+1);
        ISize sz = _gc.textExtent( c ) ;
        int wc = sz.getW();
        int cx = x; 
             if (HaveAlignment(_align,IDAlignment.LEFT)) { cx=x; } 
        else if (HaveAlignment(_align,IDAlignment.RIGHT) ) { cx=x+size.getW()-wc; } 
        else /*HCENTER*/ { cx=xctr-(wc>>1); }    
        _gc.drawText( c, cx,y );      
        y += H;
      }
    }
    else if ( HaveAlignment(_align, IDAlignment.ROTCCW90 ) )
    {
      _gc.drawText( _txt, x, y+size.getH(), Radian._PI2 );
    }
    else if( HaveAlignment(_align, IDAlignment.ROTCW90  ) )
    {
      _gc.drawText( _txt, x+size.getW(), y, Radian._3PI2 );
    }
    else //normal text
    {
      _gc.drawText( _txt, x,y );
    }
  }

  /** @return the bounding rectangle of the text. might be bigger than
   *  _r.
   * _align is SWT.LEFT/RIGHT/TOP/BOTTOM/CENTER/ VERTICAL/ROTCCW90/ROTCW90 */
  public static final Rect boundingRect( IGC _gc, IRect _r, int _align, String _txt )
  {
    return boundingRect( _gc, _r.getX(),_r.getY(),_r.getW(),_r.getH(), _align, _txt );
  }

  /** @return the bounding rectangle of the text. might be bigger than
   *  _r.
   * _align is SWT.LEFT/RIGHT/TOP/BOTTOM/CENTER/ VERTICAL/ROTCCW90/ROTCW90 */
  public static Rect boundingRect( IGC _gc, int _x,int _y, int _w, int _h, int _align, String _txt )
  {
    ISize size = textExtent( _gc, _txt, _align );
    int x=_x, y=_y;
    
    if( HaveAlignment(_align,IDAlignment.LEFT) )
    {
      x=_x;
    }
    else if ( HaveAlignment(_align,IDAlignment.RIGHT) )
    {
      x=_x+_w-size.getW();
    }
    else //center
    {
      x=_x+(_w-size.getW())/2;
    }  
    if( HaveAlignment(_align,IDAlignment.TOP) )
    {
      y=_y;
    }
    else if ( HaveAlignment(_align,IDAlignment.BOTTOM) )
    {
      y=_y+_h-size.getH();
    }
    else //center
    {
      y=_y+(_h-size.getH())/2;
    }
    return new Rect( x,y,size.getW(),size.getH());
  }       

  /** distance in pixel for 'Contains' and 'near' approximation used in various methods. */
  private static final int DistMax = 4 ;
  
  /** @return true if segment (ax,ay,bx,by) contains point (px,py), or if this segment
   * is near the segment.
   */
  public static boolean segmentContains( int _ax, int _ay, int _bx, int _by, int _px,int _py )
  {
    //optim: test if _ax==_bx then _ay==_by
    if( _ax==_bx )
    {    
      return near_segment_uv( _ay, _by, _ax, _py,_px, DistMax );
    }
    if( _ay==_by )
    {
      //vertical line
      return near_segment_uv( _ax, _bx, _ay, _px,_py, DistMax );
    }

     //neither vertical nor horizontal, but we know that two points aren't the same.
     //optim: 
     int xmin,xmax,ymin,ymax;
     if( _ax<_bx)
     {
       xmin=_ax; xmax=_bx;
     } else {
       xmin=_bx; xmax=_ax;
     }
     if( _ay<_by)
     {
       ymin=_ay; ymax=_by;
     } else {
       ymin=_by; ymax=_ay;
     }
     //too far ?
     if( _px < (xmin-DistMax) ) return false;
     if( _px > (xmax+DistMax) ) return false;
     if( _py < (ymin-DistMax) ) return false;
     if( _py > (ymax+DistMax) ) return false;

     //norm AB
     int _bax = _bx-_ax,
         _bay = _by-_ay;
     float n_ab = (float)Math.sqrt( _bax*_bax + _bay*_bay );
     float _1_n_ab = 1.0f / n_ab;
     //vect unitaire
     float u_ab_x = _bax * _1_n_ab,
           u_ab_y = _bay * _1_n_ab ;
     //vect perpendiculaire
     float v_ab_x = -u_ab_y,
           v_ab_y = +u_ab_x ;
     float kx=_bax, lx=v_ab_x, cx=_ax-_px, 
           ky=_bay, ly=v_ab_y, cy=_ay-_py;
     //a*k+b=0
     float a= ly*kx+ky*lx, b=cx*ly + cy*lx;
     float k= -b/a;
     float x=k*_bax+_ax,
           y=k*_bay+_ay;
     //verif k [0..1]
     if(k<0.0f) return false;
     if(k>1.0f) return false;
     float v1 = _px-x, v2 = _py-y ;
     float dist = (float)Math.sqrt( v1*v1 + v2*v2 );
     return ( dist <= DistMax && (k>=0.0f)&&(k<=1.0f));
  }
  
  //for vertical or horizontal segments, so sometimes (u,v) is (x,y) otherwise (u,v) is (y,x)
  protected static boolean near_segment_uv( int _su1, int _su2, int _sv, int _pu, int _pv, int _DistMax )
  {
    int umin, umax;
    if ( _su1 < _su2 )
    {
      umin = _su1; umax=_su2;
    } else {
      umin = _su2; umax=_su1;
    }
    //u below: compute distance
    if( _pu < umin )
    {
      if( _pu < (umin-_DistMax)) return false;
      if( _pv < (_sv -_DistMax)) return false;
      if( _pv > (_sv +_DistMax)) return false;
      int v1 = _pu-umin, v2 = _pv-_sv;
      float dist = (float)Math.sqrt( v1*v1 + v2*v2 );
      return ( dist <= _DistMax );
    }
    //u above: compute distance
    else if ( _pu > umax )
    {
      if( _pu > (umax+_DistMax)) return false;
      if( _pv < (_sv -_DistMax)) return false;
      if( _pv > (_sv +_DistMax)) return false;
      int v1 = _pu-umax, v2 = _pv-_sv;
      float dist = (float)Math.sqrt( v1*v1 + v2*v2 );
      return ( dist <= _DistMax );
    }
    //u exactly "on" segment area
    else
    {
      int dist = (_pv-_sv);
      if( dist < 0 ) dist=-dist; //abs()
      return ( dist <= _DistMax );
    }
  }
  
  /**
   * @return true if given polyong definition contains point (px,py).
   * @param poly array of (x,y,x,y,...) points (as GC.fillPoly() require. 
   *        must have at least 3 points (array of 6 int length).
   * @param px x coordinate of point to check.
   * @param py y corrdinate of point to check.
   */
  public static boolean polygonContains( int poly[], int px, int py )
  {
    if( poly==null ) return false;
    if( poly.length < 6 ) return false; //must have at least 3 points
    int length = poly.length;
    if( (length&0x01)!=0) length--; //have extra int (array end is [...x,y,x,y,x], skip last x)
    int last_ip = length-2;
    int lx = poly[last_ip];
    int ly = poly[last_ip+1];
    int segment_count=0;
    for( int i=0; i<length;  )
    {
      int x = poly[i++];
      int y = poly[i++];
      //vertical line py cut the segment [lx,ly]-[x,y] above py ?
      int xn,xx,yxn,yxx;
      if( lx < x ) //sort X
      {
        xn = lx; yxn = ly; 
        xx =  x; yxx =  y; 
      } else { 
        xn =  x; yxn =  y; 
        xx = lx; yxx = ly; 
      }
      if( px >= xn && px <= xx && ( py >= ((y<ly)?y:ly)))
      {
        if( xn == xx ) //vertical segment
        {
          if( py <= ((y<ly)?ly:y)) // point inside segment ?
          {
            return true;
          }
          return false;
        }
        float sy = yxn +(px-xn)*(yxx-yxn)/(xx-xn);
        if( sy == py ) //exactly on a point
        {
          return true;
        }
        //half line [py,+oo) cut segment ?  
        if( sy < py ) segment_count ++;
      }
      
      lx=x; 
      ly=y;
    }
    //point is inside polygon if number of segment cut is not pair.
    return ((segment_count & 0x01) != 0);
  }
  
  /**
   * @return true if oval inside given rectangle contains point (px,py).
   * @param _x,_y,_w,_h the rectangle where oval is defined (as GC.drawOval() require).
   * @param px x coordinate of point to check.
   * @param py y corrdinate of point to check.
   */
  public static boolean ovalContains( int _x, int _y, int _w, int _h, int px, int py )
  {
    if( !Rect.Contains(_x,_y,_w,_h, px,py) ) return false;  
    int cx = _x+_w/2;
    int cy = _y+_h/2;
    int dx = px-cx, dy = py-cy;
    float rp = (float)(Math.sqrt( dx*dx+dy*dy ));
    int rx = Math.abs(_w/2);
    int ry = Math.abs(_h/2);
    if( rx==ry ) //circle ?
    {
      return rp <= rx;
    }
    if( dy==0 ) return ( Math.abs(dx) <= rx );
    if( dx==0 ) return ( Math.abs(dy) <= ry );
    float a = (float)Math.atan2( dy, dx );
    float xa = (float)(rx*Math.cos(a));
    float ya = (float)(ry*Math.sin(a));
    float ra = (float)( Math.sqrt( xa*xa + ya*ya ) );
    return (rp<=ra);
  }
  /**
   * @return true if oval inside given rectangle is (near) under point (px,py), only the border
   * of the oval is checked (using a catch distance). Use ovalContains to check if a point is contained
   * in a whole oval (circle).
   * @param _x,_y,_w,_h the rectangle where oval is defined (as GC.drawOval() require).
   * @param px x coordinate of point to check.
   * @param py y corrdinate of point to check.
   */
  public static boolean ovalUnder( int _x, int _y, int _w, int _h, int px, int py )
  {
    if( !Rect.Contains(_x,_y,_w,_h, px,py) ) return false;  
    int cx = _x+_w/2;
    int cy = _y+_h/2;
    int dx = px-cx, dy = py-cy;
    float rp = (float)(Math.sqrt( dx*dx+dy*dy ));
    int rx = Math.abs(_w/2);
    int ry = Math.abs(_h/2);
    if( rx==ry ) //circle ?
    {
      boolean r= (rx-DistMax/2 <= rp) && (rp <= rx+DistMax/2);
      return  r;
    }
    if( dy==0 ) return ( Math.abs(dx) <= rx );
    if( dx==0 ) return ( Math.abs(dy) <= ry );
    float a = (float)Math.atan2( dy, dx );
    float xa = (float)(rx*Math.cos(a));
    float ya = (float)(ry*Math.sin(a));
    float ra = (float)( Math.sqrt( xa*xa + ya*ya ) );
    return (ra-DistMax/2<=rp)&&(rp<=ra+DistMax/2);
  }
    
  /**
   * Draw specified imag using effect in rectangle.
   */  
  public static final void drawImage( IGC gc, IImage image, Rect rect, float _scale, int effect )
  {
    drawImage( gc, image, rect.x(), rect.y(), rect.w(), rect.h(), _scale, effect );
  }
  
  /**
   * Draw an imge using an effect in rectangle.
   * For effect you can use, please refer to DWallpaper.
   * @see DWallpaper
   */
//TODO: provide clipping area will be nice to improve scalability...
  public static void drawImage( IGC _gc, IImage _image, int _x, int _y, int _w, int _h, float _scale, int _effect )
  {
    if( _effect == 0 || _w <=0 || _h <= 0 ) return ;

    int imgw = _image.getWidth();
    int imgh = _image.getHeight();
    
    int devimgw = (int)(_image.getWidth() *_scale);
    int devimgh = (int)(_image.getHeight() * _scale);
    if( devimgw==0 || devimgh==0 ) return ;
    
    int ix, iy, iw=devimgw, ih=devimgh, icx=1, icy=1 ;
    
    //special:
    if( DWallpaper.HaveEffect( _effect, DWallpaper.BORDER))
    {
      ix=_x; 
      icx = _w/devimgw;
      ix = _x;     
      iw = _w/(icx==0?1:icx);
      iy=_y;
      icy = _h/devimgh;
      iy = _y;
      ih = _h/(icy==0?1:icy);
      if( icx==0 || icy ==0 ) return ;
      IShape save = _gc.getClipping();
      _gc.setClipping( new Rect(_x, _y, _w, _h) );
      try{
        int iix=ix;
        for( int ry=0; ry<icy; ++ry )
        {
          ix=iix;
          for( int rx=0; rx<icx; ++rx )
          {
            boolean border=(ry==0||rx==0|ry==icy-1||rx==icx-1);
            if( border )
            {
              _gc.drawImage( _image, ix,iy,iw,ih );
            }
            ix += iw;
          }
          iy += ih;
        }
      }
      finally
      {
        _gc.setClipping( save );
      }
      return ;
    }
    
    //horizontally
    if ( DWallpaper.HaveEffect( _effect, DWallpaper.HREPEAT ) )
    {
//TODO combiner avec HSCALE      
      int add = (_w%devimgw==0 ? 0 : 1);
      icx = _w/devimgw + add;
      int tw = icx*devimgw;
      if ( DWallpaper.HaveEffect( _effect, DWallpaper.LEFT ) ) 
      {
        ix = _x;
      }
      else if ( DWallpaper.HaveEffect( _effect, DWallpaper.RIGHT ) ) 
      {
        ix = _x+_w-tw;
      }
      else if ( DWallpaper.HaveEffect( _effect, DWallpaper.HSCALE ) ) 
      {
        ix = _x;
        int div = icx-add;
        iw = (_w)/(div==0?1:div);
        icx -= add;
      }
      else //HCENTER
      {
        ix = _x+(_w-tw)/2;
      }    
    }
    else if ( DWallpaper.HaveEffect( _effect, DWallpaper.HSCALE ) )
    {
      ix = _x;
      iw = _w;
    }
     else if( DWallpaper.HaveEffect( _effect, DWallpaper.LEFT ) )
    {
      ix = _x;
    }
    else if ( DWallpaper.HaveEffect( _effect, DWallpaper.RIGHT ) )
    {
      ix = _x+_w - devimgw;
    }
    else //HCENTER by default
    {
      ix = _x+(_w - devimgw)/2;
    }
    
    //vertically
    if ( DWallpaper.HaveEffect( _effect, DWallpaper.VREPEAT ) ) 
    {
      int add = (_h%devimgh==0 ? 0 : 1);
      icy = _h/devimgh + add;
      int th = icy*devimgh;
      if ( DWallpaper.HaveEffect( _effect, DWallpaper.TOP ) ) 
      {
        iy = _y;
      }
      else if ( DWallpaper.HaveEffect( _effect, DWallpaper.BOTTOM ) ) 
      {
        iy = _y+_h-th;
      }
      else if ( DWallpaper.HaveEffect( _effect, DWallpaper.VSCALE ) ) 
      {
        iy = _y;
        int div = icy-add;
        ih = (_h)/(div==0 ? 1: div );
        icy-=add;
      }
      else //VCENTER
      {
        iy = _y+(_h-th)/2;
      }
    }
    else if ( DWallpaper.HaveEffect( _effect, DWallpaper.VSCALE ) )
    {
      iy = _y;
      ih = _h;
    }
    else if( DWallpaper.HaveEffect( _effect, DWallpaper.TOP ) )
    {
      iy=_y;
    }
    else if ( DWallpaper.HaveEffect( _effect, DWallpaper.BOTTOM ) )
    {
      iy = _y+_h - devimgh;
    }
    else //VCENTER by default
    {
      iy = _y+(_h - devimgh)/2;
    }
    
    if( iw<0 || ih < 0 ) return ;
    
    IShape save = _gc.getClipping();
    _gc.setClipping( new Rect( _x, _y, _w, _h ) );
    try{
      int iix=ix;
//System.out.println("IMAGE// ix="+ix+" iy="+iy+" iw="+iw+" ih="+ih+" _w="+_w+" imgw="+imgw);    
      for( int ry=0; ry<icy; ++ry )
      {
        ix=iix;
        for( int rx=0; rx<icx; ++rx )
        {
          _gc.drawImage( _image, ix,iy,iw,ih );
          ix += iw;
        }
        iy += ih;
      }
    }
    finally
    {
      _gc.setClipping( save );
    }
  }
  
}
