/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IGCDStyle.java,v 1.3 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.drawutil.internal;


import org.eclipse.tptp.platform.report.core.internal.DFont;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;

import com.ibm.icu.util.ULocale;

/**
 * This class is made to transform any IDStyle applied on a IDItem, to a valid
 * style suitable for IGC drawings.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class IGCDStyle
{
 
  /** style's current font */
  protected IFont font_;
  /** style's current fore color (RGBA) */
  protected int fore_;
  /** style's current back color (RGBA) */
  protected int back_;    
 
  /** 
   * Create default style.
   */
  public IGCDStyle()
  {
    styleOf( null, 1.0f );
  }
  
  /** 
   * Create default style.
   */
  public IGCDStyle( float font_scale )
  {
    styleOf( null, font_scale);
  }

  /**
   * Create a style from given IDItem at scale 1.0f, same result as calling styleOf(item).
   */
  public IGCDStyle( IDItem _item )
  {
    styleOf( _item, 1.0f );
  }
  
  /**
   * Create a style from given IDItem, same result as calling styleOf(item).
   */
  public IGCDStyle( IDItem _item, float font_scale )
  {
    styleOf( _item, font_scale );
  }

  /**
   * @return font of this style.
   */
  public IFont getFont() { return font_; }
  /**
   * @return fore color of this style.
   */
  public int getFore() { return fore_ ; }
  /**
   * @return back color of this style.
   */
  public int getBack() { return back_ ; }
  
  /**
   * @return the default foreground color (black).
   */
  public static int GetDefaultFore()
  {
    return RGBA.BLACK;
  }
  
  /**
   * @return the default RGBA background color (white).
   */
  public static int GetDefaultBack()
  {
    return RGBA.WHITE;
  }
  
  /**
   * @return default font for given scale 
   */
  public static Font GetDefaultFont( float font_scale )
  {	
  	return new Font( "", "", (int)(8*font_scale), IFont.NORMAL );
  } 

  //?? multi thread aware ?
  //private static SColor a_scolor = new SColor();
  //private static SFont  a_sfont   = new SFont();
  /** 
   * Change this IGCDStyle with style collected from given IDItem.
   * DStyle for item is retrieved walking upstair in tree from _item to try to find complete style.
   * On top level node, complete style by defaults.
   * The contract is after this, IGCDStyle have non null two colors and fonts .
   * The parameter scale_font is float value 0..+oo which you can obtain a font bigger or lower
   * that the original font. Used for zoom feature. (ok +oo it's not really realistic). 
   */
  public void styleOf( IDItem _item, float font_scale )
  {
    IDStyle jscrib_style=null;
    IDFont  jscrib_font =null;
    IDColor jscrib_color=null;
    int all = 0;
    Font a_sfont=null;
    int  a_scolor=0;
    
    //reset field
    font_=null;
    fore_=0;
    back_=0;
    boolean have_fore = false;
    boolean have_back = false;
    boolean have_font = false;
    
    for( IDItem item=_item; item!=null; item=item.getParent() )
    {
      if( (jscrib_style=item.getStyle())!=null )
      {
        //font
        if ( !have_font && (jscrib_font=jscrib_style.getFont())!=null ) 
        {           
          font_ = GetFont( jscrib_font, font_scale, null );
          have_font = (font_!=null);
          all++;
        }
        //fore color
        if ( !have_fore && (jscrib_color=jscrib_style.getForeColor())!=null )
        {
          fore_ = GetRGBA( jscrib_color );
          have_fore = true; //? even if transparent?
          all++;
        }
        //back color
        if ( !have_back && (jscrib_color=jscrib_style.getBackColor())!=null )
        {
          back_ = GetRGBA( jscrib_color );
          have_back = true ;
          all++;
        }
        //style completed ?
        if( have_font && have_back && have_fore ) return ;
      }
    }//for
    
    //get default for missing component
    if ( !have_font ) 
    {           
      font_ = GetDefaultFont( font_scale);
    }
    //fore color
    if ( !have_fore )
    {
      fore_ = GetDefaultFore();
    }
    //back color
    if ( !have_back )
    {
      back_ = GetDefaultBack();
    }
  }
  
  /** 
   * Convert a jscrib font to IGC one.
   * @return a IGC Font object if _scrib_font is null, return _font
   * @param _swt_device the devie used to create font
   * @param _jscrib_font the font to convert
   * @param _font_scale zoom average for the font
   * @param _font reuse this object rather than allocate a new one.
   */
  public static Font GetFont( IDFont _jscrib_font, float _font_scale, Font _font )
  {
    if( _jscrib_font == null )
    {
    	return _font;
    } 
    
    int fstyle  = Font.NORMAL;
    if( _jscrib_font.haveStyle( DFont.BOLD  ) ) fstyle |= Font.BOLD;
    if( _jscrib_font.haveStyle( DFont.ITALIC) ) fstyle |= Font.ITALIC;
    if( _jscrib_font.haveStyle( DFont.UNDERLINE)) fstyle |= Font.UNDERLINE;
    if( _jscrib_font.haveStyle( DFont.OVERLINE )) fstyle |= Font.OVERLINE ;
    if( _jscrib_font.haveStyle( DFont.STRIKE   )) fstyle |= Font.STRIKE;
    if( _jscrib_font.haveStyle( DFont.DASHED   )) fstyle |= Font.DASHED;
     
    String ffamily = _jscrib_font.getFamily();
    String fscript = _jscrib_font.getScript();
    int    fsize   = _jscrib_font.getSize();

    //SWT bark when ffamily is empty...
    if( DrawUtilIGC.isEmpty(ffamily)) ffamily="arial";
    if( DrawUtilIGC.isEmpty(fscript)) fscript=ULocale.getDefault().toString();

    if (_font_scale!=1.0f)
	    fsize = (int)(fsize*_font_scale);

    //limit font size...
    if( fsize <= 5 ) fsize=6;

    if( _font!=null )
    {
      _font.setFont( fscript, ffamily, fsize, fstyle );
    } else {
      _font = new Font( fscript, ffamily, fsize, fstyle );
    }
    return _font;
  }
  
  /**
   * Convert an jscrib component color range to a valid SWT Color range [0..255].
   * @return a value in range 0..255.
   */
  public static int getColorComponent( int jscrib_color_component )
  {
    if( jscrib_color_component > 255 ) return 255;
    if( jscrib_color_component < 0   ) return   0;
    return jscrib_color_component;
  }
  
  /**
   * Convert a jscrib IDColor to a RGBA Color.Transparent color if given one is null
   * @return RGBA color corresponding to given IDColor.
   * @param _jscrib_color the color to convert.
   */
  public static int GetRGBA( IDColor _jscrib_color)
  {
  	if( _jscrib_color==null ) return RGBA.TRANSPARENT;
    return RGBA.Get( _jscrib_color.getRed  (),
                     _jscrib_color.getGreen(),
                     _jscrib_color.getBlue () );
  }

  /** 
   * @return the RGBA foreground color use by item or one of its parent, or default foreground.
   */
  public static int GetForeColor( IDItem _item )
  {
    IDStyle jscrib_style=null;
    IDColor jscrib_color=null;
    
    for( IDItem item=_item; item!=null; item=item.getParent() )
    {
      if( (jscrib_style=item.getStyle())!=null )
      {
        if ( (jscrib_color=jscrib_style.getForeColor())!=null )
        {
          return GetRGBA( jscrib_color);
        }
      }
    }//for
    
    //return default foreground color:
    return GetDefaultFore();
  }
  
  /** 
   * @return the RGBA background color use by item or one of its parent, or default background.
   */
  public static int GetBackColor( IDItem _item )
  {
    IDStyle jscrib_style=null;
    IDColor jscrib_color=null;
    
    for( IDItem item=_item; item!=null; item=item.getParent() )
    {
      if( (jscrib_style=item.getStyle())!=null )
      {
        if ( (jscrib_color=jscrib_style.getBackColor())!=null )
        {
          return GetRGBA( jscrib_color );
        }
      }
    }//for
    
    //return default
    return GetDefaultBack();
  }  
  
  /** 
   * @return the inherited font by a style on the item or a style applied on a parent
   */
  public static Font GetFont( IDItem _item, float font_scale )
  { 
	   IDFont f = GetIDFont( _item );
	   if (f!=null)
		  return GetFont( f, font_scale, null );
	   else
		  //	return default
		  return GetDefaultFont( font_scale );
  }  
  
   /** 
	* @return the inherited font by a style on the item or a style applied on a parent.
	*/
   public static IDFont GetIDFont( IDItem _item )
   {
	  IDStyle jscrib_style=null;
	  IDFont jscrib_font=null;
    
	  for( IDItem item=_item; item!=null; item=item.getParent() )
	  {
		   if( (jscrib_style=item.getStyle())!=null )
		   {
			 if ( (jscrib_font=jscrib_style.getFont())!=null )
			 {
			   return  jscrib_font;
			 }
		   }
	  }//for
	 
	  return null;
   }
   
}
