/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ISymbol.java,v 1.2 2008/05/23 14:11:56 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 29 nov. 2003
 *
 */
package org.eclipse.tptp.platform.report.drawutil.internal;

import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IPen;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;


/**
 * A Symbol a shape, it must support:
 * - drawing inside a rectangle (@see draw)
 * - checking if a point is inside or not in the symbol's shape (@see contains)
 * 
 * Any static class defining standards symbol are provided:
 * - Square, Oval (circle), triangles (Up,Left,Right,Down), Plus (cross), ...
 * - Polygon abstract class can be subclassed for any polygon shaped symbol.
 * 
 * @author ademuyser
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public interface ISymbol
{
  /** 
   * Draw the symbol using igc's current pen and brush
   * @param gc the graphic context used to draw the symbol.
   * @param rect the rectangle where the symbol must be drawn.
   */
  public void draw( IGC gc, IRect rect );
  
  /**
   * @return true if point (px,py) is inside symbol.
   * @param gc use this to get any info you want (as text extends ...)
   * @param rect bounding rectangle of the symbol.
   * @param px x coordinate of point to check.
   * @param px y coordinate of point to check.
   */
  public boolean contains( IGC gc, IRect rect, int px, int py );
  
  /**
   * @return the id associated with this symbol ( a class name is a good idea, for example).
   */
  public String getId();
  
  /** 
   * Define a square symbol (a rectangle), this is the simpliest symbol.
   */
  public static class Square implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      gc.fillRect( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
      gc.drawRect( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
    }
    public boolean contains( IGC gc, IRect rect, int px, int py )
    {
      return rect.contains( px, py );
    }
    public String getId() { return getClass().getName(); }
  }

  /** 
   * Define a square symbol (a rectangle), with a second rectangle inside.
   * This second rectangle is filled with gc's pen is pen is a LineStylePen.
   */
  public static class Square2 implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int x = rect.getX(), y=rect.getY(), w=rect.getW(), h=rect.getH();
      gc.fillRect( x, y, w, h );
      gc.drawRect( x, y, w, h );

      IPen pen = gc.getPen();
      IBrush old_brush = gc.getBrush();
      if( pen instanceof LineStylePen )
      {
        gc.setBrush( new SolidBrush( ((LineStylePen)pen).getRGBA() ));
      }
      int x2 = rect.getX()+rect.getW()/4;
      int w2 = rect.getW()/2;
      int y2 = rect.getY()+rect.getH()/4;
      int h2 = rect.getH()/2;
      gc.fillRect( x2, y2, w2+1, h2+1 );
      gc.drawRect( x2, y2, w2, h2 );
      gc.setBrush( old_brush );
    }
    public boolean contains( IGC gc, IRect rect, int px, int py )
    {
      return rect.contains( px, py );
    }
    public String getId() { return getClass().getName(); }
  }
  
  /** 
   * Define Oval family symbol, can be used for a circle symbol (set rectangle's width=height).
   */
  public static class Oval implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      gc.fillOval( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
      gc.drawOval( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      return DrawUtilIGC.ovalContains( r.getX(),r.getY(),r.getW(),r.getH(), px,py );
    }
    public String getId() { return getClass().getName(); }
  }

  /** 
   * Define OvalDot family symbol, can be used for a circle symbol (set rectangle's width=height),
   * having a dot inside. This dot is filled with pen's color if this pen is a LineStylePen.
   */
  public static class OvalDot implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      gc.fillOval( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
      gc.drawOval( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
      IPen pen = gc.getPen();
      IBrush old = gc.getBrush();
      if( pen instanceof LineStylePen )
      {
        gc.setBrush( new SolidBrush( ((LineStylePen)pen).getRGBA()) );
      }
      int x2 = rect.getX()+rect.getW()/4;
      int w2 = rect.getW()/2;
      int y2 = rect.getY()+rect.getH()/4;
      int h2 = rect.getH()/2;
      gc.fillOval( x2, y2, w2+1, h2+1 );
      gc.drawOval( x2, y2, w2, h2 );
      gc.setBrush( old );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      return DrawUtilIGC.ovalContains( r.getX(),r.getY(),r.getW(),r.getH(), px,py );
    }
    public String getId() { return getClass().getName(); }
  }  
  /** 
   * Polygon shaped symbol. To use this class, subclass it and redefine getPoints() method.
   */
  public abstract static class PolygonSymbol implements ISymbol
  {
    /** 
     * @return the polygon definition, as IGC.fillPoly() require.
     */
    protected abstract Polygon getPolygon( IRect r );
    
    public void draw( IGC gc, IRect rect )
    {
      Polygon poly = getPolygon( rect );
      synchronized( poly )
      {
        gc.fillPoly( poly );
        gc.drawPoly( poly );
      }
    }
    public boolean contains( IGC gc, IRect rect, int px, int py )
    {      
      if( !rect.contains(px,py) ) return false;
      Polygon poly = getPolygon( rect );
      synchronized( poly )
      {
        return ((Polygon)poly).contains( px, py );        
      }
    }
    public String getId() { return getClass().getName(); }
  }
  
  
  /**
   * Diamong symbol, simpler named lozenge. (but lozenge died ...).
   */
  public static class Diamond extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {
      int xc = Rect.CenterX(r);
      int yc = Rect.CenterY(r);
      int dx = r.getW()/2;
      int dy = r.getH()/2;
      int points[] =new int[10];
      points[0] = xc;     points[1] = yc-dy;
      points[2] = xc+dx;  points[3] = yc;
      points[4] = xc;     points[5] = yc+dy;
      points[6] = xc-dx;  points[7] = yc;
      points[8] = points[0];     points[9] = points[1];
      return new Polygon(points);
    }
  }
  /**
   *  Upward triangle symbol
   */
  public static class Up extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {      
      int yb = Rect.Bottom(r);
      Polygon p = new Polygon( 4 );
      p.setPoint( 0, Rect.CenterX(r), Rect.Top(r) );
      p.setPoint( 1, Rect.Right(r),   yb );
      p.setPoint( 2, Rect.Left(r),    yb );
      p.setPoint( 3, Rect.CenterX(r), Rect.Top(r) );
      return p;
    }
  }
  /**
   * Downward triangle symbol
   */
  public static class Down extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {
      int yt = Rect.Top(r);
      Polygon p = new Polygon( 4 );
      p.setPoint( 0, Rect.Right(r),   yt );
      p.setPoint( 1, Rect.CenterX(r), Rect.Bottom(r) );
      p.setPoint( 2, Rect.Left(r),    yt );
      p.setPoint( 3, Rect.Right(r),   yt );
      return p;
    }
  }
  /** 
   * Rightward triangle symbol
   */
  public static class Right extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {
      int y2 =Rect.CenterY(r);
      int xl = Rect.Left(r);
      int points[] = new int[8];
      points[0] = xl;            points[1] = Rect.Top(r);
      points[2] = Rect.Right(r); points[3] = y2;
      points[4] = xl;            points[5] = Rect.Bottom(r);
      points[6] = points[0];     points[7] = points[1];
      return new Polygon( points );
    }
  }
  /** 
   * Leftward triangle symbol
   */
  public static class Left extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {
      int xr = Rect.Right(r);
      int points[] = new int[8];
      points[0] = xr; points[1] = Rect.Top(r);
      points[2] = xr; points[3] = Rect.Bottom(r);
      points[4] = Rect.Left(r);  points[5] = Rect.CenterY(r);
      points[6] = points[0]; points[7] = points[1];
      return new Polygon( points );
    }
  }
  /**
   * Plus, or cross symbol
   */
  public static class Plus extends PolygonSymbol
  {
    protected Polygon getPolygon( IRect r )
    {
      int dx = r.getW()/3,  dy=r.getH()/3;
      int x0 = Rect.Left(r), y0 = Rect.Top(r); 
      int x1 = x0+dx,    y1 = y0 + dy;
      int x2 = x1+dx,    y2 = y1 + dy;
      int x3 = x2+dx,    y3 = y2 + dy;
      int points[] =new int[26];
      points[ 0] = x2; points[ 1] = y0;
      points[ 2] = x2; points[ 3] = y1;
      points[ 4] = x3; points[ 5] = y1;
      points[ 6] = x3; points[ 7] = y2;
      points[ 8] = x2; points[ 9] = y2;
      points[10] = x2; points[11] = y3;
      points[12] = x1; points[13] = y3;
      points[14] = x1; points[15] = y2;
      points[16] = x0; points[17] = y2;
      points[18] = x0; points[19] = y1;
      points[20] = x1; points[21] = y1;
      points[22] = x1; points[23] = y0;
      points[24] = points[0]; points[25] = points[1];
      return new Polygon( points );
    }
  }
  /**
   * Minus, covering 1/3 of rectangle height.
   */
  public static class Minus implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int h =rect.getH()/3;
      gc.fillRect( rect.getX(), rect.getY()+h, rect.getW(), h );
      gc.drawRect( rect.getX(), rect.getY()+h, rect.getW(), h );
    }
    public boolean contains( IGC gc, IRect rect, int px, int py )
    {
      int h =rect.getH()/3;
      return Rect.Contains( rect.getX(), rect.getY()+h, rect.getW(), h,  px, py );
    }
    public String getId() { return getClass().getName(); }
  }
  
  /** 
   * Define down-half-Oval symbol
   */
  public static class MoonDown implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int y = rect.getY()-rect.getH()/4;
      int cx = Rect.CenterX(rect), cy=Rect.CenterY(rect);
      int w = rect.getW();
      int rx = w/2, ry = rect.getH()/2;
      gc.fillArc( cx,cy, 0.0,  rx,ry, Radian._PI, Radian._PI );
      gc.drawArc( cx,cy, 0.0,  rx,ry, Radian._PI, Radian._PI );
      y = rect.getY()+rect.getH()/4;
      gc.drawLine( Rect.Left(rect), cy, Rect.Right(rect), cy );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      int x= r.getX(), y=r.getY(), w=r.getW(), h=r.getH();
      return  !Rect.Contains( x, y-h/4, w, h/2, px,py )
            && DrawUtilIGC.ovalContains( x,y-h/4,w,h, px,py );
    }
    public String getId() { return getClass().getName(); }
  }
  /** 
   * Define up-half-oval symbol
   */
  public static class MoonUp implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int y = rect.getY()+rect.getH()/4;
      int cx = Rect.CenterX(rect), cy=Rect.CenterY(rect);
      int rx = rect.getW()/2, ry = rect.getH()/2;
      gc.fillArc( cx,cy, 0.0,  rx,ry, 0, Radian._PI );
      gc.drawArc( cx,cy, 0.0,  rx,ry, 0, Radian._PI );
      y = rect.getY()+3*rect.getH()/4;
      gc.drawLine( Rect.Left(rect), cy, Rect.Right(rect), cy );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      int x= r.getX(), y=r.getY(), w=r.getW(), h=r.getH();
      return  !Rect.Contains( x, y+h/4, w, h/2, px,py )
            && DrawUtilIGC.ovalContains( x, y+h/4, w, h/2, px,py );
    }
    public String getId() { return getClass().getName(); }
  }  
  /** 
   * Define left-half-oval family symbol
   */
  public static class MoonLeft implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int x = rect.getX()+rect.getW()/4;
      int cx = Rect.CenterX(rect), cy=Rect.CenterY(rect);
      int rx = rect.getW()/2, ry = rect.getH()/2;
      gc.fillArc( cx,cy, 0.0,  rx,ry, Radian._PI2, Radian._PI );
      gc.drawArc( cx,cy, 0.0,  rx,ry, Radian._PI2, Radian._PI );
      x = rect.getX()+3*rect.getW()/4;
      gc.drawLine( cx, Rect.Top(rect), cx, Rect.Bottom(rect) );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      int x= r.getX(), y=r.getY(), w=r.getW(), h=r.getH();
      return  !Rect.Contains( x+w/4, y, w/2, h, px,py )
            && DrawUtilIGC.ovalContains( x+w/4, y, w/2, h, px,py );
    }
    public String getId() { return getClass().getName(); }
  } 
  /** 
   * Define right family symbol
   */
  public static class MoonRight implements ISymbol
  {
    public void draw( IGC gc, IRect rect )
    {
      int x = rect.getX()-rect.getW()/4;
      int cx = Rect.CenterX(rect), cy=Rect.CenterY(rect);
      int rx = rect.getW()/2, ry = rect.getH()/2;
      gc.fillArc( cx,cy, 0.0,  rx,ry, Radian._3PI2, Radian._PI );
      gc.drawArc( cx,cy, 0.0,  rx,ry, Radian._3PI2, Radian._PI );
      x = rect.getX()+rect.getW()/4;
      gc.drawLine( cx, Rect.Top(rect), cx, Rect.Bottom(rect) );
    }
    public boolean contains( IGC gc, IRect r, int px, int py )
    {
      int x= r.getX(), y=r.getY(), w=r.getW(), h=r.getH();
      return  !Rect.Contains( x+w/4, y, w/2, h, px,py )
            && DrawUtilIGC.ovalContains( x+w/4, y, w/2, h, px,py );
    }
    public String getId() { return getClass().getName(); }
  } 

  /**
   * two vertical triangles
   */
  public static class VTri2 extends PolygonSymbol
  {
    protected Polygon getPolygon(IRect r) 
    {
      int points[] = new int[10];
      points[0] = r.getX();  points[1] = r.getY();
      points[2] = r.getX()+r.getW(); points[3] = r.getY()+r.getH();
      points[4] = points[0]; points[5] = points[3];
      points[6] = points[2]; points[7] = points[1];
      points[8] = points[0]; points[9] = points[1];
      return new Polygon(points);
    }
  }
  /**
   * two horizontal triangles
   */
  public static class HTri2 extends PolygonSymbol
  {
    protected Polygon getPolygon(IRect r) 
    {
      int points[] = new int[10];
      points[0] = r.getX();  points[1] = r.getY();
      points[2] = r.getX()+r.getW(); points[3] = r.getY()+r.getH();
      points[4] = points[2]; points[5] = points[1];
      points[6] = points[0]; points[7] = points[3];
      points[8] = points[0]; points[9] = points[1];
      return new Polygon(points);
    }
  }
  /**
   * Abstract class for symbol using generator.
   */
  public static abstract class GeneratedPolygonSymbol extends PolygonSymbol
  {
    protected abstract double [] getGenerator(); //must create gen_ xy array of points
    //use this to fit generator to 1x1 square
    protected static void FitGenerator( double _gen[] )
    {
      double xn=0, yn=0, xx=0, yx=0;
      for( int i=0; i<_gen.length; )
      {
        if( i==0 ) {  xn = xx = _gen[i]; }
        else if( _gen[i]<xn ) xn=_gen[i]; else if( _gen[i]>xx ) xx=_gen[i];
        i++;
        if( i==1) {  yn = yx = _gen[i]; }
        else if( _gen[i]<yn ) yn=_gen[i]; else if( _gen[i]>yx ) yx=_gen[i];
        i++;
      }
      //fit to 1x1 square
      double kx = 1.0/(xx-xn);
      double ky = 1.0/(yx-yn);
      for( int i=0;i<_gen.length; )
      {
        _gen[i] = kx*(_gen[i]-xn); i++;
        _gen[i] = ky*(_gen[i]-yn); i++;
      }
    }
    protected Polygon getPolygon( IRect r )
    {
      double []gen = getGenerator();
      //scale generator to rectangle
      double sx = r.getW(), sy=r.getH();
      int x0 = r.getX(), y0 = r.getY();
      int points[] = new int[gen.length+2];
      for( int i=0; i<gen.length;  )
      {
        points[i] = x0 + (int)Math.round(sx*gen[i]); i++;
        points[i] = y0 + (int)Math.round(sy*gen[i]); i++;
      }      
      points[gen.length  ] = points[0]; 
      points[gen.length+1] = points[1];
      return new Polygon( points );
    }
  }
  /**
   * Star, a 5 branch star.
   */
  public static class Star extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null)  return gen_;
      gen_ = new double[20];
      double rx = 0.5,  ry=0.5;
      double rix = rx/2.65, riy= ry/2.65;
      double cx = 0.5, cy = 0.5;
      double a=0.5*Math.PI;
      double da=0.2*Math.PI;
      int i=0, j=0;
      for(; j<10; ++j, a+=da )
      {
        gen_[i++] = cx+((j&0x1)==0 ? rx : rix )*Math.cos( a );
        gen_[i++] = cy-((j&0x1)==0 ? ry : riy )*Math.sin( a );
      }
      FitGenerator( gen_ );
      return gen_;
    }
  }
  /**
   * Hexagon
   */
  public static class Hexagon extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null ) return gen_;
      gen_ = new double[12];     
      double c = 0.5*Math.cos(Math.PI/3);
      double s = 0.5*Math.sin(Math.PI/3);
      gen_[0] = 1.0;    gen_[1] = 0.5;
      gen_[2] = 0.5+c;  gen_[3] = 0.5-s;
      gen_[4] = 0.5-c;  gen_[5] = gen_[3];
      gen_[6] = 0;      gen_[7] = 0.5;
      gen_[8] = gen_[4];gen_[9] = 0.5+s;
      gen_[10] =gen_[2];gen_[11]= gen_[9];
      
      FitGenerator( gen_ );
      return gen_;
    }
  }
  /**
   * Hexagon cross look like 3 triangles. 
   */
  public static class HexagonCross extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null ) return gen_;
      gen_ = new double[12];     
      double c = 0.5*Math.cos(Math.PI/3);
      double s = 0.5*Math.sin(Math.PI/3);
      gen_[0] = 1.0;    gen_[1] = 0.5;
      gen_[2] = 0;      gen_[3] = 0.5;
      gen_[4] = 0.5-c;  gen_[5] = 0.5+s;
      gen_[6] = 0.5+c;  gen_[7] = 0.5-s;
      gen_[8] = 0.5-c;  gen_[9] = 0.5-s;
      gen_[10] =0.5+c;  gen_[11] = 0.5+s;
      
      FitGenerator( gen_ );
      return gen_;
    }
  }
  /**
   * Hexagon cross look like 3 triangles (reversed) 
   */
  public static class HexagonCross2 extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null ) return gen_;
      gen_ = new double[12];     
      double c = 0.5*Math.cos(Math.PI/3);
      double s = 0.5*Math.sin(Math.PI/3);
      gen_[0] = 1.0;    gen_[1] = 0.5;
      gen_[2] = 0;      gen_[3] = 0.5;
      gen_[4] = 0.5-c;  gen_[5] = 0.5-s;
      gen_[6] = 0.5+c;  gen_[7] = 0.5+s;
      gen_[8] = 0.5-c;  gen_[9] = 0.5+s;
      gen_[10] =0.5+c;  gen_[11] = 0.5-s;
      
      FitGenerator( gen_ );
      return gen_;
    }
  }
  /**
   * Pentagon
   */
  public static class Pentagon extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null) return gen_;
      gen_ = new double[10];
      double rx = 0.5,  ry=0.5;
      double cx = 0.5, cy = 0.5;
      double a=Math.PI/2;
      double da=2*Math.PI/5;
      int i=0, j=0;
      for(; j<5; ++j, a+=da )
      {
        gen_[i++] = cx+rx*Math.cos( a );
        gen_[i++] = cy-ry*Math.sin( a );
      }
      FitGenerator( gen_ );
      return gen_;
    }  
  }
  /**
   * Cross look like 4 triangles. 
   */
  public static class Cross extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null ) return gen_;
      gen_ = new double[20]; 
      double l = 1/4.0;
      gen_[0] = 0.5;    gen_[1] = 0.5;
      gen_[2] = 0;      gen_[3] = 0.5+l;
      gen_[4] = 0;      gen_[5] = 0.5-l;
      gen_[6] = 1;      gen_[7] = 0.5+l;
      gen_[8] = 1;      gen_[9] = 0.5-l;
      gen_[10]= 0.5;    gen_[11] = 0.5;
      gen_[12]= 0.5-l;  gen_[13] = 0;
      gen_[14]= 0.5+l;  gen_[15] = 0;
      gen_[16]= 0.5-l;  gen_[17] = 1;
      gen_[18]= 0.5+l;  gen_[19] = 1;
      return gen_;
    }
  }
  /**
   * Cross look like 4 triangles,  like a Cross with "45 rotated"... 
   */
  public static class Cross2 extends GeneratedPolygonSymbol
  {
    private static double gen_[];
    protected double[] getGenerator()
    {
      if( gen_!=null ) return gen_;
      gen_ = new double[20]; 
      double l = 1/4.0;
      gen_[0] = 0.5;    gen_[1] = 0.5;
      gen_[2] = 0;      gen_[3] = 0.5-l;
      gen_[4] = 0.5-l;  gen_[5] = 0;
      gen_[6] = 0.5+l;  gen_[7] = 1;
      gen_[8] = 1;      gen_[9] = 0.5+l;
      gen_[10]= 0.5;    gen_[11]=0.5;
      gen_[12]= 0.5-l;  gen_[13]= 1;
      gen_[14]= 0;      gen_[15]=0.5+l;
      gen_[16]= 1;      gen_[17]=0.5-l;
      gen_[18]= 0.5+l;  gen_[19]=0; 
      return gen_;
    }
  }
}
