/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConfParser.java,v 1.2 2008/05/23 14:11:49 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on Sep 18, 2003
 */
package org.eclipse.tptp.platform.report.drivers.html;

import java.util.HashMap;
import java.util.Map.Entry;
import java.util.List;
import java.util.Set;
import java.util.LinkedList;

import org.eclipse.tptp.platform.report.drivers.internal.ConfSemanticException;
import org.eclipse.tptp.platform.report.drivers.internal.ConvertText;
import org.eclipse.tptp.platform.report.drivers.internal.HtmlFile;
import org.eclipse.tptp.platform.report.drivers.internal.HtmlValuators;
import org.eclipse.tptp.platform.report.drivers.internal.HtmlVariable;
import org.eclipse.tptp.platform.report.drivers.internal.Item;
import org.eclipse.xsd.util.XSDParser;
import org.xml.sax.InputSource;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Element;

/**
 * This class contains an html writer parser configuration.
 * This configuration is read from the dhtml.xml file standing in the same path than this class.
 * Configuration allows to set some of the html generated for a JScrib item.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class ConfParser 
{
  private boolean parsed; // true if data have been parsed without error

  private HashMap tt_fonts = new HashMap();
  private HtmlFile html_file; //defined only at root parser level
  private HtmlValuators variables = new HtmlValuators();
  private HashMap items = new HashMap();
  private ConvertText convert_text;
  
  private ConfParser parent; // this parser extends parent features, null for root parser
  private boolean is_root_parser;
  private String conf_file_name;

  /** Create a root html configuration parser */
  public ConfParser() 
  {
    conf_file_name = "dhtml.xml"; //default "high level configuration file
    is_root_parser=true;
    initRootVariables();
  }
  
  /**
   * Create a secondary html configuration parser.
   * In other words this parser extends parent parser features, to do it do not forget
   * to call setParentConfParser() with extended parser.
   * Html variables added to secondary parser must be initalized using initVariable() method. 
   */
  public ConfParser( String file_name, ConfParser parent )
  {
    conf_file_name=file_name;
    this.is_root_parser=parent==null;
    this.parent=parent;
    if(is_root_parser) initRootVariables();
  }
  
  /** Change actual configuration parser */
  public void setParentConfParser( ConfParser p )
  {
    parent = p;
  }
  /** @return actual parent configuration parser, null if none */
  public ConfParser getParentConfParser() { return parent; }
  
  private InputSource getConfInputSource() {
  	InputSource result;
	final Class clazz = ConfParser.class;
	try {
		 result = new InputSource(clazz.getResourceAsStream(conf_file_name));
	} catch (Throwable t) {
		String error = "Cannot load "+conf_file_name+" file";
		throw new RuntimeException(error);
	}
	return result;
  }

  public boolean parse() 
  {
    parsed=false;
    XSDParser parser = new XSDParser();
	try {
	  parser.parse(getConfInputSource());
	} catch (Exception e) {
	  e.printStackTrace();
	}
	Document document = parser.getDocument();
	try {
	  if (document == null) {
		throw new ConfSemanticException("no document read");
	  }
	  Element root = document.getDocumentElement();
	  if (root == null || !"dhtml".equals( root.getNodeName() ) ) {
		throw new ConfSemanticException("no 'dhtml' element in document");
	  }
	  checkTTFonts(root);    
	  if(is_root_parser) html_file = checkFile(root);    
	  checkItems(root);
	  convert_text = checkConvertText(root);    
	} catch (ConfSemanticException e) {
	  System.err.println("["+getConfInputSource()+"] "+e.getMsg());
	  parser = null;
	  return false;
	}
    parsed=true;
	return true;
  }

  private void checkTTFonts(Element root) throws ConfSemanticException {
    NodeList list = root.getElementsByTagName("tt_font");
    for (int i=0; i<list.getLength(); i++) {
      checkTTFont(list.item(i));
    }
  }

  private void checkTTFont(Node tt_font) throws ConfSemanticException 
  {
	NamedNodeMap attrs = tt_font.getAttributes();
	if (attrs == null || attrs.getLength() != 2) {
	  throw new ConfSemanticException("'old' and 'new' attributes allowed and mandatory for 'tt_font' tag");
	}
	Node old = attrs.getNamedItem("old");
	if (old == null) {
	  throw new ConfSemanticException("no 'old' attribute declared for 'tt_font' tag");
	}
	Node new_ = attrs.getNamedItem("new");
	if (new_ == null) {
	  throw new ConfSemanticException("no 'new' attribute declared for 'tt_font' tag");
	}
	tt_fonts.put(old.getNodeValue(), new_.getNodeValue());
  }
  
  public boolean isTTFont(String name) {
  	if( tt_fonts.get(name) != null ) return true;
  	return parent!=null && parent.isTTFont(name);
  }

  /** @return true if variable is defined by this parser or parent one */
  public boolean isDefinedVariable(String name ) {
    if( variables.isDefinedVariable(name) ) return true;
    return parent!=null && parent.isDefinedVariable(name);
  }
  
  /** @return variable value defined in this parser of parent one */
  public String getVariableValue(String name) {
    if( variables.isDefinedVariable(name)) {
      return variables.getVariableValue(name);
    }
    if( parent != null ) {
      return parent.getVariableValue(name);
    }
    return null;
  }
  
  /** Change value of variable in this parser or parent depending of which one define it. */
  public void setVariableValue(String name, String value, Object owner) 
  {
    if( variables.isDefinedVariable(name) ) {
      variables.setVariableValue(name, value, owner);  
      return ;
    }
	if( parent != null ) {
      parent.setVariableValue( name, value, owner );	  
	}
  }

  /** unset variable in this parser <b>and</b> parent parser (if any) */
  public void unsetVariables(Object owner) {
    variables.unsetVariables(owner);  
	if( parent!=null ) parent.unsetVariables( owner );
  }
  
  /** 
   * Initialize a new variable for this parser, all new variable added by a parser extension
   * must be initalized using this method.
   */
  public void initVariable( String name )
  {
    variables.initVariable( name );
  }

  private HtmlVariable checkVar(Element var) throws ConfSemanticException { 
	NamedNodeMap attrs = var.getAttributes();
	if (attrs == null || attrs.getLength() != 1) {
	  throw new ConfSemanticException("only 'name' attribute allowed for 'var' tag");
	}
	Node name = attrs.getNamedItem("name");
	if (name == null) {
	  throw new ConfSemanticException("no 'name' attribute declared for 'var' tag");
	}
	
//TODO: don't allow user variable definition	
	if (getVariableValue(name.getNodeValue()) == null) {
	  throw new ConfSemanticException("unknown value "+name.getNodeValue()+" for 'name' attribute in 'var' tag\n"+getListOfVariables());
	}
	
	return new HtmlVariable( name.getNodeValue() );
  }
  
  private String getListOfVariables()
  {
    String s = variables.getListOfVariables();
    if( parent!=null ) {
      s += parent.getListOfVariables();
    }
    return s;
  }

  
  private List checkHtmlFragment(Element parent, String tag_name,
								   boolean mandatory) throws ConfSemanticException {
	NodeList tag = parent.getElementsByTagName(tag_name);
	if (tag == null || tag.getLength() == 0) {
	  if (mandatory) {
		throw new ConfSemanticException("missing element '"+tag_name+"'");
	  }
	  return null;
	}
	if (tag.getLength() != 1) {
	  throw new ConfSemanticException("only one '"+tag_name+"' element is allowed");
	}
	NodeList childs = tag.item(0).getChildNodes();
	List result = new LinkedList();
	final int SIZE=1024;
	//as parser generate losts of TEXT_NODE and CDATA_SECTION_NODE ...
	StringBuffer buf = new StringBuffer( SIZE );
	for (int k=0; k<childs.getLength(); k++) {
	  Node n = childs.item(k);	  
	  if ( n != null) {
		if ((n.getNodeType() == Node.TEXT_NODE)
          ||(n.getNodeType() == Node.CDATA_SECTION_NODE) )
        {
		  //result.add( n.getNodeValue() );
		  buf.append( n.getNodeValue() );
		} else if ( n.getNodeType() == Node.ELEMENT_NODE) {
		  if ( ! "var".equals( childs.item(k).getNodeName()) ) {
			throw new ConfSemanticException("only 'var' tag is allowed in '"+tag_name+"' tag; found '"+childs.item(k).getNodeName()+"' tag");
		  }
		  if( buf.length()>0 ) {
		    result.add( buf.toString() );
		    buf = new StringBuffer( SIZE );
		  }
		  result.add( checkVar((Element)n));
		} else if ( n.getNodeType() == Node.COMMENT_NODE ) {
		  //eat this silently ... but I think this is the throw below which have to be removed...
		} else {		  
		  throw new ConfSemanticException("only 'var' tag is allowed in '"+tag_name+"' tag");
		}
	  }
	}
	if( buf.length() > 0 ) {
	  result.add( buf.toString() );
	}
	return result;
  }

  private HtmlFile checkFile(Element root) throws ConfSemanticException {
	NodeList file = root.getElementsByTagName("file");	
	if (file==null || file.getLength() == 0) {
	  throw new ConfSemanticException("missgin element 'file'");
	}
	if (file.getLength() != 1) {
	  throw new ConfSemanticException("only one 'file' tag allowed");
	}
	NamedNodeMap attrs = file.item(0).getAttributes();
	if (attrs == null || attrs.getLength() > 2) {
	  throw new ConfSemanticException("'extension' and 'max_title_level' attributes needed and authorized for 'file' tag");
	}
	Node extension = attrs.getNamedItem("extension");
	if (extension == null) {
	  throw new ConfSemanticException("no 'extension' attribute declared for 'file' tag");
	}
	Node max_title_level = attrs.getNamedItem("max_title_level");
	if (max_title_level == null) {
	  throw new ConfSemanticException("no 'max_title_level' attribute declared for 'file' tag");
	}
	return new HtmlFile(
		extension.getNodeValue(),
		Integer.decode(max_title_level.getNodeValue()).shortValue(),
		checkHtmlFragment((Element)file.item(0), "begin", true),
		checkHtmlFragment((Element)file.item(0), "end", true));
  }

  /** @return item for given class from this parser or parent parser */
  public Item getItem(Class class_) {
	Item item = (Item)items.get(class_);
	if( item !=null ) return item;
	if( parent==null ) return null;
	return parent.getItem(class_);
  }

  private void checkItem(Element elt) throws ConfSemanticException {
	NamedNodeMap attrs = elt.getAttributes();
	Node class_ = attrs==null ? null : attrs.getNamedItem("class");
	if (class_ == null) {
	  throw new ConfSemanticException("nmissing 'class' attribute in 'item' element");
	}
	Item item = new Item(class_.getNodeValue(),
						 checkHtmlFragment(elt, "begin", true),
						 checkHtmlFragment(elt, "end", true),
						 checkHtmlFragment(elt, "begin_child", false),
						 checkHtmlFragment(elt, "end_child", false));
	if (items.get(item.getItemClass()) != null) {
		throw new ConfSemanticException("item already defined for "+item.getItemClass()+" in 'item' tag");
	}
	items.put(item.getItemClass(), item);
	
	//parse child element as ConfParser extenders might be interested in
	for( Node n =elt.getFirstChild(); n!=null; n=n.getNextSibling() )
	{
	  if( n.getNodeType() == Node.ELEMENT_NODE )
	  {
	    Element e = (Element)n;
	    if( "begin".equals( e.getNodeName())
	     || "begin".equals( e.getNodeName())
	     || "begin_child".equals( e.getNodeName())
	     || "end_child".equals( e.getNodeName()) )
	    {
	      continue; //aready parsed	      
	    }
	  }
	  checkItemChild( item, n );
	}
  }
  
  /** redefine this method if you are interested in &lt;item&gt; child node */
  protected void checkItemChild( Item item, Node n ) throws ConfSemanticException {}

  private void checkItems(Element root) throws ConfSemanticException {
	NodeList list = root.getElementsByTagName("item");
	if (list != null) {
	  for (int i=0; i<list.getLength(); i++) {
		checkItem((Element)list.item(i));
	  }
	}
  }

  private void checkReplace(ConvertText ct, Element elt) throws ConfSemanticException {
	NamedNodeMap attrs = elt.getAttributes();
	if (attrs == null || attrs.getLength() != 2) {
	  throw new ConfSemanticException("'char' and 'string' attributes needed and authorized for 'replace' tag");
	}
	Node _char = attrs.getNamedItem("char");
	if (_char == null) {
	  throw new ConfSemanticException("no 'char' attribute declared for 'replace' tag");
	}
	if (_char == null ||
		_char.getNodeValue().length() != 1) {
	  throw new ConfSemanticException("size of 'char' attribute declared for 'replace' tag must be 1, found \""+_char.getNodeValue()+"\"");
	}
	Character c_char = new Character(_char.getNodeValue().charAt(0));
	Node string = attrs.getNamedItem("string");
	if (string == null) {
	  throw new ConfSemanticException("no 'string' attribute declared for 'replace' tag");
	}
	ct.putReplace(c_char, string.getNodeValue());
  }

  private void checkReplaces(ConvertText ct, Element elt) throws ConfSemanticException {
	NodeList list = elt.getElementsByTagName("replace");
	if (list != null) {
	  for (int i=0; i<list.getLength(); i++) {
		checkReplace(ct, (Element)list.item(i));
	  }
	}
  }

  private ConvertText checkConvertText(Element root) throws ConfSemanticException {
	NodeList ct = root.getElementsByTagName("convert_text");
	if (ct == null ||
		ct.getLength() == 0) {
	  return null;
	}
	if (ct.getLength() != 1) {
	  throw new ConfSemanticException("only one 'convert_text' tag allowed");
	}
	NamedNodeMap attrs = ct.item(0).getAttributes();
	if (attrs == null || attrs.getLength() > 2) {
	  throw new ConfSemanticException("'first_space' and 'extra_space' attributes needed and authorized for 'convert_text' tag");
	}
	Node first_space = attrs.getNamedItem("first_space");
	if (first_space == null) {
	  throw new ConfSemanticException("no 'first_space' attribute declared for 'convert_text' tag");
	}
	Node extra_space = attrs.getNamedItem("extra_space");
	if (extra_space == null) {
	  throw new ConfSemanticException("no 'extra_space' attribute declared for 'convert_text' tag");
	}
	ConvertText result = new ConvertText(first_space.getNodeValue(), extra_space.getNodeValue());
	checkReplaces(result, (Element)ct.item(0));
	return result;
  }

  static public void dump(String indent, String name, String value) {
	System.out.println(indent+name+"=\""+value+"\"");
  }

  static public void dump(String indent, String name, int value) {
	System.out.println(indent+name+"="+value);
  }

  static public void dump(String indent, String name, List list) {
	System.out.println(indent+name+"="+list);
  }

  /** @return parent's html file of this html file if this parser is root parser */
  public HtmlFile getHtmlFile() {
  	return parent==null ? html_file : parent.getHtmlFile();
  }
  
  public ConvertText getConvertText() {
	return convert_text;
  }
  
  public void dump() {
	if ( !parsed ) {
	  System.err.println("<no parser: error detected during analysis or parse() not called yet>");
	  return;
	}

	getHtmlFile().dump();

	Set entries = items.entrySet();
	Object entries_tables[] = entries.toArray();
	for (int i = 0;
		 i < entries.size();
		 i++) {
	  Entry entry = (Entry)entries_tables[i];
	  Item item = (Item)entry.getValue();
	  item.dump();
	}

	convert_text.dump();
  }

  private void initRootVariables()
  {
    // filled with default content '#'
    variables.initVariable("CONSTANT"); //NO
    variables.initVariable("DOCUMENT_TAG"); //OK
    variables.initVariable("DOCUMENT_TITLE"); //OK
    variables.initVariable("SECTION_TAG"); //OK
    variables.initVariable("PARAGRAPH_ALIGN");
    variables.initVariable("TITLE_TAG"); //OK
    variables.initVariable("TITLE_LEVEL"); //OK
    variables.initVariable("TITLE_NUMBER"); //OK
    variables.initVariable("TITLE_FULL_NUMBER"); //OK
    variables.initVariable("IMAGE_NAME"); //OK
    variables.initVariable("LIST_TYPE"); //OK
    variables.initVariable("LINK_TARGET"); //OK
    variables.initVariable("LINK_ACTION"); //OK
    variables.initVariable("LINK_TARGET_DOCUMENT"); //NO
    variables.initVariable("LINK_TARGET_SECTION"); //NO
    variables.initVariable("LINK_TARGET_TAG"); //NO
    variables.initVariable("POPUP_LINK_METHOD"); //OK
    variables.initVariable("TAG_NAME"); //OK
    variables.initVariable("TABLE_WIDTH"); //OK
    variables.initVariable("TABLE_BORDER"); //OK
    variables.initVariable("TABLE_COLUMNS"); //OK
    variables.initVariable("TABLE_ROWS"); //OK
    variables.initVariable("ROW_CELLS"); //OK
    variables.initVariable("CELL_COLSPAN"); //OK
    variables.initVariable("CELL_ROWSPAN"); //OK
    variables.initVariable("CELL_COL"); //OK
    variables.initVariable("CELL_ROW"); //OK
    variables.initVariable("CELL_WIDTH"); //OK
    variables.initVariable("CELL_ALIGN"); //OK
    variables.initVariable("CELL_VALIGN"); //OK
    variables.initVariable("POPUP_TAG"); //OK
    variables.initVariable("INDEX_OF");
    variables.initVariable("INDEX_TYPE");
    variables.initVariable("INDEX_ENTRY_TYPE");
    variables.initVariable("INDEX_ENTRY_NUMBER");
    variables.initVariable("SUMMARY_ENTRY_TEXT");
    variables.initVariable("GRAPHIC_TITLE"); //OK
    variables.initVariable("GRAPHIC_WIDTH"); //OK
    variables.initVariable("GRAPHIC_HEIGHT"); //OK
    
    variables.initVariable("HTML_FILE_NAME"); //OK
    variables.initVariable("HTML_TITLE_LEVEL"); //OK
    variables.initVariable("HTML_LIST_TYPE"); //OK
    variables.initVariable("HTML_LINK"); //OK
    variables.initVariable("HTML_LINE_SIZE"); //OK
    variables.initVariable("HTML_TABLE_WIDTH"); //OK
    variables.initVariable("HTML_CELL_COLSPAN"); //OK
    variables.initVariable("HTML_CELL_ROWSPAN"); //OK
    variables.initVariable("HTML_CELL_WIDTH"); //OK
    variables.initVariable("HTML_BACK_COLOR"); //OK
    variables.initVariable("HTML_FORE_COLOR"); //OK
    variables.initVariable("HTML_BGCOLOR"); //OK
    variables.initVariable("HTML_COLOR"); //OK
    variables.initVariable("HTML_GRAPHIC_WIDTH"); //OK
    variables.initVariable("HTML_GRAPHIC_HEIGHT"); //OK
    variables.initVariable("HTML_GRAPHIC_FILE_NAME"); //OK
    variables.initVariable("HTML_GRAPHIC_APPLET_JAR_PREFIX"); //OK
    variables.initVariable("HTML_PNG_FILE_NAME");
    variables.initVariable("HTML_FONT_ATTRS_BEGIN"); //OK
    variables.initVariable("HTML_FONT_ATTRS_END"); //OK
  }
}

