/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DHtmlGraphicApplet.java,v 1.6 2008/05/23 14:11:49 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.drivers.html;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.eclipse.tptp.platform.report.chart.internal.ChartEncoderOutputStream;
import org.eclipse.tptp.platform.report.chart.internal.DefaultChartRenderData;
import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drivers.internal.ConfSemanticException;
import org.eclipse.tptp.platform.report.drivers.internal.Item;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlWriter;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.w3c.dom.Element;
import org.w3c.dom.Node;


/**
 * This class uses the JScrib DExtensible extension mechanism to generate DGraphic
 * items to html using an applet (jsgrpahicapplet.jar)
 * If it is not delivered with the DHtmlWriter class, a default implementation for DGraphic items
 * translation is taken: do nothing.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DHtmlGraphicApplet
{
	private static final String jarfilename = "jsgraphicapplet.jar";
	private static final int F_DIRECT=0; //generate data as string
	private static final int F_BEST=1;  //generate data as best format (less size)
	private static final int F_ZIP=2;  //generate data using zip format

	private static final int C_NEVER =0;  //never copy jar file to generation directory
	private static final int C_AUTO  =1;  //copy if needed
	private static final int C_ALWAYS=2;  //copy each time an applet is generated

	private DHtmlWriter driver;
	private ConfParser confparser;
	private int format_;
	private int copy_jar_file_; 
	private int encode_; 
	
	public DHtmlGraphicApplet()
	{
	  format_ = F_BEST;
	  copy_jar_file_ = C_AUTO;
	  encode_ = ChartEncoderOutputStream.E_FORJAVASCRIPT;
	}
	
	private static byte escape_double_quote_[] = new byte[]{ '&', '#', '3', '4', ';' };
	private static byte escape_ampersand_[] = new byte[]{ '&', '#', '3', '8', ';' };
	
	private void graphicDump( Item item, DGraphic graphic, int w, int h, float scale ) 
	{
	  final ByteArrayOutputStream bos = new ByteArrayOutputStream( 16384 );
	  DXmlWriter jsml = new DXmlWriter();
	  try {
	    
	    jsml.write( bos, graphic );
	    
	    //direct: must convert " to &#34; due to string embedded in html file.
	    String nozip =null;	    
	    if( format_!= F_ZIP )
	    {
	      byte data[] = bos.toByteArray();
	      int quote=0;
	      int len=data.length;
	      //
	      for( int i=0; i<len; i++) if(data[i]=='"'||data[i]=='&') quote++;
	      if( quote==0 )
	      {
	        nozip = new String( data );
	      } else {
	        byte cnv[] = new byte[len-quote+quote*5];
	        int k=0;
	        for( int i=0; i<len; i++ )
	        {
	          if( data[i]=='"' ) {
	            cnv[k++] = '&'; cnv[k++] = '#'; cnv[k++] = '3';cnv[k++] = '4';cnv[k++] = ';';
	          } else if( data[i]=='&' ) {
	            cnv[k++] = '&'; cnv[k++] = '#'; cnv[k++] = '3';cnv[k++] = '8';cnv[k++] = ';';
	          } else {
	            cnv[k++] = data[i];
	          }
	        }
	        nozip = new String( cnv );
	      }
	    }
	    
	    String zip = null;
	    if( format_ != F_DIRECT )
	    {
	      //try to zip ... 
	      ByteArrayOutputStream zbos = new ByteArrayOutputStream(2048);
	      ZipOutputStream zos = new ZipOutputStream( zbos );
	      zos.putNextEntry( new ZipEntry("jsml"));	  
	      zos.write( bos.toByteArray() );
	      zos.closeEntry();
	      zos.close();
	    
 	      final ByteArrayOutputStream cbos = new ByteArrayOutputStream( zbos.size() );
 	      //as zip is stored in a string in HTML document we must convert few byte of zip stream
 	      ChartEncoderOutputStream cos = new ChartEncoderOutputStream( cbos, encode_ );
 	      cos.write( zbos.toByteArray() );
	    
	      zip = cbos.toString("UTF-8");
	    }

        //generate applet parameter
	    switch( format_ )
	    {
	    case F_BEST:
	      if( nozip.length() <= zip.length() )
	      {
	        confparser.setVariableValue("HTML_APPLET_PARAM_NAME", "jsml", item); 
	        confparser.setVariableValue("HTML_APPLET_PARAM_VALUE", nozip, item);
	      } else {
	        confparser.setVariableValue("HTML_APPLET_PARAM_NAME", "jsml-zip", item);
	        confparser.setVariableValue("HTML_APPLET_PARAM_VALUE", zip, item);
	      }
	      break;
	    case F_ZIP:
	      confparser.setVariableValue("HTML_APPLET_PARAM_NAME", "jsml-zip", item);
	      confparser.setVariableValue("HTML_APPLET_PARAM_VALUE", zip, item);
	      break;
	    case F_DIRECT:
	      confparser.setVariableValue("HTML_APPLET_PARAM_NAME", "jsml", item);
	      confparser.setVariableValue("HTML_APPLET_PARAM_VALUE", nozip, item);
	      break; 
	    }
	  }
	  catch( Exception e ) 
	  {
	    e.printStackTrace();
	  }
	}
	
	public ConfParser getConfParser()
	{
	  return confparser;
	}

	private boolean initialized;
	
	private void initConfParser( ConfParser parent )
	{
	  if( initialized )
	  {
	    confparser.setParentConfParser( parent );
	    return ;
	  }
	  
	  confparser = new ConfParser("dhtmlgraphicapplet.xml", parent )
	  {
	    protected void checkItemChild( Item item, Node n ) throws ConfSemanticException
	    {
	      if( n.getNodeType() == Node.ELEMENT_NODE && "property".equals( n.getNodeName() ) ) //$NON-NLS-1$
	      {
	        Element e = (Element)n;
	        String name = e.getAttribute("name");
	        String value = e.getAttribute("value");
	        if( name==null ) 
	          throw new ConfSemanticException("missing 'name' attribute in 'property' element");
	        if( "data".equals(name) ) checkPropertyData( value );
	        else if( "copyjarfile".equals(name)) checkPropertyCopyJarFile( value );
	        else if( "encode".equals(name)) checkPropertyEncode( value );
	      }
	    }
	    private void  checkPropertyData( String value ) throws ConfSemanticException
	    {
	      if( value==null || "best".equals(value) )
	        format_ = F_BEST;
	      else if( "zip".equals(value) )
	        format_ = F_ZIP;
	      else if( "direct".equals(value) )
	        format_ = F_DIRECT;
	      else
	        throw new ConfSemanticException("Unknown '"+value+"' value for 'data' property");
	    }
	    private void  checkPropertyCopyJarFile( String value ) throws ConfSemanticException
	    {
	      if( value==null || "auto".equals(value) )
	        copy_jar_file_ = C_AUTO;
	      else if( "always".equals(value) )
	        copy_jar_file_ = C_ALWAYS;
	      else if( "never".equals(value) )
	        copy_jar_file_ = C_NEVER;
	      else
	        throw new ConfSemanticException("Unknown '"+value+"' value for 'copyjarfile' property");
	    }
	    private void  checkPropertyEncode( String value ) throws ConfSemanticException
	    {
	      if( value==null || "default".equals(value) )
	        encode_ = ChartEncoderOutputStream.E_DEFAULT;
	      else if( "forjavascript".equals(value) )
	        encode_ = ChartEncoderOutputStream.E_FORJAVASCRIPT;
	      else if( "all".equals(value) )
	        encode_ = ChartEncoderOutputStream.E_ALL;
	      else
	        throw new ConfSemanticException("Unknown '"+value+"' value for 'encode' property");
	    }
	  };
	  
	  //initialize specific variable for this piece of code
	  confparser.initVariable("HTML_APPLET_PARAM_NAME");
	  confparser.initVariable("HTML_APPLET_PARAM_VALUE");
	  
      if (!confparser.parse()) 
	  {
	     System.err.println("Graphic Applet Html Configuration parsing failed");
	     confparser=null;
	     initialized=false;
	     return;
	  }
      initialized=true;
	}
	
	public void doMethod(DGraphic item, DExtensible ext, Object arg) throws DHtmlWriterException
	{
	  driver = (DHtmlWriter)ext;
	  
	  //extends driver's configuration parser.
	  ConfParser parent_parser = driver.getConfparser();
	  initConfParser( parent_parser );
	  if( confparser==null ) return ; //error in configuration, no applet.
	  driver.setConfparser( confparser );
	  
	  Item hitem = driver.getItem(item);
	  driver.updateStyleVariables(item);
	  
	  int width = (int)(500*item.getWidthScale());
	  int height= (int)(500*item.getHeightScale());
      
      String title = DefaultChartRenderData.getResourceString(item.getTitle(), 
                (DI18N)item.getChildOfClass(DI18N.class));
	  
	  confparser.setVariableValue("GRAPHIC_TITLE", title, item);
	  confparser.setVariableValue("GRAPHIC_WIDTH", Float.toString(item.getWidthScale()), item);
	  confparser.setVariableValue("GRAPHIC_HEIGHT", Float.toString(item.getHeightScale()), item);
	  confparser.setVariableValue("HTML_GRAPHIC_WIDTH", Integer.toString(width), item);
	  confparser.setVariableValue("HTML_GRAPHIC_HEIGHT", Integer.toString(height), item);

	  try {
	    graphicDump( hitem, item, width,height, 1.0f );
	    driver.htmlDump(hitem.getBegin());
	    driver.htmlDump(hitem.getEnd());
	    confparser.unsetVariables(item);
	  }
	  catch( Exception  e )
	  {
	    e.printStackTrace();
	  }
	  //restore driver parser:
	  confparser.setParentConfParser( null );
	  driver.setConfparser( parent_parser );  
	  
	  //check for applet jar file in destination directory:
	  if( driver.getGenerationPolicy()==null ) return ;	    
	  checkForAppletJarFile( driver.getGenerationPolicy().getBaseDirectoryPath() );
	}

	private void checkForAppletJarFile( String dirname )
	{
	  //DHtmlWrite was not configured with ouput directory, so I can't copy .jar file.
	  if( dirname==null ) return ;
	  if( copy_jar_file_ == C_NEVER ) return ;
	  
	  String destname = dirname + File.separator + jarfilename;
	  
	  URL url = getClass().getResource( jarfilename );
	  if( url==null ) {
	    System.err.println("Warning: JScrib graphic applet jar file missed: "+jarfilename+", can't copy jar file to HTML output directory.");
	    return ;
	  }
	  
	  if( copy_jar_file_ == C_AUTO )
	  {
	    File src  = new File( url.getFile() );
	    File dest = new File( destname );
	    //assuming this is the right file!
	    if( dest.exists() && src.length()==dest.length() && src.lastModified() <= dest.lastModified() ) 
	    {
	      return ; 
	    }
	  }

	  //copy jar file to destination:
	  int SIZE=10000; 
	  InputStream is = getClass().getResourceAsStream( jarfilename );
	  if( is==null ) {
	    //but it's already checked, let this part of code for "assertion"
	    System.err.println("Internal Error: JScrib graphic applet jar file missed: "+jarfilename);
	    return ;
	  }
	  BufferedInputStream bis = new BufferedInputStream( is, SIZE );
	  FileOutputStream fos;
	  try {
	    fos = new FileOutputStream( destname );
	  } catch( FileNotFoundException e ) {
	    System.err.println("Can't write file:"+destname);
	    System.err.println("Reason: "+e.getMessage());
	    return ;
	  }
	  BufferedOutputStream bos = new BufferedOutputStream( fos );
       
	  byte buf[] = new byte[SIZE];
	  int len;
	  try{
	    while( (len=bis.read( buf ))>=0 )
	    {
	      bos.write( buf, 0, len );
	    }
	  } catch( IOException e ) {
	    System.err.println("Error: can't copy graphic applet jar file to "+dirname);
	    System.err.println("Reason: "+e.getMessage());
	    File f = new File( destname );
	    if( !f.delete() ) {
	      System.err.println("Can't remove file: "+destname);
	    }
	    return;
	  }
	}
}
