/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DHtmlWriter.java,v 1.14 2008/05/23 14:11:49 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on Sep 19, 2003
 */
package org.eclipse.tptp.platform.report.drivers.html;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.internal.DefaultChartRenderData;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DBorder;
import org.eclipse.tptp.platform.report.core.internal.DCell;
import org.eclipse.tptp.platform.report.core.internal.DCellText;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DCurveLink;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DFolder;
import org.eclipse.tptp.platform.report.core.internal.DFolderModel;
import org.eclipse.tptp.platform.report.core.internal.DFooter;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DHeader;
import org.eclipse.tptp.platform.report.core.internal.DI18N;
import org.eclipse.tptp.platform.report.core.internal.DImage;
import org.eclipse.tptp.platform.report.core.internal.DIndex;
import org.eclipse.tptp.platform.report.core.internal.DIndexEntry;
import org.eclipse.tptp.platform.report.core.internal.DLine;
import org.eclipse.tptp.platform.report.core.internal.DLink;
import org.eclipse.tptp.platform.report.core.internal.DLinkUtil;
import org.eclipse.tptp.platform.report.core.internal.DList;
import org.eclipse.tptp.platform.report.core.internal.DPageBreak;
import org.eclipse.tptp.platform.report.core.internal.DPageCounter;
import org.eclipse.tptp.platform.report.core.internal.DParagraph;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPointLink;
import org.eclipse.tptp.platform.report.core.internal.DPopup;
import org.eclipse.tptp.platform.report.core.internal.DRow;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DSummary;
import org.eclipse.tptp.platform.report.core.internal.DTable;
import org.eclipse.tptp.platform.report.core.internal.DTag;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.DTitle;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drivers.internal.HtmlFile;
import org.eclipse.tptp.platform.report.drivers.internal.HtmlVariable;
import org.eclipse.tptp.platform.report.drivers.internal.IWriter;
import org.eclipse.tptp.platform.report.drivers.internal.Item;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.tools.internal.DParser;
import org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider;
import org.eclipse.tptp.platform.report.tools.internal.IDProgressMonitor;


/**
 * This is the Html Generator driver.
 * See DHtmlGraphic and DHtmlGraphicApplet class for DGraphic items translation details.
 * It is a DParser, so it is extensible using the JScrib DExtensible extension mechanism.
 * Do not forgot to set image provider.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DHtmlWriter extends DParser implements IWriter 
{
    
    /** Constant to generate DGraphic item in an applet, this is the default graphic extension */
    public static final String GRAPHIC_APPLET="org.eclipse.tptp.platform.report.drivers.html.DHtmlGraphicApplet"; //$NON-NLS-1$
    /** Constant to generate DGraphic item in an image (require AWT classes available) */
    public static final String GRAPHIC_IMAGE_AWT="org.eclipse.tptp.platform.report.drivers.html.DHtmlGraphicImageAWT"; //$NON-NLS-1$
    /** Constant to generate DGraphic item in an image (require SWT classes available) */
    public static final String GRAPHIC_IMAGE_SWT="org.eclipse.tptp.platform.report.drivers.html.DHtmlGraphicImageSWT"; //$NON-NLS-1$
    /** Constant to generate DGraphic item in SVG format */
    public static final String GRAPHIC_SVG="org.eclipse.tptp.platform.report.drivers.html.DHtmlGraphicSVG"; //$NON-NLS-1$
  
	private ConfParser confparser;
	private PrintStream out;

	private String graphic_extension_classname_; //HTML extension used for DGraphic
	private Object graphic_extension_instance_; //HTML generation for DGraphic used
	
	private IDGenerationPolicy generationPolicy;
	
	private String title = "";
	
	private boolean verbose;
	private boolean html_header_generated;
	private LinkedList foot = new LinkedList();
	private IDProgressMonitor progressMonitor;
	private String basePath;
	private IDIImageProvider imageProvider;

	private static Item default_hitem = new Item();
	
	/**
	 * Creates a HTML JScrib Generator and reads the HTML file configuration.
	 * It tries to read the graphic extension generator module.
	 * @see DHtmlGraphic
	 */
	public DHtmlWriter() throws DHtmlWriterException
	{
		super();
		setConfparser(new ConfParser());
		if (!getConfparser().parse()) {
		    setConfparser(null);
			throw new DHtmlWriterException("Html Configuration parsing failed");
		}
		setGraphicExtensionClass( GRAPHIC_APPLET ); //by default even if SWT isn't there...
		html_header_generated = true;
	}
	
	/** @return true if HTML and BODY header/footer must be generated, in this case HTML writer generate
	 * a full HTML page. Otherwise only HTML subset that can be included in a page is generated 
	 * (ie HTML writer used in a Servlet).
	 */ 
	public boolean isHtmlHeaderGenerated() { return html_header_generated; }
	
	/** Change generation of HML/BODY element in stream, @see isHtmlHeaderGenerated() */
	public void setHtmlHeaderGenerated(boolean b) { html_header_generated=b; }
	
	/** 
	 * @return current type of generation used for DGraphic, can return one of constant defined in DHtmlWriter,
	 * or a user defined class. 
	 */
	public String getGraphicExtensionClass() { return graphic_extension_classname_; }
	
	
	/** 
	 * @return generation policy instance. 
	 */
	public IDGenerationPolicy getGenerationPolicy() { return generationPolicy; }
	
	
	/** 
	 * Change DGraphic generation mechanism, can be one of predefined constant:<br>
	 * - DHtmlWriter.GRAPHIC_IMAGE_SWT to generate graphic in a image using SWT (require a valid setDirname())<br>
	 * - DHtmlWriter.GRAPHIC_APPLET to generate graphic in a applet (require a valid setDirname() only if jar file should be copied)<br>
	 * or can be a valid class loaded using loadClass(), this class <b>must</b> define a method
	 * with the following signature:
	 * <pre>doMethod( DGraphic g, DExtensible ext, Object arg )</pre>
	 * DExtensible is an instance of DHtmlWriter.
	 */
	public void setGraphicExtensionClass( String graphic_type )  throws DHtmlWriterException
	{
	  //check if there are a "real" changement
	  if( graphic_extension_classname_!=null && graphic_extension_classname_.equals(graphic_type)) return ;
	  //remove current extension
	  if( graphic_extension_instance_!=null ) 
	  {
        removeDoMethods( graphic_extension_instance_ );
        graphic_extension_instance_ = null;
        graphic_extension_classname_ = null;
	  }
	  graphic_extension_classname_ = graphic_type;
	  //try to load new graphic extension: 
      try {
        Class clazz ;
        clazz = getClass().getClassLoader().loadClass( graphic_extension_classname_ );
        graphic_extension_instance_ = clazz.newInstance();
        installDoMethods(graphic_extension_instance_);
      } catch(ClassNotFoundException e) {
        if (isVerbose()) {
          throw new DHtmlWriterException("No Graphic to Html Translator loaded: Class not found '"+graphic_extension_classname_+"'");
        } 
      } catch (IllegalAccessException e) {
        e.printStackTrace();
      } catch (InstantiationException e) {
        e.printStackTrace();
      }
	}
	
	/** 
	 * Change the generation policy. This method is called by the method write(String generationPolicy, IDObject)
	 * @see #write(String, IDObject)
	 * @see #write(String, IDObject[])
	 */
	protected void setGenerationPolicy( IDGenerationPolicy policy ) throws DHtmlWriterException
	{ 
	  //check if there are a "real" changement
	  if( generationPolicy == policy) return ;
	  //remove current extension
	  if( generationPolicy !=null ) 
	  {
        removeDoMethods( generationPolicy );
        generationPolicy = null;
	  }
	  generationPolicy = policy;
	  installDoMethods(generationPolicy);
	  
	}

	/**
	 * Changes the configuration parser module. It is used to read the 
	 * configuration of the HTML Generator
	 */
	public void setConfparser(ConfParser confparser) {
		this.confparser = confparser;
	}

	/**
	 * Returns the configuration parser module.
	 * @see #setConfparser
	 */
	public ConfParser getConfparser() {
		return confparser;
	}

	/**
	 * De/Activate the message output during the generation. 
	 */
	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	/**
	 * Returns true if the message output is actvate
	 * @see #setVerbose
	 */
	public boolean isVerbose() {
		return verbose;
	}

	/**
	 * Sets the output stream
	 */
	public void setOutputStream(OutputStream output) {
	    out = new PrintStream(output);
	}
	
	/**
	 * Returns the output stream used to generate HTML
	 */
	public OutputStream getOutputStream() {
		return out;
	}

    /**
     * Converts and generates a String in HTML format.
     */
    public void htmlConvertDump(String string) {
        if (string==null) return;
        
        int space_count = 0;
        try {
            for (int i = 0; i < string.length(); i++) {
                char c = string.charAt(i);
                if (c == ' ') {
                    space_count++;
                    if (space_count == 1) {
                        out.write(getConfparser().getConvertText().getFirstSpace().getBytes("UTF-8"));
                    } else {
                        out.write(getConfparser().getConvertText().getExtraSpace().getBytes("UTF-8"));
                    }
                }
                else if (c == '\n')
                    out.println("<BR>");
                else {
                    space_count = 0;
                    Character ch = new Character(c);
                    String repl = getConfparser().getConvertText().getReplace(ch);
                    String cstr = repl != null ? repl : ch.toString();
                    out.write(cstr.getBytes("UTF-8"));
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
            return;
        }
    }

    /**
     * Generates a String in HTML format.
     */
    public void htmlDump(String string) {
        try {
          byte b[] = string.getBytes("UTF-8"); //$NON-NLS-1$
       
          for (int i = 0; i < b.length; i++) {
            char c = (char)b[i];
            if (c == '\n') 
                    out.println();
            else
                out.write(b[i]); }//print(c); }
        }
        catch (UnsupportedEncodingException e)
        {
            e.printStackTrace();
        }
    }

	/**
	 * Generates an int value in HTML format.
	 */
	public void htmlDump(int value) {
	    htmlDump(Integer.toString(value));
		/*try {
			out.write(Integer.toString(value).getBytes("UTF-8"));
		} catch (IOException e) {
			e.printStackTrace();
		}*/
	}

	/**
	 * Generates a LinkedList in HTML format.
	 */
	public void htmlDump(List list) {
		if (list == null) {
			return;
		}
		for (Iterator I=list.iterator(); I.hasNext(); )
		{
            Object obj = I.next();
            if( obj instanceof String )
            {
              htmlDump( (String)obj );
            } else {
              //variable definition:
              HtmlVariable var = (HtmlVariable)obj;
              String value = getConfparser().getVariableValue( var.getName() );
              if ( "#".equals( value  ) ) {
                htmlDump("${" + var.getName() + "}");
              } else {
                htmlDump(value != null ? value : var.getName() );
              }
            }
		}
	}

	
	private void setTagVariable(String name, String tag, IDItem item) {
		getConfparser().setVariableValue(name, tag != null ? tag : "_" + item.hashCode(), item);
	}

	private String to2DigitsHexa(int color) {
		return Integer.toHexString((color % 256) / 16) + Integer.toHexString(color % 16);
	}

	private String getColorHexa(IDColor color, String default_value) {
		if (color == null) {
			return default_value;
		}
		String result = "#";
		result += to2DigitsHexa(color.getRed());
		result += to2DigitsHexa(color.getGreen());
		result += to2DigitsHexa(color.getBlue());
		return result;
	}

	private String getFontAttrsBegin(IDStyle style) {
		IDFont font = style.getFont();
		IDColor color = style.getForeColor();
		if (font == null && color == null) {
			return "";
		}
		String result = "";
		if (font != null) {
			if (font.haveStyle(IDFont.BOLD)) {
				result += "<B>";
			}
			if (font.haveStyle( IDFont.UNDERLINE|IDFont.DASHED)) {
				result += "<U>";
			}
			if (font.haveStyle(IDFont.ITALIC)) {
				result += "<I>";
			}
		}
		result += "<FONT COLOR=" + getColorHexa(color, "#000000");
		if (font != null) {
			if (font.getFamily() != null) {
				result += " FACE=\"" + font.getFamily() + "\"";
			}
			if (font.getSize() != -1) {
				int s = font.getSize() / 3; // size 12 -> 4 (normal size)
				if (s < 1) {
					s = 1; // 1 is the minimum
				}
				if (s > 7) {
					s = 7; // 7 is the maximum
				}
				result += " SIZE=\"" + Integer.toString(s) + "\"";
			}
		}
		result += ">";
		if (font != null && getConfparser().isTTFont(font.getFamily())) {
			result += "<TT>";
		}

		return result;
	}

	private String getFontAttrsEnd(IDStyle style) {
		IDFont font = style.getFont();
		IDColor color = style.getForeColor();
		if (font == null && color == null) {
			return "";
		}
		String result = "";
		if (font != null && getConfparser().isTTFont(font.getFamily())) {
			result += "</TT>";
		}
		result += "</FONT>";
		if (font != null) {
            if (font.haveStyle( IDFont.ITALIC )) {
                result += "</I>";
            }
            if (font.haveStyle( IDFont.UNDERLINE|IDFont.DASHED ) ) {
                result += "</U>";
            }
			if (font.haveStyle(IDFont.BOLD)) {
				result += "</B>";
			}
		}
		return result;
	}

	public void updateStyleVariables(IDItem item) {
		if (item.getStyle() != null) {
			getConfparser().setVariableValue("HTML_FONT_ATTRS_BEGIN", getFontAttrsBegin(item.getStyle()), item);
			getConfparser().setVariableValue("HTML_FONT_ATTRS_END", getFontAttrsEnd(item.getStyle()), item);
			String fore_color = getColorHexa(item.getStyle().getForeColor(), "#000000");
			getConfparser().setVariableValue("HTML_COLOR", "=\"" + fore_color + "\"", item);
			getConfparser().setVariableValue("HTML_FORE_COLOR", fore_color, item);
			String back_color = getColorHexa(item.getStyle().getBackColor(), "#ffffff");
			getConfparser().setVariableValue("HTML_BGCOLOR", "=\"" + back_color + "\"", item);
			getConfparser().setVariableValue("HTML_BACK_COLOR", back_color, item);
		}
	}

	/**
	 * Returns the generation rules for an IDItem.
	 */
	public Item getItem(IDItem item) throws DHtmlWriterException {
		// Perform some checks
		Item hitem = getConfparser().getItem(item.getClass());
		if (hitem == null) {
			hitem = default_hitem;
			throw new DHtmlWriterException("No html parser found for " + item.getClass());
		}
		return hitem;
	}

	public void doChildrenItem(IDItem item, Item hitem, DExtensible ext, Object arg) 
	{
		for (IDItem i = item.getFirstChild(); i != null; i = i.getNext())
		{
			htmlDump(hitem.getBeginChild());
			invokeDoMethod(i,ext,arg);
			htmlDump(hitem.getEndChild());
		}
		    
	}

	/**
	 * Generates HTML tags for DDocument.
	 */
	public void doMethod(DDocument item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		setTagVariable("DOCUMENT_TAG", item.getTag(), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DSection.
	 */
	public void doMethod(DSection item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		setTagVariable("SECTION_TAG", item.getTag(), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Do nothing, ignored
	 */
	public void doMethod(DHeader item, DExtensible ext, Object arg) {
		// ignored
	}

	/**
	 * Do nothing, ignored
	 */
	public void doMethod(DFooter item, DExtensible ext, Object arg) {
		// ignored
	}

	/**
	 * Generates HTML tags for DParagraph.
	 */
	public void doMethod(DParagraph item, DExtensible ext, Object arg) throws DHtmlWriterException {
	    
	    Item hitem = getItem(item);
	    if ((item.getParent() instanceof DFolder)||(item.getParent() instanceof DList))
	    {
	        doChildrenItem(item, hitem,ext,arg);
	    }
	    else
	    {
		   updateStyleVariables(item);
           String sa="LEFT";
           if( item.haveAlignment( DParagraph.HCENTER )) sa = "CENTER";
           else if( item.haveAlignment( DParagraph.RIGHT )) sa = "RIGHT";
		   getConfparser().setVariableValue("PARAGRAPH_ALIGN", sa, item);
		   htmlDump(hitem.getBegin());
		   doChildrenItem(item, hitem,ext,arg);
		   htmlDump(hitem.getEnd());
		   getConfparser().unsetVariables(item);
	    }
	}

	/**
	 * Generates HTML tags for DTag.
	 */
	public void doMethod(DTag item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		setTagVariable("TAG_NAME", item.getTag(), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DText.
	 */
	public void doMethod(DText item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		htmlDump(hitem.getBegin());
		htmlConvertDump(item.getText());
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DImage.
	 */
	public void doMethod(DImage item, DExtensible ext, Object arg) throws DHtmlWriterException {
	    
	    String urlimg = generationPolicy.getImageUrl(item);
	    if (urlimg!=null)
	    {	    
		   Item hitem = getItem(item);
		   updateStyleVariables(item);
		   getConfparser().setVariableValue("IMAGE_NAME", generationPolicy.getImageUrl(item), item);
		   htmlDump(hitem.getBegin());
		   // no childs
		   htmlDump(hitem.getEnd());
		   getConfparser().unsetVariables(item);
	    }
	}

	/**
	 * Generates HTML tags for DList.
	 */
	public void doMethod(DList item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		if (item.getFormat() == DStyle.GLYPH) {
			getConfparser().setVariableValue("LIST_TYPE", "BULLETED", item);
			getConfparser().setVariableValue("HTML_LIST_TYPE", "UL", item);
		} else if (item.getFormat() == DStyle.NUMBER) {
			getConfparser().setVariableValue("LIST_TYPE", "NUMBERED", item);
			getConfparser().setVariableValue("HTML_LIST_TYPE", "OL", item);
		} else {
			getConfparser().setVariableValue("LIST_TYPE", "", item);
			getConfparser().setVariableValue("HTML_LIST_TYPE", "UL", item);
		}
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DTable.
	 */
	public void doMethod(DTable item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		
		float w = 0;
		float wc[] = item.getColumnWidth();
		if (wc == null || wc.length < item.getColumnCount()) {
				w = item.getWidthScale();
		} else {
				for (int i = 0; i < wc.length; i++) {
					w += wc[i];
				}
		}
		getConfparser().setVariableValue("TABLE_WIDTH", Float.toString(w), item);
		getConfparser().setVariableValue("HTML_TABLE_WIDTH", Integer.toString((int) (w * 100.0f)), item);
		
		getConfparser().setVariableValue("TABLE_BORDER", Integer.toString(item.getBorder()), item);
		getConfparser().setVariableValue("TABLE_COLUMNS", Integer.toString(item.getColumnCount()), item);
		getConfparser().setVariableValue("TABLE_ROWS", Integer.toString(item.getRowCount()), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DRow.
	 */
	public void doMethod(DRow item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		getConfparser().setVariableValue("ROW_CELLS", Integer.toString(item.getCellCount()), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	private void setCellVariables(DCell item){
		updateStyleVariables(item);
		
		getConfparser().setVariableValue("CELL_COLSPAN", Integer.toString(item.getColSpan()), item); 
		getConfparser().setVariableValue("HTML_CELL_COLSPAN", Integer.toString(item.getColSpan() + 1), item);
		
		getConfparser().setVariableValue("CELL_ROWSPAN", Integer.toString(item.getRowSpan()), item);
		getConfparser().setVariableValue("HTML_CELL_ROWSPAN", Integer.toString(item.getRowSpan() + 1), item);
		
		getConfparser().setVariableValue("CELL_COL", Integer.toString(item.getColumnNumber()), item);
		getConfparser().setVariableValue("CELL_ROW", Integer.toString(item.getLineNumber()), item);
		DRow row = (DRow) item.getParent();
        if (row==null) return;
		DTable table = (DTable) row.getParent();
		if (table.getColumnWidth() != null && table.getColumnCount() > item.getColumnNumber()) {
			float w = table.getColumnWidth()[item.getColumnNumber()];
			getConfparser().setVariableValue("CELL_WIDTH", Float.toString(w), item);
			getConfparser().setVariableValue("HTML_CELL_WIDTH", Integer.toString((int) (w * 100.0f)), item);
		}
        else
        {
            getConfparser().setVariableValue("CELL_WIDTH", Float.toString(100/table.getColumnCount()), item);
            getConfparser().setVariableValue("HTML_CELL_WIDTH", Integer.toString((int) (100/table.getColumnCount())), item);
        }
		
		// horizontal cell alignment
		String ha = "LEFT";
		if( item.haveAlignment( DParagraph.HCENTER )) ha = "CENTER";
        else if( item.haveAlignment( DParagraph.RIGHT )) ha = "RIGHT";
        getConfparser().setVariableValue("CELL_ALIGN", ha, item);
		
		// vertical cell alignment
		String va = "MIDDLE";
		if( item.haveAlignment( DParagraph.TOP )) va = "TOP";
        else if( item.haveAlignment( DParagraph.BOTTOM )) va = "BOTTOM";
		getConfparser().setVariableValue("CELL_VALIGN", va, item);
	}

	/**
	 * Generates HTML tags for DCell.
	 */
	public void doMethod(DCell item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		setCellVariables(item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DCellText.
	 */
	public void doMethod(DCellText item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		setCellVariables(item);
		htmlDump(hitem.getBegin());
		String text = item.getText();
		if (text.length() == 0 || text.equals(" ")) {
			text = "  ";
		}
		htmlConvertDump(text);
		// no childs
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DPopup. Called by doMethod(Dlink,..
	 */
	public void doMethod(DPopup item, DExtensible ext, Object arg) throws DHtmlWriterException {

	    if (arg==null)
	       foot.add(item);
	    else
	    {
		   // This one is not extensible yet...
		   Item hitem = getItem(item);
		   updateStyleVariables(item);
		   setTagVariable("POPUP_TAG", item.getTag(), item);
		   htmlDump(hitem.getBegin());
		   doChildrenItem(item, hitem,ext,arg);
		   htmlDump(hitem.getEnd());
		   getConfparser().unsetVariables(item);
	    }
	}

	/**
	 * Generates HTML tags for DLink.
	 */
	public void doMethod(DLink item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		
		String url = generationPolicy.getLinkUrl(item);
		getConfparser().setVariableValue("HTML_LINK", url, item);
        getConfparser().setVariableValue("POPUP_LINK_METHOD", DLinkUtil.getMethod(item), item);
        
        String action="";
        if (DLinkUtil.isPopupLink(item)) 
            action = "onclick=\"return popup(this,'')\"";
        getConfparser().setVariableValue("LINK_ACTION", action, item);
        
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
	
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
		
  }

	/**
	 * Generates HTML tags for DLine.
	 */
	public void doMethod(DLine item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		getConfparser().setVariableValue("HTML_LINE_SIZE", Integer.toString(item.getSize()), item);
		htmlDump(hitem.getBegin());
		// no childs
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * do noting, ignored
	 */
	public void doMethod(DPageBreak item, DExtensible ext, Object arg) {
		// ignored
	}

	/**
	 * do nothing, ignored
	 */
	public void doMethod(DPageCounter item, DExtensible ext, Object arg) {
		// ignored
	}

	/**
	 * Generates HTML tags for DSummary.
	 */
	public void doMethod(DSummary item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DTitle.
	 */
	public void doMethod(DTitle item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		setTagVariable("TITLE_TAG", item.getTag(), item);
		getTitleLevel().increase(((DTitle) item).getLevel());
		getConfparser().setVariableValue("TITLE_LEVEL", Integer.toString(getTitleLevel().getLevel()), item);
		getConfparser().setVariableValue("HTML_TITLE_LEVEL", Integer.toString(getTitleLevel().getLevel()), item);
		getConfparser().setVariableValue("TITLE_FULL_NUMBER", getTitleLevel().toFormatString(0), item);
		getConfparser().setVariableValue("TITLE_NUMBER", Integer.toString(getTitleLevel().getCurrentEntryNumber()), item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * do nothing, ignored
	 */
	public void doMethod(DIndex item) {
	}

	/**
	 * Generates HTML tags for DIndexEntry.
	 */
	public void doMethod(DIndexEntry item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
		htmlDump(hitem.getBegin());
		doChildrenItem(item, hitem,ext,arg);
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * Generates HTML tags for DGraphic.
	 */
	public void doMethod(DGraphic item, DExtensible ext, Object arg) throws DHtmlWriterException {
		Item hitem = getItem(item);
		updateStyleVariables(item);
        
        String title = DefaultChartRenderData.getResourceString(item.getTitle(), 
                  (DI18N)item.getChildOfClass(DI18N.class));
		getConfparser().setVariableValue("GRAPHIC_TITLE", title, item);
		getConfparser().setVariableValue("GRAPHIC_WIDTH", Float.toString(item.getWidthScale()), item);
		getConfparser().setVariableValue("HTML_GRAPHIC_WIDTH", Integer.toString((int) (item.getWidthScale() * 100.0f)), item);
		getConfparser().setVariableValue("GRAPHIC_HEIGHT", Float.toString(item.getHeightScale()), item);
		getConfparser().setVariableValue("HTML_GRAPHIC_FILE_NAME", "img" + Integer.toHexString(item.hashCode()), item);
		htmlDump(hitem.getBegin());
		htmlDump(hitem.getEnd());
		getConfparser().unsetVariables(item);
	}

	/**
	 * already made from the enclosing DGraphic
	 */
	public void doMethod(DAxis item, DExtensible ext, Object arg) {
		// already made from the enclosing DGraphic
	}

	/**
	 * already made from the enclosing DGraphic
	 */
	public void doMethod(DCurve item, DExtensible ext, Object arg) {
		// already made from the enclosing DGraphic
	}

	/**
	 * already made from the enclosing DGraphic
	 */
    public void doMethod(DCurveLink item, DExtensible ext, Object arg) {
      // already made from the enclosing DGraphic
    }

    /**
     * already made from the enclosing DGraphic
     */
	public void doMethod(DPoint item, DExtensible ext, Object arg) {
		// already made from the enclosing DGraphic
	}

	/**
	 * already made from the enclosing DGraphic
	 */
    public void doMethod(DPointLink item, DExtensible ext, Object arg) {
      // already made from the enclosing DGraphic
    }
    
    
    /**
     * Generates HTML for an IDObject array in the OutputStream output and uses
     * a DOneDocumentPolicy automatically to generate attachments
     * 
     * @see DOneDocumentPolicy
     */
	public void write(OutputStream output, IDObject[] objects) throws DHtmlWriterException
	{
	    if (getConfparser() == null) 
		    throw new DHtmlWriterException("Html Configuration parsing failed");

		out = new PrintStream(output);
		
		setGenerationPolicy(new DOneDocumentPolicy(this, getDirname(), getImageProvider()));
		generationPolicy.start();

		// Some variables at document level
		
		getConfparser().setVariableValue("DOCUMENT_TITLE", getTitle(), objects[0]);
		getConfparser().setVariableValue("HTML_BGCOLOR", "=\"#ffffff\"", objects[0]);

		getTitleLevel().setMax(getConfparser().getHtmlFile().getMaxTitleLevel() - 1);
		HtmlFile hf = getConfparser().getHtmlFile();
		
		if(html_header_generated) htmlDump(hf.getBegin());
		
	    for (int i=0; i<objects.length; i++)
		   invokeDoMethod(objects[i],this,null);
		
		Iterator it = foot.iterator();
		while (it.hasNext())
		{
		    htmlDump("<HR>");
		    Object o = it.next();
		    invokeDoMethod(o, this, o);
		}
		
		if(html_header_generated) htmlDump(hf.getEnd());
		
		out.close();
	}

	/**
	 * Generates HTML for an IDObject in the OutputStream output and uses
     * a DOneDocumentPolicy automatically to generate attachments
     * 
     * @see DOneDocumentPolicy
	 */
	public void write(OutputStream output, IDObject object) throws DHtmlWriterException
	{
		if (getConfparser() == null) 
		    throw new DHtmlWriterException("Html Configuration parsing failed");

		out = new PrintStream(output);
				
		setGenerationPolicy(new DOneDocumentPolicy(this, getDirname(), getImageProvider()));
		generationPolicy.start();
		
		getConfparser().setVariableValue("DOCUMENT_TITLE", getTitle(), object);
		getConfparser().setVariableValue("HTML_BGCOLOR", "=\"#ffffff\"", object);

		getTitleLevel().setMax(getConfparser().getHtmlFile().getMaxTitleLevel() - 1);
		HtmlFile hf = getConfparser().getHtmlFile();
		
		if(html_header_generated) htmlDump(hf.getBegin());
		
		invokeDoMethod(object,this,null);
		
		Iterator it = foot.iterator();
		while (it.hasNext())
		{
		    htmlDump("<HR>");
		    Object o = it.next();
		    invokeDoMethod(o, this, o);
		}
		
		if(html_header_generated) htmlDump(hf.getEnd());
		
		out.close();
	}
	
	/**
	 * Generate the HTML with the current generation policy.
	 * By default this policy is DHtmlWriter.MULTI_FILE_POLICY.
	 * To change the generation policy calls setGenerationPolicyClass before.
	 * @see #setGenerationPolicy(String)
	 */
	public void write(IDGenerationPolicy policy, IDObject[] objects) throws  DHtmlWriterException
	{
	    if (getConfparser() == null) 
		    throw new DHtmlWriterException("Html Configuration parsing failed");

	    setGenerationPolicy(policy);
	    
	    policy.start();
	    for (int i=0; i<objects.length; i++)
		   invokeDoMethod(objects[i],this,null);
	}
	
	/**
	 * Generate the HTML with the current generation policy.
	 * By default this policy is DHtmlWriter.MULTI_FILE_POLICY.
	 * To change the generation policy calls setGenerationPolicyClass before.
	 * @see #setGenerationPolicy(String)
	 */
	public void write(IDGenerationPolicy policy, IDObject object) throws  DHtmlWriterException
	{
	    if (getConfparser() == null) 
		    throw new DHtmlWriterException("Html Configuration parsing failed");

	    setGenerationPolicy(policy);
	    
	    policy.start();
		invokeDoMethod(object,this,null);
	}
	
	/**
	 * Clears all data allocated.
	 */
	public void clear() {
		super.clear();
	}
	
	public void doMethod( DFolder folder, DExtensible ext, Object arg )
	{
	  DHtmlWriter w = (DHtmlWriter)ext;
	  
	  DFolderModel model = folder.getFolderModel();
	  if( model==null ) model = DFolderModel.GetDefault();
	  
//TODO: generation using HTML configuration file ....      
	  //symbol is generated as LI, UL is added to support folder indentation.
	  if( model.isIndented()) w.htmlDump("<UL>");
	  w.htmlDump("<LI>");
	  
	  //generate header and body:
	  IDItem item = folder.getFirstChild();
	  if( item!=null )
	  {
	    // w.htmlDump("<BR>"); LI does the work
	    //header:
	    ext.invokeDoMethod( item, ext, arg );
	    //force a break line as folder header doesn't require '\n' at header's end.
	    w.htmlDump("<BR>\n");
	    
	    //body:
	    for( item=item.getNext(); item!=null; item=item.getNext() )
	    {
	      ext.invokeDoMethod( item, ext, arg );
	    }
	  }
	  
	  w.htmlDump( "</LI>\n");
	  if( model.isIndented() ) w.htmlDump("</UL>");
	}
	
   /** Write HTML for DBorder */
   public void doMethod( DBorder border, DExtensible ext, Object arg )
   {
      DHtmlWriter w = (DHtmlWriter)ext;
      
      if( border.getLineType()==DBorder.L_NONE )
      {
        ((DHtmlWriter)ext).doChildrenItem( border, ext ,arg );
        return ;
      }
      
      String sid = Integer.toString( border.hashCode(), 16 );
      
      boolean border_contains_only_text_and_image=true;
      for( IDItem item=border.getFirstChild(); item!=null; item=item.getNext() )
      {
        if( !(item instanceof DText) && !(item instanceof DImage))
        {
          border_contains_only_text_and_image = false;
          break;
        }
      }
      
      //border color is from border foreground     
      int bc = IGCDStyle.GetForeColor( border );
      String sc = Integer.toString( (bc>>8), 16 );
      while( sc.length()<6 ) sc = "0"+sc;
      
      if( border_contains_only_text_and_image )
      {
        //please note that I don't know how to generate shadow ...
        //the only thing I try to generate using CSS is the border
        //Note: Opera 7.64/window browser doesn't print table border nor CSS border generate here.
      
        w.htmlDump("\n<STYLE type=\"text/css\"><!--");
        w.htmlDump(".styDBorder");
        w.htmlDump(sid);
        w.htmlDump(" {border: ");
        switch( border.getLineType() )
        {
        //case DBorder.L_NONE : break; //already done
        default:
        case DBorder.L_SIMPLE:
          w.htmlDump(" solid 1px "); break;
        case DBorder.L_BOLD:
          w.htmlDump(" solid 2px "); break;
        case DBorder.L_HEAVY:
          w.htmlDump(" solid 3px "); break;
        case DBorder.L_DOUBLE:
          w.htmlDump(" double 3px "); break;
        case DBorder.L_BOLD_SIMPLE:
        case DBorder.L_SIMPLE_BOLD:
          w.htmlDump(" double 5px "); break;
        case DBorder.L_TRIPLE:
          w.htmlDump(" ridge 3px "); break; //ok, far from triple border
        }
        w.htmlDump("#");
        w.htmlDump(sc);      
        w.htmlDump("}--></STYLE>\n");
        w.htmlDump("<FONT class=\"styDBorder");
        w.htmlDump(sid);
        w.htmlDump("\">");
        ((DHtmlWriter)ext).doChildrenItem( border, ext, arg );      
        w.htmlDump("</FONT>");
      }
      else
      {
        //try to generate a TABLE, no CSS style
        int table_border=0;
        int cell_padding=0;
        switch( border.getLineType() )
        {
        //case DBorder.L_NONE : break; //already done
        default:
        case DBorder.L_SIMPLE: table_border=1; break;
        case DBorder.L_BOLD: table_border=2; break;
        case DBorder.L_HEAVY: table_border=3; break;
        case DBorder.L_DOUBLE: table_border=1; cell_padding=1; break;
        case DBorder.L_BOLD_SIMPLE:
        case DBorder.L_SIMPLE_BOLD: table_border=2; cell_padding=1; break;
        case DBorder.L_TRIPLE: table_border=2; cell_padding=3; break;
        }
        
        //w.htmlDump("<TABLE width=\"100%\" BORDER=\""+table_border+"\" cellpadding=\""+cell_padding+"\" FGCOLOR=\"#");
        w.htmlDump("<TABLE BORDER=\""+table_border+"\" cellpadding=\""+cell_padding+"\" FGCOLOR=\"#");
        w.htmlDump(sc);
        w.htmlDump("\"><TR><TD>");
        ((DHtmlWriter)ext).doChildrenItem( border, ext, arg );      
        w.htmlDump("</TD></TD></TABLE>");        
      }
   }
   
    /** @return the title of the HTML. By default returns empty string */ 
    public String getTitle() {
        return title;
    }
    
    /** Sets the html title */
    public void setTitle(String title) {
        this.title = title;
    }
    
    /** @return the progress monitor */
    public IDProgressMonitor getProgressMonitor() {
        return progressMonitor;
    }
    
    /** Sets the progress monitor used by the html driver */
    public void setProgressMonitor(IDProgressMonitor progressMonitor) {
        this.progressMonitor = progressMonitor;
    }
    
    /** @return the destination directory for HTML files. */
    public String getDirname() {
        return basePath;
    }
    
    /** sets the destination directory for HTML files. */
    public void setDirname(String basePath) {
        this.basePath = basePath;
    }
    
    /** @return the image provider set for the driver */
    public IDIImageProvider getImageProvider() {
        return imageProvider;
    }
    
    /** sets the image provider  */
    public void setImageProvider(IDIImageProvider imageProvider) {
        this.imageProvider = imageProvider;
    }
    
    /** generates the documents in only one HTML file by using DOneDocumentPolicy
     * and a directory named <html filename>_files to store images and charts into it.
     * @param  html_filepath is the path to the html file which is generated. The parent's 
     * file path is used as HTML target directory */
    static public void write(String html_filepath, IDObject[] docs,  
                             String graphic_extension, IDIImageProvider imgp) 
                                  throws DHtmlWriterException
    {
        try {
  	        DHtmlWriter writer = new DHtmlWriter();
  	        writer.setGraphicExtensionClass(graphic_extension);
  	        File f = new File(html_filepath);
  	        DOneDocumentPolicy policy = new DOneDocumentPolicy(writer, f.getParent(), imgp);
  	        policy.setImageDirectoryName(f.getName()+"_files");
  	        writer.setOutputStream(new FileOutputStream(html_filepath));
  	        
  	        writer.write(policy, docs);
  	     }
  	     catch (DHtmlWriterException e)
  	     {
  	       throw new DHtmlWriterException(html_filepath+"\n\nError When Generating HTML file:"+e.getMessage());
  	     }
  	     catch (FileNotFoundException e)
  	     {
  	       throw new DHtmlWriterException(html_filepath+"\n\nError When Generating HTML file:\n"+e.getMessage());
  	     }
    }
      
}
