/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DXmlDocumentReader.java,v 1.7 2008/05/23 14:12:01 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.drivers.xml.internal;

import java.io.InputStream;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.tptp.platform.report.ReportCorePlugin;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCategory;
import org.eclipse.tptp.platform.report.core.internal.DColor;
import org.eclipse.tptp.platform.report.core.internal.DColorRegistry;
import org.eclipse.tptp.platform.report.core.internal.DCoord;
import org.eclipse.tptp.platform.report.core.internal.DCoordObject;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DFontRegistry;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DMarkerLine;
import org.eclipse.tptp.platform.report.core.internal.DMarkerRegion;
import org.eclipse.tptp.platform.report.core.internal.DPalettes;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DStyleRegistry;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDCoord;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.core.internal.IDStringSerializable;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.core.internal.IDocumentReader;
import org.eclipse.tptp.platform.report.drivers.internal.IReader;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.extension.internal.DExtensionRegistry;
import org.eclipse.tptp.platform.report.extension.internal.IDExtension;
import org.eclipse.tptp.platform.report.tools.internal.CalendarField;
import org.eclipse.tptp.platform.report.tools.internal.DAlignmentPair;
import org.eclipse.tptp.platform.report.tools.internal.DParser;
import org.eclipse.xsd.XSDDiagnostic;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.SimpleDateFormat;

	  
/**
 * DXmlReader is an extensible class that parse an InputStream and a XML stream
 * to build a jscrib IDItem tree and associated registries (colors,fonts and styles).<br>
 * The following example reads a document from the xml file named 'myfile.rxml'.<p>
 * <PRE>
 *    DXmlReader reader = new DXmlReader();
 *    try{
 *      IDObject [] docs = reader.read( new FileInputStream("rxml") );
 *      // returns only the first document. 
 *      if (docs.lenght>0 && (docs[0] instanceof DDocument)
 *          return (DDocument)docs[0];
 *    }
 *    catch(Exception e)
 *    {
 *      // an error occurs during the file parsing. 
 *      ...     
 *    }
 * 
 * </PRE> 
 * 
 * @see #read
 * @see DExtensible 
 * @see IReader
 * @see InputStream
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DXmlDocumentReader extends DParser implements IReader, IDocumentReader
{	  
  /** 
   * Argument class for doMethod(), and instance of Arg class will be given as the third parameter.
   * If you aren't aware of doMethod(), please refer to DExtension and DExtensible classes.
   */
  public static class Arg
  {
    /** current node that engage doMethod() call */
    public Node node; 

    /** 
     * Do not parse this children element name, this is set by doMethod() and
     * used in doChildrenItem().
     * @see org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlDocumentReader#doMethod(DSection,DExtensible,Object) for an example.
     */
    public String filter_children[];
    
    /**
     * id returned by DColor or DFont have parsed, if they want to be integrated in registry.
     * (id must be >=0), as a String like getNodeValue() return.
     */
    public String ret_id ;
    
    /** return value */
    public Object return_value;
    
    /** parent IDItem */
    public IDItem parent;
    
    public Arg( Node n,  IDItem parent) { node=n; ret_id=null; this.parent=parent; }
    
    /** @return true if node_name is in filtered children list */
    public boolean filterChildren( String node_name )
    {
      if( filter_children == null ) return false;
      for( int i=0; i<filter_children.length; ++i )
      {
        String s = filter_children[i];
        if( s==null ) continue;
        if( s.equals( node_name ) ) return true;
      }
      return false;
    }
  }

	private DStyleRegistry styles;
	private DColorRegistry colors;
	private DFontRegistry  fonts;
	private IDObject roots[];

    /** current map of IDFont using their id as key. */
	private HashMap fontsId;
    /** current map of IDColor using their id as key. */
	private HashMap colorsId; //
    /** current map of IDStyle using their id as key. */
    private HashMap stylesId; //
    /** current map of DAxis using their names as key. */
	private HashMap axis;
    /** Xml parser */
	private IXmlParser parser_;
 
	
	/**
     * Create a DXmlReader, ready to parse input @see read(InputStream)
	 */
	public DXmlDocumentReader( IXmlParser parser ) 
	{
	  parser_ = parser;
      fontsId  = new HashMap();
      colorsId = new HashMap();
      stylesId = new HashMap();
      axis     = new HashMap();
      DExtensionRegistry.sgn_addExtension.connect( this, "updateExtension(org.eclipse.tptp.platform.report.extension.internal.IDExtension, Class)");
      DExtensionRegistry.sgn_removeExtension.connect( this, "updateExtension(org.eclipse.tptp.platform.report.extension.internal.IDExtension, Class)");
	}
		
	/** @return the IDColor with is stored in the node. The node value can be
	 * a color id or a color defined with the format #RRGGBBAA
	 */
    private IDColor getColor(String val, String nodeName)
	{   
        DColor color = null;
        if ((val.charAt(0)!='#') && (val.length()<7))
            throw new DXmlError("Bad syntax color ("+val+") while parsing '" + nodeName +"' tag");
        
        color = (DColor)colorsId.get(val);
        if (color==null)
        {
           // color with format #RRGGBBAA
           color = new DColor();
           color.serializeFromString(val);
           if (colors==null)
               colors = new DColorRegistry();
           colors.putColor(color);
           colorsId.put(val, color);
        }
        
        return color;
    }
    
    /**
     * @return the object value which represents a coordinate value
     * Double, Date, String
     */
    private Object getCoordValue(String text, Class c, Node node)
    {
        if (c == null)
        {
            try {  return new Double(text);  }
            catch (NumberFormatException e) {
                return text;
            }
        }
        else
        if (c == Date.class)
        {
            try {
               return SimpleDateFormat.getInstance().parse(text);
            } catch (ParseException e)
            { throw new DXmlError("Bad formatted Date expected value in the node '"+node.getNodeName()+"'"); }
        }
        else
        if (c == Double.class)
        {
            try {
                return new Double(text);
            } catch (NumberFormatException e) {
                throw new DXmlError("Bad formatted Double expected value in the node '"+node.getNodeName()+"'");
            }
        }
        else
        if (c == String.class)
            return text;
        
        throw new DXmlError("Bad formatted coodinate value for the node '"+node.getNodeName()+"'");
    }
    
    
    
    /**
     * @return the object value which represents a range value
     * Double, CalendarField, String
     */
    private Object getRangeValue(String text, Class c, Node node)
    {
        if (c == null)
        {
            try {  return new Double(text);  }
            catch (NumberFormatException e) {
                return text;
            }
        }
        else
        if (c == Date.class)
        {
            CalendarField cf = new CalendarField();
			cf.serializeFromString(text);
			return cf;
	    }
        else
        if (c == Double.class)
        {
            try {
                return new Double(text);
            } catch (NumberFormatException e) {
                throw new DXmlError("Bad formatted Double expected value in the node '"+node.getNodeName()+"'");
            }
        }
        else
        if (c == String.class)
            return text;
        
        throw new DXmlError("Bad formatted coodinate value for the node '"+node.getNodeName()+"'");
    }
	
	/** @return the color registry */
	public DColorRegistry getColorRegistry()
	{
		return colors;
	}
	
	public DFontRegistry getFontRegistry()
	{
		return fonts;
	}
	
	public DStyleRegistry getStyleRegistry()
    {
		return styles;
    }
    
    /** override current color registry using the new one */
    public void setColorRegistry( DColorRegistry r )
    {
      colors = r;
    }
  
    /** override current font registry using the new one */
    public void setFontRegistry( DFontRegistry r)
    {
      fonts = r;
    }
  
    /** override current style registry using the new one */
    public void setStyleRegistry( DStyleRegistry r )
    {
      styles =r;
    }
	
    
    public void clear()
    {
      fontsId.clear();
      colorsId.clear();
      stylesId.clear();
      axis.clear();
      folder_model_read_ = null;
    }
    
     /**
      * Parse input stream building one or more documents updating colors,fonts and
      * styles registry (@see getStyleRegistry()).
      * @param in input stream to use for xml input.
      * @return one or mode DDocument in an array (can be null).
      * @throws various exception (ie java.lang.IOException), or DXmlError
      */
    public IDObject[] read( InputStream in )
    {
    	return read(new InputSource(in));
    }
    
    /**
     * This is a convience method that reads in an input stream that contains an xml input and
     * return a DOM model.
     * 
     * @param in input stream to use for xml input.
     * @return a DOM that represents the xml input.
     * @throws various exception (ie java.lang.IOException), or DXmlError
     */
   public Document readDOM( InputStream in )
    {
    	return readDOM(new InputSource(in));
    }    
   /**
    * This is a convience method that reads in an input source that contains an xml input and
    * return a DOM model.
    * 
    * @param in input source to use for xml input.
    * @return a DOM that represents the xml input.
    * @throws various exception (ie java.lang.IOException), or DXmlError
    */
   
    public Document readDOM( InputSource is )
    {
      clear();      
      
     /* InputSource inputSource = new InputSource(in);
      
      XMLLoader xmlLoader = new XMLLoader("./config/chart.xsd");
      xmlLoader.setValidating(true);
      
      
      if (xmlLoader.load(inputSource) == false) {
          Exception e = xmlLoader.getException();
          if (e!=null)
          throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")+"\n"+e.getMessage()); //$NON-NLS-1$
      }

      Document dom_doc = xmlLoader.getDOM();
       
      //parser_.parse( new InputSource(in) );
	   //Document dom_doc = parser_.getDocument();

	   if (dom_doc == null) throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")); //$NON-NLS-1$*/
	   
      parser_.parse( is );
	   Document dom_doc = parser_.getDocument();
	   Collection diagnostics = parser_.getDiagnostics();
	   if (diagnostics.size() > 0) {
		   Iterator iter = diagnostics.iterator();
		   StringBuffer sb = new StringBuffer("Validation Error:\n");
		   while (iter.hasNext()){
			   XSDDiagnostic diagnostic = (XSDDiagnostic)iter.next();
			   sb.append(diagnostic.getMessage()+"\n");
		   }
		   throw new DXmlError(sb.toString());
	   }
	   if (dom_doc == null) throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")); //$NON-NLS-1$*/
	   return dom_doc;
    }
    
     public IDObject[] read( InputSource is )
     {
       clear();      
       
      /* InputSource inputSource = new InputSource(in);
       
       XMLLoader xmlLoader = new XMLLoader("./config/chart.xsd");
       xmlLoader.setValidating(true);
       
       
       if (xmlLoader.load(inputSource) == false) {
           Exception e = xmlLoader.getException();
           if (e!=null)
           throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")+"\n"+e.getMessage()); //$NON-NLS-1$
       }

       Document dom_doc = xmlLoader.getDOM();
        
       //parser_.parse( new InputSource(in) );
	   //Document dom_doc = parser_.getDocument();

	   if (dom_doc == null) throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")); //$NON-NLS-1$*/
	   
       parser_.parse( is );
	   Document dom_doc = parser_.getDocument();
	   Collection diagnostics = parser_.getDiagnostics();
	   if (diagnostics.size() > 0) {
		   Iterator iter = diagnostics.iterator();
		   StringBuffer sb = new StringBuffer("Validation Error:\n");
		   while (iter.hasNext()){
			   XSDDiagnostic diagnostic = (XSDDiagnostic)iter.next();
			   sb.append(diagnostic.getMessage()+"\n");
		   }
		   throw new DXmlError(sb.toString());
	   }
	   if (dom_doc == null) throw new DXmlError(ReportCorePlugin.translate("BAD_XML_FORMAT")); //$NON-NLS-1$*/
	   return read(dom_doc);
     }
     
     /** reads a document directly form a DOM */ 
     public IDObject[] read(Document in) {
 	  
       Element root = in.getDocumentElement();   

	   // initialize all registries 
	  
       ArrayList docs = new ArrayList();
       NodeList l = root.getChildNodes();
       Arg arg = new Arg(null, null);
       
       if( "chart".equals(root.getNodeName()))
       {
         DDocument doc = new DDocument();
         DGraphic gph = new DGraphic();
         doc.addChild(gph);
         arg.node = root;      
         invokeDoMethod( gph, this, arg );
         docs.add( doc );
       }
       else
       {
          throw new DXmlError("'chart' element is expected as root element");
       }

       if( docs.size() == 0 ) return null;
       roots = new IDObject[ docs.size() ];
       for( int i=0; i<docs.size(); ++i )
       {
         roots[i] = (IDObject)docs.get(i);
       }
	   return roots;
	 }

     	 
	/**
     * Replace few xml entity '&xxx;' to their char value in string.
     * @return modified string, or s directly.
	 */
	public static String decode(String s)
	{
		if (s == null) return s;
		s = s.replaceAll("&apos;", "'");
		s = s.replaceAll("&lt;", "<");
		s = s.replaceAll("&gt;", ">");
		s = s.replaceAll("&quot;", "\"");
		s = s.replaceAll("&amp;","&");
		return s;
	}   	
	
    /** 
     * Check if attribute attr_tag is defined in attribute map attr.
     * @return attribute found
     * @throws DXmlError if attribute attr_tag is not found in node map attr.
     */
	protected Node getRequiredAttribut(NamedNodeMap attr, String attr_tag, String elt_tag  )
	{
      Node n = attr.getNamedItem(attr_tag);
	  if (n == null) 
		throw new DXmlError("The required '" +attr_tag +"' attribute is not defined in the " + elt_tag +" element");
      return n;
	}  	
	
    /**
     * @return text from node converted from UTF-8 encoding.
     */
	public static String getNodeText(Node node )
	{
		String s = "";
		for ( Node child = node.getFirstChild(); child!=null; child=child.getNextSibling() )
		{
		  if ( child.getNodeType() == Node.TEXT_NODE )
		  {
            String sc = child.getNodeValue();
            /*InputSource (?from InputStream?)  seems to handle directly UTF-8 decoding
             * if not, try this:
			try
			{
				s = decode(new String( sc.getBytes(), "UTF-8"));
			}
			catch (UnsupportedEncodingException e)
			{
				s = decode(sc);
			}*/
//TODO: if UTF-8 is done, perhaps decoding &xx; is done too...      
            s += decode(sc);
		  }
		}
		return s;
	}
	
	/** 
      * Create a IDItem instance from given str.
      * @param str the node name to use to create IDItem instance.
      * @param ext a DXmlReader.
      * @param o_arg a CreateIDItemArg instance.
      * @return an IDItem or null if node isn't an iditem element.
      */
     protected IDItem createIDItemForNode( Node node )
     {          
       String name = node.getNodeName();
       if( name==null ) return null ;
       
       switch( name.length() )
       {
         case 4: if ("axis".equals(name)) return new DAxis("unnamed");
                 break;
         case 5: if ("coord".equals(name))
                 {
                    NamedNodeMap attr = node.getAttributes();   
                    String CoordNodeName = node.getNodeName();    
                    Node n = getRequiredAttribut(attr, "value", CoordNodeName);
                    Object c = getCoordValue(n.getNodeValue(), null, n);
                    if (c instanceof Double)
                        return new DCoord(null, ((Double)c).doubleValue());
                    else
                        return new DCoordObject(null, c);
                 }
                 if ("point".equals(name)) return new DPoint();
                 
                 if ("chart".equals(name)) return new DGraphic();
                 break;
         //case 6: if ("shapes".equals(name)) return new DShapes();
         //        break;
         case 7: //if ("include".equals(name)) return new DInclude();     
                 if ("dataset".equals(name)) return new DCurve();
                 break;
         case 8: if ("category".equals(name))  return new DCategory(); 
                 if ("palettes".equals(name))  return new DPalettes();
                 break;
         //case 9: if ("timeStamp".equals(name)) return new DTimeStamp();
         //        break;
         case 10:
                 if ("markerLine".equals(name)) return new DMarkerLine();
                 break;
                 
         //case 11:
         //        if ("preferences".equals(name)) return new DPreferences();   
         //        break;
                 
         default:
           if ("markerRegion".equals(name)) return new DMarkerRegion();
           //if ("eventHandler".equals(name)) return new DEvent();
           //if ("accessibilityTitle".equals(name)) return new DAccessibilityTitle();
           //if ("accessibilityDesc".equals(name)) return new DAccessibilityDesc();
           //if ("internationalization".equals(name)) return new DI18N();
         
       }       
        throw new DXmlError( name + " is unknown node.");
     }
     
     protected IDItem createFromClass( String class_name )
     {
       try
       {
         Class clazz = Class.forName( class_name );
         Object inst = clazz.newInstance();        
         return (IDItem)inst;
       }
       catch (Exception e)
       {
         if (e instanceof InstantiationException) 
           //|| (e instanceof IllegalAccessException))
         {
           System.out.println("Can't instanciate class '"+class_name+"',  haven't a public empty constructor ?");
           return null;
         }
         if (e instanceof ClassNotFoundException) 
           //|| (e instanceof InstantiationException) 
           //|| (e instanceof IllegalAccessException))
         {
           System.out.println("Can't found class = " + class_name);
         }
       }
       return null;
     }

     /**
      * invokeDoMethod() for all node contained in arg (@see class Arg), that aren't
      * filtered (@see Arg class again).
      */
     public void doChildrenItem( IDItem parent, DExtensible ext, Object a_arg)
	 {	
        Arg arg = (Arg)a_arg;
        Node n = arg.node ;
		if (!n.hasChildNodes()) return;
	 	
	 	Element e = null;
	 	if (n.getNodeType()==Node.TEXT_NODE) {
	 		return;
	 	}
	 	else if (n.getNodeType() == Node.ELEMENT_NODE) 
	 	{
	 		e = (Element)n;
	 	}
	 	
	 	NodeList list = e.getChildNodes();
	 	IDItem previous = null;
	 	for (int i=0; i<list.getLength(); i++)
	 	{
	 		IDItem child = null;
	 		Node node = list.item(i);
	 		
			if (node.getNodeType()!=Node.ELEMENT_NODE) {
				continue;
			}	
       
            //filtered node ?
            if( arg.filterChildren( node.getNodeName() ) )continue;

            child = createIDItemForNode( node );
	 		 		
	 		if (child == null) continue; //node is not a IDItem
            invokeDoMethod( child, this, new Arg(node, parent) );

	 		if (parent == child)
	 		    throw new DXmlError("Parent = " + parent.getClass());
	 		
	 		if (parent != null)
	 		{	
			    parent.insertChild(child, previous);
			    previous = child;
	 		}
	 	}
	 }

     /** configure a DGraphic and it children */
	 public void doMethod( DGraphic graph, DExtensible ext, Object the_arg )
	 {
        Arg arg = (Arg)the_arg;
		NamedNodeMap attr = arg.node.getAttributes();
		
		Node n;
	
		if ((n=attr.getNamedItem("backgroundColor")) != null) 
	     {
	         DStyle s = new DStyle();
	         DColor c = new DColor();
	         c.serializeFromString(n.getNodeValue());
	         s.setBackColor(c);
	         graph.setStyle(s);
	     }
		if ((n=attr.getNamedItem("width")) != null) 
		    graph.getProperties().store(DGraphic.P_MAX_WIDTH, Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("minWidth")) != null) 
		    graph.getProperties().store(DGraphic.P_MIN_WIDTH, Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("height")) != null)
		    graph.getProperties().store(DGraphic.P_MAX_HEIGHT, Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("minHeight")) != null) 
		    graph.getProperties().store(DGraphic.P_MIN_HEIGHT, Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("backgroundColor")) != null) 
		    graph.getProperties().store(DGraphic.P_BACKCOLOR, getColor(n.getNodeValue(), "chart"));
	 	
	 	axis.clear(); //axis are defined inside graphic, no share from one graphic to other one.
	 	
        for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          if ("legend".equals(c.getNodeName()))
              parseLegend( graph, ext , new Arg(c, graph));
          else
          if ("chartTitle".equals(c.getNodeName()))
              parseChartTitle( graph, ext , new Arg(c, graph));
          else
          if ( "plotArea".equals(c.getNodeName()))
              parsePlotArea( graph, ext, new Arg(c, graph));
        } 
        
        String save[] = arg.filter_children;
        arg.filter_children = filter_for_DGraphic;
	 	doChildrenItem(graph, ext, arg );
        arg.filter_children = save;
        
        axis.clear();
	 }
	 
	 //...same filtering PROPERTIES and AXIS, this is for DGraphic purpose.
     private static String[] filter_for_DGraphic = { "legend", "chartTitle", "plotArea"};
                                                    
	 
	 /** parses 'internationalization' element */
	 /*public void doMethod( DI18N i18n,  DExtensible ext, Object the_arg )
	 {     
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	    
	     Node n;
	     if ((n=attr.getNamedItem("language")) != null) i18n.setLanguage(n.getNodeValue());
	     if ((n=attr.getNamedItem("country")) != null) i18n.setCountry(n.getNodeValue());
	     if ((n=attr.getNamedItem("timezone")) != null) i18n.setTimeZone(n.getNodeValue());
	     if ((n=attr.getNamedItem("textDirection")) != null) 
	     {
	         if ("RTL".equals(n.getNodeValue()))
	             i18n.setTextDirection(DI18N.DEF_RTL);
	         else
	             i18n.setTextDirection(DI18N.DEF_LTR);
	     }
	     if ((n=attr.getNamedItem("resourceBundle")) != null) i18n.setResourceBundle(n.getNodeValue());
	 }*/
	 
	 /** parses 'preferences' element */
	 /*public void doMethod( DPreferences pref,  DExtensible ext, Object the_arg )
	 {	
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	    
	     Node n;
	     if ((n=attr.getNamedItem("url")) != null) pref.setUrl(n.getNodeValue());
	     if ((n=attr.getNamedItem("tooltip")) != null) pref.setTooltip(n.getNodeValue());
	     if ((n=attr.getNamedItem("show")) != null) pref.setShow(Boolean.valueOf(n.getNodeValue()).booleanValue());
	     if ((n=attr.getNamedItem("browserCookieName")) != null) pref.setBrowserCookieName(n.getNodeValue());
	     if ((n=attr.getNamedItem("updatedPreferencesVariable")) != null) pref.setUpdatedPreferencesVariable(n.getNodeValue());
	     if ((n=attr.getNamedItem("storedPreferences")) != null) pref.setStoredPreferences(n.getNodeValue());
	 }*/
	 
	 /** parses 'legend' element */
	 protected void parseLegend( DGraphic g,  DExtensible ext, Object the_arg )
	 {	     
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	    
	     if ((n=attr.getNamedItem("sizingFactor")) != null) 
	         g.getProperties().store(DGraphic.P_LEGEND_LIMIT, Integer.valueOf(n.getNodeValue()));
         
         if ((n=attr.getNamedItem("show")) != null) 
             g.getProperties().store(DGraphic.P_SHOW_LEGEND, Boolean.valueOf(n.getNodeValue()).booleanValue());
	     
	     DAlignmentPair p = new DAlignmentPair();
	     if ((n=attr.getNamedItem("alignment")) != null) 
	     {
	         Node old = arg.node;
	         arg.node = n;
	         invokeDoMethod(p, ext, arg);
	         if ((p.getAlignment()==IDAlignment.TOP)||(p.getAlignment()==IDAlignment.BOTTOM))
	             p.setSecondaryAlignment(IDAlignment.RIGHT);
	         else
	             p.setSecondaryAlignment(IDAlignment.TOP);
	         arg.node = old;
	     }
	     else
	     {
	         p.setAlignment(IDAlignment.RIGHT);
	         p.setSecondaryAlignment(IDAlignment.TOP);
	     }
	     g.getProperties().store(DGraphic.P_LEGEND_LAYOUT,p);
	         
	     //parseChartItem( arg.node, g.getProperties(), DGraphic.P_LEGEND_EVENT_LIST, 
	     //                DGraphic.P_LEGEND_ACC_TITLE, DGraphic.P_LEGEND_ACC_DESC);
	    
	 }
	 
	 /** parses common attribute for 'category' node */
	 public void doMethod(DCategory cat, DExtensible ext, Object arg)
	 {
	     NamedNodeMap attr = ((Arg)arg).node.getAttributes();
	     Node n;
	     if ((n = getRequiredAttribut(attr, "id", "category"))!=null) cat.setId(n.getNodeValue());
	     if ((n=attr.getNamedItem("label")) != null) cat.setLabel(n.getNodeValue());
	     
	     doChildrenItem(cat, ext, arg);
	 }
	 
	 /** parses common attribute for 'eventHandler' node */
	 /*public void doMethod(DEvent evt, DExtensible ext, Object arg)
	 {
	     NamedNodeMap attr = ((Arg)arg).node.getAttributes();
	     Node n;
	     if ((n=attr.getNamedItem("type")) != null) evt.setType(n.getNodeValue());
	     evt.setHandler(getNodeText(((Arg)arg).node));
	 }*/
	     
	 /** parses common attribute for 'accessibilityTitle' node */
	 /*public void doMethod(DAccessibilityTitle acc, DExtensible ext, Object a_arg)
	 {
	     Arg arg = (Arg)a_arg;
	     acc.setText(getNodeText(arg.node));
	 }*/
	 
	 /** parses common attribute for 'accessibilityDesc' node */
	 /*public void doMethod(DAccessibilityDesc acc, DExtensible ext, Object a_arg)
	 {
	     Arg arg = (Arg)a_arg;
	     acc.setText(getNodeText(arg.node));
	 }*/
	 
	 /** parses common attribute for ChartItem node */
	 /*protected void parseChartItem( Node node, DPropertyStore ps, String keyEvent, String keyAcc_t,  String keyAcc_desc)
	 {
	     LinkedList eventList = null;
	     
	     for (Node c = node.getFirstChild(); c!=null; c=c.getNextSibling())
	     { 
	        if ("eventHandler".equals(c.getNodeName())) 
	        {
	           DEvent evt = new DEvent();
	           NamedNodeMap attr2 = c.getAttributes();
	           Node n2;
	           if ((n2=attr2.getNamedItem("type")) != null) evt.setType(n2.getNodeValue());
	         
	           evt.setHandler(getNodeText(c));
	         
	           if (eventList==null)
	               eventList = new LinkedList();
	           eventList.add(evt);
	        }
	        else
	        if ("accessibilityTitle".equals(c.getNodeName())) 
	        {
	           ps.store(keyAcc_t, getNodeText(c));
	        }
	        else
	        if ("accessibilityDesc".equals(c.getNodeName())) 
	        {
	           ps.store(keyAcc_desc, getNodeText(c));
	        }
	     }
	   
	     if (eventList!=null)
	        ps.store(keyEvent, eventList);
	 }*/
	 
	 /** parses 'chartTitle' element */
	 protected void parseChartTitle( DGraphic g,  DExtensible ext, Object the_arg )
	 {	   
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     
	     if ((n=attr.getNamedItem("show")) != null) 
	         g.getProperties().store(DGraphic.P_SHOW_TITLE, Boolean.valueOf(n.getNodeValue()));

	     g.setTitle(getRequiredAttribut(attr, "label", "chartTitle").getNodeValue());
	         
	     DAlignmentPair p = new DAlignmentPair();
	     p.setAlignment(IDAlignment.TOP);
         p.setSecondaryAlignment(IDAlignment.LEFT);
     
	 /*    if ((n=attr.getNamedItem("alignment")) != null) 
	     {
	         Node old = arg.node;
	         arg.node = n;
	         invokeDoMethod(p, ext, arg);
	         arg.node = old;
	         
	         if (p.getAlignment()==IDAlignment.LEFT)
	             p.setSecondaryAlignment(IDAlignment.ROTCCW90|IDAlignment.TOP);
	         else
	         if (p.getAlignment()==IDAlignment.RIGHT)
	            p.setSecondaryAlignment(IDAlignment.ROTCW90|IDAlignment.TOP);    
	         else
	             p.setSecondaryAlignment(IDAlignment.LEFT);
	     }
	     else
	     {
	         p.setAlignment(IDAlignment.TOP);
             p.setSecondaryAlignment(IDAlignment.LEFT);
	     }*/
         
	     g.getProperties().store(DGraphic.P_TITLE_LAYOUT,p);
	     
	     //parseChartItem( arg.node, g.getProperties(), DGraphic.P_TITLE_EVENT_LIST,
	     //                DGraphic.P_TITLE_ACC_TITLE, DGraphic.P_TITLE_ACC_DESC);
	    
	 }
	 
	 /** parses 'timeStamp' element */
	/* public void doMethod( DTimeStamp ts,  DExtensible ext, Object the_arg )
	 {	     
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     if ((n=attr.getNamedItem("dateFormat")) != null) ts.setDatePattern( n.getNodeValue());
	     if ((n=attr.getNamedItem("value")) != null)
	         ts.setValue(n.getNodeValue());
	     
	     if ((n=attr.getNamedItem("show")) != null) 
	         ts.setShow(Boolean.valueOf(n.getNodeValue()).booleanValue());
	     if ((n=attr.getNamedItem("prefix")) != null) ts.setPrefix( n.getNodeValue());
	 }*/

	 /** parses 'plotArea' element */
	 protected void parsePlotArea( DGraphic g,  DExtensible ext, Object the_arg )
	 {
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     String ctype = getRequiredAttribut(attr, "type", "plotArea").getNodeValue();
	     g.setGraphicType(ctype);
	      
	     Node n;
	    	     
	     if ((n=attr.getNamedItem("backgroundColor")) != null) 
	         g.getProperties().store(DGraphic.P_PLOTAREA_BACKCOLOR, getColor(n.getNodeValue(), "plotArea"));
	    // if ((n=attr.getNamedItem("valueFormat")) != null) 
	    //     g.getProperties().store(DGraphic.P_VALUE_FORMAT, n.getNodeValue());
	     if ((n=attr.getNamedItem("showPercentage")) != null) 
	         g.getProperties().store(DGraphic.P_SHOW_PERCENTAGE, Boolean.valueOf(n.getNodeValue()));

	     if ((n=attr.getNamedItem("showValues")) != null) 
	         g.getProperties().store(DGraphic.P_SHOW_VALUES, Boolean.valueOf(n.getNodeValue()));

	     if ((n=attr.getNamedItem("margin")) != null) 
	         g.getProperties().store(DGraphic.P_MARGIN, Integer.parseInt(n.getNodeValue()));
	    
	   	 
         //	this is important to get AXIS now, otherwise DCoord can't retrieve them..
	     for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
		 {	         
	         if ("axes".equals(c.getNodeName()))
	             parseAxes(g, ext, new Arg(c, g));
		 }
	     
		 for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
		 {
		     if ("datasets".equals(c.getNodeName()))
		         parseCurves(g, ext, new Arg(c, g));
		 }
		 
		 String save[] = arg.filter_children;
	     arg.filter_children = filter_for_plotArea;
		 doChildrenItem(g, ext, arg );
	     arg.filter_children = save;
	 }
	 
	 // filter already loaded element
     private static String[] filter_for_plotArea = { "axes", "datasets" };
	 
     
	 /** parses 'include' element */
	 /*public void doMethod( DInclude inc,  DExtensible ext, Object the_arg )
	 {	     
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     if ((n=attr.getNamedItem("href")) != null) inc.setHref(n.getNodeValue());
	     inc.setJavaScript(getNodeText(arg.node));
	 }*/
	 
	 /** parses 'curves' item */
	 protected void parseCurves( DGraphic graph,  DExtensible ext, Object the_arg )
	 {
	     doChildrenItem(graph, ext, the_arg );
	 }
	 
	 /** parses 'axes' item */
	 protected void parseAxes( DGraphic graph,  DExtensible ext, Object the_arg )
	 {
	     doChildrenItem(graph, ext, the_arg );
	 }
	 
	 /** parses 'shapes' element */
	 /*public void doMethod(DShapes s, DExtensible ext, Object the_arg )
	 {
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();    
	     Node n;
	     
	     if ((n=attr.getNamedItem("location")) != null) s.setLocation( n.getNodeValue());
	     if ((n=attr.getNamedItem("show")) != null) 
	         s.setShow(Boolean.valueOf(n.getNodeValue()).booleanValue());
	 }*/
	 
	 /** parses 'palettes' element */
	 public void doMethod(DPalettes pal, DExtensible ext, Object the_arg )
	 {
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();    
	     Node n;
	     
	     if ((n=attr.getNamedItem("location")) != null) pal.setLocation( n.getNodeValue());
	     if ((n=attr.getNamedItem("paletteSetId")) != null) pal.setPaletteSetId( n.getNodeValue());
	     if ((n=attr.getNamedItem("paletteId")) != null) pal.setPaletteId( n.getNodeValue());
	 }
   
	 
     /** configure a DAxis */
	 public void doMethod( DAxis ax, DExtensible ext, Object the_arg )
	 {
        Arg arg = (Arg)the_arg;
		NamedNodeMap attr = arg.node.getAttributes();
		String axisNodeName = arg.node.getNodeName();
		Node n;
	    
		
		if ((n = getRequiredAttribut(attr, "id", "axis"))!=null) ax.setName(n.getNodeValue());
		
		if ((n=attr.getNamedItem("title")) != null) ax.setTitle(n.getNodeValue());
        if ((n=attr.getNamedItem("scale")) != null) 
        {
            if ("log".equals(n.getNodeValue()))
                ax.setScaleType(DAxis.S_LOG);
            else
                ax.setScaleType(DAxis.S_LIN);   
        }
       
        if ((n=attr.getNamedItem("showTitle")) != null) 
            ax.getProperties().store(DAxis.P_SHOW_TITLE, Boolean.valueOf(n.getNodeValue()));
        
        if ((n=attr.getNamedItem("rotateLabel")) != null)
        {
			   double degrees = Double.parseDouble(n.getNodeValue());
			   String degreeId = DAxis.S_ZERO_DEGREE;
			   if (degrees >= 45) degreeId = DAxis.S_PLUS_NINTY_DEGREE;
			   else if ((degrees < 45) && (degrees > 0)) degreeId = DAxis.S_ZERO_DEGREE;
			   else if ((degrees <= -45)) degreeId = DAxis.S_MINUS_NINTY_DEGREE;
			   else if ((degrees > -45) && (degrees < 0)) degreeId = DAxis.S_ZERO_DEGREE;
			   else degreeId = DAxis.S_ZERO_DEGREE;
               ax.getProperties().store(DAxis.P_LABEL_ROTATION, degreeId);
        }
        
        String type = "number";
        if ((n=attr.getNamedItem("dataFormatType")) != null)
        {
            type = n.getNodeValue();
            if ("date".equals(type))
                ax.getProperties().store(DAxis.P_DATA_CLASS, Date.class);  
            else
            if ("number".equals(type))
                ax.getProperties().store(DAxis.P_DATA_CLASS, Double.class);
            else
            if ("string".equals(type))
                ax.getProperties().store(DAxis.P_DATA_CLASS, String.class);
            else
                throw new DXmlError("Unkown type '"+type+"' in the dataFormatType definition");
        }
        
        Class pdclass = (Class)ax.getProperties().get(DAxis.P_DATA_CLASS);
        
        if ((n=attr.getNamedItem("min")) != null) 
              ax.getProperties().store(DAxis.P_MIN, getCoordValue(n.getNodeValue(), pdclass, n));
       
        if ((n=attr.getNamedItem("max")) != null) 
              ax.getProperties().store(DAxis.P_MAX, getCoordValue(n.getNodeValue(), pdclass, n)); 

        //register axis for future ... access (DCoord)
        axis.put(ax.getName(), ax);
        
        //have a property store ?
        for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          if ("labelFormat".equals(c.getNodeName()))
              parseLabelFormat(ax, ext, new Arg(c, ax));
          else
          if ("categories".equals(c.getNodeName()))
              parseCategories(ax, ext, new Arg(c, ax));
          else
          if ("majorUnit".equals(c.getNodeName()))    
              parseMajorUnit(ax, ext, new Arg(c, ax));
          else
          if ("minorUnit".equals(c.getNodeName()))
              parseMinorUnit(ax, ext, new Arg(c, ax));
          
        }

        String save[] = arg.filter_children;
        arg.filter_children = filter_for_DAxis;
        doChildrenItem(ax, ext, arg );
        arg.filter_children = save;
        
	 }
	 

     // filter already loaded element
     private static String[] filter_for_DAxis = {  "categories", "majorUnit", "minorUnit", "labelFormat"};
	 
	 /** parses 'markerLine' element */
	 public void doMethod(DMarkerLine ml, DExtensible ext, Object the_arg)
	 {
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     if ((n=attr.getNamedItem("id")) != null) ml.setId(n.getNodeValue());
	     if ((n=attr.getNamedItem("value")) != null)
	     {
            Class f = (Class)((DAxis)arg.parent).getProperties().get(DAxis.P_DATA_CLASS);
            ml.setValue(getCoordValue(n.getNodeValue(), f, n));
	     }
	     
	     if ((n=attr.getNamedItem("label")) != null) ml.setLabel(n.getNodeValue());
	     if ((n=attr.getNamedItem("color")) != null) ml.setColor(getColor(n.getNodeValue(), "markerLine"));
	     if ((n=attr.getNamedItem("thickness")) != null) ml.setThickness(Double.parseDouble(n.getNodeValue()));
	 }
	 
	 /** parses 'markerRegion' element */
	 public void doMethod(DMarkerRegion mr, DExtensible ext, Object the_arg)
	 {
	     Arg arg = (Arg)the_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	    
	     Node n;
	     if ((n=attr.getNamedItem("id")) != null) mr.setId(n.getNodeValue());
	     if ((n=attr.getNamedItem("label")) != null) mr.setLabel(n.getNodeValue());
	     if ((n=attr.getNamedItem("color")) != null) mr.setColor(getColor(n.getNodeValue(), "markerRegion"));
	      	    
	     Class f = (Class)((DAxis)arg.parent).getProperties().get(DAxis.P_DATA_CLASS);      
	     if ((n=attr.getNamedItem("fromValue")) != null)
	         mr.setFromValue(getCoordValue(n.getNodeValue(), f, n));

	     if ((n=attr.getNamedItem("toValue")) != null)
	         mr.setFromValue(getCoordValue(n.getNodeValue(), f, n));
	 }
	 
	 /** parses 'categories' element */
	 protected void parseCategories(DAxis ax, DExtensible ext, Object the_arg)
	 {
	     doChildrenItem(ax, ext, the_arg );
	 }
	 
	 /** parses 'majorUnit' element */
	 protected void parseMajorUnit(DAxis ax, DExtensible ext, Object a_arg)
	 {
	     Arg arg = (Arg)a_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     if ((n=attr.getNamedItem("value")) != null)
	     {
	         Class f = (Class)ax.getProperties().get(DAxis.P_DATA_CLASS);
	         ax.getProperties().store(DAxis.P_MAJUNIT_VALUE, getRangeValue(n.getNodeValue(), f, n));
	     }
	//     else
	//	     ax.getProperties().store(DAxis.P_MAJUNIT_VALUE, new Double(1.0));
	        
	     if ((n=attr.getNamedItem("showTick")) != null) 
	         ax.getProperties().store(DAxis.P_MAJUNIT_SHOWTICK, Boolean.valueOf(n.getNodeValue()).booleanValue());
	     if ((n=attr.getNamedItem("showGridLine")) != null)  
	         ax.getProperties().store(DAxis.P_MAJUNIT_SHOWGRID, Boolean.valueOf(n.getNodeValue()).booleanValue());
	    
	     boolean showTick = ax.getProperties().get(DAxis.P_MAJUNIT_SHOWTICK, true);
	     Object value = ax.getProperties().get(DAxis.P_MAJUNIT_VALUE);
	     
	     if (showTick && (value!=null))
	         ax.getProperties().store(DAxis.P_STEP_UNIT, value);
	     
	     boolean showGrid = ax.getProperties().get(DAxis.P_MAJUNIT_SHOWGRID, true);
	     if (showGrid && (value!=null))
	         ax.getProperties().store(DAxis.P_STEP_LINE, value);
	 }
	 
	 /** parses 'minorUnit' element */
	 protected void parseMinorUnit(DAxis ax, DExtensible ext, Object a_arg)
	 {
	     Arg arg = (Arg)a_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     if ((n=attr.getNamedItem("value")) != null)
	     {
	        Class f = (Class)ax.getProperties().get(DAxis.P_DATA_CLASS);
	        ax.getProperties().store(DAxis.P_MINUNIT_VALUE, getRangeValue(n.getNodeValue(), f, n));
	     }
	//     else
	//         ax.getProperties().store(DAxis.P_MINUNIT_VALUE, new Double(1.0));
	         
	        
	     if ((n=attr.getNamedItem("showTick")) != null) 
	         ax.getProperties().store(DAxis.P_MINUNIT_SHOWTICK, Boolean.valueOf(n.getNodeValue()).booleanValue());
	     if ((n=attr.getNamedItem("showGridLine")) != null)  
	         ax.getProperties().store(DAxis.P_MINUNIT_SHOWGRID, Boolean.valueOf(n.getNodeValue()).booleanValue());
	    
	     boolean showTick = ax.getProperties().get(DAxis.P_MINUNIT_SHOWTICK, true);
	     Object value = ax.getProperties().get(DAxis.P_MINUNIT_VALUE);
	     if (showTick && (value!=null))
	         ax.getProperties().store(DAxis.P_STEP_DOT, value);
	 }
	 
	 /** parses 'labelFormat' element */
	 protected void parseLabelFormat(DAxis ax, DExtensible ext, Object a_arg)
	 {
	     Arg arg = (Arg)a_arg;
	     NamedNodeMap attr = arg.node.getAttributes();
	     
	     Node n;
	     String type = "number";
	     String pattern=null;
	     if ((n=attr.getNamedItem("type")) != null) {
	         type = n.getNodeValue();
	         ax.getProperties().store(DAxis.P_LABEL_FORMATTYPE, type);
	     }
	     if ((n=attr.getNamedItem("pattern")) != null) {
	         
	         pattern = n.getNodeValue();
	         ax.getProperties().store(DAxis.P_LABEL_FORMAT, pattern);
	     }
	     
	   /* EM: bug 126078 unable to determine the right format number
        * because the locale is unknown at this point
        *  Object f = null;
        * 
	     if ("date".equals(type))
	     {
	         if (pattern!=null)
	             f = new SimpleDateFormat(pattern);
	         else
	             f = new SimpleDateFormat();
	     }
	     else
	     if ("number".equals(type))
	     {
	         if (pattern!=null)
	            f = new DecimalFormat(pattern);
	         else
	            f = new DecimalFormat();
	     }
	     else
	     {
	        try
	        {
	           Class c = Class.forName(type);
	           Object o = c.newInstance();
	        }
			catch (ClassNotFoundException e) {
				System.err.println("(W) unable to find class '"+type+"' for labelFormat element in type attribute");
			} catch (InstantiationException e) {
				System.err.println("(W) unable to find class '"+type+"' for labelFormat element in type attribute");
			} catch (IllegalAccessException e) {
				System.err.println("(W) unable to find class '"+type+"' for labelFormat element in type attribute");
			}
	     }
         
	     if (f!=null)
	        ax.getProperties().store(DAxis.P_UNIT_FORMAT, f);*/
	 }
  
     /** configure a DCurve and its children */
	 public void doMethod( DCurve curv, DExtensible ext, Object a_arg )
	 {
        Arg arg = (Arg)a_arg;
		NamedNodeMap attr = arg.node.getAttributes();
   
        Node n;
        if((n=attr.getNamedItem("label")) !=null ) curv.setName(n.getNodeValue());
		if((n=attr.getNamedItem("symbol")) !=null) 
		    curv.getProperties().store(DCurve.P_SYMBOL, n.getNodeValue());
		
		if((n=attr.getNamedItem("color")) !=null) 
		    curv.getProperties().store(DCurve.P_COLOR, getColor(n.getNodeValue(), "dataset"));

		doChildrenItem(curv, ext, arg );
	 }

     /** configure a DPoint and it children */
	 public void doMethod( DPoint pnt, DExtensible ext, Object arg )
	 {
		//NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		//setNodeStyle(attr,pnt);
		
		//Node n;
				
        doChildrenItem(pnt, ext, arg );
	 }
 	 
 	/** configure a DCoord */
 	 public void doMethod( IDCoord coord, DExtensible ext, Object a_arg )
	 {
 	    Arg arg = (Arg)a_arg; 
 	     
		NamedNodeMap attr = arg.node.getAttributes();
		String CoordNodeName = arg.node.getNodeName();
		
        String axisId = getRequiredAttribut(attr, "axis", CoordNodeName).getNodeValue();
   		DAxis ax = (DAxis)axis.get(axisId);
   		if (ax==null)
   		    throw new DXmlError("Unknown axis id = '"+ axisId + "' in coord definition");
   		coord.setAxis(ax);
	 }
     
     /** configure a Data using node text as setTime() */
     public void doMethod( Date date, DExtensible ext, Object t_arg )
     {
        Arg arg = (Arg)t_arg;
        try
        {
            arg.return_value = SimpleDateFormat.getInstance().parse(getNodeText(arg.node));
        }
        catch( ParseException e )
        {
          throw new DXmlError(e.getMessage());
        }
     }
     
     /** configure a String using node text as string */
     public void doMethod( String date, DExtensible ext, Object t_arg )
     {
        Arg arg = (Arg)t_arg;
        arg.return_value = getNodeText(arg.node);
     }
     
     /** configure a String using node text as string */
     public void doMethod( DAlignmentPair align, DExtensible ext, Object t_arg )
     {
        Arg arg = (Arg)t_arg;
        Node n = arg.node;
        
        if ("top".equals(n.getNodeValue()))
            align.setAlignment(IDAlignment.TOP);
        else
        if ("bottom".equals(n.getNodeValue()))
            align.setAlignment(IDAlignment.BOTTOM);
        else
        if ("right".equals(n.getNodeValue()))    
            align.setAlignment(IDAlignment.RIGHT);
        else
        if ("left".equals(n.getNodeValue()))    
            align.setAlignment(IDAlignment.LEFT);
        
        align.setSecondaryAlignment(IDAlignment.LEFT);
     }
	 
  
   /** 
    * Create an object for class name using node data.
    * This is for simple class like "java.lang.Byte" this object creation is not extensible
    * nor overridable using DExtensible mechanism.
    * IDXmlSerializable class name is also supported, in this case instance if configured
    * using extensible mechanism.
    * @return class instance, or null if this is a unknown class.
    */
   protected Object createObjectForClassName( String class_name, Node node )
   {
     try
     {
       Class c = Class.forName(class_name);
       
       if ( c == String .class) return (Object)(               getNodeText(node));
       if ( c == Byte   .class) return (Object)(Byte   .decode(getNodeText(node)));
       if ( c == Short  .class) return (Object)(Short  .decode(getNodeText(node)));   
       if ( c == Integer.class) return (Object)(Integer.decode(getNodeText(node)));
       if ( c == Long   .class) return (Object)(Long   .decode(getNodeText(node)));   
       if ( c == Float  .class) return (Object)(Float  .valueOf(getNodeText(node)));
       if ( c == Double .class) return (Object)(Double .valueOf(getNodeText(node)));
       if ( c == Boolean.class) return (Object)(Boolean.valueOf(getNodeText(node)));
       //a IDColor class but this is a reference in the color registry
       if ( IDColor.class.isAssignableFrom( c )) 
       {
         return getColor(getNodeText(node), node.getNodeName()); //colorsId.get( getNodeText(node) );
       }
       //a IDFont class but this is a reference in the font registry
       if ( IDFont.class.isAssignableFrom( c ))
       {
         return fontsId.get( getNodeText(node) ); 
       } 
       //a IDStyle class but this is a reference in the style registry
       if ( IDFont.class.isAssignableFrom( c ))
       {
         return stylesId.get( getNodeText(node) ); 
       } 
       // a string serializable object ?
       if( IDStringSerializable.class.isAssignableFrom( c ) )
       {
         try
         {
            IDStringSerializable v = (IDStringSerializable)c.newInstance();
            Arg arg = new Arg( node, null );
            invokeDoMethod( v, this, arg );
            return v;
         }
         catch (Exception e)
         {
            if (e instanceof InstantiationException) 
          //|| (e instanceof IllegalAccessException))
            {
             System.out.println("IDStringSerializable class should have an empty public constructor");
             return null;
            }
         }       }
       // a serializable object ?
       if( IDXmlSerializable.class.isAssignableFrom( c ) )
       {
         try
         {
            IDXmlSerializable v = (IDXmlSerializable)c.newInstance();
            Arg arg = new Arg( node, null );
            invokeDoMethod( v, this, arg );
            return v;
         }
         catch (Exception e)
         {
            if (e instanceof InstantiationException) 
          //|| (e instanceof IllegalAccessException))
            {
             System.out.println("IDXmlSerializable class should have an empty public constructor");
             return null;
            }
         }
       }
       else if ( c == Class.class ) return Class.forName( getNodeText(node) );
       else if ( c == DecimalFormat.class )
       {
         return new DecimalFormat( getNodeText(node ) );
       }
       else if ( c == com.ibm.icu.text.SimpleDateFormat.class )
       {
         return new com.ibm.icu.text.SimpleDateFormat( getNodeText(node ) );
       }
       else if ( c== java.util.Date.class )
       {
         try{
             return SimpleDateFormat.getInstance().parse(getNodeText(node));
         }
         catch( ParseException e)
         {
           throw new DXmlError("Failed to parse Date");
         }
       }
/*TODO       else if ( Serializable.class.isAssignableFrom( c ) )
       {
         try{
           ByteArrayInputStream bais = new ByteArrayInputStream( getNodeText(node).getBytes() );
           ObjectInputStream ois = new ObjectInputStream(bais);
           Object o = ois.readObject();
System.err.println("read object of class="+c.getName());
           return o;
         }
         catch( IOException e )
         {
           throw new DXmlError( e.getMessage() );
         }
       }*/
       else
       {
         System.err.println("(W) don't know how to handle property class '"+c.getName()+"' for the element '"+node.getNodeName()+"'");
       }
     }
     catch (Exception e)
     {
       if (e instanceof ClassNotFoundException) 
        //|| (e instanceof InstantiationException) 
        //|| (e instanceof IllegalAccessException))
       {
         System.err.println("(W) Unable to instanciate class '" + class_name + "'");
       }
     }
     return null;   
   }
   
   /** doMethod() for a IDXmlSerializable instance, call readXml() on this object */
   public void doMethod( IDXmlSerializable v, DExtensible ext, Object arg )
   {
     v.readXml( ((Arg)arg).node );
   }

   /** doMethod() for a IDStringSerializable instance, call fromString() on this object */
   public void doMethod( IDStringSerializable v, DExtensible ext, Object arg )
   {
     Node node = ((Arg)arg).node;
     v.serializeFromString( getNodeText(node) );
   }
   
   /** This method is public due to Signal connection, but must not be called directly. */ 
   public void updateExtension( IDExtension ext, Class for_extension)
   {
     if( getClass().equals( for_extension ) )
     {
       //redo extensions updates for take new extension
       DExtensionRegistry.updateExtensible( this );
     }
   }
   
   /** decode 6 hexa-digit string RRGGBB into a IDColor */
   public static IDColor DecodeColor( String s )
   {
     try {
       int rgb = Integer.parseInt(s,16);
       return new DColor( (rgb>>16)&0xFF, (rgb>>8)&0xFF, rgb&0xFF );
     } catch( NumberFormatException e ) {
       return null;
     }
   }

   private HashMap folder_model_read_ = new HashMap();
   
   
 
   private int parseInt( String s, int _default )
   {
     try {
       int v = Integer.parseInt( s );
       return v;
     } catch( Throwable t ) {
       return _default;
     }
   }

 



   /**
    * @see org.eclipse.tptp.platform.report.core.IDocumentReader#getDocuments()
    */
   public IDObject[] getDocuments() {
       int nbdoc=0;
       for (int i=0; i<roots.length; i++)
           if (roots[i] instanceof DDocument)
               nbdoc++;
       IDObject docs[] = new IDObject[nbdoc];
       for (int i=0; i<roots.length; i++)
           if (roots[i] instanceof DDocument)
               docs[i] = roots[i];
       return docs;
   }

   /**
    * @see org.eclipse.tptp.platform.report.core.IDocumentReader#getStyles()
    */
   public IDObject[] getStyles() {
     
      Iterator it = styles.iterator();
      int nbsts = 0;
      while(it.hasNext()) 
      {
          IDStyle s = (IDStyle)it.next();
          nbsts ++;
      }
      
      IDObject sts[] = new IDObject[nbsts];
      
      it = styles.iterator();
      int i=0;
      while(it.hasNext())
      {
          IDStyle s = (IDStyle)it.next();
          sts[i++] = s;
      }
      return sts;
   }
}
