/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DXmlDocumentWriter.java,v 1.5 2008/05/23 14:12:01 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.drivers.xml.internal;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;

import org.eclipse.tptp.platform.report.chart.svg.internal.part.IGraphicTypeConstants;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCategory;
import org.eclipse.tptp.platform.report.core.internal.DCoord;
import org.eclipse.tptp.platform.report.core.internal.DCoordObject;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DMarkerLine;
import org.eclipse.tptp.platform.report.core.internal.DMarkerRegion;
import org.eclipse.tptp.platform.report.core.internal.DPalettes;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPropertyStore;
import org.eclipse.tptp.platform.report.core.internal.DShapes;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.core.internal.IDStringSerializable;
import org.eclipse.tptp.platform.report.drivers.internal.IWriter;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.tools.internal.DAlignmentPair;
import org.eclipse.tptp.platform.report.tools.internal.DParser;

import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.SimpleDateFormat;

/**
 * Provides an object to write JScrib document in an XML format.<br>
 * The following example writes a JScrib IDDocument item in an xml file named
 * 'myfile.rxml'.<p>
 * 
 * <PRE>
 *    IDDocument doc = new DDocument();
 *    ...
 *    DXmlWriter aw = new DXmlWriter();
 *    try{
 *        aw.write( new FileOutputStream("myfile.rxml"), doc );
 *        // Good, the file is generated 
 *        ...
 *    }
 *    catch(Exception e ) 
 *    {
 *      // An error occurs in the writer
 *      ...
 *    }
 * </PRE>
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DXmlDocumentWriter extends DParser implements IWriter {
    private OutputStream out;
    private boolean write_complete_document;

    /** info on element's stack currently in write process */
    private static class Element {
        public String name;
        public int indent;
        public boolean can_indent;
        public boolean have_attr;
        public boolean empty;
        public Element up;
        public boolean start_part_closed;

        public Element(String n, Element u) {
            name = n;
            up = u;
            indent = 0;
            can_indent = true;
            have_attr = false;
            empty = true;
            start_part_closed = false;
        }
    }

    private Element curr_element;
    private int curr_indent;
    private boolean auto_indent;

    /**
     * Create a JScrib XML writer, by default auto indent mode is not engaged.
     * @see #setAutoIndentMode
     */
    public DXmlDocumentWriter() {
        super();
        auto_indent = false;
        write_complete_document = true;
    }

    /** @return current used output stream */
    public OutputStream getOutputStream() {
        return out;
    }

    /** 
     * @return true if auto indent mode is engaged.
     * @see #setAutoIndentMode
     */
    public boolean isAutoIndentMode() {
        return auto_indent;
    }

    /** 
     * Change current value of auto indent mode, is this mode xml elements will be indented
     * using spaces if they don't contains CData.
     * Using this mode produce more human readable stream (files), but their size will grow.
     * (depending on the number of element and max element stack level encountered,
     *  but a rate from 10% to 50% seems to be common).
     * @see #outCData
     */
    public void setAutoIndentMode(boolean b) {
        auto_indent = b;
    }

    /** @return true if writer will write complete jscrib xml document for each write() calls */
    public boolean isWriteCompleteDocument() {
        return write_complete_document;
    }

    /** 
     * Change 'complete document' mode.<br>
     * Default value is <b>true</b>.
     * If parameter is true, next call to write() method will generate a complete XML document
     * including &lt;?xml ...>, &ltJSCRIB> and registries elements. In this mode all colors, fonts,
     * styles and any registries seen in write() parameter are merged into current writer's registries,
     * and written once at the end of XML document.<br>
     * If parameter is false, next call to write() method will generate only the part of XML
     * elements corresponding to given IDObject. In this mode any colors, fonts, styles are merged
     * into writer's registries <b>but</b> no output of writer's registries are done.
     * <b>But</b> any registries seen in write() call parameters is generated as simple object. 
     */
    public void setWriteCompleteDocument(boolean b) {
        write_complete_document = b;
    }

    /**
     * Modifying this name means you need to modify it in DXmlReader
     * @return xml root element name for jscrib, default is 'JSCRIB'.
     */
    public String getRootElementName() {
        return "chart";
    }

    /**
     * Writes an IDObject array in the OutputStream in XML format
     */
    public void write(OutputStream output, IDObject[] objects) throws Exception {

        out = output;
        folder_model_written_ = null; //reset folder model saved
        String root_name = getRootElementName();

        if (write_complete_document) {
            out.write("<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n".getBytes("UTF-8"));
            curr_element = null;
            curr_indent = 0;
            //startElement(root_name);
        }
        
        for (int i = 0; i < objects.length; i++) {
            Object obj = objects[i];
            if (obj == null) continue;
            
            invokeDoMethod(obj, this, null);
        }
        
        curr_element = null;
    }

    /**
     * Writes an IDObject in OutputStream in XML Formal.
     * @see write(OutputStream, IDObject[]).
     */
    public void write(OutputStream output, IDObject object) throws Exception {
        write(output, new IDObject[] { object});
    }

    /**
     * Clears all internal datas. 
     * @see org.eclipse.tptp.platform.report.tools.internal.DParser#clear()
     */
    public void clear() {
        super.clear();
        curr_element = null;
        curr_indent = 0;
    }

    /** 
     * Writes a \n and a number of space depending on parameter, if auto indent mode is engaged.
     * @param idt proportinal to the number of space to write.
     * @throws IOException is the ouput failed.
     */
    protected void outIndent(int idt) throws IOException {
        if (!auto_indent) return;
        out.write('\n');
        for (int i = 0; i < idt; ++i) {
            out.write(' ');
            out.write(' ');
        }
    }

    private void closeStartElement() throws IOException {
        if (curr_element != null && !curr_element.start_part_closed) {
            if (curr_element.empty) {
                out.write('>');
                out.write('\n');
            }
            curr_element.empty = false;
            if (curr_element.can_indent) {
                outIndent(curr_element.indent);
            }
            curr_element.start_part_closed = true;
        }
    }

    /**
     * Calls this to start an XML element, do the ouput and data used to close the element
     * correctly. Support for indentation.
     * @see #endElement
     * @see #outAttribute
     * @param name the name of the element to start.
     */
    public void startElement(String name) {
        try {
            closeStartElement();
            out.write('<');
            out.write(name.getBytes("UTF-8"));
            curr_element = new Element(name, curr_element);
            curr_element.indent = curr_indent;
            curr_indent++;
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Closes a previously started element, choose the best way to write the end ("/>" or "&lt;/ELM>",
     * depending on data collected such as outCData was called.
     * @see #startElement
     * @see #outCData
     */
    public void endElement() {
        if (curr_element == null) return;
        try {
            // /> or (indent)?</XXX>
            if (curr_element.empty) {
                out.write('/');
                out.write('>');
                out.write('\n');
            } else {
                if (curr_element.can_indent) {
                    outIndent(curr_element.indent);
                }
                out.write('<');
                out.write('/');
                out.write(curr_element.name.getBytes("UTF-8"));
                out.write('>');
                out.write('\n');
            }
            curr_indent--;
            curr_element = curr_element.up;
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /** 
     * Writes CData: text outside element area (&gt;..&lt), text is encoded and converted to UTF-8 before wrote.
     * You must call this if you want auto indentation works correctly.
     * @param data the text to write.
     */
    public void outCData(String data) {
        if (data == null) return;
        try {
            if (curr_element != null) {
                curr_element.can_indent = false;
                curr_element.empty = false;
                out.write('>');
            }
            out.write(encode(data).getBytes("UTF-8")); //$NON-NLS-1$
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /** 
     * Writes an attribute in current element. If required and value is null, an error is thrown,
     * if not required and the value is null and error is thrown.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     */
    public void outAttribute(String attr_name, String attr_value, boolean required) {
        if (attr_value == null) {
            if (required) {
                throw new DXmlError("Required attribute '" + attr_name + "' can't have null value, correct the document.");
            }
            return;
        }
        try {
            out.write(' ');
            out.write(attr_name.getBytes("UTF-8"));
            out.write('=');
            out.write('"');
            //need " detection here ?
            out.write(encode(attr_value).getBytes("UTF-8"));
            out.write('"');
            curr_element.have_attr = true;
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /** 
     * Writes an attribute only if value differ than default (do nothing if value is null)
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     * @param default_value the default value, if equals to the value, no output are done.
     */
    public final void outAttribute(String attr_name, String attr_value, String default_value) {
        if (attr_value == null) return;
        if (attr_value == default_value || attr_value.equals(default_value)) return;
        outAttribute(attr_name, attr_value, false);
    }

    /** 
     * Writes an integer attribute
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     */
    public final void outAttribute(String attr_name, int attr_value) {
        outAttribute(attr_name, Integer.toString(attr_value), true);
    }

    /** 
     * Writes an integer attribute, if value differ than default one.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     * @param default_value the default value, if equals to the value, no output are done.
     */
    public final void outAttribute(String attr_name, int attr_value, int default_value) {
        if (attr_value == default_value) return;
        outAttribute(attr_name, Integer.toString(attr_value), false);
    }

    /** 
     * Write boolean attribute
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     */
    public final void outAttribute(String attr_name, boolean attr_value) {
        outAttribute(attr_name, Boolean.toString(attr_value), true);
    }

    /** 
     * Writes a boolean attribute, if value differ than default one.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     * @param default_value the default value, if equals to the value, no output are done.
     */
    public final void outAttribute(String attr_name, boolean attr_value, boolean default_value) {
        if (attr_value == default_value) return;
        outAttribute(attr_name, Boolean.toString(attr_value), false);
    }

    /** 
     * Writes float attribute.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     */
    public final void outAttribute(String attr_name, float attr_value) {
        outAttribute(attr_name, Float.toString(attr_value), true);
    }

    /**
     * Writes a float attribute, if value differ than default one.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     * @param default_value the default value, if equals to the value, no output are done.
     */
    public final void outAttribute(String attr_name, float attr_value, float default_value) {
        if (attr_value == default_value) return;
        outAttribute(attr_name, Float.toString(attr_value), false);
    }

    /** 
     * Writes float attribute.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     */
    public final void outAttribute(String attr_name, double attr_value) {
        outAttribute(attr_name, Double.toString(attr_value), true);
    }

    /**
     * Writes a float attribute, if value differ than default one.
     * @param attr_name the name of the attribute
     * @param attr_value the value of the attribute
     * @param default_value the default value, if equals to the value, no output are done.
     */
    public final void outAttribute(String attr_name, double attr_value, double default_value) {
        if (attr_value == default_value) return;
        outAttribute(attr_name, Double.toString(attr_value), false);
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * @param default_value the boolean default value
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key, boolean default_value) {
        if (ps.hasKey(key)) outAttribute(attr_name, ps.get(key, default_value));
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * @param default_value the string default value
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key, String default_value) {
        if (ps.hasKey(key)) outAttribute(attr_name, ps.get(key, default_value), false);
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * @param default_value the double default value
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key, double default_value) {
        if (ps.hasKey(key)) outAttribute(attr_name, ps.get(key, default_value), default_value);
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * @param default_value the float default value
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key, float default_value) {
        if (ps.hasKey(key)) outAttribute(attr_name, ps.get(key, default_value), default_value);
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * @param default_value the integer default value
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key, int default_value) {
        if (ps.hasKey(key)) outAttribute(attr_name, ps.get(key, default_value), default_value);
    }

    /** Writes a property contained in a property store as an element attribute only
     * if the key property exists in the property store
     * @param attr_name the name of the attribute
     * @param ps the property store
     * @param key the property's key
     * 
     */
    public final void outPropertyAttr(String attr_name, DPropertyStore ps, String key) {
        if (ps.hasKey(key)) {
            Object o = ps.get(key);
            try {
                out.write(' ');
                out.write(attr_name.getBytes("UTF-8"));
                out.write('=');
                out.write('"');
                //need " detection here ?
                Element olde = curr_element;
                curr_element = null;
                if (o instanceof IDStringSerializable)
                    out.write(encode(((IDStringSerializable) o).serializeToString()).getBytes("UTF-8"));
                else
                    invokeDoMethod(o, this, null);
                out.write('"');
                curr_element = olde;
                curr_element.have_attr = true;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

    }

    /** 
     * Starts element and write common IDItem attribute (style).
     * It's your responsability to call endElement().
     * @see #startElement
     * @param name the name of the element.
     * @param item the item to write.
     */
    public void startElement(String name, IDItem item) {
        startElement(name);
    }

    /**
     * Encodes given string replacing a trap character by their xml entity names
     * (ie > by &amp;gt;, & by &amp;amp;). Replacement is done directly in s.
     * @param s the string to encode.
     * @return s after replacement
     */
    public static String encode(String s) {
        if (s == null) return ""; //$NON-NLS-1$
        s = s.replaceAll("&", "&amp;"); //$NON-NLS-1$ //$NON-NLS-2$
        s = s.replaceAll("'", "&apos;"); //$NON-NLS-1$ //$NON-NLS-2$
        s = s.replaceAll("<", "&lt;"); //$NON-NLS-1$ //$NON-NLS-2$
        s = s.replaceAll(">", "&gt;"); //$NON-NLS-1$ //$NON-NLS-2$
        s = s.replaceAll("\"", "&quot;"); //$NON-NLS-1$ //$NON-NLS-2$
        return s;
    }

    /**
     * Writes xml for a DDocument.
     */
    public void doMethod(DDocument d, DExtensible p, Object arg) {
        doChildrenItem(d, p, arg);
    }

    /**
     * Writes xml for a DGraphic
     */
    public void doMethod(DGraphic g, DExtensible p, Object arg) {
        
        startElement("chart", g);

        outPropertyAttr("width", g.getProperties(), DGraphic.P_MAX_WIDTH);
        outPropertyAttr("minWidth", g.getProperties(), DGraphic.P_MIN_WIDTH);
        outPropertyAttr("height", g.getProperties(), DGraphic.P_MAX_HEIGHT);
        outPropertyAttr("minHeight", g.getProperties(), DGraphic.P_MIN_HEIGHT);
        outPropertyAttr("backgroundColor", g.getProperties(), DGraphic.P_BACKCOLOR);

        writeLegend(g, p, arg);
        writeChartTitle(g, p, arg);
        writePlotArea(g, p, arg);

        // writes DGraphic child items expect DAxis and DCurve which are written
        // by writePlotArea
        for (IDItem c = g.getFirstChild(); c != null; c = c.getNext()) {
            if (c instanceof DGraphic) continue;
            
            if ((c instanceof DAxis) || (c instanceof DCurve) || (c instanceof DShapes) || (c instanceof DPalettes)) continue;
            invokeDoMethod(c, p, arg);
        }

        endElement();
    }

    /**
     * Writes 'internationalization' node
     */
    /*public void doMethod(DI18N i18n, DExtensible p, Object arg) {
        startElement("internationalization");
        outAttribute("language", i18n.getLanguage(), null);
        outAttribute("country", i18n.getCountry(), null);
        outAttribute("timezone", i18n.getTimeZone(), null);
        outAttribute("textDirection", i18n.getTextDirection(), "LTR");
        outAttribute("resourceBundle", i18n.getResourceBundle(), null);
        endElement();

    }*/

    /**
     * Writes 'preferences' node
     */
    /*public void doMethod(DPreferences pref, DExtensible p, Object arg) {
        startElement("preferences");
        outAttribute("url", pref.getUrl(), null);
        outAttribute("tooltip", pref.getTooltip(), null);
        outAttribute("show", pref.isShow());
        outAttribute("browserCookieName", pref.getBrowserCookieName(), null);
        outAttribute("updatedPreferencesVariable", pref.getUpdatedPreferencesVariable(), null);
        outAttribute("storedPreferences", pref.getStoredPreferences(), null);
        endElement();
    }*/

    /**
     * Writes DCategory element
     */
    public void doMethod(DCategory event, DExtensible ext, Object arg) {
        startElement("category");
        outAttribute("id", event.getId(), null);
        outAttribute("label", event.getLabel(), null);

        doChildrenItem(event, ext, arg);
        endElement();
    }

    /**
     * Writes DEvent element
     */
   /*public void doMethod(DEvent event, DExtensible ext, Object arg) {
        startElement("eventHandler");
        outAttribute("type", event.getType(), null);
        outCData(event.getHandler());
        endElement();
    }*/

    /**
     * Writes DAccessibility element
     */
   /* public void doMethod(DAccessibilityTitle acc, DExtensible ext, Object arg) {
        if (acc.getText() != null) {
            startElement("accessibilityTitle");
            outCData(acc.getText());
            endElement();
        }
    }*/

    /**
     * Writes 'accessibilityDesc' element
     */
    /*public void doMethod(DAccessibilityDesc acc, DExtensible ext, Object arg) {
        if (acc.getText() != null) {
            startElement("accessibilityDesc");
            outCData(acc.getText());
            endElement();
        }
    }*/

    /**
     * Writes common attibute and content for ChartItem
     */
    /*protected void writeChartItem(List events, String acc_t, String acc_desc, DExtensible p, Object arg) {
        if (events != null) {
            Iterator it = events.iterator();
            while (it.hasNext())
                invokeDoMethod((DEvent) it.next(), p, arg);
        }

        invokeDoMethod(new DAccessibilityTitle(acc_t), p, arg);
        invokeDoMethod(new DAccessibilityDesc(acc_desc), p, arg);
    }*/

    /**
     * Writes 'legend' node
     */
    protected void writeLegend(DGraphic g, DExtensible p, Object arg) {
        startElement("legend");
        
        outPropertyAttr("width", g.getProperties(), DGraphic.P_LEGEND_WIDTH, 100);
        outPropertyAttr("height", g.getProperties(), DGraphic.P_LEGEND_HEIGHT, 100);
        
        outPropertyAttr("sizingFactor", g.getProperties(), DGraphic.P_LEGEND_LIMIT, 4);
        outPropertyAttr("show", g.getProperties(), DGraphic.P_SHOW_LEGEND, true);

        DAlignmentPair align = (DAlignmentPair) g.getProperties().get(DGraphic.P_LEGEND_LAYOUT);
        if (align != null) {
           try { 
             Element oldelem = curr_element;
             curr_element = null;
             out.write(" alignment=\"".getBytes("UTF-8"));
             invokeDoMethod(align, p, arg);
             out.write('"');
             curr_element = oldelem;
         
           } catch (IOException e) {
              e.printStackTrace();
           }
        }

        //writeChartItem((List) g.getProperties().get(DGraphic.P_LEGEND_EVENT_LIST), (String) g.getProperties().get(DGraphic.P_LEGEND_ACC_TITLE), (String) g.getProperties().get(
        //        DGraphic.P_LEGEND_ACC_DESC), p, arg);

        endElement();
    }

    /**
     * Writes 'chartTitle' node
     */
    protected void writeChartTitle(DGraphic g, DExtensible p, Object arg) {
        startElement("chartTitle");
        outAttribute("label", g.getTitle() == null ? "" : g.getTitle(), true);
        outPropertyAttr("show", g.getProperties(), DGraphic.P_SHOW_TITLE, true);

        /*DAlignmentPair align = (DAlignmentPair) g.getProperties().get(DGraphic.P_TITLE_LAYOUT);
        if (align != null) {
           try { 
             Element oldelem = curr_element;
             curr_element = null;
             out.write(" alignment=\"".getBytes());
             invokeDoMethod(align, p, arg);
             out.write('"');
             curr_element = oldelem;
         
           } catch (IOException e) {
              e.printStackTrace();
           }
        }*/
        

       // writeChartItem((List) g.getProperties().get(DGraphic.P_TITLE_EVENT_LIST), (String) g.getProperties().get(DGraphic.P_TITLE_ACC_TITLE), (String) g.getProperties().get(
       //         DGraphic.P_TITLE_ACC_DESC), p, arg);

        endElement();
    }
    
    /**
     * Write DAlignmentPair value
     */
    
    public void doMethod(DAlignmentPair align, DExtensible p, Object arg) {
        String val = null;
        switch (align.getAlignment())
        {
           case IDAlignment.BOTTOM:
                val = "bottom"; break;
           case IDAlignment.TOP:
                val = "top"; break;
           case IDAlignment.LEFT:
                val = "left"; break;
           case IDAlignment.RIGHT:
                val = "right"; break;
        }
        
        if (val!=null)
            outCData(val);
    }

    /**
     * Writes 'timeStamp' node
     */
    /*public void doMethod(DTimeStamp ts, DExtensible p, Object arg) {
        startElement("timeStamp");
        outAttribute("dateFormat", ts.getDatePattern(), false);
        outAttribute("show", ts.isShow(), true);
        outAttribute("prefix", ts.getPrefix(), false);
        if (ts.getValue() != null) outAttribute("value", ts.getValue(), false);
        endElement();
    }*/

    private String getType(DGraphic _dgraphic, String type){
    	String returnType = type;
		boolean threeD = _dgraphic.getProperties().get(DGraphic.P_3D, false);
		if (DGraphic.T_HBARS.equals(type)){
			if (threeD)
				returnType = IGraphicTypeConstants.HBAR_CHART3D;			
			else
				returnType = IGraphicTypeConstants.HBAR_CHART;			
		}
		else if (DGraphic.T_HISTOGRAM.equals(type)){
			if (threeD)
				returnType = IGraphicTypeConstants.VBAR_CHART3D;
			else
				returnType = IGraphicTypeConstants.VBAR_CHART;
		}
		else if (DGraphic.T_STACKBARS.equals(type)){
			if (threeD)
				returnType = IGraphicTypeConstants.VSTACKBAR_CHART3D;
			else
				returnType = IGraphicTypeConstants.VSTACKBAR_CHART;
		}
		else if (DGraphic.T_SECTORS3D.equals(type))
			returnType = IGraphicTypeConstants.PIE_CHART3D;
		else if (DGraphic.T_SECTORS.equals(type)){
			returnType = IGraphicTypeConstants.PIE_CHART;
		}
		else if (DGraphic.T_METER.equals(type))
			returnType = IGraphicTypeConstants.METER;
		else if (DGraphic.T_XY.equals(type))
			returnType = IGraphicTypeConstants.LINE_CHART;
		return returnType;
		
    	
    }
    /**
     * Writes 'plotArea' node
     */
    protected void writePlotArea(DGraphic g, DExtensible p, Object arg) {
        startElement("plotArea");

        outAttribute("type", getType(g, g.getRenderableId()), true);
        outPropertyAttr("backgroundColor", g.getProperties(), DGraphic.P_PLOTAREA_BACKCOLOR);
        //outPropertyAttr("valueFormat", g.getProperties(), DGraphic.P_VALUE_FORMAT, null);
        outPropertyAttr("showPercentage", g.getProperties(), DGraphic.P_SHOW_PERCENTAGE, true);
        outPropertyAttr("showValues", g.getProperties(), DGraphic.P_SHOW_VALUES, true);
        outPropertyAttr("margin", g.getProperties(), DGraphic.P_MARGIN, 10);

        // shapes
        DShapes s = (DShapes) g.getChildOfClass(DShapes.class);
        if (s != null) invokeDoMethod(s, p, arg);

        // palettes
        DPalettes pal = (DPalettes) g.getChildOfClass(DPalettes.class);
        if (pal != null) invokeDoMethod(pal, p, arg);

        writeAxes(g, p, arg);
        writeCurves(g, p, arg);

        endElement();
    }

    /**
     * Writes 'include' node
     */
    /*public void doMethod(DInclude inc, DExtensible p, Object arg) {
        startElement("include");
        outAttribute("href", inc.getHref(), false);
        if (inc.getJavaScript() != null) outCData(inc.getJavaScript());
        endElement();
    }*/

    /**
     * Writes 'shapes' node
     */
    /*public void doMethod(DShapes shapes, DExtensible p, Object arg) {
        startElement("shapes");
        outAttribute("location", shapes.getLocation(), false);
        outAttribute("show", shapes.isShow());
        endElement();
    }*/

    /**
     * Writes 'palettes' node
     */
    public void doMethod(DPalettes pal, DExtensible p, Object arg) {
        startElement("palettes");
        outAttribute("location", pal.getLocation(), false);
        outAttribute("paletteSetId", pal.getPaletteSetId(), false);
        outAttribute("paletteId", pal.getPaletteId(), false);
        endElement();
    }

    /**
     * Writes 'axes' node
     */
    protected void writeAxes(DGraphic g, DExtensible p, Object arg) {
        startElement("axes");
        for (IDItem c = g.getFirstChild(); c != null; c = c.getNext()) {
            if (c instanceof DAxis) invokeDoMethod(c, p, arg);
        }
        endElement();

    }

    /**
     * Writes 'curves' node
     */
    protected void writeCurves(DGraphic g, DExtensible p, Object arg) {
        startElement("datasets");
        for (IDItem c = g.getFirstChild(); c != null; c = c.getNext()) {
            if (c instanceof DCurve) invokeDoMethod(c, p, arg);
        }
        endElement();
    }

    /**
     * Writes xml for a DAxis
     */
    public void doMethod(DAxis a, DExtensible p, Object arg) {
        startElement("axis", a);
        outAttribute("id", a.getName(), true);
        outAttribute("title", a.getTitle(), false);
        if (a.getScaleType() == DAxis.S_LIN)
            outAttribute("scale", "linear", "linear");
        else
            outAttribute("scale", "log", "linear");
        
        outPropertyAttr("min", a.getProperties(), DAxis.P_MIN, 0.0);
        outPropertyAttr("max", a.getProperties(), DAxis.P_MAX, 0.0);
        
		String degreeId = a.getProperties().get(DAxis.P_LABEL_ROTATION, DAxis.S_ZERO_DEGREE);
		double degree = 0;
		if (degreeId.equals(DAxis.S_MINUS_NINTY_DEGREE)) degree = -90;
		else if (degreeId.equals(DAxis.S_PLUS_NINTY_DEGREE)) degree = 90;
        outPropertyAttr("rotateLabel", a.getProperties(), DAxis.P_LABEL_ROTATION, degree);
        
        outPropertyAttr("showTitle", a.getProperties(), DAxis.P_SHOW_TITLE, true);
//        outPropertyAttr("dataformatType", a.getProperties(), DAxis.P_LABEL_FORMATTYPE);
        //outPropertyAttr("labelAlignment", a.getProperties(), DAxis.P_LABEL_ALIGNMENT);
        
        writeLabelFormat(a, p, arg);
        invokeDoMethod(a.getProperties(), p, arg);
        writeCategories(a, p, arg);
        writeMinorUnit(a, p, arg);
        writeMajorUnit(a, p, arg);

        // write other iditem except DCategory which are written by writeCategories
        for (IDItem c = a.getFirstChild(); c != null; c = c.getNext()) {
            if (c instanceof DCategory) continue;
            invokeDoMethod(c, p, arg);
        }

        endElement();
    }

    /** Writes 'labelFormat' element */
    protected void writeLabelFormat(DAxis a, DExtensible p, Object arg) {
        String labelFormatType = null;
        Object o = a.getProperties().get(DAxis.P_UNIT_FORMAT);
        if (o == null) {
            o = a.getProperties().get(DAxis.P_LABEL_FORMAT);
            labelFormatType = (String) a.getProperties().get((DAxis.P_LABEL_FORMATTYPE));
        }
        if (o == null) return;

        startElement("labelFormat");

        if (o instanceof SimpleDateFormat) {
            outAttribute("type", "date", false);
            outAttribute("pattern", ((SimpleDateFormat) o).toPattern(), false);
        } else if (o instanceof DecimalFormat) {
            outAttribute("type", "number", false);
            outAttribute("pattern", ((DecimalFormat) o).toPattern(), false);
        } else if (o instanceof String) {
            outAttribute("type", labelFormatType, false);
            outAttribute("pattern", (String) o, false);
        } else {
            outAttribute("type", o.getClass().getName(), false);
        }
        endElement();
    }

    /** Writes 'markerLine' element */
    public void doMethod(DMarkerLine ml, DExtensible p, Object arg) {
        startElement("markerLine");
        outAttribute("id", ml.getId(), false);

        if (ml.getValue() instanceof Date)
            outAttribute("value", SimpleDateFormat.getInstance().format(ml.getValue()), null);
        else if (ml.getValue() != null) 
        	 outAttribute("value", Double.toString(((Double) ml.getValue()).doubleValue()), false);

        outAttribute("label", ml.getLabel(), false);
        if (ml.getColor() != null) outAttribute("color", ml.getColor().serializeToString(), false);
        outAttribute("thickness", ml.getThickness(), 0.0);
        endElement();
    }

    /** Writes 'markerRegion' element */
    public void doMethod(DMarkerRegion ml, DExtensible p, Object arg) {
        startElement("markerRegion");
        outAttribute("id", ml.getId(), false);

        if (ml.getFromValue() != null) {
            if (ml.getFromValue() instanceof Date)
                outAttribute("fromValue", SimpleDateFormat.getInstance().format(ml.getFromValue()), null);
            else
                outAttribute("fromValue", ((Double) ml.getFromValue()).doubleValue(), 0.0);
        }

        if (ml.getToValue() != null) {
            if (ml.getToValue() instanceof Date)
                outAttribute("toValue", SimpleDateFormat.getInstance().format(ml.getToValue()), null);
            else
                outAttribute("toValue", ((Double) ml.getToValue()).doubleValue(), 0.0);
        }

        outAttribute("label", ml.getLabel(), false);
        if (ml.getColor() != null) outAttribute("color", ml.getColor().serializeToString(), false);
        endElement();
    }

    /** Writes 'category' element */
    protected void writeCategories(DAxis a, DExtensible p, Object arg) {
    	boolean categoryExist = false;
        for (IDItem c = a.getFirstChild(); c != null; c = c.getNext()) {
        	if (c instanceof DCategory){
        		categoryExist = true;
        		break;
        	}
        }
    	if (!categoryExist) return;
        startElement("categories");
        for (IDItem c = a.getFirstChild(); c != null; c = c.getNext()) {
            if (c instanceof DCategory) invokeDoMethod(c, p, arg);
        }

        endElement();
    }

    /** Writes 'majorUnit' element */
    protected void writeMajorUnit(DAxis a, DExtensible p, Object arg) {
        if (a.getProperties().get(DAxis.P_MAJUNIT_VALUE) != null || a.getProperties().get(DAxis.P_MAJUNIT_SHOWTICK) != null
                || a.getProperties().get(DAxis.P_MAJUNIT_SHOWGRID) != null) {
            startElement("majorUnit");
            outPropertyAttr("value", a.getProperties(), DAxis.P_MAJUNIT_VALUE, 0.0);
            outPropertyAttr("showTick", a.getProperties(), DAxis.P_MAJUNIT_SHOWTICK, false);
            outPropertyAttr("showGridLine", a.getProperties(), DAxis.P_MAJUNIT_SHOWGRID, false);
            endElement();
        }
    }

    /** Writes 'minorUnit' element */
    protected void writeMinorUnit(DAxis a, DExtensible p, Object arg) {
        if (a.getProperties().get(DAxis.P_MINUNIT_VALUE) != null || a.getProperties().get(DAxis.P_MINUNIT_SHOWTICK) != null
                || a.getProperties().get(DAxis.P_MINUNIT_SHOWGRID) != null) {
            startElement("minorUnit");
            outPropertyAttr("value", a.getProperties(), DAxis.P_MINUNIT_VALUE, 0.0);
            outPropertyAttr("showTick", a.getProperties(), DAxis.P_MINUNIT_SHOWTICK, false);
            outPropertyAttr("showGridLine", a.getProperties(), DAxis.P_MINUNIT_SHOWGRID, false);
            endElement();
        }
    }

    /**
     * Writes xml for a DCurve
     */
    public void doMethod(DCurve c, DExtensible p, Object arg) {
        startElement("dataset", c);
        outAttribute("label", c.getName(), false);
        outPropertyAttr("symbol", c.getProperties(), DCurve.P_SYMBOL, null);
        outPropertyAttr("color", c.getProperties(), DCurve.P_COLOR);

        doChildrenItem(c, p, arg);
        endElement();
    }

    /**
     * Writes xml for a DPoint
     */
    public void doMethod(DPoint p, DExtensible prs, Object arg) {
        startElement("point", p);

        doChildrenItem(p, prs, arg);
        endElement();
    }

    /**
     * Writes xml for a DCoord
     */
    public void doMethod(DCoord c, DExtensible p, Object arg) {
        startElement("coord");
        outAttribute("axis", c.getAxis().getName(), true);
        outAttribute("value", c.getValue()); //required
        endElement();
    }

    /**
     * Writes xml for a DCoord
     */
    public void doMethod(DCoordObject c, DExtensible p, Object arg) {
       
        startElement("coord");
        outAttribute("axis", c.getAxis().getName(), true);
        
        try {
           Element oldelem = curr_element;
           curr_element = null;
           out.write(" value=\"".getBytes("UTF-8"));
           invokeDoMethod(c.getValue(null), p, arg);
           out.write('"');
           curr_element = oldelem;
        
        } catch (IOException e) {
           e.printStackTrace();
        }
        endElement();
    
    }

    protected ArrayList folder_model_written_;

    /** encode IDColor into a 6 hexa-digit string RRGGBB */
    public static String EncodeColor(IDColor c) {
        int r = c.getRed();
        int g = c.getGreen();
        int b = c.getBlue();
        int rgb = (r << 16) | (g << 8) | b;
        String s = Integer.toString(rgb, 16);
        StringBuffer buf = new StringBuffer(6);
        for (int i = s.length(); i < 6; i++)
            buf.append('0');
        buf.append(s);
        return buf.toString();
    }

    /** aml dump for IDXmlSerializable class, write as cdata the writeXml() result. */
    public void doMethod(IDXmlSerializable i, DExtensible ext, Object arg) {
        outCData(i.writeXml());
    }

    /** aml dump for IDStringSerializable class, write as cdata the writeXml() result. */
    public void doMethod(IDStringSerializable i, DExtensible ext, Object arg) {
        outCData(i.serializeToString());
    }

    /** encode string using any '&xx;' xml entity and dump coded string */
    public void doMethod(String str, DExtensible ext, Object arg) {
        outCData(encode(str));
    }

    /** write xml for Byte objects. */
    public void doMethod(Byte v, DExtensible ext, Object arg) {
        outCData(((Byte) v).toString());
    }

    /** write xml for Short v. */
    public void doMethod(Short v, DExtensible ext, Object arg) {
        outCData(((Short) v).toString());
    }

    /** write xml for Integer objects. */
    public void doMethod(Integer v, DExtensible ext, Object arg) {
        outCData(((Integer) v).toString());
    }

    /** write xml for Long objects. */
    public void doMethod(Long v, DExtensible ext, Object arg) {
        outCData(((Long) v).toString());
    }

    /** write xml for Float objects. */
    public void doMethod(Float v, DExtensible ext, Object arg) {
        outCData(((Float) v).toString());
    }

    /** write xml for Double objects. */
    public void doMethod(Double v, DExtensible ext, Object arg) {
        outCData(((Double) v).toString());
    }

    /** write xml for Boolean objects. */
    public void doMethod(Boolean v, DExtensible ext, Object arg) {
        outCData(((Boolean) v).toString());
    }

    /** write xml for Class object, onl name of class is save */
    public void doMethod(Class c, DExtensible ext, Object arg) {
        outCData(c.getName());
    }

    /** write xml for java.util.Date, save the time in milliseconds */
    public void doMethod(Date date, DExtensible ext, Object arg) {
        outCData(SimpleDateFormat.getInstance().format(date));
    }

    /** write xml for DecimalFormat save pattern */
    public void doMethod(com.ibm.icu.text.DecimalFormat df, DExtensible ext, Object arg) {
        outCData(df.toPattern());
    }

    /** write xml for SimpleDateFormat save pattern */
    public void doMethod(com.ibm.icu.text.SimpleDateFormat df, DExtensible ext, Object arg) {
        outCData(df.toPattern());
    } 

    /**
     * @return an internal ID to saves colors, fonts, styles in jsml. Object must not be null.
     */
    private String getInternalID(Object o) {
        return Integer.toString(o.hashCode(), 16);
    }

}