/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DXmlReader.java,v 1.5 2008/05/23 14:12:01 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.drivers.xml.internal;


import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DBorder;
import org.eclipse.tptp.platform.report.core.internal.DCell;
import org.eclipse.tptp.platform.report.core.internal.DCellText;
import org.eclipse.tptp.platform.report.core.internal.DColor;
import org.eclipse.tptp.platform.report.core.internal.DColorRegistry;
import org.eclipse.tptp.platform.report.core.internal.DCoord;
import org.eclipse.tptp.platform.report.core.internal.DCoordObject;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DCurveLink;
import org.eclipse.tptp.platform.report.core.internal.DData;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DFolder;
import org.eclipse.tptp.platform.report.core.internal.DFolderModel;
import org.eclipse.tptp.platform.report.core.internal.DFont;
import org.eclipse.tptp.platform.report.core.internal.DFontRegistry;
import org.eclipse.tptp.platform.report.core.internal.DFooter;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DHeader;
import org.eclipse.tptp.platform.report.core.internal.DImage;
import org.eclipse.tptp.platform.report.core.internal.DIndex;
import org.eclipse.tptp.platform.report.core.internal.DIndexEntry;
import org.eclipse.tptp.platform.report.core.internal.DItem;
import org.eclipse.tptp.platform.report.core.internal.DItemContainer;
import org.eclipse.tptp.platform.report.core.internal.DLine;
import org.eclipse.tptp.platform.report.core.internal.DLink;
import org.eclipse.tptp.platform.report.core.internal.DList;
import org.eclipse.tptp.platform.report.core.internal.DPageBreak;
import org.eclipse.tptp.platform.report.core.internal.DPageCounter;
import org.eclipse.tptp.platform.report.core.internal.DParagraph;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPointLink;
import org.eclipse.tptp.platform.report.core.internal.DPopup;
import org.eclipse.tptp.platform.report.core.internal.DPropertyStore;
import org.eclipse.tptp.platform.report.core.internal.DRow;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DStyleRegistry;
import org.eclipse.tptp.platform.report.core.internal.DSummary;
import org.eclipse.tptp.platform.report.core.internal.DTable;
import org.eclipse.tptp.platform.report.core.internal.DTag;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.DTitle;
import org.eclipse.tptp.platform.report.core.internal.DWallpaper;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.core.internal.IDStringSerializable;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.DSymbolRegistry;
import org.eclipse.tptp.platform.report.drawutil.internal.ISymbol;
import org.eclipse.tptp.platform.report.drivers.internal.IReader;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.extension.internal.DExtensionRegistry;
import org.eclipse.tptp.platform.report.extension.internal.IDExtension;
import org.eclipse.tptp.platform.report.tools.internal.DAlignment;
import org.eclipse.tptp.platform.report.tools.internal.DParser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.ibm.icu.text.DecimalFormat;
	  
/**
 * DXmlReader is an extensible class that parse an InputStream and a XML stream
 * to build a jscrib IDItem tree and associated registries (colors,fonts and styles).<br>
 * The following example reads a document from the xml file named 'myfile.jsml'.<p>
 * <PRE>
 *    DXmlReader reader = new DXmlReader();
 *    try{
 *      IDObject [] docs = reader.read( new FileInputStream("myfile.jsml") );
 *      // returns only the first document. 
 *      if (docs.lenght>0 && (docs[0] instanceof DDocument)
 *          return (DDocument)docs[0];
 *    }
 *    catch(Exception e)
 *    {
 *      // an error occurs during the file parsing. 
 *      ...     
 *    }
 * 
 * </PRE> 
 * 
 * @see #read
 * @see DExtensible 
 * @see IReader
 * @see InputStream
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DXmlReader extends DParser implements IReader
{	  
  
  /** 
   * "Null" item a factory could return to said "node you give me is Ok, but it doesn't create any IDItem",
   * so the factory caller could forget this item without reported 'unknown node' error 
   */
  public static class DDummy extends DItem {}
  
  /** 
   * DXmlReader extenders may want to create their IDItem from their xml node name.
   * This factory must be registered at updateExtensible() time.
   * @see #addIDItemFactory
   */
  public static interface IDItemFactory
  {
    IDItem createIDItem( Node node );
  }

  /** 
   * Argument class for doMethod(), and instance of Arg class will be given as the third parameter.
   * If you aren't aware of doMethod(), please refer to DExtension and DExtensible classes.
   */
  public static class Arg
  {
    /** current node that engage doMethod() call */
    public Node node; 

    /** 
     * Do not parse this children element name, this is set by doMethod() and
     * used in doChildrenItem().
     * @see org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlReader#doMethod(DSection,DExtensible,Object) for an example.
     */
    public String filter_children[];
    
    /**
     * id returned by DColor or DFont have parsed, if they want to be integrated in registry.
     * (id must be >=0), as a String like getNodeValue() return.
     */
    public String ret_id ;
    
    public Arg( Node n ) { node=n; ret_id=null; }
    
    /** @return true if node_name is in filtered children list */
    public boolean filterChildren( String node_name )
    {
      if( filter_children == null ) return false;
      for( int i=0; i<filter_children.length; ++i )
      {
        String s = filter_children[i];
        if( s==null ) continue;
        if( s.equals( node_name ) ) return true;
      }
      return false;
    }
  }
   
    /** list of registered factoryies used to create IDItem object from element names */
    private List iditem_factories;

	private DStyleRegistry styles;
	private DColorRegistry colors;
	private DFontRegistry  fonts;

    /** current map of IDFont using their id as key. */
	private HashMap fontsId;
    /** current map of IDColor using their id as key. */
	private HashMap colorsId; //
    /** current map of IDStyle using their id as key. */
    private HashMap stylesId; //
    /** current map of DAxis using their names as key. */
	private HashMap axis;
    /** Xml parser */
	private IXmlParser parser_;
 
	
	/**
     * Create a DXmlReader, ready to parse input @see read(InputStream)
	 */
	public DXmlReader( IXmlParser parser ) 
	{
	  parser_ = parser;
      fontsId  = new HashMap();
      colorsId = new HashMap();
      stylesId = new HashMap();
      axis     = new HashMap();
      DExtensionRegistry.sgn_addExtension.connect( this, "updateExtension(org.eclipse.tptp.platform.report.extension.internal.IDExtension, Class)");
      DExtensionRegistry.sgn_removeExtension.connect( this, "updateExtension(org.eclipse.tptp.platform.report.extension.internal.IDExtension, Class)");
	}
		

	public DColorRegistry getColorRegistry()
	{
		return colors;
	}
	
	public DFontRegistry getFontRegistry()
	{
		return fonts;
	}
	
	public DStyleRegistry getStyleRegistry()
    {
		return styles;
    }
    
    /** override current color registry using the new one */
    public void setColorRegistry( DColorRegistry r )
    {
      colors = r;
    }
  
    /** override current font registry using the new one */
    public void setFontRegistry( DFontRegistry r)
    {
      fonts = r;
    }
  
    /** override current style registry using the new one */
    public void setStyleRegistry( DStyleRegistry r )
    {
      styles =r;
    }
	
    /**
     * Add an IDItem factory called to create an IDItem instance from a node (name).
     * All registered factories are called before default one.
     */
    public void addIDItemFactory( IDItemFactory f )
    {
      if( iditem_factories==null ) iditem_factories = new ArrayList();
      iditem_factories.add( f );
    }
    
    /**
     * Remove an IDItem factory previously registered by addIDItemFactory.
     */
    public void removeIDItemFactory( IDItemFactory f )
    {
      if( iditem_factories==null ) return;
      iditem_factories.remove( f );
    }
    
    public void clear()
    {
      fontsId.clear();
      colorsId.clear();
      stylesId.clear();
      axis.clear();
      folder_model_read_ = null;
    }
    
    /** 
     * Modify this name means you need to modify the same in DmlReader.
     * @return Xml root element name for jscrib documents
     */
    public String getRootElementName() { return "JSCRIB"; }

     /**
      * Parse input stream building one or more documents updating colors,fonts and
      * styles registry (@see getStyleRegistry()).
      * @param in input stream to use for xml input.
      * @return one or mode DDocument in an array (can be null).
      * @throws various exception (ie java.lang.IOException), or DXmlError
      */
     public IDObject[] read( InputStream in ) throws Exception
     {
       clear();      
       parser_.parse( new InputSource(in) );
	   Document dom_doc = parser_.getDocument();

	   if (dom_doc == null) throw new DXmlError("no document to read");
 	  
       Element root = dom_doc.getDocumentElement();
	   String root_name = getRootElementName();	 
	   if (root == null || !root.getNodeName().equals( root_name )) {
		  throw new DXmlError("no '"+root_name+"' root element found in document");
	   } 

	   // initialize all registries 
	   parseStyles( dom_doc );
	  
       ArrayList docs = new ArrayList();
       NodeList l = root.getChildNodes();
       Arg arg = new Arg(null);
       //do not parse again FONTS/COLORS/STYLES elements
       for( int i=0; i<l.getLength(); ++i )
       {
         Node n = l.item(i);         
         if( n==null ) continue;
         if( n.getNodeType()!=Node.ELEMENT_NODE ) continue;
         if( "DOCUMENT".equals( n.getNodeName() ) )
         {
           DDocument doc = new DDocument();
           arg.node = n;      
           invokeDoMethod( doc, this, arg );
           docs.add( doc );
         }
//TODO: as it's required to chart-applet test by as it's require XSD and EMF ....         
         else if( "GRAPHIC".equals( n.getNodeName() ) )
         {
           DGraphic gph = new DGraphic();
           arg.node = n;      
           invokeDoMethod( gph, this, arg );
           docs.add( gph );
         }
       }

       if( docs.size() == 0 ) return null;
       IDObject a[] = new IDObject[ docs.size() ];
       for( int i=0; i<docs.size(); ++i )
       {
         a[i] = (IDObject)docs.get(i);
       }
	   return a;
	 }

	 /** configure DColor from xml. */
	 public void doMethod( DColor color, DExtensible ext, Object arg)
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		Node att_id = getRequiredAttribut(attr, "ID", "COLOR");
        Node n;
        int r=0, g=0, b=0;
        if( (n=attr.getNamedItem("R"))!=null) r= Integer.parseInt(n.getNodeValue());
        if( (n=attr.getNamedItem("G"))!=null) g= Integer.parseInt(n.getNodeValue());
        if( (n=attr.getNamedItem("B"))!=null) b= Integer.parseInt(n.getNodeValue());
		        
		color.setRGB( r, g, b );
        ((Arg)arg).ret_id = att_id.getNodeValue();    
	 }

     /** configure DFont from xml . */
	 public void doMethod( DFont font, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		Node att_id = getRequiredAttribut(attr, "ID", "FONT");
			
        Node n = attr.getNamedItem("SIZE");
		if (n != null) font.setSize(Integer.parseInt( n.getNodeValue()) );
        n = attr.getNamedItem("FAMILY");
		if ( n != null) font.setFamily( n.getNodeValue() );
        n = attr.getNamedItem("SCRIPT");
		if ( n != null) font.setScript( n.getNodeValue() );
							   
        n = attr.getNamedItem("STYLE");
        if ( n != null) font.setStyle( n.getNodeValue() );

		((Arg)arg).ret_id = att_id.getNodeValue();   
	 }
	 
     /** configure DStyle from xml. */
	 public void doMethod(DStyle style,DExtensible ext, Object a_arg)
     {
        Arg arg = (Arg)a_arg;
		NamedNodeMap attr = arg.node.getAttributes();
		Node n ;
    
        arg.ret_id = getRequiredAttribut(attr, "ID", "STYLE").getNodeValue();
        
        n = attr.getNamedItem("NAME");
        if( n!=null )
        {
		  style.setName( n.getNodeValue() );
        }
        
        n = attr.getNamedItem("FONT");
		if ( n != null) {
		   DFont font = (DFont)fontsId.get( n.getNodeValue() );   
		   if (font == null) 
			  throw new DXmlError("Undeclared FONT ID ("+n.getNodeValue()+") while parsing STYLE tag");
		   style.setFont(font);
		}
        n = attr.getNamedItem("FORECOLOR");
		if ( n != null) {
     	   DColor fore = (DColor)colorsId.get( n.getNodeValue() );
     	   if (fore == null) 
			  throw new DXmlError("Undeclared FORECOLOR ID ("+n.getNodeValue()+") while parsing STYLE tag");
		   style.setForeColor(fore);
     	}
        n = attr.getNamedItem("BACKCOLOR");
		if ( n != null) {
		   DColor back = (DColor)colorsId.get( n.getNodeValue() );
		   if (back == null) 
			  throw new DXmlError("Undeclared BACKCOLOR ID ("+n.getNodeValue()+") while parsing STYLE tag");
		   style.setBackColor(back);
		}
        n = attr.getNamedItem("WALLPAPER");
        if( n!=null )
        {
          DWallpaper wp = new DWallpaper(null);
          wp.serializeFromString( n.getNodeValue() );
          style.setWallpaper( wp );
        }
     }
     
     /**
      * Parse styles from DOM document building font,color and styles registry 
      */
	 private void parseStyles( Document dom_doc )
	 {	 	
	 	// registry initialisation
	 	colorsId.clear();
	 	fontsId.clear();
        stylesId.clear();
	 	if(styles!=null) styles.clear();
        if(colors!=null) colors.clear();
        if(fonts !=null) fonts.clear();
	 	
	 	// get colors
	 	NodeList List = dom_doc.getDocumentElement().getElementsByTagName("COLORS");
		for (int i = 0; i<List.getLength(); i++) 
		{
			NodeList colorList = (List.item(i)).getChildNodes();			
			for(int j=0; j < colorList.getLength(); j++)
			{
				if ( "COLOR".equals( colorList.item(j).getNodeName()) )
                {
                  DColor c = new DColor();
                  Arg arg = new Arg( colorList.item(j) );
			      invokeDoMethod( c, this, arg  );
                  if( arg.ret_id != null )
                  {
                    colorsId.put( arg.ret_id, c);
                    if(colors==null) colors = new DColorRegistry();
                    colors.putColor( c );
                  }
                }
			}
		}
		
	 	// get fonts
		List = dom_doc.getDocumentElement().getElementsByTagName("FONTS");
		for (int i = 0; i<List.getLength(); i++) 
		{
			NodeList fontList = (List.item(i)).getChildNodes();
			for(int j=0; j < fontList.getLength(); j++)
			{
				if ( "FONT".equals( fontList.item(j).getNodeName()) )
                {
                  DFont f = new DFont();
                  Arg arg = new Arg( fontList.item(j) );
 			      invokeDoMethod( f, this, arg );
                  if( arg.ret_id != null )
                  {
                    fontsId.put( arg.ret_id, f );
                    if( fonts==null ) fonts = new DFontRegistry();
                    fonts.putFont( f );
                  }
                }
			}
		}
	 	
	 	// get styles
		List = dom_doc.getDocumentElement().getElementsByTagName("STYLES");
		for (int i = 0; i<List.getLength(); i++) 
		{
			NodeList styleList = (List.item(i)).getChildNodes();
			for(int j=0; j < styleList.getLength(); j++)
			{
				if ( "STYLE".equals(styleList.item(j).getNodeName()))
                {
                  DStyle s = new DStyle();
                  Arg arg = new Arg(styleList.item(j) );
  			      invokeDoMethod( s,this, arg );
                  if( arg.ret_id != null )
                  {
                     stylesId.put( arg.ret_id, s );
                     if( styles==null ) styles = new DStyleRegistry();
                     styles.putStyle(s);
                  }
                }
			}
		}
	 }
	 
	 
	/**
     * Replace few xml entity '&xxx;' to their char value in string.
     * @return modified string, or s directly.
	 */
	public static String decode(String s)
	{
		if (s == null) return s;
		s = s.replaceAll("&apos;", "'");
		s = s.replaceAll("&lt;", "<");
		s = s.replaceAll("&gt;", ">");
		s = s.replaceAll("&quot;", "\"");
		s = s.replaceAll("&amp;","&");
		return s;
	}   	
	
    /** 
     * Check if attribute attr_tag is defined in attribute map attr.
     * @return attribute found
     * @throws DXmlError if attribute attr_tag is not found in node map attr.
     */
	protected Node getRequiredAttribut(NamedNodeMap attr, String attr_tag, String elt_tag  )
	{
      Node n = attr.getNamedItem(attr_tag);
	  if (n == null) 
		throw new DXmlError(attr_tag +" attribute not defined while parsing " + elt_tag +" tag");
      return n;
	}  	
	
    /** 
     * If attribute map contains 'STYLE' attribute, get it from style registry and apply this
     * style on givent item.
     */
	public void setNodeStyle( NamedNodeMap attr , IDItem item)
	{
      Node n = attr.getNamedItem("STYLE");
	  if ( n != null)
		   item.setStyle((IDStyle)stylesId.get(n.getNodeValue()));		
	}
	
    /**
     * @return text from node converted from UTF-8 encoding.
     */
	public static String getNodeText(Node node )
	{
		String s = "";
		for ( Node child = node.getFirstChild(); child!=null; child=child.getNextSibling() )
		{
		  if ( child.getNodeType() == Node.TEXT_NODE )
		  {
            String sc = child.getNodeValue();
            /*InputSource (?from InputStream?)  seems to handle directly UTF-8 decoding
             * if not, try this:
			try
			{
				s = decode(new String( sc.getBytes(), "UTF-8"));
			}
			catch (UnsupportedEncodingException e)
			{
				s = decode(sc);
			}*/
//TODO: if UTF-8 is done, perhaps decoding &xx; is done too...      
            s += decode(sc);
		  }
		}
		return s;
	}
  
     /** 
      * Create a IDItem instance from given str.
      * @param str the node name to use to create IDItem instance.
      * @param ext a DXmlReader.
      * @param o_arg a CreateIDItemArg instance.
      * @return an IDItem or null if node isn't an iditem element.
      */
     protected IDItem createIDItemForNode( Node node )
     {          
       //an extension may provide this:
       if ( iditem_factories !=null )
       {
         for( int i=0; i<iditem_factories.size(); ++i )
         {
           IDItemFactory f = (IDItemFactory)iditem_factories.get(i);
           IDItem item = f.createIDItem( node );
           if( item instanceof DDummy ) return null;
           if( item!=null ) return item;
         }
       }  
       
       //'default' factory is here:
       String name = node.getNodeName();
       
       if( name==null ) return null ;
       
       switch( name.length() )
       {
         case 3: if (name.equals("TAG"))return new DTag();
                 if (name.equals("ROW")) return new DRow();
                 break;
         case 4: if (name.equals("TEXT"))  return new DText();
                 if (name.equals("CELL")) return  new DCell();
                 if (name.equals("LIST"))return new DList();
                 if (name.equals("LINK"))return new DLink();
                 if (name.equals("DATA"))return new DData();
                 if (name.equals("AXIS"))return new DAxis("unnamed");
                 if (name.equals("LINE"))return new DLine();
                 break;
         case 5: if (name.equals("COORD"))return new DCoord(null,0.0f);
                 if (name.equals("POINT"))return new DPoint();
                 if (name.equals("IMAGE"))return new DImage();
                 if (name.equals("CURVE"))return new DCurve();
                 if (name.equals("TITLE"))return new DTitle();
                 if (name.equals("TABLE"))return new DTable();
                 if (name.equals("POPUP"))return new DPopup();
                 if (name.equals("INDEX"))return new DIndex();
                 break;
         case 6: if (name.equals("HEADER")) return new DHeader();
                 if (name.equals("FOOTER"))return new DFooter();
                 if (name.equals("FOLDER"))return new DFolder();
                 if (name.equals("BORDER"))return new DBorder();
                 if (name.equals("IDITEM"))
                 {  
                   NamedNodeMap attr = node.getAttributes();
                   Node a_class = getRequiredAttribut( attr, "CLASS", "IDITEM" );
                   String class_name = a_class.getNodeValue();
                   return createFromClass( class_name );
                 }
                 break;
         case 7: if (name.equals("GRAPHIC"))return new DGraphic();
                 if (name.equals("SECTION"))return new DSection();
                 if (name.equals("SUMMARY"))return new DSummary();
                 break;
         case 8: if (name.equals("DOCUMENT"))return new DDocument(); 
                 break;
         case 9: if (name.equals("PARAGRAPH")) return new DParagraph();
                 if (name.equals("CELL_TEXT")) return new DCellText();
                 if (name.equals("COORD_OBJ")) return new DCoordObject();
                 break;
         case 10:if (name.equals("PAGE_BREAK"))return new DPageBreak();
                 if (name.equals("POINT_LINK"))return new DPointLink();
                 if (name.equals("CURVE_LINK"))return new DCurveLink();
                 break;
                 
         default: 
           if (name.equals("INDEX_ENTRY"))return new DIndexEntry();
           if (name.equals("PAGE_COUNTER"))return new DPageCounter();
           
           if( name.equals("JSML_EXTENSION"))
           {
             NamedNodeMap attr = node.getAttributes();
             Node a_class = getRequiredAttribut( attr, "CLASS", "JSML_EXTENSION" );
             String class_name = a_class.getNodeValue();
             IDItem item= createFromClass( class_name );
             //this element serve only to create and load .class containing xml reader (and others)
             //extension installers..
             return null;
           }
       }       
        throw new DXmlError( name + " is unknown node.");
//System.err.println( name + " is unknown in node "+node.getParentNode().getNodeName());
//       return null;
     }
     
     protected IDItem createFromClass( String class_name )
     {
       try
       {
         Class clazz = Class.forName( class_name );
         Object inst = clazz.newInstance();        
         return (IDItem)inst;
       }
       catch (Exception e)
       {
         if (e instanceof InstantiationException) 
           //|| (e instanceof IllegalAccessException))
         {
           System.out.println("Can't instanciate class '"+class_name+"',  haven't a public empty constructor ?");
           return null;
         }
         if (e instanceof ClassNotFoundException) 
           //|| (e instanceof InstantiationException) 
           //|| (e instanceof IllegalAccessException))
         {
           System.out.println("Can't found class = " + class_name);
         }
       }
       return null;
     }

     /**
      * invokeDoMethod() for all node contained in arg (@see class Arg), that aren't
      * filtered (@see Arg class again).
      */
     public void doChildrenItem( IDItem parent, DExtensible ext, Object a_arg)
	 {	
        Arg arg = (Arg)a_arg;
        Node n = arg.node ;
		if (!n.hasChildNodes()) return;
	 	
	 	Element e = null;
	 	if (n.getNodeType()==Node.TEXT_NODE) {
	 		return;
	 	}
	 	else if (n.getNodeType() == Node.ELEMENT_NODE) 
	 	{
	 		e = (Element)n;
	 	}
	 	
	 	NodeList list = e.getChildNodes();
	 	IDItem previous = null;
	 	for (int i=0; i<list.getLength(); i++)
	 	{
	 		IDItem child = null;
	 		Node node = list.item(i);
	 		
			if (node.getNodeType()!=Node.ELEMENT_NODE) {
				continue;
			}	
       
            //filtered node ?
            if( arg.filterChildren( node.getNodeName() ) )continue;

            child = createIDItemForNode( node );
	 		 		
	 		if (child == null) continue; //node is not a IDItem
            invokeDoMethod( child, this, new Arg(node) );

	 		
/*documents	 		if (l != null)
	 		   l.add(child);
	 		*/

	 		if (parent == child)
	 		    throw new DXmlError("Parent = " + parent.getClass());
	 		
	 		if (parent != null)
	 		{	
			    parent.insertChild(child, previous);
			    previous = child;
	 		}
	 	}
	 }

	 
     /** configure document, and its children */
	 public void doMethod( DDocument doc, DExtensible ext, Object arg)
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
        Node n = attr.getNamedItem("TITLE");
	 	if( n!=null ) doc.setTitle( n.getNodeValue() );
        doc.setTag( getRequiredAttribut( attr, "TAG", node.getNodeName() ).getNodeValue() );

		setNodeStyle(attr,doc);

        doChildrenItem( doc, ext, arg );
	 }

     /** configure section and its children */
     public void doMethod( DSection sec, DExtensible ext, Object arg)
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
        sec.setTag( getRequiredAttribut( attr, "TAG", node.getNodeName() ).getNodeValue() );
		setNodeStyle(attr,sec);

        for( Node c=node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          String name = c.getNodeName();
          if( "HEADER".equals( name ) )
          {
            if( sec.getHeader()!=null )
            {
              throw new DXmlError("more than one HEADER defined in section");
            }
            DHeader header = new DHeader();
            invokeDoMethod( header, ext, new Arg( c ) );
            sec.setHeader( header );
          }
          else if ( "FOOTER".equals( name ) )
          {
            if( sec.getFooter()!=null )
            {
              throw new DXmlError("more than one FOOTER defined in section");
            }
            DFooter footer = new DFooter();
            invokeDoMethod( footer, ext, new Arg( c ) );
            sec.setFooter( footer );
          }
        }
        
        Arg aarg = (Arg)arg;
        String save[] = aarg.filter_children;
        aarg.filter_children = filter_HEADER_FOOTER;

		doChildrenItem(sec, ext, arg);
    
        aarg.filter_children = save;
	 }
   
     private static String[] filter_HEADER_FOOTER = { "HEADER", "FOOTER" };
	
     /** configure footer and add it in last parsed section */
	 public void doMethod( DFooter foot, DExtensible ext, Object arg)
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		setNodeStyle(attr,foot);
		
		doChildrenItem( foot, ext, arg );		
	 }
   
	 /** configure header and add it in last parsed section */
	 public void doMethod( DHeader head, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		setNodeStyle(attr,head);
	 	   
		doChildrenItem( head, ext, arg );
	 }
   
	 /** configure paragraph */
	 public void doMethod(DParagraph para, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
	 	Node n = attr.getNamedItem("ALIGNMENT");
		if ( n != null)
		   para.setAlignment( DAlignment.StringToAlignment(n.getNodeValue()));
		setNodeStyle(attr,para);
		
		doChildrenItem(para, ext, arg );
	 }
	 
//	 public DData doNode( DData data, Node node) throws AmlException
//	 {
//		NamedNodeMap attr = node.getAttributes();
//		
//   	setNodeStyle(attr,data);
//	 	   
//	 	if (node.hasChildNodes())
//	 	{
//	 		
//	 	}
//	 	else
//	 	{
//	 		doNode
//	 	}
//	 	
//		NamedNodeMap attr = node.getAttributes();
//
//		NodeList list = node.getChildNodes();
//		for (int i=0; i<list.getLength(); i++)
//		{
//		  Node child = list.item(i);
//		  if (child.getNodeType()== Node.TEXT_NODE)
//			 data.setData(node.getNodeValue());
//		}
//		
//		setNodeStyle(attr,data);
//	 	   
//		return data;
//	 }
	    
    /** configure DItem applying  style attribute */
//TODO: quand c'est appel ca ?  puisque le reader detecte la classe pour creer l'objet et appeler
//le bon doMethod() ... si on passe ici ce n'est que pour le style, pas pour l'attribut CLASS... 
	public void doMethod( DItem it, DExtensible ext, Object arg )
	{
	   NamedNodeMap attr = ((Arg)arg).node.getAttributes();

	   setNodeStyle(attr,it);
	}

    /** configure a DItemContainer, and its children */       
	public void doMethod( DItemContainer ic,  DExtensible ext, Object arg )
	{
	   NamedNodeMap attr = ((Arg)arg).node.getAttributes();
	   setNodeStyle(attr,ic);
		
	   doChildrenItem( ic, ext, arg );
	 }
     
     /** configure a DText */
	 public void doMethod( DText text,  DExtensible ext, Object arg )
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
		setNodeStyle(attr,text);
		
		text.setText(getNodeText(node));
	 }
 
     /** configure DImage */
 	 public void doMethod( DImage img, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		img.setName(getRequiredAttribut(attr, "NAME", "IMAGE").getNodeValue());
        Node n ;
        if((n=attr.getNamedItem("SCALEW"))!=null )
		  img.setWidthScale(Float.parseFloat(n.getNodeValue()));
        if((n=attr.getNamedItem("SCALEH"))!=null )
		  img.setHeightScale(Float.parseFloat(n.getNodeValue()));
		setNodeStyle(attr,img);
	 }

     /** configure DList */
     public void doMethod( DList list, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		
		list.setFormat(Integer.parseInt(getRequiredAttribut(attr, "TYPE", "LIST").getNodeValue()));
		setNodeStyle(attr,list);
	 	
		doChildrenItem( list, ext, arg );
	 }
 
     /** configure a DTable */
	 public void doMethod( DTable tab, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		Node n ;
		if ((n=attr.getNamedItem("SCALEW")) != null)
		   tab.setWidthScale(Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("SCALEH")) != null)
		   tab.setHeightScale(Float.parseFloat(n.getNodeValue()));
		if ((n=attr.getNamedItem("BORDER")) != null)
		   tab.setBorder(Integer.parseInt(n.getNodeValue()));
		
		if ((n=attr.getNamedItem("WIDTH")) != null)
		{
			String s = n.getNodeValue();
			String[] split = s.split("/");
			float[] cw = new float[split.length];
			try {
			  for (int i=0; i<split.length; i++)
			  {
			    String sp = split[i];
			    cw[i] = Float.parseFloat(sp);
			  }
			  tab.setColumnWidth(cw);
			} catch( NumberFormatException e ) {
			  throw new DXmlError("Invalid table's width attribute, valid format is: WIDTH=\"(float ('/' float)*)\"");
			}
		}
		setNodeStyle(attr,tab);
	 	   
		doChildrenItem(tab, ext, arg );
	 }
     
     /** configure a DRow */
	 public void doMethod( DRow row, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		setNodeStyle(attr,row);

		doChildrenItem(row, ext, arg );
	 }

     /** configure a DCell */
	 public void doMethod( DCell cell, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
        Node n;
		if ((n=attr.getNamedItem("COLSPAN")) != null)
		   cell.setColSpan(Integer.parseInt(n.getNodeValue()));
		if ((n=attr.getNamedItem("ROWSPAN")) != null)
		   cell.setRowSpan(Integer.parseInt(n.getNodeValue()));
		if ((n=attr.getNamedItem("ALIGNMENT")) != null)
		   cell.setAlignment( DAlignment.StringToAlignment(n.getNodeValue()));
		setNodeStyle(attr,cell);
		

		doChildrenItem(cell, ext, arg );
	 }

     /** configure a DCellText */
	 public void doMethod( DCellText ct, DExtensible ext, Object arg )
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
        Node n;
		if ((n=attr.getNamedItem("COLSPAN")) != null)
		   ct.setColSpan(Integer.parseInt(n.getNodeValue()));
		if ((n=attr.getNamedItem("ROWSPAN")) != null)
		   ct.setRowSpan(Integer.parseInt(n.getNodeValue()));
		if ((n=attr.getNamedItem("ALIGNMENT")) != null)
			   ct.setAlignment( DAlignment.StringToAlignment(n.getNodeValue()));
		setNodeStyle(attr,ct);
		
		ct.setText(getNodeText(node));
		
	 	doChildrenItem( ct, ext, arg );
	 }

     /** configure a DLink */
	 public void doMethod( DLink ln, DExtensible ext, Object arg )
	 { 
        Node node=((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
		ln.setTarget(getRequiredAttribut(attr, "TARGET", node.getNodeName()).getNodeValue());
		setNodeStyle(attr,ln);
		
		doChildrenItem(ln, ext, arg );
	 }
     /** configure a DTag */
	 public void doMethod( DTag tag, DExtensible ext, Object arg )
	 {
        Node node=((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
		tag.setTag(getRequiredAttribut(attr, "NAME", node.getNodeName()).getNodeValue());
		setNodeStyle(attr,tag);
	 }

     /** configure a DLine */
	 public void doMethod( DLine line, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
        Node n;
		if ((n=attr.getNamedItem("SIZE")) != null)
		   line.setSize(Integer.parseInt( n.getNodeValue()));
        if ((n=attr.getNamedItem("TYPE")) != null)
           line.setType( n.getNodeValue() );
		setNodeStyle(attr,line);

       doChildrenItem(line, ext, arg);
	 }

     /** configure a DPageBreak */
	 public void doMethod( DPageBreak pb, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();

		setNodeStyle(attr,pb);
	 }

     /** configure a DPageCounter */
	 public void doMethod( DPageCounter pc, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		setNodeStyle(attr,pc);
	 }

     /** configure a DPopup, it children */
	 public void doMethod( DPopup ppp, DExtensible ext, Object arg )
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
	    ppp.setTag(getRequiredAttribut(attr, "TAG", node.getNodeName() ).getNodeValue());		
		setNodeStyle(attr,ppp);	
    	
		doChildrenItem(ppp, ext, arg);
	 }
    
     /**configure a DSummary */
	 public void doMethod(DSummary sum, DExtensible ext, Object arg )
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
	    sum.setTitleLevel(Integer.parseInt(getRequiredAttribut(attr, "LEVEL", node.getNodeName()).getNodeValue()));
        Node n;
		if ((n=attr.getNamedItem("TAG")) != null) sum.setTag(n.getNodeValue());
		setNodeStyle(attr,sum);
	 }

     /** configure a DTitle and it children */
	 public void doMethod( DTitle ttl, DExtensible ext, Object arg )
	 {
        Node node = ((Arg)arg).node;
		NamedNodeMap attr = node.getAttributes();
		ttl.setLevel(Integer.parseInt(getRequiredAttribut(attr, "LEVEL", node.getNodeName()).getNodeValue()));
        Node n;
		if ((n=attr.getNamedItem("TAG")) != null) ttl.setTag(n.getNodeValue());
		setNodeStyle(attr,ttl);

        doChildrenItem(ttl, ext, arg);
	 }
 
     /** configure a DIndex and it children */
	 public void doMethod( DIndex ndx, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
        setNodeStyle(attr,ndx);
        Node n;
        if( (n=attr.getNamedItem("PROPERTY"))!=null )
        {
          String prop = n.getNodeValue();
          ndx.setProperty( prop );
          if( (n=attr.getNamedItem("CLASS"))!=null )
          {
            String clname = n.getNodeValue();
            Object o = createObjectForClassName( clname, n );
            ndx.setValue( o );
          }
        }
	 }

     /** configure a DIndexEntry and it children */
	 public void doMethod( DIndexEntry ie, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
        setNodeStyle(attr,ie);
	
		doChildrenItem(ie, ext, arg );
	 }

     /** configure a DGraphic and it children */
	 public void doMethod( DGraphic graf, DExtensible ext, Object the_arg )
	 {
        Arg arg = (Arg)the_arg;
		NamedNodeMap attr = arg.node.getAttributes();

		graf.setRenderableId(getRequiredAttribut(attr, "TYPE", "GRAPHIC").getNodeValue());
        Node n;
		if ((n=attr.getNamedItem("TITLE")) != null) graf.setTitle(n.getNodeValue());
		setNodeStyle(attr,graf);
	 	
	 	axis.clear(); //axis are defined inside graphic, no share from one graphic to other one.

        //have a property store ?
        //this is important to get AXIS now, otherwise DCoord can't retrieve them..
        for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          if( "PROPERTIES".equals( c.getNodeName() ))
          {         
            DPropertyStore ps = graf.getProperties();
            if( ps==null ) ps = new DPropertyStore();
            invokeDoMethod( ps, ext, new Arg(c) );
            graf.setProperties( ps );
          }
          else if ( "AXIS".equals( c.getNodeName() ))
          {
            DAxis the_axis = new DAxis(null);
            invokeDoMethod( the_axis, ext, new Arg(c) );
            graf.addChild( the_axis );
            //axis register itself in axis map.
          }
        }
	 	
        String save[] = arg.filter_children;
        arg.filter_children = filter_for_DGraphic;
	 	doChildrenItem(graf, ext, arg );
        arg.filter_children = save;
        
        axis.clear(); //this was my axis, don't let other knowing them'
	 }
   
     //can be used by any doMethod wanted to filter PROPERTIES child node (@see doMethod(DGraphic,,)
     private static String[] filter_PROPERTIES = { "PROPERTIES" };
     //...same filtering PROPERTIES and AXIS, this is for DGraphic purpose.
     private static String[] filter_for_DGraphic = { "PROPERTIES", "AXIS" };
	 
     /** configure a DAxis */
	 public void doMethod( DAxis ax, DExtensible ext, Object the_arg )
	 {
        Arg arg = (Arg)the_arg;
		NamedNodeMap attr = arg.node.getAttributes();
		Node n;
	    
        ax.setName(getRequiredAttribut(attr, "NAME", "AXIS").getNodeValue());
		if ((n=attr.getNamedItem("UNIT" )) != null) ax.setUnit(n.getNodeValue());
		if ((n=attr.getNamedItem("TITLE")) != null) ax.setTitle(n.getNodeValue());
        if ((n=attr.getNamedItem("TYPE" )) != null) ax.setScaleType( n.getNodeValue() );
		setNodeStyle(attr,ax);

        //register axis for future ... access (DCoord)
        axis.put(ax.getName(), ax);
        
        //have a property store ?
        for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          if( "PROPERTIES".equals( c.getNodeName() ))
          {
            DPropertyStore ps = ax.getProperties();
            if( ps ==null ) ps = new DPropertyStore();
            invokeDoMethod( ps, ext, new Arg(c) );
            ax.setProperties( ps );
            break; //only only property store
          }
        }

        String save[] = arg.filter_children;
        arg.filter_children = filter_PROPERTIES;
        doChildrenItem(ax, ext, arg );
        arg.filter_children = save;
	 }
  
     /** configure a DCurve and its children */
	 public void doMethod( DCurve curv, DExtensible ext, Object a_arg )
	 {
        Arg arg = (Arg)a_arg;
		NamedNodeMap attr = arg.node.getAttributes();
        setNodeStyle(attr,curv);
        Node n;
        if((n=attr.getNamedItem("NAME")) !=null ) curv.setName(n.getNodeValue());
		if((n=attr.getNamedItem("TYPE")) !=null) curv.setType(n.getNodeValue());
    
        //have a property store ?
        for( Node c = arg.node.getFirstChild(); c!=null; c=c.getNextSibling() )
        {
          if( "PROPERTIES".equals( c.getNodeName() ))
          {         
            DPropertyStore ps = curv.getProperties();
            if( ps==null ) ps = new DPropertyStore();
            invokeDoMethod( ps, ext, new Arg(c) );
            curv.setProperties( ps );
            break; //only only property store
          }
        }

        String save[] = arg.filter_children;
        arg.filter_children = filter_PROPERTIES;
		doChildrenItem(curv, ext, arg );
        arg.filter_children = save;
	 }
	 
     /**configure a DCurveLink */
	 public void doMethod( DCurveLink cl, DExtensible ext, Object a_arg )
	 {
        Arg arg = (Arg)a_arg;
	    NamedNodeMap attr = arg.node.getAttributes();
        cl.setTarget(getRequiredAttribut(attr, "TARGET", "CURVE_LINK").getNodeValue());
        this.doMethod( (DCurve)cl, ext, a_arg );
 	 }

     /** configure a DPoint and it children */
	 public void doMethod( DPoint pnt, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		setNodeStyle(attr,pnt);
		
        doChildrenItem(pnt, ext, arg );
	 }

     /** configure a DPointLink and it children */
	 public void doMethod( DPointLink pl, DExtensible ext, Object arg )
	 {
	    NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		pl.setTarget(getRequiredAttribut(attr, "TARGET", "POINT_LINK").getNodeValue());
	    setNodeStyle(attr,pl);
		
	    doChildrenItem(pl, ext, arg );
	 }
	
     /** configure a DCoord */
 	 public void doMethod( DCoord coord, DExtensible ext, Object arg )
	 {
		NamedNodeMap attr = ((Arg)arg).node.getAttributes();
		DAxis ax = (DAxis)axis.get(getRequiredAttribut(attr, "AXIS", "COORD").getNodeValue());
        coord.setAxis(ax);
        coord.setValue(Float.parseFloat(getRequiredAttribut(attr, "VALUE", "COORD").getNodeValue()));       
	 }

     /** configure a DCoord */
     public void doMethod( DCoordObject coord, DExtensible ext, Object arg )
     {
       NamedNodeMap attr = ((Arg)arg).node.getAttributes();
       DAxis ax = (DAxis)axis.get(getRequiredAttribut(attr, "AXIS", "COORD_OBJ").getNodeValue());
       coord.setAxis(ax);

       String cl_name = getRequiredAttribut( attr, "CLASS", "COORD_OBJ" ).getNodeValue();
       
       if( Object.class.getName().equals( cl_name ) )
       {
         coord.setValue( null ); //by convention
       }
       else
       {
         try{
            Class c = Class.forName( cl_name );
            Object val = c.newInstance();
            invokeDoMethod( val, ext, arg ); //configure value
            coord.setValue( val );
         }
         catch( Exception e )
         {
           throw new DXmlError( e.getMessage() );
         }
       }
     }
     
     /** configure a Data using node text as setTime() */
     public void doMethod( Date date, DExtensible ext, Object arg )
     {
        String s = getNodeText( ((Arg)arg).node );
        try
        {
          date.setTime( Long.parseLong( s ));
        }
        catch( NumberFormatException e )
        {
          throw new DXmlError(e.getMessage());
        }
     }
     
	 
    /** configure a DPropertyStore, parsing all 'PROPERTY' children nodes. */
	public void doMethod(DPropertyStore ps, DExtensible ext, Object arg )
	{	 	
	   Node node = ((Arg)arg).node ;
	 	
	   // get property ... ies
	   for (Node c=node.getFirstChild(); c!=null; c=c.getNextSibling() )
	   {
         if( ! "PROPERTY".equals( c.getNodeName() ) ) continue; //? but...
          
		 NamedNodeMap attr = c.getAttributes();
		 String name  = getRequiredAttribut( attr, "NAME", c.getNodeName() ).getNodeValue();
         String clazz = getRequiredAttribut( attr, "CLASS", c.getNodeName() ).getNodeValue();

         Object o = createObjectForClassName( clazz, c );
//System.out.println("[XMLREAD]/PROP/ property='"+name+"' class="+clazz+"' object="+o);
		 ps.store( name, o  );
	  }
   }
	 
  
   /** 
    * Create an object for class name using node data.
    * This is for simple class like "java.lang.Byte" this object creation is not extensible
    * nor overridable using DExtensible mechanism.
    * IDXmlSerializable class name is also supported, in this case instance if configured
    * using extensible mechanism.
    * @return class instance, or null if this is a unknown class.
    */
   protected Object createObjectForClassName( String class_name, Node node )
   {
     try
     {
       Class c = Class.forName(class_name);
       
       if ( c == String .class) return (Object)(               getNodeText(node));
       if ( c == Byte   .class) return (Object)(Byte   .decode(getNodeText(node)));
       if ( c == Short  .class) return (Object)(Short  .decode(getNodeText(node)));   
       if ( c == Integer.class) return (Object)(Integer.decode(getNodeText(node)));
       if ( c == Long   .class) return (Object)(Long   .decode(getNodeText(node)));   
       if ( c == Float  .class) return (Object)(Float  .valueOf(getNodeText(node)));
       if ( c == Double .class) return (Object)(Double .valueOf(getNodeText(node)));
       if ( c == Boolean.class) return (Object)(Boolean.valueOf(getNodeText(node)));
       //a IDColor class but this is a reference in the color registry
       if ( IDColor.class.isAssignableFrom( c )) 
       {
         return colorsId.get( getNodeText(node) );
       }
       //a IDFont class but this is a reference in the font registry
       if ( IDFont.class.isAssignableFrom( c ))
       {
         return fontsId.get( getNodeText(node) ); 
       } 
       //a IDStyle class but this is a reference in the style registry
       if ( IDFont.class.isAssignableFrom( c ))
       {
         return stylesId.get( getNodeText(node) ); 
       } 
       // a string serializable object ?
       if( IDStringSerializable.class.isAssignableFrom( c ) )
       {
         try
         {
            IDStringSerializable v = (IDStringSerializable)c.newInstance();
            Arg arg = new Arg( node );
            invokeDoMethod( v, this, arg );
            return v;
         }
         catch (Exception e)
         {
            if (e instanceof InstantiationException) 
          //|| (e instanceof IllegalAccessException))
            {
             System.out.println("IDStringSerializable class should have an empty public constructor");
             return null;
            }
         }       }
       // a serializable object ?
       if( IDXmlSerializable.class.isAssignableFrom( c ) )
       {
         try
         {
            IDXmlSerializable v = (IDXmlSerializable)c.newInstance();
            Arg arg = new Arg( node );
            invokeDoMethod( v, this, arg );
            return v;
         }
         catch (Exception e)
         {
            if (e instanceof InstantiationException) 
          //|| (e instanceof IllegalAccessException))
            {
             System.out.println("IDXmlSerializable class should have an empty public constructor");
             return null;
            }
         }
       }
       else if ( c == Class.class ) return Class.forName( getNodeText(node) );
       else if ( c == DecimalFormat.class )
       {
         return new DecimalFormat( getNodeText(node ) );
       }
       else if ( c == com.ibm.icu.text.SimpleDateFormat.class )
       {
         return new com.ibm.icu.text.SimpleDateFormat( getNodeText(node ) );
       }
       else if ( c== java.util.Date.class )
       {
         try{
           return new java.util.Date( Long.parseLong( getNodeText(node) ) );
         }
         catch( NumberFormatException e)
         {
           throw new DXmlError("Failed to parse Date");
         }
       }
/*TODO       else if ( Serializable.class.isAssignableFrom( c ) )
       {
         try{
           ByteArrayInputStream bais = new ByteArrayInputStream( getNodeText(node).getBytes() );
           ObjectInputStream ois = new ObjectInputStream(bais);
           Object o = ois.readObject();
System.err.println("read object of class="+c.getName());
           return o;
         }
         catch( IOException e )
         {
           throw new DXmlError( e.getMessage() );
         }
       }*/
       else
       {
         System.err.println("(W) don't know how to handle property class '"+c.getName()+"'");
       }
     }
     catch (Exception e)
     {
       if (e instanceof ClassNotFoundException) 
        //|| (e instanceof InstantiationException) 
        //|| (e instanceof IllegalAccessException))
       {
         System.out.println("(W) Unable to instanciate class = " + class_name);
       }
     }
     return null;   
   }
   
   /** doMethod() for a IDXmlSerializable instance, call readXml() on this object */
   public void doMethod( IDXmlSerializable v, DExtensible ext, Object arg )
   {
     v.readXml( ((Arg)arg).node );
   }

   /** doMethod() for a IDStringSerializable instance, call fromString() on this object */
   public void doMethod( IDStringSerializable v, DExtensible ext, Object arg )
   {
     Node node = ((Arg)arg).node;
     v.serializeFromString( getNodeText(node) );
   }
   
   /** This method is public due to Signal connection, but must not be called directly. */ 
   public void updateExtension( IDExtension ext, Class for_extension)
   {
     if( getClass().equals( for_extension ) )
     {
       //redo extensions updates for take new extension
       DExtensionRegistry.updateExtensible( this );
     }
   }
   
   /** decode 6 hexa-digit string RRGGBB into a IDColor */
   public static IDColor DecodeColor( String s )
   {
     try {
       int rgb = Integer.parseInt(s,16);
       return new DColor( (rgb>>16)&0xFF, (rgb>>8)&0xFF, rgb&0xFF );
     } catch( NumberFormatException e ) {
       return null;
     }
   }

   private HashMap folder_model_read_ = new HashMap();
   
   /** parse XML tree to get DFolder including any DFolderModel saved with */
   public void doMethod( DFolder folder, DExtensible ext, Object a_arg )
   {
     DXmlReader r = (DXmlReader)ext;
     Arg arg = (Arg)a_arg;
     NamedNodeMap attr = arg.node.getAttributes();
     
     if( folder_model_read_==null ) folder_model_read_ = new HashMap();
     
     Node n = arg.node;
     for( ;n.getParentNode()!=null; n=n.getParentNode());     

     r.setNodeStyle( attr, folder );

     if((n=attr.getNamedItem("OPEN")) !=null ) folder.setOpen( "true".equals(n.getNodeValue()) );
     //is this folder define a new  model ?
     if((n=attr.getNamedItem("MODELDEF")) !=null) 
     {
       int id=-1;
       try {
         id = Integer.parseInt( n.getNodeValue(), 16 );
       }
       catch( NumberFormatException e )
       {
         throw new DXmlError("Folder model id must be a number");
       }
       
       DFolderModel model = new DFolderModel();
       folder.setFolderModel( model );
       
       folder_model_read_.put( new Integer(id), model );
       
       if( (n=attr.getNamedItem("INDENTED"))!=null)
       {
         model.setIndented( "true".equals(n.getNodeValue()));
       }
       if( (n=attr.getNamedItem("OPENED"))!=null)
       {
         ISymbol symbol = DSymbolRegistry.GetCoreSymbol( n.getNodeValue() );
         model.setOpenedSymbol( true,  symbol );
         model.setOpenedSymbol( false, symbol );
       } else {
         if( (n=attr.getNamedItem("OPENEDL2R"))!=null)
         {
           model.setOpenedSymbol( true, DSymbolRegistry.GetCoreSymbol( n.getNodeValue()) );
         }
         if( (n=attr.getNamedItem("OPENEDR2L"))!=null)
         {
           model.setOpenedSymbol( false, DSymbolRegistry.GetCoreSymbol( n.getNodeValue()) );
         }
       }
       if( (n=attr.getNamedItem("CLOSED"))!=null)
       {
         ISymbol symbol = DSymbolRegistry.GetCoreSymbol( n.getNodeValue() );
         model.setClosedSymbol( true,  symbol );
         model.setClosedSymbol( false, symbol );
       } else {
         if( (n=attr.getNamedItem("CLOSEDL2R"))!=null)
         {
           model.setClosedSymbol( true, DSymbolRegistry.GetCoreSymbol( n.getNodeValue()) );
         }
         if( (n=attr.getNamedItem("CLOSEDR2L"))!=null)
         {
           model.setClosedSymbol( false, DSymbolRegistry.GetCoreSymbol( n.getNodeValue()) );
         }
       }
       if( (n=attr.getNamedItem("OPENEDFORE"))!=null)
       {
         model.setOpenedForeColor( DecodeColor( n.getNodeValue() ));
       }
       if( (n=attr.getNamedItem("OPENEDBACK"))!=null)
       {
         model.setOpenedBackColor( DecodeColor( n.getNodeValue() ));
       }
       if( (n=attr.getNamedItem("CLOSEDFORE"))!=null)
       {
         model.setClosedForeColor( DecodeColor( n.getNodeValue() ));
       }
       if( (n=attr.getNamedItem("CLOSEDBACK"))!=null)
       {
         model.setClosedBackColor( DecodeColor( n.getNodeValue() ));
       }
     }
     //this folder use a defined model ?
     else if((n=attr.getNamedItem("MODEL")) !=null) 
     {
       int id=-1;
       try {
         id = Integer.parseInt( n.getNodeValue(), 16 );
         DFolderModel model = (DFolderModel)folder_model_read_.get( new Integer(id) );
         if( model==null )
         {
           throw new DXmlError("Can't retrieve folder model id="+n.getNodeValue());
         }
         folder.setFolderModel( model );
       }
       catch( NumberFormatException e )
       {
         throw new DXmlError("Folder model id must be a number");
       }
       catch( ArrayIndexOutOfBoundsException e2 )
       {
         throw new DXmlError("Folder model id="+id+" isn't a known model.");
       }
       catch( IndexOutOfBoundsException e3 )
       {
         throw new DXmlError("Folder model id="+id+" isn't a known model.");
       }
     }
     //else: this folder use default model
    
	 r.doChildrenItem( folder, ext, arg );
   }
 
   private int parseInt( String s, int _default )
   {
     try {
       int v = Integer.parseInt( s );
       return v;
     } catch( Throwable t ) {
       return _default;
     }
   }

   public void doMethod( DBorder border, DExtensible ext, Object a_arg )
   {
     DXmlReader r = (DXmlReader)ext;
     Arg arg = (Arg)a_arg;
     NamedNodeMap attr = arg.node.getAttributes();
     
     Node n = arg.node;

     r.setNodeStyle( attr, border );

     if((n=attr.getNamedItem("SHADOW_TYPE")) !=null ) 
       border.setShadowType( DBorder.ToShadowType( n.getNodeValue() ) );
     else
       border.setShadowType( DBorder.S_NONE );
     
     if((n=attr.getNamedItem("SHADOW_DEPTH")) !=null ) 
       border.setShadowDepth( parseInt( n.getNodeValue(), 0) );
     else
       border.setShadowDepth( 8 );
     
     if((n=attr.getNamedItem("LINE_TYPE")) !=null ) 
       border.setLineType( DBorder.ToLineType( n.getNodeValue() ) );
     else
       border.setLineType( DBorder.L_SIMPLE );
     
     if((n=attr.getNamedItem("MARGIN")) !=null ) 
       border.setMargin( parseInt( n.getNodeValue(), 0 ));
     else
       border.setMargin( 3 );
     
     if((n=attr.getNamedItem("SHADOW_COLOR")) !=null )
     {
       border.setShadowColor( DXmlReader.DecodeColor( n.getNodeValue() ) );
     }
      
	 r.doChildrenItem( border, ext, arg );
   }
}
