/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DXmlWriter.java,v 1.5 2008/05/23 14:12:01 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.drivers.xml.internal;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DBorder;
import org.eclipse.tptp.platform.report.core.internal.DCell;
import org.eclipse.tptp.platform.report.core.internal.DCellText;
import org.eclipse.tptp.platform.report.core.internal.DColor;
import org.eclipse.tptp.platform.report.core.internal.DColorRegistry;
import org.eclipse.tptp.platform.report.core.internal.DCoord;
import org.eclipse.tptp.platform.report.core.internal.DCoordObject;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DCurveLink;
import org.eclipse.tptp.platform.report.core.internal.DData;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DFolder;
import org.eclipse.tptp.platform.report.core.internal.DFolderModel;
import org.eclipse.tptp.platform.report.core.internal.DFont;
import org.eclipse.tptp.platform.report.core.internal.DFontRegistry;
import org.eclipse.tptp.platform.report.core.internal.DFooter;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DHeader;
import org.eclipse.tptp.platform.report.core.internal.DImage;
import org.eclipse.tptp.platform.report.core.internal.DIndex;
import org.eclipse.tptp.platform.report.core.internal.DIndexEntry;
import org.eclipse.tptp.platform.report.core.internal.DLine;
import org.eclipse.tptp.platform.report.core.internal.DLink;
import org.eclipse.tptp.platform.report.core.internal.DList;
import org.eclipse.tptp.platform.report.core.internal.DPageBreak;
import org.eclipse.tptp.platform.report.core.internal.DPageCounter;
import org.eclipse.tptp.platform.report.core.internal.DParagraph;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPointLink;
import org.eclipse.tptp.platform.report.core.internal.DPopup;
import org.eclipse.tptp.platform.report.core.internal.DPropertyStore;
import org.eclipse.tptp.platform.report.core.internal.DRow;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DStyleRegistry;
import org.eclipse.tptp.platform.report.core.internal.DSummary;
import org.eclipse.tptp.platform.report.core.internal.DTable;
import org.eclipse.tptp.platform.report.core.internal.DTag;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.DTitle;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDObject;
import org.eclipse.tptp.platform.report.core.internal.IDStringSerializable;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.ISymbol;
import org.eclipse.tptp.platform.report.drivers.internal.IWriter;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.tools.internal.DAlignment;
import org.eclipse.tptp.platform.report.tools.internal.DParser;



/**
 * Provides an object to write JScrib document in an XML format.<br>
 * The following example writes a JScrib IDDocument item in an xml file named
 * 'myfile.jsml'.<p>
 * 
 * <PRE>
 *    IDDocument doc = new DDocument();
 *    ...
 *    DXmlWriter aw = new DXmlWriter();
 *    try{
 *        aw.write( new FileOutputStream("myfile.jsml"), doc );
 *        // Good, the file is generated 
 *        ...
 *    }
 *    catch(Exception e ) 
 *    {
 *      // An error occurs in the writer
 *      ...
 *    }
 * </PRE>
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DXmlWriter extends DParser implements IWriter 
{   
   private OutputStream out;
   private boolean write_complete_document ;
   private DStyleRegistry styles= new DStyleRegistry();
   private DColorRegistry colors= new DColorRegistry();
   private DFontRegistry  fonts = new DFontRegistry();
	
   /** info on element's stack currently in write process */
   private static class Element
   {
     public String name;
     public int indent;
     public boolean can_indent;
     public boolean have_attr;
     public boolean empty;
     public Element up;
     public boolean start_part_closed;
     
     public Element( String n, Element u )
     {
       name=n;
       up=u;
       indent=0;
       can_indent=true;
       have_attr=false;
       empty=true;
       start_part_closed=false;
     }
   }
   
   private Element curr_element;
   private int curr_indent;
   private boolean auto_indent;
   
   /**
    * Create a JScrib XML writer, by default auto indent mode is not engaged.
    * @see #setAutoIndentMode
	*/
   public DXmlWriter() 
   {
	 super();
     auto_indent=false;
     write_complete_document = true;
   }
   
   /** @return current used output stream */
   public OutputStream getOutputStream()
   {
     return out ;
   }
   
   /** 
    * @return true if auto indent mode is engaged.
    * @see #setAutoIndentMode
    */
   public boolean isAutoIndentMode() { return auto_indent; }
   
   /** 
    * Change current value of auto indent mode, is this mode xml elements will be indented
    * using spaces if they don't contains CData.
    * Using this mode produce more human readable stream (files), but their size will grow.
    * (depending on the number of element and max element stack level encountered,
    *  but a rate from 10% to 50% seems to be common).
    * @see #outCData
    */
   public void  setAutoIndentMode( boolean b ) { auto_indent=b; }
   
   /** @return true if writer will write complete jscrib xml document for each write() calls */
   public boolean isWriteCompleteDocument() { return write_complete_document; }
   
   /** 
    * Change 'complete document' mode.<br>
    * Default value is <b>true</b>.
    * If parameter is true, next call to write() method will generate a complete XML document
    * including &lt;?xml ...>, &ltJSCRIB> and registries elements. In this mode all colors, fonts,
    * styles and any registries seen in write() parameter are merged into current writer's registries,
    * and written once at the end of XML document.<br>
    * If parameter is false, next call to write() method will generate only the part of XML
    * elements corresponding to given IDObject. In this mode any colors, fonts, styles are merged
    * into writer's registries <b>but</b> no output of writer's registries are done.
    * <b>But</b> any registries seen in write() call parameters is generated as simple object. 
    */
   public void setWriteCompleteDocument( boolean b ) 
   {
     write_complete_document = b ;
   }
   
   /**
    * Modifying this name means you need to modify it in DXmlReader
    * @return xml root element name for jscrib, default is 'JSCRIB'.
    */
   public String getRootElementName() { return "JSCRIB"; }
   
   /**
     * Writes an IDObject array in the OutputStream in XML format.<br>
     * Note: refere to setWriteCompleteDocument() method for colors, fonts, styles and corresponding
     * registries handling.
     * @see setAutoIndentMode(), setWriteCompleteDocument()
	 */
	public void write(OutputStream output, IDObject[] objects) throws Exception
	{		
		out = output;
		extensions = null; //reset outputed extensions (see writeExtension)
		folder_model_written_ = null; //reset folder model saved
	    String root_name = getRootElementName();
      
	    if( write_complete_document )
	    {
	      out.write( "<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n".getBytes("UTF-8") );
	      curr_element = null;
	      curr_indent=0;
	      startElement( root_name );
	    }
     
		for (int i = 0; i < objects.length; i++)
        {
          Object obj = objects[i];          
          if( obj == null ) continue;
          
          //special handling for DStyleRegistry, DColorRegistry and DFontRegistry user want to write
          //intern them into writer's registry
          if( write_complete_document )
          {
            if( obj instanceof DColorRegistry )
            {
              for( Iterator it=((DColorRegistry)obj).iterator(); it.hasNext(); )
              {
                colors.putColor( (IDColor)it.next() );
              }
            }
            else if( obj instanceof DFontRegistry )
            {
              for( Iterator it=((DFontRegistry)obj).iterator(); it.hasNext(); )
              {
                fonts.putFont( (IDFont)it.next() );
              }
            }
            else if( obj instanceof DStyleRegistry )
            {
              for( Iterator it=((DStyleRegistry)obj).iterator(); it.hasNext(); )
              {
                IDStyle sty = (IDStyle)it.next();              
                styles.putStyle( sty );
                if (sty.getBackColor()!=null) colors.putColor(sty.getBackColor());
                if (sty.getForeColor()!=null) colors.putColor(sty.getForeColor());
                if (sty.getFont()     !=null)  fonts.putFont (sty.getFont());
              }
            }
            else
            {
              invokeDoMethod( obj, this,null);
            }
          }
          else
          {
            invokeDoMethod( obj, this,null);
          }
		}
		if( write_complete_document )
		{
		  invokeDoMethod(fonts, this,null);
		  invokeDoMethod(colors,this,null);
		  invokeDoMethod(styles,this,null);
		//other IDRegistry to save here:

  		  endElement();
		}
        curr_element=null;
        extensions=null; 
	}

	/**
	 * Writes an IDObject in OutputStream in XML Formal.
	 * @see write(OutputStream, IDObject[]).
	 */
	public void write(OutputStream output, IDObject object) throws Exception
	{
		write(output, new IDObject[] { object } );
	}

	/**
	 * Clears all internal datas. 
	 * @see org.eclipse.tptp.platform.report.tools.internal.DParser#clear()
	 */
	public void clear() {
		fonts.clear();
		colors.clear();
		styles.clear();
		super.clear();
        curr_element=null;
        curr_indent=0;
	}
   
   /** 
    * Writes a \n and a number of space depending on parameter, if auto indent mode is engaged.
    * @param idt proportinal to the number of space to write.
    * @throws IOException is the ouput failed.
    */
   protected void outIndent( int idt ) throws IOException
   {
     if( !auto_indent ) return ;
     out.write('\n');
     for( int i=0; i<idt; ++i ) 
     {
       out.write(' ');
       out.write(' ');
     }
   }
   
   private void closeStartElement() throws IOException
   {
     if( curr_element!=null && !curr_element.start_part_closed )
     {
       if( curr_element.empty )
       {
         out.write('>');
       }
       curr_element.empty = false;
       if( curr_element.can_indent )
       {
         outIndent( curr_element.indent );
       }
       curr_element.start_part_closed = true;
     }
   }
   
   /**
    * Calls this to start an XML element, do the ouput and data used to close the element
    * correctly. Support for indentation.
    * @see #endElement
    * @see #outAttribute
    * @param name the name of the element to start.
    */
   public void startElement( String name )
   {
     try {
       closeStartElement();
       out.write('<');
       out.write( name.getBytes("UTF-8") );
       curr_element = new Element( name, curr_element );
       curr_element.indent = curr_indent;
       curr_indent++;
     }
     catch (IOException e) {
       e.printStackTrace();
     }
   }
   
   /**
    * Closes a previously started element, choose the best way to write the end ("/>" or "&lt;/ELM>",
    * depending on data collected such as outCData was called.
    * @see #startElement
    * @see #outCData
    */
   public void endElement()
   {
     if( curr_element==null ) return ;
     try{
       // /> or (indent)?</XXXX>
       if( curr_element.empty )
       {
         out.write('/');
         out.write('>');
       } else {
         if( curr_element.can_indent )
         {
           outIndent( curr_element.indent );
         }
         out.write('<');
         out.write('/');
         out.write( curr_element.name.getBytes("UTF-8") );
         out.write('>');
       }
       curr_indent--;
       curr_element = curr_element.up;
     }
     catch( IOException e ) {
       e.printStackTrace();
     }
   }

   /** 
    * Writes CData: text outside element area (&gt;..&lt), text is encoded and converted to UTF-8 before wrote.
    * You must call this if you want auto indentation works correctly.
    * @param data the text to write.
    */ 
   public void outCData( String data )
   {
     try{
       if( curr_element!=null )
       {
         curr_element.can_indent=false;
         curr_element.empty = false;
         out.write('>');
       }
       out.write( encode(data).getBytes("UTF-8") );
     }
     catch( IOException e )
     {
       e.printStackTrace();
     }
   }
   
   /** 
    * Writes an attribute in current element. If required and value is null, an error is thrown,
    * if not required and the value is null and error is thrown.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    */
   public void outAttribute( String attr_name, String attr_value, boolean required )
   {
     if( attr_value==null )
     {
       if( required )
       {
         throw new DXmlError("Required attribute '"+attr_name+"' can't have null value, correct the document.");
       }
       return;
     }
     try{
       out.write(' ');
       out.write( attr_name.getBytes("UTF-8"));
       out.write('=');
       out.write('"');
       //need " detection here ?
       out.write( encode(attr_value).getBytes("UTF-8"));
       out.write('"');
       curr_element.have_attr = true;
     }
     catch( IOException e){
       e.printStackTrace();
     }
   }
  
   /** 
    * Writes an attribute only if value differ than default (do nothing if value is null)
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    * @param default_value the default value, if equals to the value, no output are done.
    */
   public void outAttribute( String attr_name, String attr_value, String default_value )
   {
     if( attr_value==null ) return ;
     if( attr_value==default_value || attr_value.equals( default_value ) ) return ;
     outAttribute( attr_name, attr_value, false );
   }
   
   /** 
    * Writes an integer attribute
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    */
   public final void outAttribute( String attr_name, int attr_value )
   {
     outAttribute( attr_name, Integer.toString(attr_value), true );
   }
   /** 
    * Writes an integer attribute, if value differ than default one.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    * @param default_value the default value, if equals to the value, no output are done.
    */
   public final void outAttribute( String attr_name, int attr_value, int default_value )
   {
     if( attr_value==default_value) return ;
     outAttribute( attr_name, Integer.toString(attr_value), false );
   }
   
   /** 
    * Write boolean attribute
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    */
   public final void outAttribute( String attr_name, boolean attr_value )
   {
     outAttribute( attr_name, Boolean.toString(attr_value), true );
   }
   /** 
    * Writes a boolean attribute, if value differ than default one.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    * @param default_value the default value, if equals to the value, no output are done.
    */
   public final void outAttribute( String attr_name, boolean attr_value, boolean default_value )
   {
     if( attr_value==default_value) return ;
     outAttribute( attr_name, Boolean.toString(attr_value), false );
   }
   
   /** 
    * Writes float attribute.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    */
   public final void outAttribute( String attr_name, float attr_value )
   {
     outAttribute( attr_name, Float.toString(attr_value), true );
   }
   /**
    * Writes a float attribute, if value differ than default one.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    * @param default_value the default value, if equals to the value, no output are done.
    */
   public final void outAttribute( String attr_name, float attr_value, float default_value )
   {
     if( attr_value==default_value) return ;
     outAttribute( attr_name, Float.toString(attr_value), false );
   }

   /** 
    * Writes float attribute.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    */
   public final void outAttribute( String attr_name, double attr_value )
   {
     outAttribute( attr_name, Double.toString(attr_value), true );
   }
   /**
    * Writes a float attribute, if value differ than default one.
    * @param attr_name the name of the attribute
    * @param attr_value the value of the attribute
    * @param default_value the default value, if equals to the value, no output are done.
    */
   public final void outAttribute( String attr_name, double attr_value, double default_value )
   {
     if( attr_value==default_value) return ;
     outAttribute( attr_name, Double.toString(attr_value), false );
   }

   /** 
    * Starts element and write common IDItem attribute (style).
    * It's your responsability to call endElement().
    * @see #startElement
    * @param name the name of the element.
    * @param item the item to write.
    */
   public void startElement( String name, IDItem item )
   {
     startElement( name );
     
     IDStyle sty = item.getStyle();
     if ( sty != null )  
     {
       outAttribute( "STYLE", sty.getID(), true);
       //register style 
       styles.putStyle( sty );
       if (sty.getBackColor()!=null) colors.putColor(sty.getBackColor());
       if (sty.getForeColor()!=null) colors.putColor(sty.getForeColor());
       if (sty.getFont()     !=null)  fonts.putFont (sty.getFont());
     }
   }
 
   /**
    * Encodes given string replacing a trap character by their xml entity names
    * (ie > by &amp;gt;, & by &amp;amp;). Replacement is done directly in s.
    * @param s the string to encode.
    * @return s after replacement
    */
   public static String encode(String s)
   {
	   if (s == null) return s;
	   s = s.replaceAll("&","&amp;");
	   s = s.replaceAll("'","&apos;");
	   s = s.replaceAll("<","&lt;");
	   s = s.replaceAll(">","&gt;");
	   s = s.replaceAll("\"","&quot;");
	   return s;
   }   
   
   /**
    * Writes xml for a DDocument.
	*/
   public void doMethod(DDocument d, DExtensible p, Object arg) 
   {
	 startElement("DOCUMENT", d.getDocument());
	 outAttribute( "TITLE", d.getTitle(), false);
	 outAttribute( "TAG", d.getTag(), true );
	  
	 doChildrenItem(d,p,arg);	 
	 endElement();     
   }
   
   /**
    * Writes xml for a DSection
	*/
   public void doMethod(DSection s, DExtensible p, Object arg) throws DParser.ParserException
   {
	  startElement("SECTION",s); 
	  outAttribute( "TAG", s.getTag(), true );
	  //super: to reach both footer and header, and to children
	  super.doMethod(s,p,arg); 
	  endElement();
   }
   
   /**
    * Writes xml for a DFooter
	*/
   public void doMethod(DFooter f, DExtensible p, Object arg) 
   {
	  startElement("FOOTER",f);		
	  doChildrenItem(f,p,arg); 
      endElement();
   }
   
   /**
    * Writes xml for a DHeader
	*/
   public void doMethod(DHeader h, DExtensible p, Object arg) 
   {
	 startElement("HEADER",h); 		
	 doChildrenItem(h,p,arg); 
	 endElement();
   }
  
   /**
    * Writes xml for a DParagraph
	*/
   public void doMethod(DParagraph p, DExtensible pr, Object arg) 
   {
	 startElement("PARAGRAPH",p);
     if( p.getAlignment()!=DParagraph.DEFAULT_ALIGN )
  	   outAttribute( "ALIGNMENT", DAlignment.AlignmentToString(p.getAlignment()), false );
 		
	 doChildrenItem(p,pr,arg); 	  
	 endElement();
   }
 
   /**
    * Writes xml for a DData ... 
    */
   public void doMethod(DData ip, DExtensible p, Object arg)
   {
	  startElement("DATA", ip);
	  outAttribute( "CLASS", ip.getClass().getName(), true);
	  invokeDoMethod(ip.getData(),p,arg);
	  endElement();
   }
   
   /** Saves generic attribute and element for any unknown IDItem implementor */
   public void doMethod(IDItem i, DExtensible p, Object arg)
   {
	  startElement("IDITEM", i);
	  outAttribute( "CLASS", i.getClass().getName(), true);
      boolean have_elm = (i.getFirstChild()!=null);
      if( have_elm )
	   {
          doChildrenItem( i, p, arg );
	   }
      endElement();
    }
    	
   /**
    * Writes xml for a DText
	*/
   public void doMethod(DText t, DExtensible p, Object arg) 
   {
	 startElement("TEXT",t);
	 outCData( t.getText() );
	 endElement();
   }
  
   /**
    * Writes xml for a DImage
	*/
   public void doMethod(DImage i, DExtensible p, Object arg) 
   {
	 startElement("IMAGE",i);
	 outAttribute( "NAME",  i.getName(), true);
	 outAttribute( "SCALEW", i.getWidthScale(), 1.0f);
	 outAttribute( "SCALEH", i.getHeightScale(), 1.0f);
	 endElement();
   }
 
   /**
    * Writes xml for a DColor
    */
   public void doMethod(DColor c, DExtensible p, Object arg)
   {
	 startElement( "COLOR");
	 outAttribute( "ID", c.getID(), true ); //required
	 outAttribute( "R", c.getRed());
	 outAttribute( "G", c.getGreen());
	 outAttribute( "B", c.getBlue());
	 endElement();
   
   }
   
   /**
    * Writes xml for a DStyle, style is already in registries.
    */
   public void doMethod(DStyle s, DExtensible p, Object arg)
   {
	 startElement( "STYLE");
     outAttribute( "ID", s.getID(), true);
     String n = s.getName();
     if( n!=null )
     {
	   outAttribute( "NAME", n, false);
     }
     IDFont font = s.getFont();
	 if (font!=null)
	 {
	   outAttribute( "FONT", font.getID(), true );
	 }
     IDColor color = s.getForeColor();
	 if (color!=null)
	 {
	   outAttribute( "FORECOLOR", color.getID(), true );
	 }
     color = s.getBackColor();
	 if ( color!=null )
	 {
	   outAttribute( "BACKCOLOR", color.getID(), true);
	 }	 
     if (s.getWallpaper()!=null)
     {
       outAttribute( "WALLPAPER", s.getWallpaper().serializeToString(), false );
     }
	 endElement();
   }
   
   /**
    * Writes xml for a DFont
    */
   public void doMethod(DFont f, DExtensible p, Object arg)
   {
	 startElement( "FONT");
	 outAttribute( "ID", f.getID(), true); //required
	 outAttribute( "SIZE", f.getSize());
	 outAttribute( "FAMILY", f.getFamily(),"");
	 outAttribute( "SCRIPT", f.getScript(), false);
	 outAttribute( "STYLE", f.getStringStyle(),"");	
	 endElement();
   }
   
   /**
    * Writes xml for a DList
	*/
   public void doMethod(DList l, DExtensible p, Object arg) 
   { 
	 startElement("LIST",l);
	 outAttribute( "TYPE", l.getFormat()); //required
 		
	 doChildrenItem(l,p,arg); 
	 
	 endElement();
   }
  
   /**
    * Writes xml for a DTable
	*/
   public void doMethod(DTable t, DExtensible p, Object arg) 
   {
	  startElement("TABLE",t);
	  outAttribute( "BORDER", t.getBorder());
	  outAttribute( "SCALEW", t.getWidthScale(),1.0f);
	  outAttribute( "SCALEH", t.getHeightScale(),1.0f);
	  String s = null;
      float  col_width[] = t.getColumnWidth();
	  if (col_width!=null)
	  {
	  	s = "";
		for (int i=0; i<col_width.length;i++)
		{
		    s = s + col_width[i];
		    if ( i < (col_width.length-1))
			  s = s + "/";
		}
	  }
	  
	  outAttribute( "WIDTH", s, false);
 		
	  doChildrenItem(t,p,arg);    

	  endElement();
   }
  
   /**
    * Writes xml for a DRow
	*/
   public void doMethod(DRow r, DExtensible p, Object arg) 
   {
	  startElement("ROW",r); 		
	  doChildrenItem(r,p,arg);    
	  endElement();
   }
   
	 
   /**
    * Writes xml for a DCell
    */
   public void doMethod(DCell c, DExtensible p, Object arg) 
   {
	 startElement("CELL",c);
	 outAttribute( "COLSPAN", c.getColSpan(), DCell.DEFAULT_COLSPAN );
	 outAttribute( "ROWSPAN", c.getRowSpan(), DCell.DEFAULT_ROWSPAN );
	 if( c.getAlignment()!=DParagraph.DEFAULT_ALIGN )
	  	   outAttribute( "ALIGNMENT", DAlignment.AlignmentToString(c.getAlignment()), false );
		
	 doChildrenItem(c,p,arg);    
	
	 endElement();
   }
   
   /**
    * Writes xml for a DCellText
    */
   public void doMethod(DCellText c, DExtensible p, Object arg) 
   {
	 startElement("CELL_TEXT",c);
     outAttribute( "COLSPAN", c.getColSpan(), DCell.DEFAULT_COLSPAN );
     outAttribute( "ROWSPAN", c.getRowSpan(), DCell.DEFAULT_ROWSPAN );
     if( c.getAlignment()!=DParagraph.DEFAULT_ALIGN )
    	   outAttribute( "ALIGNMENT", DAlignment.AlignmentToString(c.getAlignment()), false );
		
	 outCData( c.getText() );
 	 endElement();
   }
 
	/**
     * Writes xml for a DLink
	 */
	public void doMethod(DLink l, DExtensible p, Object arg) 
	{
	  startElement("LINK",l);
	  outAttribute( "TARGET", l.getTarget(), true);	
	  doChildrenItem(l,p,arg);	    
	  endElement();
	}

	/**
     * Writes xml for a DTag
	 */
	public void doMethod(DTag t, DExtensible p, Object arg) 
	{
	   startElement("TAG",t);
	   outAttribute( "NAME", t.getTag(), true);
	   endElement();
	}

    /**
     * Writes xml for a DLine
     */
	public void doMethod(DLine l, DExtensible p, Object arg)
	{
	   startElement("LINE",l);
	   outAttribute( "SIZE", l.getSize(), DLine.DEFAULT_SIZE );
       outAttribute( "TYPE", l.getType(), DLine.DEFAULT_TYPE );
	   endElement();
	}
	
	/**
     * Writes xml for a DPageBreak
	 */
	public void doMethod(DPageBreak pb, DExtensible p, Object arg) 
	{
	  startElement("PAGE_BREAK",pb);
	  endElement();
	}
   
	/**
     * Writes xml for a DPageCounter
	 */
	public void doMethod(DPageCounter pc, DExtensible p, Object arg) 
	{
		startElement("PAGE_COUNTER",pc);
		endElement();
    }
 
	/**
     * Writes xml for a DPopup
	 */
	public void doMethod(DPopup p, DExtensible prs, Object arg) 
	{
	  startElement("POPUP",p);
	  outAttribute( "TAG", p.getTag(), true);	   
	  doChildrenItem(p,prs,arg); 
	  endElement();
	}
    
    /**
     * Writes xml for a DSummary
     */
	public void doMethod(DSummary s, DExtensible p, Object arg)
	{
	  startElement("SUMMARY",s);
	  outAttribute( "LEVEL", s.getTitleLevel()); //required
	  outAttribute( "TAG",  s.getTag(), false);
	  endElement();
	}
 
   /**
    * Writes xml for a DTitle
    */
   public void doMethod(DTitle t, DExtensible p, Object arg) 
   {
	 startElement("TITLE",t);
	 outAttribute( "LEVEL", t.getLevel()); //required
	 outAttribute( "TAG", t.getTag(), false);	   
 	 doChildrenItem(t,p,arg);
	 endElement();
   }
   
   /**
    * Writes xml for a DIndex
    */
   public void doMethod(DIndex i, DExtensible p, Object arg)
   {
     startElement("INDEX",i);
//TODO revoir ca c'est incorrect avec la DTD	        
     if( i.getProperty() != null )
     {
       outAttribute("PROPERTY", i.getProperty(), false);
       Object value = i.getValue();
       if( value != null )
       {
         outAttribute("CLASS", value.getClass().getName(), false);
         invokeDoMethod( value, p, arg );
       }
       outAttribute(i.getProperty(), (String)i.getValue(), false);  
     }
     endElement();
   }

   /**
    * Writes xml for a DIndexEntry
    */
   public void doMethod(DIndexEntry i, DExtensible p, Object arg)
   {
	 startElement("INDEX_ENTRY",i);	
	 invokeDoMethod(i.getProperties(),p,arg);  
	 doChildrenItem(i,p,arg); 
	 endElement();
   }

   /**
    * Writes xml for a DGraphic
	*/
   public void doMethod(DGraphic g, DExtensible p, Object arg) 
   {  
	  startElement("GRAPHIC",g);
	  outAttribute( "TYPE", g.getRenderableId(), true);
	  outAttribute( "TITLE", g.getTitle(), false);
	  
	  invokeDoMethod(g.getProperties(),p,arg);
	   
	  doChildrenItem(g,p,arg);
		
	  endElement();
   }
   
   /**
    * Writes xml for a DAxis
    */
   public void doMethod(DAxis a, DExtensible p, Object arg)
   {
	  startElement("AXIS",a);
	  outAttribute( "NAME", a.getName(), true);
	  outAttribute( "TITLE", a.getTitle(), false);
	  outAttribute( "UNIT",  a.getUnit(), false);
	  outAttribute( "TYPE",  a.getScaleType(), DAxis.S_LIN );
    
	  invokeDoMethod(a.getProperties(),p,arg);
	  
	  endElement();
   }
   
   /**
    * Writes xml for a DCurve
    */
   public void doMethod(DCurve c, DExtensible p, Object arg)
   {
	  startElement("CURVE",c);
      outAttribute( "TYPE", c.getType(), false);
      outAttribute( "NAME", c.getName(), false);
      invokeDoMethod( c.getProperties(), p, arg );
      doChildrenItem(c,p,arg);
	  endElement();
   } 
   
   /**
    * Writes xml for a DCurveLink
    */
   public void doMethod(DCurveLink c, DExtensible p, Object arg)
   {
	  startElement("CURVE_LINK",c);
      outAttribute( "TYPE", c.getType(), false);
      outAttribute( "NAME", c.getName(), false);
      outAttribute( "TARGET", c.getTarget(), true);   
      invokeDoMethod( c.getProperties(), p, arg );
      doChildrenItem(c,p,arg);
	  endElement();
   }

   /**
    * Writes xml for a Dpoint
    */
   public void doMethod(DPoint p, DExtensible prs, Object arg)
   {
	  startElement("POINT",p);	  
	  doChildrenItem(p,prs,arg);
	  endElement();
   }

   /**
    * Writes xml for a DPointLink
    */
   public void doMethod(DPointLink p, DExtensible prs, Object arg)
   {
	  startElement("POINT_LINK",p);
	  outAttribute( "TARGET", p.getTarget(), true);	  
	  doChildrenItem(p,prs,arg);
	  endElement();
   }

   /**
    * Writes xml for a DCoord
    */
   public void doMethod(DCoord c, DExtensible p, Object arg)
   {
	  startElement( "COORD");
	  outAttribute( "AXIS", c.getAxis().getName(), true);
	  outAttribute( "VALUE", c.getValue()); //required
	  endElement();
   }
   
   /**
    * Writes xml for a DCoord
    */
   public void doMethod(DCoordObject c, DExtensible p, Object arg)
   {
    startElement( "COORD_OBJ");
    outAttribute( "AXIS", c.getAxis().getName(), true);
    Object val = c.getValue(null);
    outAttribute( "CLASS", (val==null ? Object.class.getName() : val.getClass().getName()), true );
    invokeDoMethod( val, p, arg );
    endElement();
   }
   
   
   protected ArrayList folder_model_written_;
   /** encode IDColor into a 6 hexa-digit string RRGGBB */
   public static String EncodeColor( IDColor c )
   {
     int r = c.getRed();
     int g = c.getGreen();
     int b = c.getBlue();
     int rgb = (r<<16)|(g<<8)|b;
     String s = Integer.toString( rgb, 16 );
     StringBuffer buf = new StringBuffer(6);
     for( int i=s.length(); i<6; i++ ) buf.append('0');
     buf.append( s );
     return buf.toString();
   }

   /**
    * Writes xml for a DFolder, model is automatically saved as folder's attribute
    * if it not already written.
    */
   public void doMethod( DFolder folder, DExtensible ext, Object arg )
   {
     DXmlWriter w = (DXmlWriter)ext;
     
     DFolderModel model = folder.getFolderModel();
     boolean is_default_model = (model==null || model == DFolderModel.GetDefault());
     //default model is not saved
     if( !is_default_model )
     {
       if( folder_model_written_==null ) folder_model_written_ = new ArrayList();
     }
     
     w.startElement( "FOLDER", folder );
     w.outAttribute( "OPEN", folder.isOpen() );
     if( !is_default_model )
     {
       int id = model.hashCode();
       if( !folder_model_written_.contains( model ) )
       {
         //define new model using Folder's attributes...
         folder_model_written_.add( model );
         w.outAttribute( "MODELDEF", Integer.toString(id,16), true );
         w.outAttribute( "INDENTED", model.isIndented() );
         ISymbol l2r, r2l;
         l2r = model.getOpenedSymbol( true  );
         r2l = model.getOpenedSymbol( false );
         if( l2r!=null && r2l !=null && l2r==r2l )
         {
           w.outAttribute( "OPENED", l2r.getId(), true );
         } else {         
           if( l2r!=null ) w.outAttribute( "OPENEDL2R", l2r.getId(), true);
           if( r2l!=null ) w.outAttribute( "OPENEDR2L", r2l.getId(), true);
         }
         l2r = model.getClosedSymbol( true  );
         r2l = model.getClosedSymbol( false );
         if( l2r!=null && r2l !=null && l2r==r2l )
         {
           w.outAttribute( "CLOSED", l2r.getId(), true );
         } else {         
           if( l2r!=null ) w.outAttribute( "CLOSEDL2R", l2r.getId(), true);
           if( r2l!=null ) w.outAttribute( "CLOSEDR2L", r2l.getId(), true);
         }
         IDColor c;
         if( (c=model.getOpenedForeColor())!=null ) w.outAttribute( "OPENEDFORE", EncodeColor(c), true );
         if( (c=model.getOpenedBackColor())!=null ) w.outAttribute( "OPENEDBACK", EncodeColor(c), true );
         if( (c=model.getClosedForeColor())!=null ) w.outAttribute( "CLOSEDFORE", EncodeColor(c), true );
         if( (c=model.getClosedBackColor())!=null ) w.outAttribute( "CLOSEDBACK", EncodeColor(c), true );
       }
       else
       {
         //reuse already defined model
         w.outAttribute( "MODEL",  Integer.toString(id,16), true );
       }
     }
     
     w.doChildrenItem( folder, ext, arg );
     
     w.endElement();
   }

   /**
    * Writes xml for a DColorRegistry
    */
   public void doMethod(DColorRegistry cr, DExtensible p, Object arg)
   {
     startElement( "COLORS");
     for( Iterator i=cr.iterator(); i.hasNext(); )
     {       
	   invokeDoMethod( i.next(),p,arg);
     }
     endElement();
   }
	  
   /**
    * Writes xml for a DStyleRegistry
    */
	public void doMethod(DStyleRegistry sr, DExtensible p, Object arg)
	{
	  startElement("STYLES");
	  for( Iterator i = sr.iterator(); i.hasNext(); )
	  {
	    invokeDoMethod( i.next(),p,arg);
	  }
	  endElement();
	}
	 
   /**
    * Writes xml for a DFontRegistry
    */
	public void doMethod(DFontRegistry fr, DExtensible p, Object arg)
	{ 
	  startElement("FONTS");
	  for (Iterator i = fr.iterator(); i.hasNext() ;) 
	  {
		invokeDoMethod( i.next(),p,arg);
	  }		
	  endElement();
	}
	  
      /** aml dump for IDXmlSerializable class, write as cdata the writeXml() result. */
	  public void doMethod( IDXmlSerializable i, DExtensible ext, Object arg )
      {
        outCData( i.writeXml() );
      }
      
      /** aml dump for IDStringSerializable class, write as cdata the writeXml() result. */
      public void doMethod( IDStringSerializable i, DExtensible ext, Object arg )
      {
        outCData( i.serializeToString() );
      }

      /** encode string using any '&xx;' xml entity and dump coded string */
      public void doMethod( String str, DExtensible ext, Object arg )
      {
        outCData( encode( str) );
      }
      
      /** write xml for Byte objects. */
      public void doMethod( Byte v, DExtensible ext, Object arg )
      {
        outCData( ((Byte)v).toString() );
      }
      /** write xml for Short v. */
      public void doMethod( Short v, DExtensible ext, Object arg )
      {
        outCData( ((Short)v).toString() );
      }
      /** write xml for Integer objects. */
      public void doMethod( Integer v, DExtensible ext, Object arg )
      {
        outCData( ((Integer)v).toString() );
      }
      /** write xml for Long objects. */
      public void doMethod( Long v, DExtensible ext, Object arg )
      {
        outCData( ((Long)v).toString() );
      }
      /** write xml for Float objects. */
      public void doMethod( Float v, DExtensible ext, Object arg )
      {
        outCData( ((Float)v).toString() );
      }
      /** write xml for Double objects. */
      public void doMethod( Double v, DExtensible ext, Object arg )
      {
        outCData( ((Double)v).toString() );
      }
      /** write xml for Boolean objects. */
      public void doMethod( Boolean v, DExtensible ext, Object arg )
      {
        outCData( ((Boolean)v).toString() );
      }
      /** write xml for Class object, onl name of class is save */
      public void doMethod( Class c, DExtensible ext, Object arg )
      {
        outCData( c.getName() );
      }
      /** write xml for java.util.Date, save the time in milliseconds */
      public void doMethod( Date date, DExtensible ext, Object arg )
      {
        outCData( Long.toString( date.getTime() ) );
      }
      /** write xml for DecimalFormat save pattern */
      public void doMethod( com.ibm.icu.text.DecimalFormat df, DExtensible ext, Object arg )
      {
        outCData( df.toPattern() );
      }
      /** write xml for SimpleDateFormat save pattern */
      public void doMethod( com.ibm.icu.text.SimpleDateFormat df, DExtensible ext, Object arg )
      {
        outCData( df.toPattern() );
      }
      /** write xml for Serializable object, onl name of class is save */
/*TODO      public void doMethod( Serializable ser, DExtensible ext, Object arg )
   What a lots of byte to save ... ! beurk.
      {
        try
        {
          ByteArrayOutputStream baos = new ByteArrayOutputStream();          
          ObjectOutputStream oos = new ObjectOutputStream( baos );
          oos.writeObject( ser );
          String s = baos.toString();
          outCData( s );
        }
        catch( IOException e )
        {
          throw new DXmlError( e.getMessage() );
        }
      }*/
	  
    /**
     * Writes xml  for a DPropertyStore, save properties using keys, classes of objects
     */
	public void doMethod(DPropertyStore ps, DExtensible p, Object arg)
	{
	  Map m = ps.elements();
	  if ( m == null ) return;
	  Set s = m.keySet();
	  if ( s == null ) return;
	    
	  startElement("PROPERTIES");
	  for (Iterator i = s.iterator(); i.hasNext() ;)
	  {
		String key = (String)i.next();
		Object value = m.get(key);
		startElement( "PROPERTY");
		if (key != null)
		{
		   outAttribute( "NAME", key, true);
           Class clazz = value.getClass();
		   outAttribute( "CLASS", clazz.getName(), true);
           if( IDColor.class.isAssignableFrom( clazz ) )
           {
             //prefer to save color in color registry, here put only id
             IDColor color = (IDColor)value ;
             colors.putColor( color );
             outCData( color.getID() );
           }
           else if( IDFont.class.isAssignableFrom( clazz ) )
           {
             //prefer to save font in font registry, here put only id
             IDFont font = (IDFont)value;
             fonts.putFont( font );
             outCData( font.getID() );
           }
           else if( IDStyle.class.isAssignableFrom( clazz ) )
           {
             //prefer to save style in style registry, here put only id
             IDStyle style = (IDStyle)value;
             styles.putStyle( style );
             outCData( style.getID() );
           }
           else
           {
             invokeDoMethod(value,p,arg);
           }
		}
        endElement();
	  }
	  endElement();
	}
  
    protected HashMap extensions;
    
    /**
     * This method must be called by DXmlWriter extension in order to print in output stream
     * a XML element allowing the DXmlReader to load extension.
     * Only extension using their own xml element name should call this method, extension
     * that uses the default doMethod(IDItem,...) haven't to call it.
     * You can call this method more than once (each time item is written to DXmlWriter).
     * @param extension an item for which the creation of an instance will register all
     * extension this item require (ie: at least extension for DXmlReader).
     * @see org.eclipse.tptp.platform.report.chart.internal.DXYSurface for example.
     */
    public void writeExtension( Class extension )
    {
      if( extension==null ) return;
      if( extensions==null ) {
        extensions = new HashMap();
      } else if ( extensions.containsKey( extension.getClass() )) {
        //already wrote.
        return ;
      }
      try {
        //finish current element if any
        closeStartElement();        
        out.write( "<JSML_EXTENSION CLASS=\"".getBytes() );
        out.write( extension.getName().getBytes("UTF-8") );
        out.write( "\" />".getBytes() );
        extensions.put( extension.getClass(), null );
      } catch( IOException e ) {
        throw new DXmlError( e.getMessage() );
      }
    }
    
    /**
     * Similar to:
     * <PRE>
     * writeExtension( extension.getClass() )
     * </PRE>
     * @param extension Extension class to load and instanciate is same as IDItem class
     */
    public final void writeExtension( IDItem extension )
    {
      writeExtension( extension.getClass() );
    }
    
    /** Write XML for a DBorder */
    public void doMethod( DBorder border, DExtensible ext, Object arg )
    {
      DXmlWriter w = (DXmlWriter)ext;
      
      w.startElement( "BORDER", border );
      
      w.outAttribute( "SHADOW_TYPE", DBorder.StrShadowType(border.getShadowType()), "NONE" );
      w.outAttribute( "SHADOW_DEPTH", border.getShadowDepth(), 8 );
      if( border.getShadowColor()!=null )
      {
        w.outAttribute( "SHADOW_COLOR", DXmlWriter.EncodeColor( border.getShadowColor() ), true );
      }
      w.outAttribute( "LINE_TYPE", DBorder.StrLineType( border.getLineType()), "SIMPLE" );
      w.outAttribute( "MARGIN", border.getMargin(), 3 );
      
      w.doChildrenItem( border, ext, arg );
      
      w.endElement();
    }
    
    /** @return the used color registry */
    public DColorRegistry getColorRegistry() { return colors; }
    /** @return the used font registry */
    public DFontRegistry getFontRegistry() { return fonts; }
    /** @return the used style registry */
    public DStyleRegistry getStyleRegistry() { return styles; }
    
    /** @return the used color registry */
    public void setColorRegistry(DColorRegistry r) { colors = r ; }
    /** @return the used font registry */
    public void setFontRegistry(DFontRegistry r) { fonts = r; }
    /** @return the used style registry */
    public void setStyleRegistry(DStyleRegistry r) { styles = r; }
}