/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CircleAlg.java,v 1.2 2008/05/23 14:11:58 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.alg.internal;

import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IVector;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.Vector;



/**
 * Algorithm to generate circle point, tangent vector and fill circle and arc.
 *
 * TODO: why not removing CircleAlg and use only OvalAlg ??
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class CircleAlg
{
  /** store points for circle generation, this circle is centered at (0,0) */
  private static class Model extends ModelMap.AbstractModel
  {
    int radius_;
    int px_[],py_[]; // generated point of circle, store only 1/8th of circle
    float tx_[],ty_[]; //tangent normalized vector (for 1/8th of circle) 
    int size_;
    boolean shared_endpoint_;//true  if px_[size-1]==py_[size-1], important for path and fill
    int usage_; //count number of usage, for model map flush.
    
    public Model( int radius )
    {
      radius_=radius;
      computePoints();
      usage_=1;
    }
    
    public Object getKey() { return GetKey( radius_ ); }
    
    int ptx_,pty_;
    float tgx_,tgy_;
    /** available result through ptx_,pty_,tgx_,tgy_; */
    public void getPoint( int index, int octant )
    {
      switch( octant )
      {
        case 0: ptx_ = px_[index]; pty_ = py_[index];
                tgx_ = tx_[index]; tgy_ = ty_[index]; 
                return;
        case 1: ptx_ = py_[index]; pty_ = px_[index];
                tgx_ = -ty_[index]; tgy_ = -tx_[index];
                return;
        case 2: ptx_ = -py_[index]; pty_ = px_[index];
                tgx_ = -ty_[index]; tgy_ = tx_[index];
                return;
        case 3: ptx_ = -px_[index]; pty_ = py_[index];
                tgx_ = tx_[index]; tgy_ = -ty_[index];
                return;
        case 4: ptx_ = -px_[index]; pty_ = -py_[index];
                tgx_ = -tx_[index]; tgy_ = -ty_[index];
                return;
        case 5: ptx_ = -py_[index]; pty_ = -px_[index];
                tgx_ = ty_[index]; tgy_ = tx_[index];
                return ;
        case 6: ptx_ = py_[index]; pty_ = -px_[index];
                tgx_ = ty_[index]; tgy_ = -tx_[index];
                return;
        case 7: ptx_ = px_[index]; pty_ = -py_[index];
                tgx_ = -tx_[index]; tgy_ = ty_[index];
                return ;
      }
    }
    private void computePoints()
    {
      if( radius_==0 )
      {
        //at least the center
        px_=new int[1]; px_[0]=0;
        py_=new int[1]; py_[0]=0;
        size_=1; 
        return;
      }
      //use Bresenham algorithm:
      int size = radius_; //very bad estimation of size
      int d = 3  - 2*radius_;
      int x = radius_;
      int y = 0;
      size_=0;
      px_ = new int[size];
      py_ = new int[size];

      int di = 10 - 4*radius_;
      int ri = 6;
      x = 0;
      y = radius_;
      while (x <= y) {
        px_[size_] =y;
        py_[size_] =x;        
        size_++;
        if (d >=  0) {
          d += di; 
          di += 8;
          y -= 1;
        }
        else {
          d += ri;
          di += 4;
        }
        ri += 4;
        x += 1;
      }
  
      //reduce data:
      int np[] = new int[size_];
      for( int i=0; i<size_; ++i) np[i] = px_[i];
      px_ = np;
      np = new int[size_];
      for( int i=0; i<size_; ++i) np[i] = py_[i];
      py_ = np;     
      
      shared_endpoint_ = (px_[size_-1]==py_[size_-1]);
      
      //compue tangent vector:
      tx_ = new float[size_];
      ty_ = new float[size_];
      for( int i=0; i<size_; ++i )
      {
        //vector 0P:
        int px =px_[i];
        int py =py_[i];
        float norm = (float)Math.sqrt( px*px + py*py );
        if( norm==0 )
        {
          tx_[i] = ty_[i] = 0;
        } else {          
          //tangent vector is OP with PI/2 rotation.        
          tx_[i] = -py/norm;
          ty_[i] =  px/norm;
        }
      }
    }    
  }
  
  protected int cx_,cy_; // center of circle.
  protected double arc_start_, arc_length_; //angle of arc start, and arc length.
  protected Model model_;
  protected int istart_, iend_; //index of start/end in model's array.
  protected int ostart_, oend_; //octant start/end

  /** Create empty cirle, must call setCircle or setCircle arc to define circle. */
  public CircleAlg()
  {
    setCircle(0,0,0);
  }
  public CircleAlg( int cx, int cy, int radius )
  {
    setCircle( cx,cy,radius);
  }
  public CircleAlg( int cx, int cy, int radius, double arc_start, double arc_length )
  {
    setArc( cx,cy,radius, arc_start, arc_length );
  }
  public CircleAlg( CircleAlg c ) 
  {
    setCircle( c );
  }
  
  public void setCircle( int cx, int cy, int radius )
  {
    cx_=cx; cy_=cy;
    arc_start_ = 0;
    arc_length_ = Radian._2PI;
    model_ = GetModel( radius );
    istart_ = 0; ostart_=0;
    iend_=0; oend_=0; //strange but not ?
    state_=1;
//System.out.println("Circle.setCircle() istart="+istart_+" ostart="+ostart_+" iend="+iend_+" oend="+oend_);    
  }
  public void setArc( int cx, int cy, int radius, double arc_start, double arc_length )
  {
    cx_=cx; cy_=cy;
    arc_start_ = Radian.normalize(arc_start);
    arc_length_ = arc_length;
    model_ = GetModel( radius );  
    double as = Radian.normalize( arc_start_ );
    double ae = Radian.normalize( arc_start_+arc_length );
    ostart_ = GetOctant( as );
    oend_   = GetOctant( ae );
    istart_ = GetIndex( as );
    iend_   = GetIndex( ae );
    state_=1;
//System.out.println("Circle.setCircleArc() istart_="+istart_+" ostart_"+ostart_+" iend="+iend_+" oend="+oend_+" as="+as+" ae="+ae);
//System.out.println("--Circle.setArc() model.size_ (octant)="+model_.size_+" shared.endpoint="+model_.shared_endpoint_);    
//System.out.println("arc_start="+arc_start_+"->as="+as+" ostart="+ostart_+" istart="+istart_);
//System.out.println("arc_length="+arc_length_+"->ae="+ae+" oend="+oend_+" iend="+iend_);
    
  }
  public void setCircle( CircleAlg c )
  {
    cx_=c.cx_; cy_=c.cy_;
    arc_start_ = c.arc_start_;
    arc_length_= c.arc_length_;
    model_ = c.model_;
    istart_=c.istart_;
    iend_=c.iend_;
    ostart_=c.ostart_;
    oend_=c.oend_;
    state_=1;
  }
  
  public int getCenterX() { return cx_; }
  public int getCenterY() { return cy_; }
  public int getRadius() { return model_.radius_; }
  public double getArcStart() { return arc_start_; }
  public double getArcLength() { return arc_length_; }
  
  //expected normalized angle (0..2PI(
  private static int GetOctant( double angle )
  {
    if( angle<Radian._PI4  ) return 0;
    if( angle<Radian._PI2  ) return 1;
    if( angle<Radian._3PI4 ) return 2;
    if( angle<Radian._PI   ) return 3;
    if( angle<Radian._5PI4 ) return 4;
    if( angle<Radian._3PI2 ) return 5;
    if( angle<Radian._7PI4 ) return 6;
    return 7;
  }
  private static int NextOctant( int octant ) { return (octant+1)%8; }
  private static int PrevOctant( int octant ) {
    octant--;
    if( octant<0 ) octant=7;
    return octant;
  }
  private int GetIndex( double angle )
  {
    int index;
    if( angle<Radian._PI4 )
    {
      index = (int)(((model_.size_-1)*angle)/Radian._PI4);      
    } 
    else if( angle < Radian._PI2 )
    {
      index = (int)(((model_.size_-1)*(Radian._PI2-angle))/Radian._PI4);      
    }
    else if( angle < Radian._3PI4 )
    {
      index = (int)(((model_.size_-1)*(angle-Radian._PI2))/Radian._PI4);
    }
    else if ( angle < Radian._PI )
    {
      index = (int)(((model_.size_-1)*(Radian._PI-angle))/Radian._PI4);
    }
    else if ( angle < Radian._5PI4 )
    {
      index = (int)(((model_.size_-1)*(angle-Radian._PI))/Radian._PI4);
    }
    else if ( angle < Radian._3PI2 )
    {
      index = (int)(((model_.size_-1)*(Radian._3PI2-angle))/Radian._PI4);
    }
    else if ( angle < Radian._7PI4 )
    {
      index = (int)(((model_.size_-1)*(angle-Radian._3PI2))/Radian._PI4);
    }
    else 
    {
      index = (int)(((model_.size_-1)*(Radian._2PI-angle))/Radian._PI4);
    }
    return index;
  }
  
  protected static ModelMap models_ = new ModelMap( 5,50,10);
  
  private static Object GetKey( int radius )
  {
    return new Integer(radius);
  }
  private static Model GetModel( int radius )
  {
    Object key = GetKey( radius );
    Model model = (Model)models_.getModel( key );
    if( model==null )
    {
      model = new Model( radius );
      models_.addModel( model );
    }
    return model;
  }
 
  protected int curr_index_, curr_octant_;
  protected Vector vector_ = new Vector();
  protected byte state_;

  /** restart nextPoint generation */
  public void restart()
  { 
    state_=1;
  }
  public void endPathElement(IGC gc) {}

  private int curr_ptx_, curr_pty_;
  private float curr_tgx_, curr_tgy_;
  
  /**@return X coordinate of point computed by nextPoint(), available if nextPoint() return true */
  public int getX() { return curr_ptx_; }
  /**@return X coordinate of point computed by nextPoint(), available if nextPoint() return true */
  public int getY() { return curr_pty_; }
  /**@return X coordinate of tangent vector computed by nextPoint(), available if nextPoint() return true */
  public float getTX() { return curr_tgx_; }
  /**@return X coordinate of tangent vector computed by nextPoint(), available if nextPoint() return true */
  public float getTY() { return curr_tgy_; }
  
private int cnt;
  public boolean nextPoint(IPoint point) 
  {    
    switch( state_ )
    {
      case 0: return false; //finished
      case 1: //not started
        if( (model_.radius_==0)||(arc_length_==0) ) { state_=0; return false; } //nothing to generate
        curr_index_=istart_;
        curr_octant_=ostart_;
cnt=0;        
        state_=2; //generating
        break;
    }
    
//System.out.println("Circle.nextPoint() curr_index="+curr_index_+" curr_octant="+curr_octant_);
//TODO:remove debug here    
cnt++;
if(cnt>9*model_.size_) {
  System.out.println("--- obviously there are a bad end condition, stop here");
  System.out.println(" istart="+istart_+" ostart="+ostart_+" iend="+iend_+" oend="+oend_);
  System.out.println(" arc_start_="+arc_start_+" arc.length="+arc_length_);
  return false;
}
    model_.getPoint( curr_index_, curr_octant_ );
    curr_ptx_ = cx_+model_.ptx_;
    curr_pty_ = cy_-model_.pty_;
    curr_tgx_ = -model_.tgx_;
    curr_tgy_ = model_.tgy_;

    if( point!=null) point.setPoint( curr_ptx_, curr_pty_ );
    //next index: (depending on octant index orientation AND arc orientation )
    if( arc_length_ >= 0 )
    {
      if( (curr_octant_&0x1)==0 )
      {
        //incr octant
        curr_index_++;
        if( model_.shared_endpoint_ )
        {
          if( curr_index_>= model_.size_-1 ) {
            curr_index_ = model_.size_-1;
            curr_octant_ = NextOctant( curr_octant_ );
          }
        } else {
          if( curr_index_ >= model_.size_ ) {
            curr_index_ = model_.size_-1;
            curr_octant_ = NextOctant( curr_octant_ );
          }
        }
      } else {
        //decr octant
        curr_index_--;
        if( curr_index_ == 0 ) 
        {
          curr_octant_ = NextOctant( curr_octant_ );
        }
      }
    }
    else // arc_length < 0 
    {
      if( (curr_octant_&0x1)==0 )
      {
        curr_index_ --;
        if( curr_index_ < 0 ) 
        {
          curr_index_ = 1;
          curr_octant_ = PrevOctant( curr_octant_ );
        }
      } else {
        curr_index_++;
        if( curr_index_ >= model_.size_ )
        {
          if( model_.shared_endpoint_ )
          {
            curr_index_ = model_.size_-2;
          } else {
            curr_index_ = model_.size_-1;
          }
          curr_octant_ = PrevOctant( curr_octant_ );
        }
      }
    }
    //stop test for next time
    if( curr_octant_ == oend_ && curr_index_==iend_ ) 
    {
      //no more points
      state_=0;
    }
    //but have a point for this time         
    return true;       
  }

  /** @return in vector current tangent, available if nextPoint() return true. */
  public void getCurrentTangent(IVector vector) 
  {
    vector.setVector( curr_tgx_, curr_tgy_  );
  }

  /** @return true if (x,y) point is inside given circle */
  public static boolean Contains( int x, int y, int cx, int cy, int radius )
  {
    //check for circle bounds:
    if( x < cx-radius || y < cy-radius || x > cx+radius || y > cy+radius ) return false;
    // test for point inside circle:
    int dx = x-cx, dy=y-cy;
    int pr2 = dx*dx + dy*dy;
    if( pr2 > radius*radius ) return false; //outside circle
    return true;
  }
  
  /** @return true if (x,y) point is inside given circle arc */
  public static boolean Contains( int x, int y, int cx, int cy, int radius, double arc_start, double arc_length )
  {
    //check for circle bounds:
    if( x < cx-radius || y < cy-radius || x > cx+radius || y > cy+radius ) return false;
    // test for point inside circle:
    int dx = x-cx, dy=y-cy;
    int pr2 = dx*dx + dy*dy;
    if( pr2 > radius*radius ) return false; //outside circle
    //this is a full circle:
    if( arc_length >= Radian._2PI || -arc_length <= Radian._2PI )
    {
      return true;
    } 
    //test for point's angle, must be in start_ .. end_angle
    double angle = Math.atan2( dy, dx );
    if( angle < 0 ) angle += Radian._2PI;
    if( arc_length == 0 ) 
    {
      return angle == arc_start; //?or false as arc length is 0 !
    }
    else if ( arc_length > 0 )
    {
      if (angle < arc_start ) angle += Radian._2PI;
      return angle <= arc_start+arc_length;
    } else {
      if( angle > arc_start ) angle -= Radian._2PI;
      return angle >= arc_start+arc_length;
    }
  }
  
  public boolean contains(int x, int y) 
  {
    return Contains( x,y, cx_,cy_,model_.radius_, arc_start_, arc_length_ );
  }

  public boolean contains(IPoint p) {
    return Contains( p.getX(),p.getY(), cx_,cy_,model_.radius_, arc_start_, arc_length_ ); 
  }
//TODO: Contains( IRect, .... ) at least for a circle is usefull too, or better contains(IShape)...
  
  public IRect getBounds() 
  {
    int r = model_.radius_;
    int r2 = r<<1; //2*r;
    return new Rect(cx_-r, cy_-r, r2, r2 );
  }

  /** fill current circle as it was really a full circle, see FillCircle(). */
  public void fillCircle( IGCDirect gc )
  {
    FillCircle( gc, model_, cx_,cy_ );
  }
  
  public void fill( IGCDirect gd )
  {
    FillArc( gd, cx_, cy_, model_.radius_, arc_start_, arc_length_ );
  }
  
   /**
   * Fill circle (even if this is an arc definition), brush must be started by caller,
   * use only drawHLineDirect().
   */
  public static void FillCircle( IGCDirect gc, int cx, int cy, int radius )
  {
    FillCircle( gc, GetModel(radius), cx,cy );
  }
  
  /**
   * internal use only.
   */
  protected static void FillCircle( IGCDirect gc, Model model, int cx, int cy )
  {
    //but can also be filled using compute points without model.
    int lx=model.px_[0], ly=model.py_[0];
    gc.drawHLineDirect( cx-lx, cx+lx, cy );
    int lty=lx, lxx=ly; 
    int px=lx, py=ly;
    for( int index=1; index<model.size_; index++ )
    {
      px = model.px_[index];
      py = model.py_[index];
      gc.drawHLineDirect( cx-px, cx+px, cy+py );
      gc.drawHLineDirect( cx-px, cx+px, cy-py );
      if( px != lty )
      {
        gc.drawHLineDirect( cx-lxx, cx+lxx, cy+lty );
        gc.drawHLineDirect( cx-lxx, cx+lxx, cy-lty );
        lty=px;
        lxx=py;
      } else {
        if ( py > lxx ) lxx=py;
      }
    }
    if( px!=py) { //missing h line if last point is not on line y=x;
      gc.drawHLineDirect( cx-lxx, cx+lxx, cy+lty );
      gc.drawHLineDirect( cx-lxx, cx+lxx, cy-lty );
    } 
  }
  
  public static void FillArc( IGCDirect gd, int cx, int cy, int radius, double arc_start, double arc_length )
  {
    if( arc_length >= Radian._2PI || arc_length <= -Radian._2PI )
    {
      FillCircle( gd, cx, cy, radius );
    } 
    else
    {
//TODO: take a look at OvalAlg.FillArc() and think if it's really optimized to rewrite same
//thing for CircleAlg ... for the moment I prefer to delegate works to OvalAlg, even if
//more in memory data is required to this..
      OvalAlg.FillArc( gd, cx,cy, radius,radius, arc_start, arc_length );
    }
  }
}
