/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LineAlg.java,v 1.2 2008/05/23 14:11:58 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.alg.internal;

import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IVector;

/**
 * Algorythm for segment's point generation, can access to each points.
 * This is a implementation of Bresenham's line algorithm.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class LineAlg implements IAlg
{
  protected int S_NOTSTARTED=0;
  protected int S_GENERATING=1;
  protected int S_ENDED=2;
  
  protected int x1_, y1_, x2_, y2_ ;
  protected int state_;


  /**
   * Build a segment between two 2d points 
   */
  public LineAlg( int _x1, int _y1, int _x2, int _y2 )
  {
    setLine( _x1, _y1, _x2, _y2 );
  }
  
  /** Create line from 0,0 to 0,0, call setLine() to change this values. */
  public LineAlg()
  {
    setLine(0,0,0,0);
  }

  /**
   * set a new line a segment between two 2d points. 
   */
  public void setLine( int _x1, int _y1, int _x2, int _y2 )
  {
    x1_ = _x1; y1_ = _y1 ;
    x2_ = _x2; y2_ = _y2 ;
    state_=S_NOTSTARTED;
    update_tangent_=true;
  }
  
  /** @return X coordinate of first points of line */
  public int getX1() { return x1_; }
  /** @return Y coordinate of first points of line */
  public int getY1() { return y1_; }
  /** @return X coordinate of second points of line */
  public int getX2() { return x2_; }
  /** @return Y coordinate of second points of line */
  public int getY2() { return y2_; }
  
  /** restart current segment */
  public void restart()
  {
    state_=S_NOTSTARTED;
  }
  
  protected int xd, yd;//internal use
  protected int px_, py_ ; //current point coordinate
  protected int ax, ay; //internal use
  protected int sx, sy; //internal use

  /** @return current X point  coordinate point available after call of and if nextPoint() return true. */
  public int getX() { return px_; }
  /** @return current Y point coordinate available after call of and if nextPoint() return true. */
  public int getY() { return py_; }
  
  /**
   * Return next point in line, or false if they are no more point.
   * point can be accessed getX() and getY(). if point is not null it will be set with result
   * if there are a next point.
   * computed point is available through getX(),getY() or point parameter if it's not null.
   * @return true if there are a next point in segment, false otherwise.
   * @param point if not null, nextPoint when returning true, fill this point with coordinates.
   */
  public boolean nextPoint( IPoint point )
  {
    if( state_==S_ENDED ) return false;
    if ( state_==S_NOTSTARTED )
    {
      int dx = x2_ - x1_;
      int dy = y2_ - y1_;

      ax = (dx>=0?dx:-dx) << 1; //2*abs(dx)
      ay = (dy>=0?dy:-dy) << 1; //2*abs(dy)
      sx = (dx>0) ? +1 : (dx==0)? 0 : -1; //sign
      sy = (dy>0) ? +1 : (dy==0)? 0 : -1; //sign

      px_ = x1_;
      py_ = y1_;
 
      state_=S_GENERATING;

      if (ax >= ay)            /* x dominant */
      {
        yd = ay - (ax >> 1);
      } else {
        xd = ax - (ay >> 1);
      }
    }
    else
    {
      //compute next pos.
      if (ax >= ay)            /* x dominant */
      {
        //the end
        if (px_ == x2_)
        {
          state_=S_ENDED;
          return false;
        }

        if (yd >= 0)
        {
          py_ += sy;
          yd -= ax;
        }

        px_ += sx;
        yd += ay;
      }
      else
      {
        if (py_ == y2_)
        {
          state_=S_ENDED;
          return false;
        }

        if (xd >= 0)
        {
          px_ += sx;
          xd -= ay;
        }

        py_ += sy;
        xd += ax;
      }
    }

    if ( point!=null) point.setPoint( px_, py_ );
    return true;
  }
  
  /** 
   * Clip current segment to given rectangle.  Segment might change even if this method return false.
   * @param rect Rectangle used to clip segment
   * @return true if part of segment is visible, false if segment is outside rect.
   */
  public boolean clipTo( IRect rect )
  {
    return clipTo( rect.getX(), rect.getY(), rect.getW(), rect.getH() );
  }
  
  /**@return Sutherland-Cohen code (in a hexa decimal readable maner) */
  protected int code( int x, int y, int xn, int yn, int xx, int yx )
  {
    int code = 0;
    if( y < yn ) code |=0x0100; else if ( y > yx ) code |=0x1000;
    if( x < xn ) code |=0x0001; else if ( x > xx ) code |=0x0010;
    return code;
  }
  
  /** 
   * Clip current segment to given rectangle.Segment might change even if this method return false.
   * @param x,y,w,h rectangle to clip segment to, follow IRect definition of rectangle.  
   * @return true if part of segment is visible, false if segment is outside rect.
   */
  public boolean clipTo( int x, int y, int w, int h )
  {
    int xn = x, xx=x+w-1;
    int yn = y, yx=y+h-1;
    if( xn>xx) { int t=xn; xn=xx; xx=xn; }
    if( yn>yx) { int t=yn; yn=yx; yx=t; }
    return clipToBounds( xn,yn,xx,yx);
  }
  
  /**
   * Clip segment to given bounds. Segment might change even if this method return false.
   * @param xn left side of bounds, must be lower than xx.
   * @param yn bottom side of bounds, must be lower than yx.
   * @param xx right side of bounds, must be greater than xn.
   * @param yx top side of bounds, must be greater than yn.
   * @return true is current segment have visible part in this bounds, false otherwise.
   */
  public boolean clipToBounds( int xn, int yn, int xx, int yx )
  {
    //based on Sutherland-Cohen Line Clipping
    int code_p1 = code( x1_, y1_, xn,yn,xx,yx);
    int code_p2 = code( x2_, y2_, xn,yn,xx,yx);
    
    while( true )
    {
      if( code_p1 == 0 && code_p2 == 0 ) return true; //all segment is inside rect
      if( ( code_p1 & code_p2 ) != 0 ) return false; //both point in same region
    
      //does segment intersects xn vertical line ?
      if( (code_p1&0x0001)!=0 )
      {
        //yes, replace p1 with intersection:
        y1_ = y1_ + ((y2_-y1_)*(xn-x1_))/(x2_-x1_);
        x1_ = xn;
        code_p1 = code( x1_, y1_, xn,yn,xx,yx);
        continue;
      }
      if( (code_p2&0x0001)!=0 )
      {
        //yes replace p2 with intersection :
        y2_ = y1_ + ((y2_-y1_)*(xn-x1_))/(x2_-x1_);
        x2_ = xn;
        code_p2 = code( x2_, y2_, xn,yn,xx,yx);
        continue;
      }
      //does segment intersect xx vertical line ?
      if( (code_p1 & 0x0010)!=0 )
      {
        //yes, replace p1 with intersection:
        y1_ = y1_ + ((y2_-y1_)*(xx-x1_))/(x2_-x1_);
        x1_ = xx;
        code_p1 = code( x1_, y1_, xn,yn,xx,yx);
        continue;
      }
      if( (code_p2 &0x0010)!=0 )
      {
        //yes, replace p2 with intersection:
        y2_ = y1_ + ((y2_-y1_)*(xx-x1_))/(x2_-x1_);
        x2_ = xx;
        code_p2 = code( x2_, y2_, xn,yn,xx,yx);
        continue;
      }
      //does segment intersects yn horizontal line ?
      if( (code_p1 &0x0100)!=0)
      {
        //yes replace p1 with intersection:
        x1_ = x1_ + ((yn-y1_)*(x2_-x1_))/(y2_-y1_);
        y1_ = yn;
        code_p1 = code( x1_, y1_, xn,yn,xx,yx);
        continue;
      }
      if( (code_p2 &0x0100)!=0)
      {
        //yes replace p2 with intersection:
        x2_ = x1_ + ((yn-y1_)*(x2_-x1_))/(y2_-y1_);
        y2_ = yn;
        code_p2 = code( x2_, y2_, xn,yn,xx,yx);
        continue;
      }
      //does segment intersects yx horizontal line ?
      if( (code_p1 &0x1000)!=0)
      {
        //yes replace p1 with intersection:
        x1_ = x1_ + ((yx-y1_)*(x2_-x1_))/(y2_-y1_);
        y1_ = yx;
        code_p1 = code( x1_, y1_, xn,yn,xx,yx);
        continue;
      }
      if( (code_p2 &0x1000)!=0)
      {
        //yes replace p2 with intersection:
        x2_ = x1_ + ((yx-y1_)*(x2_-x1_))/(y2_-y1_);
        y2_ = yx;
        code_p2 = code( x2_, y2_, xn,yn,xx,yx);
        continue;
      }
      //can't pass here:
      throw new Error("Assertion failed in clipToBounds()");
    }
  }
  
  
  public String toString()
  {
    return super.toString()+"{p1="+x1_+","+y1_+", p2="+x2_+","+y2_+"}";
  }
  
  protected double tgx_,tgy_;
  protected boolean update_tangent_;
  public void tangent( IVector vector )
  {
    if( update_tangent_ )
    {
      //compute normalized tangent once.
      double tgx_ = x2_-x1_;
      double tgy_ = y2_-y1_;    
      //normalize vector once.
      double n = Math.sqrt( tgx_*tgx_ + tgy_*tgy_ );
      if( n!=1.0 && n!=0.0 )
      {
        tgx_ /= n;
        tgy_ /= n;
      }
      update_tangent_=false;
    }
    vector.setVector( tgx_,tgy_);
  }
  public void backTangent( IVector vector )
  {
    tangent( vector );
  }
  public void frontTangent( IVector vector )
  {
    tangent( vector ); 
  }
};
