/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: OvalAlg.java,v 1.2 2008/05/23 14:11:58 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.alg.internal;

import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IVector;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.Vector;


/**
 * Algorithm to generate oval point, tangent vector and fill oval and arc.
 * Oval is an ellipse with one of it axis aligned to one X or Y axis of coordinates.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class OvalAlg
{
  /** store points for oval generation, this oval is centered at (0,0) */
  private static class Model extends ModelMap.AbstractModel
  {
    int rx_,ry_;//radius along X and Y
    int px_[],py_[]; // generated point of oval, store only 1/4th of oval
    float tx_[],ty_[]; //tangent normalized vector (for 1/4th of oval) 
    int size_;
    int usage_; //count number of usage, for model map flush.
    
    public Model( int rx, int ry )
    {
      rx_=rx; ry_=ry;
      computePoints();
      usage_=1;
    }
    
    public Object getKey() { return GetKey(rx_,ry_); }
    
    int ptx_,pty_;
    float tgx_,tgy_;
    /** available result through ptx_,pty_,tgx_,tgy_; */
    public void getPoint( int index, int quadrant )
    {
      switch( quadrant )
      {
        case 0: ptx_ = px_[index]; pty_ = py_[index];
                tgx_ = tx_[index]; tgy_ = ty_[index]; 
                return;        
        case 1: ptx_ = -px_[index]; pty_ = py_[index];
                tgx_ =  tx_[index]; tgy_ = -ty_[index];
                return;
        case 2: ptx_ = -px_[index]; pty_ = -py_[index];
                tgx_ = -tx_[index]; tgy_ = -ty_[index];
                return;        
        case 3: ptx_ = px_[index]; pty_ = -py_[index];
                tgx_ = -tx_[index]; tgy_ = ty_[index];
                return ;
      }
    }
    private void computePoints()
    {
      if( rx_==0 || ry_==0 )
      {
        //at least the center
        px_=new int[1]; px_[0]=0;
        py_=new int[1]; py_[0]=0;
        size_=1; 
        return;
      }
      //use "A Fast Bresenham Type Algorithm For Drawing Ellipses" by John Kennedy
      int size = Math.max(rx_,ry_); // bad estimation of size
      int [] px1 = new int[size];
      int [] py1 = new int[size];
      long _2a2 = 2*rx_*rx_;
      long _2b2 = 2*ry_*ry_;
      int x = rx_;
      int y = 0;
      long xchg = ry_*ry_*(1-2*rx_);
      long ychg = rx_*rx_;
      long err  = 0;
      long stopx = _2b2*rx_;
      long stopy = 0;
      int arc1=0;
      
      while( stopx >= stopy )
      {
        px1[arc1] = x;
        py1[arc1] = y;
        arc1++;
        
        y++;
        stopy += _2a2;
        err += ychg;
        ychg += _2a2;
        if( (2*err+xchg) > 0 )
        {
          x--;
          stopx -= _2b2;
          err += xchg;
          xchg += _2b2;
        }
      }
      
      x = 0;
      y = ry_;
      int px2[] = new int[size];
      int py2[] = new int[size];
      int arc2=0;
      xchg = ry_*ry_;
      ychg = rx_*rx_*(1-2*ry_);
      err = 0;
      stopx = 0;
      stopy = _2a2*ry_;
      while ( stopx <= stopy ) 
      {
        px2[arc2] = x;
        py2[arc2] = y;
        arc2++;
        x++;
        stopx += _2b2;
        err += xchg;
        xchg += _2b2;
        if( (2*err+ychg) > 0 ) 
        {
          y--;
          stopy -= _2a2;
          err += ychg;
          ychg += _2a2;
        }
      }
      
      //now merge both arc:
      size_ = arc1+arc2;
      px_ = new int[size_];
      py_ = new int[size_];
      for( int i=0; i<arc1; i++ )
      {
        px_[i] = px1[i];
        py_[i] = py1[i];
      }
//System.out.println("arc1="+arc1+" arc2="+arc2+" size="+size+"-> size_="+size_);      
      for( int i=arc2-1,j=arc1; i>=0; i--,j++ )
      {
        px_[j] = px2[i];
        py_[j] = py2[i];
      }
//System.out.println("Oval rx_="+rx_+" ry_="+ry_);      
//System.out.println(" first point="+px_[0]+","+py_[0]+" last point="+px_[size_-1]+","+py_[size_-1]);
//System.out.println("Oval.Model: real size="+size_);      
      
      //compue tangent vector:
      tx_ = new float[size_];
      ty_ = new float[size_];
      double inv_rx2 =2.0/(double)(rx_*rx_);
      double inv_ry2 =2.0/(double)(ry_*ry_);
      for( int i=0; i<size_; ++i )
      {
        //vector 0P:
        int px =px_[i];
        int py =py_[i];
        double tx = px_[i] * inv_rx2;
        double ty = py_[i] * inv_ry2;
        double n = Math.sqrt( tx*tx + ty*ty );
        if( n==0.0 )
        {
          tx_[i] = ty_[i]= 0.0f;
        } else {
          tx_[i] = -(float)( ty/n );
          ty_[i] =  (float)( tx/n );
        }
      }
    }        
    /** fill the full oval with center at given location using drawHLineDirect */
    public void fillOval( IGCDirect gd, int cx, int cy )
    {
      int px=px_[0], py=py_[0];
      gd.drawHLineDirect( cx-px, cx+px, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          int lx = cx-px, rx=cx+px;
          gd.drawHLineDirect( lx, rx, cy+py );
          gd.drawHLineDirect( lx, rx, cy-py );
          ly=py;
        }      
      }
    }      
    /** fill the quadrant 0 and 1 (top hemi-oval) */
    public void fillQuadrant01( IGCDirect gc, int cx, int cy )
    {
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx-px, cx+px, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx-px, cx+px, cy-py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 2 and 3 (bottom hemi-oval) */
    public void fillQuadrant23( IGCDirect gc, int cx, int cy )
    {
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx-px, cx+px, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx-px, cx+px, cy+py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 1 and 2 (left hemi-oval) */
    public void fillQuadrant12( IGCDirect gc, int cx, int cy )
    {    
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx-px, cx, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          int lx=cx-px;
          gc.drawHLineDirect( lx, cx, cy-py );
          gc.drawHLineDirect( lx, cx, cy+py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 3 and 0 */
    public void fillQuadrant30( IGCDirect gc, int cx, int cy )
    { 
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx, cx+px, cy ); 
      int ly=py;;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          int rx=cx+px;
          gc.drawHLineDirect( cx, rx, cy-py );
          gc.drawHLineDirect( cx, rx, cy+py );
          ly=py;
        }      
      }
    }  
    /** fill the quadrant 0 */
    protected void fillQuadrant0( IGCDirect gc, int cx, int cy )
    { 
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx, cx+px, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx, cx+px, cy-py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 1 */
    public void fillQuadrant1( IGCDirect gc, int cx, int cy )
    {
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx-px, cx, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx-px, cx, cy-py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 2 */
    public void fillQuadrant2( IGCDirect gc, int cx, int cy )
    {  
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx-px, cx, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx-px, cx, cy+py );
          ly=py;
        }      
      }
    }
    /** fill the quadrant 3 */
    public void fillQuadrant3( IGCDirect gc, int cx, int cy )
    {  
      int px=px_[0], py=py_[0];
      gc.drawHLineDirect( cx, cx+px, cy ); 
      int ly=py;
      for( int index=1; index<size_; index++ )
      {
        px = px_[index];
        py = py_[index];
        if( py!=ly )
        {
          gc.drawHLineDirect( cx, cx+px, cy+py );
          ly=py;
        }      
      }
    }
    /**
     * fill given arc, check for various optimized way to do this.
     */
    public void fillArc( IGCDirect gc, int cx, int cy, double arc_start, double arc_length )
    {
      double aalen = Math.abs( arc_length );
      double ast = Radian.normalize( arc_start );
      if( aalen == 0.0 ) return ; //nothing to fill
      if( aalen >= Radian._2PI  )
      {
        //full oval
        fillOval( gc, cx, cy );
        return;
      }
      //top hemi-oval:
      else if( ast==0.0        && arc_length== Radian._PI
          ||  ast==Radian._PI && arc_length==-Radian._PI )
      {
        fillQuadrant01( gc, cx, cy );
        return;
      }
      //bottom hemi-oval
      else if( ast==0.0        && arc_length==-Radian._PI
          ||  ast==Radian._PI && arc_length== Radian._PI )
      {
        fillQuadrant23( gc, cx, cy );
        return;
      }
      //left hemi-oval
      else if ( ast==Radian._PI2  && arc_length ==  Radian._PI 
          ||  ast==Radian._3PI2 && arc_length == -Radian._PI )
      {
        fillQuadrant12( gc, cx, cy );      
        return;
      }
      //left hemi-oval
      else if ( ast==Radian._3PI2 && arc_length ==  Radian._PI 
          ||  ast==Radian._PI2  && arc_length == -Radian._PI )
      {
        fillQuadrant30( gc, cx, cy );
        return;
      }
      //first quadrant:
      else if( ast==0.0         && arc_length== Radian._PI2 
          || ast==Radian._PI2 && arc_length==-Radian._PI2 ) 
      {
        fillQuadrant0( gc, cx, cy );
        return;
      }
      //second quadrant:
      else if( ast==Radian._PI2 && arc_length== Radian._PI2 
          || ast==Radian._PI  && arc_length==-Radian._PI2 ) 
      {
        fillQuadrant1( gc, cx, cy );
        return;
      }
      //third quadrant:
      else if( ast==Radian._PI   && arc_length== Radian._PI2 
          || ast==Radian._3PI2 && arc_length==-Radian._PI2 ) 
      {
        fillQuadrant2( gc, cx, cy );
        return;
      }
      //fourth quadrant:
      else if( ast==Radian._PI   && arc_length== Radian._PI2 
          || ast==Radian._3PI2 && arc_length==-Radian._PI2 ) 
      {
        fillQuadrant3( gc, cx, cy );
        return ;
      }
      
      double aed = ast+arc_length;
      double naed = Radian.normalize( aed );
      int qs = GetQuadrant( ast );
      int qe = GetQuadrant( naed );
System.out.println("OvalAlg.Model.fillArc() qs="+qs+" qe="+qe); //TODO: debug here!
      switch( qs )
      {
      case 0:
        switch( qe ) 
        {
        case 0: //both angle in quadrant 0
          if( arc_length>=0 )
          {
            if( ast <= naed ) {
              fillArc00Tiny( gc, cx,cy, ast, naed );
            } else { 
              fillArc00Full( gc, cx,cy, ast, naed );           
            }
          } else {
            if( ast <= naed ) {
              fillArc00Full( gc, cx,cy, naed, ast );
            } else {
              fillArc00Tiny( gc, cx,cy, naed, ast );
            }
          }
          return ;
        case 1: //qs==0, qe==1
          if( arc_length>=0 )
            fillArc01Top( gc, cx,cy, ast, naed );
          else
            fillArc01Bottom( gc, cx,cy, ast, naed );
          return;
        case 2: //qs==0, qe==2
          if( arc_length>=0 )
            fillArc02Top( gc, cx,cy, ast,naed );
          else
            fillArc02Bottom( gc, cx,cy, ast, naed );
          return;
        case 3: //qs==0 qe==3
          if( arc_length >=0 )
            fillArc03Left( gc, cx,cy,ast, naed );
          else 
            fillArc03Right( gc, cx,cy,ast,naed );        
          return;
        }//switch
      case 1:
        switch( qe )
        {
        case 0:  
          if(  arc_length >= 0 )
            fillArc01Bottom( gc, cx,cy, naed, ast );
          else
            fillArc01Top( gc, cx,cy, naed, ast );       
          return;
        case 1:
          if( arc_length>=0 )
          {
            if( ast <= naed ) {
              fillArc11Tiny( gc, cx,cy, ast, naed );
            } else { 
              fillArc11Full( gc, cx,cy, ast, naed );           
            }
          } else {
            if( ast <= naed ) {
              fillArc11Full( gc, cx,cy, naed, ast );
            } else {
              fillArc11Tiny( gc, cx,cy, naed, ast );
            }
          }
          return;
        case 2:
          if( arc_length >=0 )
            fillArc12Left( gc, cx,cy,ast, naed );
          else
            fillArc12Right( gc, cx,cy,ast,naed );
          return;
        case 3: //qs==1 qe==3
          if( arc_length >=0 )
            fillArc13Left( gc, cx, cy, ast, naed );
          else
            fillArc13Right( gc, cx,cy,ast, naed );
          return;
        }
      case 2:
        switch( qe ) 
        {
        case 0: //qs==0, qe==2
          if( arc_length >= 0 )
            fillArc02Bottom( gc, cx,cy, naed, ast );
          else
            fillArc02Top( gc, cx,cy, naed, ast );
          return;
        case 1: //qs==2 qs==1
          if( arc_length >= 0 )
            fillArc12Right( gc, cx,cy,naed,ast );
          else
            fillArc12Left( gc, cx,cy,naed,ast );
          return;
        case 2: //qs==2, qe==2
          if( arc_length>=0 )
          {
            if( ast <= naed ) {
              fillArc22Tiny( gc, cx,cy, ast, naed );
            } else { 
              fillArc22Full( gc, cx,cy, ast, naed );           
            }
          } else {
            if( ast <= naed ) {
              fillArc22Full( gc, cx,cy, naed, ast );
            } else {
              fillArc22Tiny( gc, cx,cy, naed, ast );
            }
          }
          return;
        case 3: //qs==2 qe==3
          if( arc_length>=0 )
            fillArc23Bottom( gc, cx,cy,ast, naed );
          else
            fillArc23Top( gc, cx,cy,ast,naed );
          return;
        }
      case 3:
        switch( qe ) 
        {
        case 0: //qs==0 qe==3
          if( arc_length>=0 )
            fillArc03Right( gc, cx,cy,naed,ast );
          else
            fillArc03Left( gc, cx,cy,naed,ast );
          return;
        case 1: //qs==3 qe==1
          if( arc_length>=0 )
            fillArc13Right( gc, cx,cy,naed,ast );
          else
            fillArc13Left( gc, cx,cy,naed,ast );
          return;
        case 2: //qs==3 qe==2
          if( arc_length>=0 )
            fillArc23Top( gc, cx,cy,naed, ast );
          else
            fillArc23Bottom( gc, cx,cy,naed, ast );
          return;
        case 3://qs==3 qe==3
          if( arc_length>=0 )
          {
            if( ast <= naed ) {
              fillArc33Tiny( gc, cx,cy, ast, naed );
            } else { 
              fillArc33Full( gc, cx,cy, ast, naed );           
            }
          } else {
            if( ast <= naed ) {
              fillArc33Full( gc, cx,cy, naed, ast );
            } else {
              fillArc33Tiny( gc, cx,cy, naed, ast );
            }
          }
          return;
        }//switch
      }//switch qs    
    }
    //fill from yscan line, curr_index to top of oval
    private void fillArc_Bottom( IGCDirect gd, int cx,int cy, int yscan, int curr_index) 
    {
      //top arc
      int ytop=cy+ry_;//+model.py_[model.size_-1];
      while( yscan <= ytop )
      {
        gd.drawHLineDirect( cx-px_[curr_index], cx+px_[curr_index], yscan );
        yscan++;
        while( (curr_index<size_)&&(cy+py_[curr_index]<yscan) ) curr_index++;
      }
    }
    //fill from yscan lein, curr_index to bottom of oval
    private void fillArc_Top( IGCDirect gd, int cx, int cy, int yscan, int curr_index )
    {
      //bottom arc to the end
      int ybtm = cy-ry_;
      while( yscan >= ybtm )
      {
        gd.drawHLineDirect( cx-px_[curr_index], cx+px_[curr_index], yscan );
        yscan--;
        while( (curr_index<size_)&&(cy-py_[curr_index]>yscan) ) curr_index++;
      }
    }
    private int fillArc_BtmLeftArcToRightLine( IGCDirect gd, int cx,int cy, int yscan, int ystop, int ic, LineAlg lr )
    {
      while( yscan<=ystop )
      {
        int rx=lr.getX();
        while( (lr.getY()<=yscan)&&lr.nextPoint(null)) { if(lr.getY()>yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;
        }
        gd.drawHLineDirect( cx-px_[ic], rx, yscan );
        yscan++;
        while( (ic<size_)&&(cy+py_[ic])<yscan ) ic++;
      }
      return ic;
    }
    private int fillArc_TopLeftArcToRightLine( IGCDirect gd, int cx,int cy, int yscan, int ystop, int ic, LineAlg lr )
    {
      while( yscan >= ystop )
      {
        int rx=lr.getX();
        while( (lr.getY()>=yscan)&&(lr.nextPoint(null))) { if(lr.getY()<yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;
        }      
        gd.drawHLineDirect( cx-px_[ic], rx, yscan );
        yscan--;
        while( (ic<size_)&&(cy-py_[ic])>yscan ) ic++;
      } 
      return ic;
    }
    private void fillArc_BtmLeftLineToRightLine(  IGCDirect gd, int yscan, int ystop, LineAlg ll, LineAlg lr )
    {
      while( yscan <= ystop )
      {
        int lx=ll.getX();
        while( (ll.getY()<=yscan)&&ll.nextPoint(null)) { if(ll.getY()>yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;
        }
        int rx=lr.getX();
        while( (lr.getY()<=yscan)&&lr.nextPoint(null)) { if(lr.getY()>yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;
        }
        gd.drawHLineDirect( lx, rx, yscan );
        yscan++;
      }
    }
    private void fillArc_TopLeftLineToRightLine( IGCDirect gd, int yscan, int ystop, LineAlg ll, LineAlg lr )
    {
      while( yscan >= ystop )
      {
        int lx = ll.getX();
        while( (ll.getY()>=yscan)&&ll.nextPoint(null)) { if(ll.getY()<yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;     
        }
        int rx = lr.getX();
        while( (lr.getY()>=yscan)&&lr.nextPoint(null)) { if(lr.getY()<yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;        
        } 
        gd.drawHLineDirect( lx, rx, yscan );
        yscan--;
      }
    }  
    //scan line fill from left line to right arc into top hemioval.
    private int fillArc_BtmLeftLineToRightArc(  IGCDirect gd, int cx,int cy, int yscan, int ystop, LineAlg ll, int ic )
    {
      while( yscan <= ystop )
      {
        int lx=ll.getX();
        while( (ll.getY()<=yscan)&&ll.nextPoint(null)) { if(ll.getY()>yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;
        }
        gd.drawHLineDirect( lx, cx+px_[ic], yscan );
        yscan++;
        while((ic<size_)&&((cy+py_[ic]<yscan)))ic++;
      }
      return ic;
    }
    //scan line fill from left line to right arc into top hemioval.
    private int fillArc_TopLeftLineToRightArc(  IGCDirect gd, int cx,int cy, int yscan, int ystop, LineAlg ll, int ic )
    {
      while( yscan >= ystop )
      {
        int lx=ll.getX();
        while( (ll.getY()>=yscan)&&ll.nextPoint(null)) { if(ll.getY()<yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;        
        }
        gd.drawHLineDirect( lx, cx+px_[ic], yscan );
        yscan--;
        while((ic<size_)&&((cy-py_[ic]>yscan)))ic++;
      }
      
      return ic;
    }
    //fill part of top oval between left arc to a right line and left line to right arc, return arc index.
    private int fillArc_BtmLeftArcToRightLine_LeftLineToRightArc( IGCDirect gd, int cx,int cy, int yscan,int ystop, int ic, LineAlg lr, LineAlg ll )
    {
      while( yscan <= ystop )
      {
        int rx=lr.getX();
        while( (lr.getY()<=yscan)&&lr.nextPoint(null)) { if(lr.getY()>yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;
        }
        int lx=ll.getX();
        while( (ll.getY()<=yscan)&&ll.nextPoint(null)) { if(ll.getY()>yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;
        }
        if( rx==lx || rx==lx-1 ) {
          gd.drawHLineDirect( cx-px_[ic], cx+px_[ic], yscan );        
        } else {
          gd.drawHLineDirect( cx-px_[ic], rx, yscan );
          gd.drawHLineDirect( lx, cx+px_[ic], yscan );
        }
        yscan++;
        while((ic<size_)&&((cy+py_[ic]<yscan)))ic++;
      }
      return ic;
    }
    //fill part of bottom oval between left arc to a right line and left line to right arc, return arc index.
    private int fillArc_TopLeftArcToRightLine_LeftLineToRightArc( IGCDirect gd, int cx,int cy, int yscan,int ystop, int ic, LineAlg lr, LineAlg ll )
    {
      while( yscan >= ystop )
      {
        int rx=lr.getX();
        while( (lr.getY()>=yscan)&&lr.nextPoint(null)) { if(lr.getY()<yscan) break;
        int x=lr.getX(); if(x>rx) rx=x;
        }
        int lx=ll.getX();
        while( (ll.getY()>=yscan)&&ll.nextPoint(null)) { if(ll.getY()<yscan) break;
        int x=ll.getX(); if(x<lx) lx=x;
        }
        if( rx==lx || rx==lx-1 ) {
          gd.drawHLineDirect( cx-px_[ic], cx+px_[ic], yscan );        
        } else {
          gd.drawHLineDirect( cx-px_[ic], rx, yscan );
          gd.drawHLineDirect( lx, cx+px_[ic], yscan );
        }
        yscan--;
        while((ic<size_)&&((cy-py_[ic]>yscan)))ic++;
      }
      return ic;
    }
    
    //fill tiny arc id arc start and arc end are in quadrant 0
    private void fillArc00Tiny( IGCDirect gd, int cx, int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );    
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];   
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iaed], yaed );
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iast], yast );
      ll.nextPoint(null);
      lr.nextPoint(null);
      fillArc_TopLeftLineToRightLine( gd, cy, yast+1, ll, lr );
      fillArc_TopLeftLineToRightArc( gd, cx,cy, yast, yaed, ll, iast );
    }
    //fill the complementary arc if fillArc00Tiny.
    private void fillArc00Full( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      fillQuadrant23( gd, cx,cy );
      //hemi-oval with ar
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];
      int ystop1, ystop2;
      if( yast >= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      int yscan=cy-1;
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iast], yast);
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);    
      int ic=fillArc_TopLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy, yscan,ystop1, 1, lr,ll );
      ic=fillArc_TopLeftArcToRightLine( gd, cx,cy, ystop1-1, ystop2, ic, lr );
      fillArc_Top( gd, cx,cy, ystop2-1, ic);
    }
    private void fillArc01Top( IGCDirect gd, int cx, int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );    
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];
      int ystop1, ystop2;
      if( yast >= yaed ) {
        ystop1=yast; ystop2=yaed;
      } else {
        ystop1=yaed; ystop2=yast;
      }
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iaed], cy-py_[iaed] );
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iast], cy-py_[iast] );
      lr.nextPoint(null);
      ll.nextPoint(null);
      fillArc_TopLeftLineToRightLine( gd, cy, ystop1+1, ll, lr );
      int ic;
      if( ystop1 ==yast )
        ic = fillArc_TopLeftLineToRightArc( gd, cx,cy, ystop1, ystop2+1, ll, iast );
      else
        ic = fillArc_TopLeftArcToRightLine( gd, cx,cy, ystop1, ystop2+1, iaed, lr );
      fillArc_Top( gd, cx,cy, ystop2, ic );    
    }    
    private void fillArc01Bottom( IGCDirect gd, int cx, int cy, double nast, double naed )
    {
      fillQuadrant23( gd, cx, cy );
      //hemi-oval with both lines
      int iast = GetIndex( this, nast );    
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];
      int ystop1, ystop2;
      if( yast >= yaed ) {
        ystop1=yast; ystop2=yaed;
      } else {
        ystop1=yaed; ystop2=yast;
      }
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iast], cy-py_[iast] );
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iaed], cy-py_[iaed] );
      lr.nextPoint(null);
      ll.nextPoint(null);
      
      int ic = fillArc_TopLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy, cy-1,ystop1, 1, lr, ll );
      if( ystop1==yast )
        ic = fillArc_TopLeftArcToRightLine( gd, cx,cy, ystop1-1, ystop2, ic, lr );
      else
        ic = fillArc_TopLeftLineToRightArc( gd, cx,cy, ystop1-1, ystop2, ll, ic );
    }
    private void fillArc02Top( IGCDirect gd, int cx, int cy, double nast, double naed )
    {
      //top hemi-oval arc:
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iast], yast );
      lr.nextPoint(null);
      int ic=fillArc_TopLeftArcToRightLine( gd, cx,cy, cy,yast, 0, lr );
      fillArc_Top( gd, cx,cy, yast-1, ic );    
      //bottom hemi-oval arc:
      int iaed = GetIndex( this, naed );
      if( iaed > 0 ) //0 already done.
      {
        int yaed = cy+py_[iaed];
        lr.setLine( cx,cy, cx-px_[iaed], yaed );
        lr.nextPoint(null);
        fillArc_BtmLeftArcToRightLine( gd, cx,cy, cy+1, yaed, 1, lr );      
      }
    }
    private void fillArc02Bottom( IGCDirect gd, int cx, int cy, double nast, double naed )
    {
      //top part:
      int iast = GetIndex( this, nast );
      LineAlg ll=null;
      if( iast > 0 )
      {
        int yast = cy-py_[iast];
        ll = new LineAlg( cx,cy, cx+px_[iast], yast );
        ll.nextPoint(null);
        fillArc_TopLeftLineToRightArc( gd, cx,cy, cy,yast, ll, 0 );      
      }
      //bottom part:
      int iaed = GetIndex( this, naed );    
      int yaed = cy+py_[iaed];
      if( ll!=null )
        ll.setLine( cx,cy, cx-px_[iaed], yaed );
      else
        ll = new LineAlg( cx,cy, cx-px_[iaed], yaed );
      ll.nextPoint(null);
      int ic=fillArc_BtmLeftLineToRightArc( gd, cx,cy, cy+1, yaed, ll, 1 );    
      fillArc_Bottom( gd, cx,cy, yaed+1, ic );
    }  
    private void fillArc03Right( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      //top part:
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iast], yast );
      ll.nextPoint(null);
      fillArc_TopLeftLineToRightArc( gd, cx,cy, cy,yast, ll, 0 );
      //bottom part:
      int iaed = GetIndex( this, naed );
      int yaed = cy+py_[iaed];
      ll.setLine( cx,cy, cx+px_[iaed], yaed );
      ll.nextPoint(null);
      fillArc_BtmLeftLineToRightArc( gd, cx,cy, cy+1, yaed, ll, 1 );    
    }
    private void fillArc03Left( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );
      int ystop = cy-py_[iast];
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iast], ystop );
      lr.nextPoint(null);
      int yscan=cy;
      int ic=0;
      ic=fillArc_TopLeftArcToRightLine( gd, cx,cy, cy,ystop, 0,lr );
      fillArc_Top( gd,cx,cy, ystop-1, ic);
      //bottom part:
      int iaed = GetIndex( this, naed );
      yscan=cy+1;
      ic=1;
      ystop = cy+py_[iaed];
      lr.setLine( cx,cy, cx+px_[iaed], ystop );
      lr.nextPoint(null);
      ic=fillArc_BtmLeftArcToRightLine( gd, cx,cy, cy+1,ystop, ic,lr );
      fillArc_Bottom( gd, cx,cy, ystop+1, ic );
    }
    private void fillArc11Tiny( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];
      int yscan=cy;
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);
      fillArc_TopLeftLineToRightLine( gd, yscan, yaed+1, ll, lr );
      fillArc_TopLeftArcToRightLine( gd, cx,cy, yaed, yast, iaed, lr );    
    }
    private void fillArc11Full( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      fillQuadrant23( gd, cx,cy );
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy-py_[iast];
      int yaed = cy-py_[iaed];
      int ystop1, ystop2;
      if( yast >= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);    
      int ic=fillArc_TopLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy,  cy-1,ystop1, 1, lr,ll );
      ic=fillArc_TopLeftLineToRightArc( gd, cx,cy, ystop1-1,ystop2, ll, ic );
      fillArc_Top( gd, cx,cy, ystop2-1, ic);
    }
  
    private void fillArc12Left( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      //top part:
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg lr =new LineAlg( cx,cy, cx-px_[iast], yast );
      lr.nextPoint(null);
      fillArc_TopLeftArcToRightLine( gd, cx,cy, cy,yast, 0,lr );
      //bottom part:
      int iaed = GetIndex( this, naed );
      int yaed = cy+py_[iaed];
      lr.setLine( cx,cy, cx-px_[iaed], yaed );
      lr.nextPoint(null);
      fillArc_BtmLeftArcToRightLine( gd, cx,cy, cy+1,yaed, 1, lr );
    }  
    private void fillArc12Right( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      //top part:
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg ll =new LineAlg( cx,cy, cx-px_[iast], yast );
      ll.nextPoint(null);
      int ic=fillArc_TopLeftLineToRightArc( gd, cx,cy, cy,yast, ll,0 );
      fillArc_Top( gd, cx,cy, yast-1, ic );
      //bottom part:
      int iaed = GetIndex( this, naed );
      int yaed = cy+py_[iaed];
      ll.setLine( cx,cy, cx-px_[iaed], yaed );
      ll.nextPoint(null);
      ic = fillArc_BtmLeftLineToRightArc( gd, cx,cy, cy+1,yaed, ll, 1 );
      fillArc_Bottom( gd, cx,cy, yaed+1, ic );
    }
    private void fillArc13Left( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      //top part
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg lr =new LineAlg( cx,cy, cx-px_[iast], yast );
      lr.nextPoint(null);
      fillArc_TopLeftArcToRightLine( gd, cx,cy, cy, yast, 0,lr );
      //bottom part:
      int iaed = GetIndex( this, naed );
      int yaed = cy+py_[iaed];
      lr.setLine( cx,cy, cx+px_[iaed], yaed );
      lr.nextPoint(null);
      int ic=fillArc_BtmLeftArcToRightLine( gd,cx,cy, cy+1,yaed, 1, lr);
      fillArc_Bottom( gd, cx,cy, yaed+1, ic);
    }
    private void fillArc13Right( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      //top part
      int iast = GetIndex( this, nast );
      int yast = cy-py_[iast];
      LineAlg ll =new LineAlg( cx,cy, cx-px_[iast], yast );
      ll.nextPoint(null);
      int ic=fillArc_TopLeftLineToRightArc( gd, cx,cy, cy, yast, ll,0 );
      fillArc_Top( gd, cx,cy, yast-1,ic );
      //bottom part:
      int iaed = GetIndex( this, naed );
      int yaed = cy+py_[iaed];
      ll.setLine( cx,cy, cx+px_[iaed], yaed );
      ll.nextPoint(null);
      fillArc_BtmLeftLineToRightArc( gd,cx,cy, cy+1,yaed, ll,1);   
    }
    private void fillArc22Tiny( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];    
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);
      fillArc_BtmLeftLineToRightLine( gd, cy, yast-1, ll, lr );
      fillArc_BtmLeftArcToRightLine( gd, cx,cy, yast, yaed, iast, lr );    
    }
    private void fillArc22Full( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      fillQuadrant01( gd, cx,cy );
      //bottom part:
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];
      int ystop1, ystop2;
      if( yast <= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      int yscan=cy+1;
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);    
      int ic=fillArc_BtmLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy, yscan,ystop1, 1, lr,ll );
      ic=fillArc_BtmLeftLineToRightArc( gd, cx,cy, ystop1+1,ystop2, ll, ic );
      fillArc_Bottom( gd, cx,cy, ystop2+1, ic);
    }
    private void fillArc23Bottom( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];
      int ystop1, ystop2;
      if( yast <= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      LineAlg ll = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);
      fillArc_BtmLeftLineToRightLine( gd, cy,ystop1-1, ll, lr );
      int ic;
      if( ystop1==yast )
        ic=fillArc_BtmLeftArcToRightLine( gd, cx,cy, ystop1, ystop2-1, iast, lr );
      else
        ic=fillArc_BtmLeftLineToRightArc( gd, cx,cy, ystop1, ystop2-1, ll, iaed );
      fillArc_Bottom( gd, cx,cy, ystop2, ic );
    }
    private void fillArc23Top( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
      fillQuadrant01( gd, cx,cy );  
      //bottom part:
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];
      int ystop1, ystop2;
      if( yast <= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      LineAlg lr = new LineAlg( cx,cy, cx-px_[iast], yast);
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);
      int ic=fillArc_BtmLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy, cy+1,ystop1, 1, lr, ll );
      if( ystop1==yast ) {
        fillArc_BtmLeftLineToRightArc( gd, cx,cy, ystop1+1, ystop2, ll, ic );
      } else {
        fillArc_BtmLeftArcToRightLine( gd, cx,cy, ystop1+1, ystop2, ic, lr );
      }
    }
    private void fillArc33Tiny( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
System.out.println("fillArc33 Tinu");      
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iast], yast);
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);
      fillArc_BtmLeftLineToRightLine( gd, cy, yaed-1, ll, lr );
      fillArc_BtmLeftLineToRightArc( gd, cx,cy, yaed, yast, ll, iaed );    
    }
    private void fillArc33Full( IGCDirect gd, int cx,int cy, double nast, double naed )
    {
System.out.println("fillArc33 Full");      
      fillQuadrant01( gd, cx,cy );
      //bottom part:
      int iast = GetIndex( this, nast );
      int iaed = GetIndex( this, naed );
      int yast = cy+py_[iast];
      int yaed = cy+py_[iaed];
      int ystop1, ystop2;
      if( yast <= yaed ) { 
        ystop1 = yast; ystop2=yaed;
      } else {
        ystop1 = yaed; ystop2=yast;
      }
      int yscan=cy+1;
      LineAlg ll = new LineAlg( cx,cy, cx+px_[iast], yast);
      LineAlg lr = new LineAlg( cx,cy, cx+px_[iaed], yaed);
      lr.nextPoint(null);
      ll.nextPoint(null);    
      int ic=fillArc_BtmLeftArcToRightLine_LeftLineToRightArc( gd, cx,cy, yscan,ystop1, 1, lr,ll );
      ic=fillArc_BtmLeftArcToRightLine( gd, cx,cy, ystop1+1,ystop2, ic, lr );
      fillArc_Bottom( gd, cx,cy, ystop2+1, ic);
    }
  }  //END-OF-MODEL
  
  protected int cx_,cy_; // center of oval.
  protected double arc_start_, arc_length_; //angle of arc start, and arc length.
  protected Model model_;
  protected int istart_, iend_; //index of start/end in model's array.
  protected int qstart_, qend_; //quadrant start/end

  /** Create empty oval, must call setOval or setArc to define it. */
  public OvalAlg()
  {
    setOval(0,0,0,0);
  }
  public OvalAlg( int cx, int cy, int xradius, int yradius )
  {
    setOval( cx,cy,xradius,yradius);
  }
  public OvalAlg( int cx, int cy, int xradius, int yradius, double arc_start, double arc_length )
  {
    setArc( cx,cy,xradius,yradius, arc_start, arc_length );
  }
  public OvalAlg( OvalAlg c ) 
  {
    setOval( c );
  }
  
  public void setOval( int cx, int cy, int xradius, int yradius )
  {
    cx_=cx; cy_=cy;
    arc_start_ = 0;
    arc_length_ = Radian._2PI;
    model_ = GetModel( xradius, yradius );
    istart_ = 0; qstart_=0;
    iend_=0; qend_=0; //strange but not ?
    state_=1;
//System.out.println("oval.setoval() istart="+istart_+" ostart="+ostart_+" iend="+iend_+" oend="+oend_);    
  }
  public void setArc( int cx, int cy, int xradius,int yradius, double arc_start, double arc_length )
  {
    cx_=cx; cy_=cy;
    arc_start_ = Radian.normalize(arc_start);
    arc_length_ = arc_length;
    model_ = GetModel( xradius, yradius );  
    double as = Radian.normalize( arc_start_ );
    double ae = Radian.normalize( arc_start_+arc_length );
    qstart_ = GetQuadrant( as );
    qend_   = GetQuadrant( ae );
    istart_ = GetIndex( model_, as );
    iend_   = GetIndex( model_, ae );
//System.out.println("OvalAlg.setArc() model.size_ (quadrant)="+model_.size_);    
//System.out.println("arc_start="+arc_start_+"->as="+as+" qstart="+qstart_+" istart="+istart_);
//System.out.println("arc_length="+arc_length_+"->ae="+ae+" qend="+qend_+" iend="+iend_);
    state_=1;
//System.out.println("oval.setovalArc() istart_="+istart_+" ostart_"+ostart_+" iend="+iend_+" oend="+oend_+" as="+as+" ae="+ae);
    
  }
  public void setOval( OvalAlg c )
  {
    cx_=c.cx_; cy_=c.cy_;
    arc_start_ = c.arc_start_;
    arc_length_= c.arc_length_;
    model_ = c.model_;
    istart_=c.istart_;
    iend_=c.iend_;
    qstart_=c.qstart_;
    qend_=c.qend_;
    state_=1;
  }
  
  public int getCenterX() { return cx_; }
  public int getCenterY() { return cy_; }
  public int getRadiusX() { return model_.rx_; }
  public int getRadiusY() { return model_.ry_; }
  public double getArcStart() { return arc_start_; }
  public double getArcLength() { return arc_length_; }
  
  //expected normalized angle (0..2PI(
  private static int GetQuadrant( double angle )
  {
    //need to follow angle test in GetIndex.
    if( angle<Radian._PI2  ) return 0;
    if( angle<Radian._PI   ) return 1;
    if( angle<Radian._3PI2 ) return 2;
    return 3;
  }
  private static int NextQuadrant( int quadrant ) { return (quadrant+1)%4; }
  private static int PrevQuadrant( int quadrant ) {
    quadrant--;
    if( quadrant<0 ) quadrant=3;
    return quadrant;
  }
  private static int GetIndex( Model model, double angle )
  {
    int index;
    if( angle<Radian._PI2 )
    {
      index = (int)Math.round(((model.size_-1)*angle)/Radian._PI2);      
    } 
    else if( angle < Radian._PI )
    {
      index = (int)Math.round(((model.size_-1)*(Radian._PI-angle))/Radian._PI2);      
    }
    else if( angle < Radian._3PI2 )
    {
      index = (int)Math.round(((model.size_-1)*(angle-Radian._PI))/Radian._PI2);
    }
    else 
    {
      index = (int)Math.round(((model.size_-1)*(Radian._2PI-angle))/Radian._PI2);
    }    
    return index;
  }
  
  protected static ModelMap models_ = new ModelMap( 5,50,10);
  
  private static Object GetKey( int xradius, int yradius )
  {
    return Integer.toString(xradius)+":"+Integer.toString(yradius);
  }
  private static Model GetModel( int xradius, int yradius )
  {
    Object key = GetKey( xradius, yradius );
    Model model = (Model)models_.getModel( key );
    if( model==null )
    {
      model = new Model( xradius, yradius );
      models_.addModel( model );
    }
    return model;
  }

  protected int curr_index_, curr_quadrant_;
  protected Vector vector_ = new Vector();
  protected byte state_;

  /** restart nextPoint generation */
  public void restart()
  { 
    state_=1;
  }
  public void endPathElement(IGC gc) {}

  private int curr_ptx_, curr_pty_;
  private float curr_tgx_, curr_tgy_;
  
  /**@return X coordinate of point computed by nextPoint(), available if nextPoint() return true */
  public int getX() { return curr_ptx_; }
  /**@return X coordinate of point computed by nextPoint(), available if nextPoint() return true */
  public int getY() { return curr_pty_; }
  /**@return X coordinate of tangent vector computed by nextPoint(), available if nextPoint() return true */
  public float getTX() { return curr_tgx_; }
  /**@return X coordinate of tangent vector computed by nextPoint(), available if nextPoint() return true */
  public float getTY() { return curr_tgy_; }
  
private int cnt;
  public boolean nextPoint(IPoint point) 
  {    
    switch( state_ )
    {
      case 0: return false; //finished
      case 1: //not started
        if( (model_.rx_==0)||(model_.ry_==0)||(arc_length_==0) ) { state_=0; return false; } //nothing to generate
        curr_index_=istart_;
        curr_quadrant_=qstart_;
cnt=0;        
        state_=2; //generating
        break;
    }
    
//System.out.println("OvalAlg.nextPoint() curr_index="+curr_index_+" curr_quadrant="+curr_quadrant_);
//TODO:remove debug here    
cnt++;
if(cnt>5*model_.size_) {
  System.out.println("--- obviously there are a bad end condition, stop here");
  System.out.println(" istart="+istart_+" qstart="+qstart_+" iend="+iend_+" qend="+qend_+"  /size="+model_.size_);
  System.out.println(" arc_start_="+arc_start_+" arc.length="+(arc_start_+arc_length_));
  return false;
}
    model_.getPoint( curr_index_, curr_quadrant_ );
    curr_ptx_ = cx_+model_.ptx_;
    curr_pty_ = cy_-model_.pty_;
    curr_tgx_ = -model_.tgx_;
    curr_tgy_ = model_.tgy_;

    if( point!=null) point.setPoint( curr_ptx_, curr_pty_ );
    //next index: (depending on quadrant index orientation AND arc orientation )
    if( arc_length_ >= 0 )
    {
      if( (curr_quadrant_&0x1)==0 )
      {
        //incr. quadrant
        curr_index_++;
        if( curr_index_ >= model_.size_-1 )
        {
          curr_index_ = model_.size_-1;
          curr_quadrant_ = NextQuadrant( curr_quadrant_ );
        }
      } else {
        //decr quadrant
        curr_index_--;
        if( curr_index_==0 )
        {
          //curr_index_ = 0;
          curr_quadrant_ = NextQuadrant( curr_quadrant_ );
        }
      }
    } 
    else //arc_length < 0  
    {
      if( (curr_quadrant_&0x1)==0 )      
      {
        //decr. quadrant (as it's a negative arc length)
        curr_index_--;
        if( curr_index_ < 0 )
        {
          curr_index_ = 1;
          curr_quadrant_ = PrevQuadrant( curr_quadrant_ );
        }
      } else {
        curr_index_++;
        if( curr_index_ >= model_.size_ )
        {
          curr_index_ = model_.size_-2;
          curr_quadrant_ = PrevQuadrant( curr_quadrant_ );
        }
      }
    }
    //stop test for next time
    if( curr_quadrant_==qend_ && curr_index_==iend_ ) 
    {
      //no more points
      state_=0;
    }
    //but have a point for this time         
    return true;       
  }

  /** @return in vector current tangent, available if nextPoint() return true. */
  public void getCurrentTangent(IVector vector) 
  {
    vector.setVector( curr_tgx_, curr_tgy_  );
  }

  /** @return true if (x,y) point is inside given oval */
  public static boolean Contains( int x, int y, int cx, int cy, int xradius, int yradius )
  {
    //check for oval bounds:
    if( x < cx-xradius || y < cy-yradius || x > cx+xradius || y > cy+yradius ) return false;
    // test for point inside oval:
    int dx = x-cx, dy=y-cy;
    int pr2 = dx*dx + dy*dy;
    double a = Math.atan2(dy,dx);
    double r = xradius*Math.cos(a) + yradius*Math.sin(a);
    if( pr2 > r*r ) return false; //outside oval
    return true;
  }
  
  /** @return true if (x,y) point is inside given oval arc */
  public static boolean Contains( int x, int y, int cx, int cy, int xradius, int yradius, double arc_start, double arc_length )
  {
    //check for oval bounds:
    if( x < cx-xradius || y < cy-yradius || x > cx+xradius || y > cy+yradius ) return false;
    // test for point inside oval:
    int dx = x-cx, dy=y-cy;
    int pr2 = dx*dx + dy*dy;
    double angle = Math.atan2(dy,dx);
    double r = xradius*Math.cos(angle) + yradius*Math.sin(angle);
    if( pr2 > r*r ) return false; //outside oval
    //this is a full oval:
    if( arc_length >= Radian._2PI || -arc_length <= Radian._2PI )
    {
      return true;
    } 
    //test for point's angle, must be in start_ .. end_angle
    if( angle < 0 ) angle += Radian._2PI;
    if( arc_length == 0 ) 
    {
      return angle == arc_start; //?or false as arc length is 0 !
    }
    else if ( arc_length > 0 )
    {
      if (angle < arc_start ) angle += Radian._2PI;
      return angle <= arc_start+arc_length;
    } else {
      if( angle > arc_start ) angle -= Radian._2PI;
      return angle >= arc_start+arc_length;
    }
  }
  
  public boolean contains(int x, int y) 
  {
    return Contains( x,y, cx_,cy_,model_.rx_,model_.ry_, arc_start_, arc_length_ );
  }

  public boolean contains(IPoint p) {
    return Contains( p.getX(),p.getY(), cx_,cy_,model_.rx_,model_.ry_, arc_start_, arc_length_ ); 
  }
//TODO: Contains( IRect, .... ) at least for a oval is usefull too, or better contains(IShape)...
  
  public IRect getBounds() 
  {
    int rx = model_.rx_;
    int ry = model_.ry_;
    return new Rect( cx_-rx, cy_-ry, 2*rx, 2*ry );
  }

  /** fill current oval as it was really a full oval, see Filloval(). */
  public void fillOval( IGCDirect gc )
  {
    model_.fillOval( gc, cx_,cy_ );
  }
  /** fill current oval arc */
  public void fillArc( IGCDirect gc )
  {
    model_.fillArc( gc, cx_,cy_,arc_start_,arc_length_ );
  }
  
  /**
   * Fill oval (even if this is an arc definition), brush must be started by caller,
   * use only drawHLineDirect().
   */
  public static void FillArc( IGCDirect gd, int cx, int cy, int xradius, int yradius, double arc_start, double arc_length )
  {
    GetModel(xradius,yradius).fillArc( gd, cx,cy, arc_start, arc_length );
  }
  
  /**
   * Fill oval (even if this is an arc definition), brush must be started by caller,
   * use only drawHLineDirect().
   */
  public static void FillOval( IGCDirect gc, int cx, int cy, int xradius, int yradius )
  {
    GetModel(xradius,yradius).fillOval( gc, cx,cy );
  }

  
  
 
}

