/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FullGradientBrush.java,v 1.2 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.brushes.internal;

import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.util.internal.IGradientGenerator;
import org.eclipse.tptp.platform.report.igc.util.internal.Point;


/**
 * Linear, Circular, Conic (and more!) gradient defined by two points and a IGradientColor.
 * For simple horizontal/vertical gradient using two color take a look at GradientBrush.
 * 
 * @see IGradientColor, GradientColor, GradientBrush
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class FullGradientBrush implements IBrush
{
  /** Gradient is linear along the line defined by the two points */ 
  public static final int T_LINEAR=0;
  /** Gradient is circular, first point define the center, second one define the radius */
  public static final int T_CIRCULAR=1;
  /** Gradient is conic (infinite), first point define the center, second start angle, repeat property
   * haven't any effect on conic gradient (but raise fall yes).
   */
  public static final int T_CONIC=2;
  /** Same gradient as T_CONIC but only inside circle defined by both points.
   * repeat property can ve used. */
  public static final int T_CONIC2=3;
  /** The two points define focals of an ellipse. */
  public static final int T_ELLIPTIC=4;
  /** The two points define origin of two electromagnetic field. */
  public static final int T_ELECTROMAGNETIC=5;
  /** The two points define a rectangle in which the gradient is rectangular to,
   *  repeat is concentric */
  public static final int T_RECTANGULAR=6;
  /** Same as T_RECTANGULAR but repeat mode define rectangles side by side. */
  public static final int T_RECTANGULAR2=7;
  
  private int x1_,y1_, x2_,y2_;
  private boolean repeat_, raise_fall_;
  private int type_;  
  private IGradientGenerator gradient_;
  
  /** Create a gradient using type T_LINEAR */
  public FullGradientBrush( int _x1,int _y1, int _x2,int _y2, IGradientGenerator _gradient )
  {
    x1_=_x1; y1_=_y1;   
    x2_=_x2; y2_=_y2;   
    gradient_=_gradient;
    type_=T_LINEAR;
    updateCache();
  }
  
  /** Create a gradient */
  public FullGradientBrush( int _x1,int _y1, int _x2,int _y2, IGradientGenerator _gradient, int _type )
  {
    gradient_=_gradient;
    x1_=_x1; y1_=_y1;  
    x2_=_x2; y2_=_y2;  
    type_=_type;
    updateCache();
  }
  
  /** Create an horizontal/vertical gradient. if vertical _cX are y coordinates. */
  public FullGradientBrush( int _c1, int _c2, IGradientGenerator _gradient, boolean _vertical )
  {
    type_=T_LINEAR;
    gradient_=_gradient;
    if( _vertical )
    {
      x1_=x2_=0;
      y1_=_c1; y2_=_c2;
    } else {
      y1_=y2_=0;
      x1_=_c1; x2_=_c2;
    }
  }
  
  
  /** Create a gradient using given gradient and type, points must be defined before used.
   * @see setPoints() */
  public FullGradientBrush( IGradientGenerator _gradient, int _type )
  {
    gradient_=_gradient;
    type_=_type;
  }
  
  public FullGradientBrush( FullGradientBrush b ) 
  {
    //need to copy gradient too
    gradient_=b.gradient_;
    type_=b.type_;
    repeat_=b.repeat_;
    raise_fall_=b.raise_fall_;
    setPoints( b.x1_, b.y1_, b.x2_, b.y2_ );
  }
  
  public IBrush copyBrush() { return new FullGradientBrush(this); }
  
  /** Change both points defining the brush */
  public void setPoints( int _x1, int _y1, int _x2, int _y2 )
  {
    x1_=_x1; y1_=_y1;
    x2_=_x2; y2_=_y2;
    updateCache();
  }
  
  /** Change first points defined the brush */
  public void setPoint1( int _x1, int _y1 )
  {
    x1_=_x1; y1_=_y1;
    updateCache();
  }

  /** Change first points defined the brush */
  public void setPoint2( int _x2, int _y2 )
  {
    x2_=_x2; y2_=_y2;
    updateCache();
  }
  
  /** returns the X coordinate of the first points defined the brush */
  public int getPoint1X()
  {
    return x1_;
  }

  /** returns the X coordinate of the first points defined the brush */
  public int getPoint2X()
  {
    return x2_;
  }
  
  /** returns the Y coordinate of the second points defined the brush */
  public int getPoint1Y()
  {
    return y1_;
  }

  /** returns the Y coordinate of the second points defined the brush */
  public int getPoint2Y()
  {
    return y2_;
  }
  
  /** returns the coordinates of the first points defined the brush */
  public IPoint getPoint1()
  {
    return new Point(x1_, y1_);
  }

  /** returns the coordinates of the first points defined the brush */
  public IPoint getPoint2()
  {
    return new Point(x2_, y2_);
  }
  
  
  /** @return gradient color used by this brush */
  public IGradientGenerator getGradientColor() { return gradient_; }
  /** 
   * Change gradient color used by this brush.
   * @return previous used gradient.
   */
  public IGradientGenerator setGradientColor( IGradientGenerator _gradient )
  {
    IGradientGenerator g=gradient_;
    gradient_=_gradient;
    return g;
  }

  /**@return true if gradient is repeated (along line, or concentric circles */
  public boolean isRepeated() { return repeat_; }
  /** change repeat property */
  public void setRepeated( boolean b) { repeat_=b; }
  
  /** @return true if gradient is raise/fall, or false for only-raising gradient */
  public boolean isRaiseFall() { return raise_fall_; }
  /** Change raise-fall porperty */
  public void setRaiseFall( boolean b ) { raise_fall_=b; }

  /**@return current gradient mode */
  public int getType() { return type_; }
  /** change gradient mode */
  public void setType( int type )
  {
    type_=type;
    updateCache();
  }
  
  private int c_dx_, c_dy_, c_xr_,c_yr_,c_wr_,c_hr_;
  private double c_K_,c_A_; //meaning change following gradient type
  
  private void updateCache()
  {
    c_dx_ = x2_-x1_;
    c_dy_ = y2_-y1_;
    double c_radius2 = c_dx_*c_dx_ + c_dy_*c_dy_;    
    switch( type_ )
    {
      case T_LINEAR  : c_K_ = 1.0/(double)c_radius2; break;
      case T_CIRCULAR: c_K_ = 1.0/Math.sqrt(c_radius2); break;
      case T_CONIC   : c_A_ = Math.atan2( c_dy_,c_dx_);  break;
      case T_CONIC2  : c_A_ = Math.atan2( c_dy_,c_dx_);  
                       c_K_ = 1.0/Math.sqrt(c_radius2); break;
      case T_ELLIPTIC: c_K_ = 0.5/Math.sqrt(c_radius2); break;
      case T_ELECTROMAGNETIC: c_K_ =1.0/Math.sqrt(c_radius2); c_K_ *=c_K_; break;
      case T_RECTANGULAR:
      case T_RECTANGULAR2: 
        c_xr_ =Math.min(x1_,x2_);
        c_yr_ =Math.min(y1_,y2_);
        c_wr_ =Math.abs(c_dx_);
        c_hr_ =Math.abs(c_dy_);
        c_K_ = c_wr_==0.0 ? 0.0 :  2/(double)c_wr_;
        c_A_ = c_hr_==0.0 ? 0.0 :  2/(double)c_hr_;
        break;
    }
  }
int smp=0;  
  /* (non-Javadoc)
   * @see v2d.IBrush#getBrushColor(int, int, int, int)
   */
  public int getBrushColor(int x, int y, int curr_clr)
  {
    if( gradient_==null ) return 0;
    if( c_dx_==0 && c_dy_==0 )
    {
      return 0; //fully transparent
    }
    
    int X = x-x1_, Y=y-y1_;
    double k=0.0,k1=0.0;

    boolean is_rectangular2=false;
    switch( type_ )
    {
      default:
      case T_LINEAR  : k = (X*c_dx_+Y*c_dy_)*c_K_; break;///(double)(c_dx*c_dx+c_dy_*c_dy_); break;
      case T_CIRCULAR: k = Math.sqrt(X*X+Y*Y)*c_K_;break;//Math.sqrt( c_dx*c_dx+c_dy_*c_dy_); break;
      case T_CONIC2:
        k = k1=Math.sqrt(X*X+Y*Y)*c_K_;
        if( k > 1.0 ) 
        {
          if( true || repeat_ )
          {
            if( (((int)k1)&0x1)!=0 ) return 0;
          } 
          else return 0;//outside circle
        }
      case T_CONIC   : 
        {    
          double da = Math.atan2( Y,X) - c_A_;
          if(da<0) da+=2*Math.PI; else if (da>2*Math.PI) da-=2*Math.PI;
          k = da/(2*Math.PI);
          break;
        }        
      case T_ELLIPTIC:
        {
          int X2=x-x2_, Y2=y-y2_;
          double r = X*X2+Y*Y2;
          double d1 = Math.sqrt(X*X+Y*Y);
          double d2 = Math.sqrt(X2*X2+Y2*Y2);
          k = (d1+d2)*c_K_;
          break;
        }
      case T_ELECTROMAGNETIC:
        {
          int X2=x-x2_, Y2=y-y2_;
          double r = X*X2+Y*Y2;
          double d1 = Math.sqrt(X*X+Y*Y);
          double d2 = Math.sqrt(X2*X2+Y2*Y2);
          //d1 *= c_K_;
          //d2 *= c_K_;
          k=d1*d2*c_K_;
          break;
        }
      case T_RECTANGULAR2: is_rectangular2=true;
      case T_RECTANGULAR:
        {
          if(repeat_)
          {
            if( is_rectangular2 )
            {
              if( c_wr_>0 )
              {
                for( ; x>c_xr_+c_wr_; x-=c_wr_);
                for( ; x<c_xr_; x+=c_wr_);
              }
              if( c_hr_>0)
              {
                for( ; y>c_yr_-c_hr_; y-=c_hr_);
                for( ; y<c_yr_; y+=c_hr_);
              }
              X=x-x1_;
              Y=y-y1_;
            }
            //do nothing: repeat is concentric
          } else {
            if(x<c_xr_ || y<c_yr_ || x>c_xr_+c_wr_ || y>c_yr_+c_hr_ ) return 0;
          }
          int a = X*c_dy_;
          int s1 = (+a-Y*c_dx_) >=0 ? +1 : -1;
          int s2 = (-a-(y-y2_)*c_dx_) >=0 ? +1 : -1;
          if( s1 != s2  )
          {
            int distx = Math.abs( c_xr_+c_wr_/2-x );
            k = 1.0-distx *c_K_;
          } else {
            int disty = Math.abs( c_yr_+c_hr_/2-y );
            k = 1.0-disty *c_A_;
          }
          break;
        }
    }
  
    //repeat ?
    if( repeat_ )
    {
      //found k 0..1
      for( ; k>1.0; k-=1.0);
      for( ; k<0.0; k+=1.0);
    }
    else
    {
      //outside segment.
      if( k<0.0 || k >1.0 ) return 0; //transparent.
    }
    if( raise_fall_ )
    {
      if( k<=0.5 ) k *=2; else k = 2*(1.0-k);//2*(k-0.5); 
    }

    return gradient_.getRGBA( k );
  }
  
  protected int sx1_,sy1_,sx2_,sy2_;
  public void brushBegin( IGC gc, IGCDirect gd ) 
  {
    sx1_=x1_; sy1_=y1_; sx2_=x2_; sy2_=y2_;
    setPoints( gd.devX(x1_), gd.devY(y1_), gd.devX(x2_), gd.devY(y2_) );
  }
  
  public void brushEnd() 
  {
    setPoints( sx1_, sy1_, sx2_, sy2_ );
  }
   
}
