/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GradientBrush.java,v 1.5 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.brushes.internal;

import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;


/**
 * IBrush implementation for a vertical/horizontal Gradient brush using two colors.
 * Call setBounds() de define where the brush is applied.
 * For more complete gradient brush refer to FullGradientBrush.
 * 
 * @see FullGradientBrush
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class GradientBrush implements IBrush
{
  protected int rgba1_; //solid color or first color of gradient
  protected int rgba2_; //second color of gradient
  protected boolean vertical_;
  protected int x_,y_,w_,h_; //bounds
  
  /** Create a gradient brush */
  public GradientBrush( int _rgba1, int _rgba2, boolean _vertical )
  {
    rgba1_=_rgba1;
    rgba2_=_rgba2;
    vertical_=_vertical;
  }
  /** Create a copy of given brush */
  public GradientBrush( GradientBrush brush )
  {
    rgba1_ = brush.rgba1_;
    rgba2_ = brush.rgba2_;
    vertical_ = brush.vertical_;
    x_=brush.x_; y_=brush.y_;
    w_=brush.w_; h_=brush.h_;    
  }
  
  public IBrush copyBrush() { return new GradientBrush(this); }
  
  /** @return current color of solid brush (or first color of gradient) using RGBA encoding */
  public int getRGBA1() { return rgba1_; }
  /** @return current second color of gradient brush using RGBA encoding */
  public int getRGBA2() { return rgba2_; }
  /** @return current brush type */
  public boolean isVertical() { return vertical_; }
  
  /** Change current color of brush */
  public void setRGBA1( int _rgba )
  {
    rgba1_=_rgba;
  }
  /** Change current color of brush */
  public void setRGBA2( int _rgba )
  {
    rgba2_=_rgba;
  }
  /** Change current orientation of gradient brush */
  public void setVertical( boolean _v )
  {
    vertical_=_v;
  }
  
  /** Change gradient color but not orientation */
  public void setRGBA( int rgba1, int rgba2 )
  {
    rgba1_=rgba1;
    rgba2_=rgba2;
  }
  /** Change gradient parameters */
  public void setGradient( int rgba1, int rgba2, boolean vertical )
  {
    rgba1_=rgba1;
    rgba2_=rgba2;
    vertical_=vertical;
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IBrush#getBrushColor(int, int, int)
   */
  public int getBrushColor(int x, int y, int curr_rgba) 
  {
    //brush applies only in bounds
    //if( x<x_ || x>x_+w_ || y<y_ || y>y_+h_ ) return curr_rgba;
    double k ;
    if( vertical_ )
    {
      k = (y-dy_)/(double)(dh_);
    } else {
      k = (x-dx_)/(double)(dw_);
    }
    int grgba = RGBA.Gradient( rgba1_, rgba2_, k);
    int res = RGBA.Combine( grgba, curr_rgba );    
    return res;
  }

  protected int dx_,dy_,dw_,dh_; //bounds in device coordinates available between start/end
  public void brushBegin( IGC gc, IGCDirect gd ) 
  {
    if( gd.usePixelCoordinates()) {
      dx_=x_; dy_=y_; dw_=w_; dh_=h_;
    } else {
      dx_=gd.devX(x_);
      dy_=gd.devY(y_);
      dw_=gd.devX(w_);
      dh_=gd.devY(h_);
    }
//System.out.println("GradientBrush.startBrush() dev.bounds="+x_+","+y_+","+w_+"x"+h_);    
  }
  public void brushEnd() {
//System.out.println("GradientBrush.endBrush() restore.bounds="+x_+","+y_+","+w_+"x"+h_);    
  }
  
  /** Change bounds of brush (pixel coordinates). */
  public void setBounds(int x, int y, int w, int h) 
  {
    x_=x; y_=y; w_=w; h_=h;    
  }
  /** Change bounds of brush (pixel coordinates). */
  public void setBounds( IRect r ) 
  {
    x_=r.getX(); y_=r.getY(); w_=r.getW(); h_=r.getH();
  }
  
  /**@return current bounds of brush (pixel coordinates) */
  public Rect getBounds() { return new Rect(x_,y_,w_,h_); }

  public String toString()
  {
    String s = super.toString()+"{rgba1="+RGBA.Str(rgba1_)+",rgba2="+RGBA.Str(rgba2_)
            +",vertical="+vertical_+",x="+x_+",y="+y_+",w="+w_+",h="+h_+"}";
    return s;
  }
  /** 
   * Help method for IGC implementor.<br>
   * Fill rectangle using this brush and gc's methods.
   * This method can be used by any IGC without transparentcy support (NT: no transparency).
   * This method could be used more efficiency is gradient is opaque too.
   * x,y,w,h are in pixel coordinate, brush is started/ended in by this method.
   * @see fillRectWT
   */
  public void fillRectNT( IGC gc, IGCDirect gd, int _x, int _y, int _w, int _h )
  {
    IBrush old = gc.getBrush();    
    int x = gd.devX(_x), y=gd.devY(_y), w=gd.devX(_w), h=gd.devY(_h);
    this.brushBegin( gc, gd );
    SolidBrush b = new SolidBrush(RGBA.TRANSPARENT);
    if ( isVertical() )
    {
      int X=x+w, Y=y+h;   
      for( int i=y; i<=Y; i++ )
      {
        int rgba = getBrushColor( x, i, RGBA.WHITE ); //but isnot real curr_rgba value !
        b.setRGBA( rgba );
        gc.setBrush( b );
        gd.drawHLineDirect( x, X, i );
      }      
    } else {
      int X=x+w, Y=y+h;      
      for( int i=x; i<=X; i++ )
      {
        int rgba = getBrushColor( i, y, RGBA.WHITE ); //but isnot real curr_rgba value !
        b.setRGBA( rgba );
        gc.setBrush( b );
        gd.drawVLineDirect( i, y, Y );
      }
    }
    this.brushEnd();
    gc.setBrush( old );
  }
  
  /** 
   * Help method for IGC implementor.<br>
   * Fill rectangle using this brush and gc's methods.
   * This method can be used by any IGC with transparency support (WT: with transparency).
   * If gradient contains is opaque, call fillRectNT() instead.
   * x,y,w,h are in pixel coordinates, brush is started/ended in by this method.
   * @see fillRectNT
   */
  public void fillRectWT( IGC gc, IGCDirect gd, int _x, int _y, int _w, int _h )
  {
    boolean opaque = RGBA.IsOpaque(rgba1_) && RGBA.IsOpaque(rgba2_);
    //opaque is same code as:
    if( opaque ) {
      fillRectNT( gc, gd, _x, _y, _w, _h );
      return;
    }
    int x=gd.devX(_x), y=gd.devY(_y), w=gd.devX(_w), h=gd.devY(_h);
    this.brushBegin( gc, gd );    
    //as every pixel is visited, no consideration of vertical,horizontal, other...
    int X=x+w, Y=y+h;   
    for( int j=y; j<=Y; j++ )
    {
      for( int i=x; i<=X; i++ )
      {
        int curr = gd.getPointDirect( i,j );
        int rgba = getBrushColor( i,j, curr );      
        gd.drawPointDirect( i,j, rgba );
      }
    }
    this.brushEnd();
  }
 
}
