/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: OGC.java,v 1.4 2008/05/23 14:12:00 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.ogc.internal;

import java.util.Arrays;

import org.eclipse.tptp.platform.report.igc.alg.internal.FillPolygonAlg;
import org.eclipse.tptp.platform.report.igc.alg.internal.LineAlg;
import org.eclipse.tptp.platform.report.igc.alg.internal.OvalAlg;
import org.eclipse.tptp.platform.report.igc.internal.*;
import org.eclipse.tptp.platform.report.igc.util.internal.Ellipse;
import org.eclipse.tptp.platform.report.igc.util.internal.ImageProxy;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Oval;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBAImage;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.Segment;
import org.eclipse.tptp.platform.report.igc.util.internal.Size;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;


/**
 * TODO !!! UNDER TEST / DEVELOPMENT ...
 * implementation of IGC using an off screen buffer,
 * this IGC support transparency (this IGC implementor use none of swt.GC method).
 * User of this GC must call getBuffer() to get off screen buffer and draw it on their real GC.
 * Support clipping using IRect only.
 * OGC need an adapter to the underlaying graphic system.
 * 
 * @see IOGCAdapter.
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class OGC implements IGC
{
  /** current brush used, null if not already created. */
  protected IBrush brush_;
  /** current pen used, null if not already created. */
  protected IPen pen_;
  /** current font used, null if not created */
  protected IFont font_;
  /** used IGCDirect */
  protected IGCDirect gd_;
  
  protected int width_, height_, background_rgba_;
  /**offscreen buffer, use one dimension array as it's really better than a two dimension array.*/
  protected int buf_[]; //index is y*width_+x
  
    
  private IOGCAdapter adapter_;
  
  /** 
   * Create an Extended SWTGC, device is used for internal image creation, widthxheight 
   * off screen buffer size. 
   * @param background_rgba color used to initialize buffer, must not contains transparency. 
   */
  public OGC( IOGCAdapter adapter, int width, int height, int background_rgba ) 
  {
    adapter_ = adapter; 
    background_rgba_ = background_rgba;
    gd_ = new GCDirect();
    resize( width, height );    
  }
  
  /** @return true if current brush is null or a SolidBrush with invisible color */
  protected boolean haveNoBrush()
  {
    if( brush_==null ) return true;
    if( brush_ instanceof SolidBrush )
    {
      if( RGBA.IsInvisible( ((SolidBrush)brush_).getRGBA()) ) return true; 
    }
    return false;
  }
  /** @return true if current pen is null or a LineStylePen with invisible color */
  protected boolean haveNoPen()
  {
    if( pen_==null ) return true;
    if( pen_ instanceof LineStylePen )
    {
      return RGBA.IsInvisible( ((LineStylePen)pen_).getRGBA() );
    }
    return false;
  }
  
  public IGCDirect getIGCDirect() { return gd_; }
  
  /** resize off screen image used and fill it using background color. */
  public void resize( int width, int height )
  {
    if( width_!=width || height_!=height )
    {
      width_=width;
      height_=height;
      int nlen = width_*height_;
      if( buf_==null || nlen != buf_.length ) {
        buf_ = new int[nlen];
      }
    }
    restart();
  }
  
  /** fill current buffer with background color */
  public void restart()
  {
    Arrays.fill( buf_, background_rgba_ );
    //reset adapter
    adapter_.restart();
    font_=null;
  }
  
  public RGBAImage getBuffer()
  {
    return new RGBAImage( width_, height_, buf_ );
  }
  
  /** @return no null brush but do not copy it */
  private IBrush getBrushInternal() {
    if( brush_==null )
    {
      brush_ = new SolidBrush( background_rgba_ );
    }    
    return brush_;      
  }
  
  public IBrush getBrush() {
    //return a copy as caller might modify it.
    return getBrushInternal().copyBrush();      
  }

  /**
   * Change current brush, special support for SolidBrush and SWTFillGradientRectangleBrush,
   * others will be used/or not depending on shape to fill.
   * GradientBrush is used for fillRect() method only
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setBrush(org.eclipse.tptp.platform.report.igc.internal.IBrush)
   */
  public IBrush setBrush(IBrush _brush) 
  {
    if( brush_==null ) brush_=getBrush(); //force brush creation    
    IBrush old = brush_;
    brush_ = _brush; //but doesn't changed gc's color...    
    return old; //as brush isn't changed.
  }

   /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#pen()
   */
  public IPen getPen() {
    //return a copy, caller can do what he want without ennoying me.
    return getPenInternal().copyPen();
  }
  
  /** @return not null pen, but doesn't copy it . */
  private IPen getPenInternal() {
    if( pen_==null )
    {
      pen_ = new LineStylePen( RGBA.BLACK, LineStylePen.SOLID, 0 );
    }
    return pen_; 
  }

  /**
   * Change current pen, special supports for:<BR>
   *  -LineStylePen: use directly swt.GC. methods.<BR>
   * Others are taken into account depending on path to draw.<BR>
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setPen(org.eclipse.tptp.platform.report.igc.internal.IPen)
   */
  public IPen setPen(IPen _pen) {
    if( pen_==null ) pen_=getPenInternal(); //force pen creation    
    IPen old =pen_;
    pen_ = _pen;
    return old;
  }

  //internal use
  protected LineAlg line_alg_ = new LineAlg();
  
  
  
  

  

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawLine(int, int, int, int)
   */
  public void drawLine(int x1, int y1, int x2, int y2) 
  {
    if( pen_!= null )
    {
      //line clipping is done when pen startPath() on segment
      //and after segment was converted to device coodinates
      pen_.drawPath( this,gd_,getSegment( x1, y1, x2, y2 ) );
    }
    else Unsupported("drawLine(): without pen");
  }
  
  
  private Segment segment_; //see drawLine(), internal use only
  private Segment getSegment( int x1, int y1, int x2, int y2 ) { 
    if(segment_==null) 
    {
      segment_ = new Segment( x1,y1,x2,y2) {
        public boolean pathBegin(IGC gc, IGCDirect gd ) {
          boolean r = super.pathBegin( gc,gd );
          if( !r ) return false;
          //clip to image bounds:
          if( !this.clipTo( 0,0,width_-1,height_-1) ) return false;
          return true;
        }
      };
    } else {
      segment_.setLine(x1,y1,x2,y2);
    }
    return segment_;
  }

  /* *
   * @return RGBA.TRANSPARENT if point is outside drawing area, current RGBA color otherwise.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getPixel(int, int)
   */
  public int getPoint(int x, int y) 
  {
    int px=gd_.devX(x);
    int py=gd_.devY(y);
    if( px<0 || py <0 || px >=width_ || py>=height_ ) return RGBA.TRANSPARENT; //outside
    return buf_[py*width_+px];
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPixel(int, int)
   */
  public void drawPoint(int x, int y) 
  {    
    int px=gd_.devX(x);
    int py=gd_.devY(y);
    if( px<0 || py <0 || px >=width_ || py>=height_ ) return ; //outside
    int index = py*width_+px;
    buf_[index] = brush_.getBrushColor( px, py, buf_[index] );
  }

  

  protected Oval oval_ = new Oval(0,0,0,0);
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawCircle(int, int, int)
   */
  public void drawCircle(int cx, int cy, int radius) 
  {
    if( pen_ !=null )
    {     
      if( haveNoPen() ) return ;
      int dcx=gd_.devX(cx), dcy=gd_.devY(cy), drx=gd_.devX(radius), dry=gd_.devY(radius);
      if( dcx-drx >= width_ || dcy+dry>=height_ || dcx+drx < 0 || dcy+dry < 0 ) return ; //outside
      //pen will change oval to device coordinates ... 
      //this is silly since we compute it above ...
      oval_.setOval( cx, cy, radius, radius );
      pen_.drawPath( this, gd_, oval_ );
    } else {
      Unsupported("drawOval() without pen");
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillCircle(int, int, int)
   */
  public void fillCircle(int cx, int cy, int radius) 
  {
    if( brush_!=null )
    {
      if( haveNoPen() ) return ;
      int dcx=gd_.devX(cx), dcy=gd_.devY(cy), drx=gd_.devX(radius), dry=gd_.devY(radius);
      if( dcx-drx >= width_ || dcy+dry>=height_ || dcx+drx < 0 || dcy+dry < 0 ) return ; //outside
      oval_.setOval( dcx, dcy, drx, dry );
      fillShapeInternal( oval_ );
    }
    else Unsupported("fillCircle() without brush_");    
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawOval(int, int, int, int)
   */
  public void drawOval(int x, int y, int w, int h) 
  {
    if( pen_ !=null )
    {
      if( haveNoPen() ) return ;
      int dx = gd_.devX(x), dy = gd_.devY(y), dw=gd_.devX(w), dh=gd_.devY(h);
      if( dx>=width_ || dy >= height_ || dx+dw-1 < 0 || dy+dh-1 < 0 ) return ; //outside
      int rx = dw/2, ry = dh/2;
      oval_.setOval( dx+rx, dy+ry, rx, ry );
      pen_.drawPath( this, gd_, oval_ );
    } else { 
      Unsupported("drawOval() without pen");
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillOval(int, int, int, int)
   */
  public void fillOval(int x, int y, int w, int h) 
  {
    if( x >= width_ || x+h<0 || y+h<0 || y >= height_ ) return ;
    if( brush_!=null )
    {
      //clipping;
      int dx = gd_.devX(x);
      if( dx >= width_ ) return ;
      int dy = gd_.devY(y);
      if( dy >= height_) return ;
      int dX = dx+gd_.devX(w)-1;
      if( dX < 0 ) return ;
      int dY = dy+gd_.devY(h)-1;
      if( dY < 0 ) return ;
      int rx=w/2, ry=h/2;
       brush_.brushBegin( this, gd_ );
      OvalAlg.FillOval( gd_, x+rx, y+ry, rx, ry );
      brush_.brushEnd();            
    } else { 
      Unsupported("fillOval() without brush");
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawArc(int, int, double, int, int, double, double)
   */
  public void drawArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length)
  {
    if( pen_==null )
    {
      Unsupported("drawArc() without pen");
      return ;
    }
    if( haveNoPen() ) return ;
    double ar1 = Radian.normalize( _r1_angle );
    if( ar1 == 0.0 )
    {
      Oval oval = new Oval( xc, yc, r1, r2, _start_angle, _arc_length );
      pen_.drawPath( this, gd_, oval );
    } else { 
System.out.println("OGC DRAW ARC USE EllipseAlg ar1="+ar1);      
      Ellipse ellipse = new Ellipse( xc, yc, _r1_angle, r1, r2, _start_angle, _arc_length );
      pen_.drawPath( this, gd_, ellipse );
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillArc(int, int, double, int, int, double, double)
   */
  public void fillArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length) 
  {
    if( brush_ == null )
    {
      Unsupported("fillArc() without brush");
      return ;
    }
    if( haveNoBrush() ) return ;//nothing todo
    double ar1 = Radian.normalize( _r1_angle );
    if( ar1 == 0.0 )
    {
      Oval oval = new Oval( xc, yc, r1, r2, _start_angle, _arc_length );
      oval.fillShape( this, gd_, brush_, oval );
    } else { 
System.out.println("OGC FILL ARC USE EllipseAlg ar1="+ar1);      
      Ellipse ellipse = new Ellipse( xc, yc, _r1_angle, r1, r2, _start_angle, _arc_length );
      ellipse.fillShape( this, gd_, brush_, ellipse );
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawEllipse(int, int, double, int, int)
   */
  public void drawEllipse(int xc, int yc, double _r1_angle, int r1, int r2)
  {
    if( pen_==null )
    {
      Unsupported("drawEllipse() without pen");
      return ;
    }    
    pen_.drawPath( this, gd_, new Ellipse( xc, yc, _r1_angle, r1, r2 ));
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillEllipse(int, int, double, int, int)
   */
  public void fillEllipse(int xc, int yc, double _r1_angle, int r1, int r2) 
  {  
    if( brush_==null )
    {
      Unsupported("fillEllipse() without brush");
      return ;
    }
    Ellipse ellipse = new Ellipse(xc,yc,_r1_angle,r1,r2);
    ellipse.fillShape( this, gd_, brush_, ellipse );
  }

  

  /**
   * If current pen is null or a LineStylePen, use swt.GC.drawRectangle().
   * Otherwise use pen's methods.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawRect(int, int, int, int)
   */
  public void drawRect(int x, int y, int w, int h) 
  {
    if( pen_==null )
    {
      Unsupported("drawRect() without pen_");
      return ;
    }    
//System.out.println("OGC.drawRect("+x+","+y+","+w+","+h+") (pixel coords)");    
    drawPath( new Rect( x,y,w,h) );
  }
  
  public void fillRect( IRect r )
  {
    fillRect( r.getX(), r.getY(), r.getW(), r.getH() );
  }
  public void drawRect( IRect r )
  {
    if( r instanceof IPath )
    {
      drawPath( (IPath)r );
    } else {
      drawPath( new Rect(r) );
    }
  }

  public void fillRect( int x, int y, int w, int h )
  {
    if( gd_.usePixelCoordinates() )
    {
      gd_.fillRectDirect( x, y, w, h );
    } else {
      gd_.fillRectDirect( gd_.devX(x), gd_.devY(y), gd_.devX(w), gd_.devY(h) );
    }
  }
  /** Support for ImageProxy and SWTImage.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(int, int, org.eclipse.tptp.platform.report.igc.internal.IImage)
   */
  public void drawImage( IImage image, int _x, int _y ) 
  {
    int x= gd_.devX(_x), y=gd_.devY(_y);
    if( x >= width_ || y >= height_ ) return ; //outside;
    if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    }
    if( ! adapter_.drawImage( buf_,width_,height_, image, _x, _y ) )
    Unsupported("drawImage(IImage,int,int) with IImage = "+image);
  }
  public void drawImage( IImage _image, int _x, int _y, int _w, int _h )  
  {
    int x= gd_.devX(_x), y=gd_.devY(_y);
    if( x >= width_ || y >= height_ ) return ; //outside;
    if( _image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)_image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      _image = proxy.getLoaded();
    }
    if( !adapter_.drawImage( buf_,width_,height_, _image, _x,_y,_w,_h) )
    {
      Unsupported("drawImage(IImage,int,int,int,int) with IImage = "+_image);
    }
  }
  
  public void drawImage( IImage image, int srcX, int srcY, int srcW, int srcH, int dstX, int dstY, int dstW, int dstH )
  {
    int x= gd_.devX(dstX), y=gd_.devY(dstY);
    if( x >= width_ || y >= height_ ) return ; //outside;
    if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    }
    if( !adapter_.drawImage( buf_,width_,height_, image, srcX,srcY,srcW,srcH, dstX,dstY,dstW,dstH ) )
    {
      Unsupported("drawImage(IImage,...) with IImage = "+image);
    }
  }


  /**
   * If current pen is a LineStylePen use swt.gc.drawPolyline.
   * If polygon is a IPath:
   *  - If pen implements IGCHelperDrawPath, try to use this drawPath().
   *  - If polygon implements IGCHelperDrawPath, try to use this drawPath().
   *  
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void drawPoly(IPolygon poly) 
  {
    if( pen_==null )
    {
      Unsupported("drawPoly() without pen");
      return ;      
    }
    if( poly instanceof IPath )
    {   
      pen_.drawPath( this, gd_, (IPath)poly );
      return ;
    }
    //convert to a polygon
    Polygon p = new Polygon( poly );
    pen_.drawPath( this, gd_, p );    
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void fillPoly(IPolygon poly) 
  {
    if( brush_ == null )
    {
      Unsupported("fillPoly() without brush");
      return;
    }
    if( poly instanceof IShape )
    {
      if( brush_ instanceof IShapeFiller )
      {
        if( ((IShapeFiller)brush_).fillShape( this, gd_, brush_, (IShape)poly ) ) return ;
      }
      if( poly instanceof IShapeFiller )
      {
        if( ((IShapeFiller)poly).fillShape( this, gd_, brush_, (IShape)poly ) ) return ;
      }
    }
    //try using hand made FillPolygonAlg ... with risk ?
    if( gd_.usePixelCoordinates() )
    {
      if( fill_poly_alg_==null ) fill_poly_alg_ = new FillPolygonAlg();
      if( brush_!=null) brush_.brushBegin( this, gd_ );
      fill_poly_alg_.fillPoly( gd_, poly, 0,0,width_,height_ );
      if( brush_!=null) brush_.brushEnd();
    } else {
      //need to convert to device coordinate first:
      int size = poly.getPolySize();
      Polygon dev_poly = new Polygon( size );
      for( int i=0; i<size; ++i )
      {
        dev_poly.setPoint( i, gd_.devX(poly.getPolyX(i)), gd_.devY(poly.getPolyY(i)) );
      }
      if( fill_poly_alg_==null ) fill_poly_alg_ = new FillPolygonAlg();
      if( brush_!=null) brush_.brushBegin( this, gd_ );
      fill_poly_alg_.fillPoly( gd_, dev_poly, 0,0,width_,height_ );
      if( brush_!=null) brush_.brushEnd();
    }
    /*TODO: Polygon p = new Polygon(poly);
    p.fillShape( this, brush_, p ); */
    //Unsupported( "fillPoly() for poly="+poly);
  }
  protected FillPolygonAlg fill_poly_alg_;


  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String)
   */
  public ISize textExtent(String text) 
  {
    Size size = adapter_.textExtent( text );
    //as adapter return using right device coordinates, convert result to pixel coordinates 
    return new Size( gd_.pixX(size.getW()), gd_.pixY(size.getH()) );
  }

  public ISize textExtent( String text, double angle )
  {
    double a = Radian.normalize( angle );
    Size size = adapter_.textExtent( text );
    return RGBAImage.rotateImageSize( size.getW(), size.getH(), a );
  }
    
  public IPolygon textExtent(String text, int x, int y, double angle) 
  {
    double a = Radian.normalize( angle );
    Size size = adapter_.textExtent( text );
    return RGBAImage.rotateImageBounds( x, y, size.getW(), size.getH(), a );
  }
  
  /** devx,devy is top left corner of image (! be careful) */
  protected void drawTextImage( RGBAImage data, int devx, int devy, boolean transparent )
  {
    int white = RGBA.WHITE;
    int black = RGBA.BLACK;
    boolean no_transp = !( transparent || brush_==null);
    boolean use_brush = no_transp && brush_!=null;
    int brush_rgba = RGBA.BLACK; //for solid brush only
    boolean solid_brush  = brush_ instanceof SolidBrush;
    if( solid_brush ) brush_rgba = ((SolidBrush)brush_).getRGBA();
    int brush_rgb_ = (brush_rgba & 0xFFFFFF00);
    int brush_alpha = brush_rgba&0xFF; 
    
    brush_.brushBegin( this, gd_ );
    int Y=devy;
    int height = data.getHeight();
    int width  = data.getWidth();
    int image[] = data.getImage();
    int index0 = devy*width_+devx;
    int ir0 = 0; //as we start at ix=0 and iy=0, otherwise: iy*width+iy
    for( int iy=0; (iy<height)&&(Y<height_); iy++,Y++,index0+=width_, ir0+=width )
    {
      if( Y < 0 ) continue; //replace this with start value in iy, index0,...
      int X=devx;
      for( int ix=0, index=index0, ir=ir0; (ix<width)&&(X<width_); ix++, X++, index++, ir++ )
      {
        if( X < 0 ) continue;
        
        int rgba = image[ ir ];        
        
        if( rgba==black )
        { 
          //fully opaque
          if( solid_brush ) {
            buf_[index] = RGBA.Combine( brush_rgba, buf_[index] );
          } else {              
            buf_[index] = brush_.getBrushColor(X,Y, buf_[index]);
          }
        }
        else if( rgba != white ) //white means fully transparent 
        {
          //extract alpha value from blue channel (but same as red or green channel).
          //remember: text image is black on white background.
          int  alpha = 255-RGBA.GetB(rgba); //255:white:transparent, 0: black: opaque
          if( solid_brush ) {
            alpha = (alpha*brush_alpha)/255; //scale to brush alpha value.  
            buf_[index] = RGBA.Combine( brush_rgb_+alpha, buf_[index] );              
          } else {              
            int bck = buf_[index];
            int brgba = brush_.getBrushColor(X,Y, bck );
            alpha = (alpha*(brgba&0xFF))/255; //combine alpha from brush and image           
            buf_[index] = RGBA.Combine( (brgba&0xFFFFFF00)|alpha, bck );
          }
        }        
      }
    }
    brush_.brushEnd();
  }
  

  /**
   * Use current pen to draw font, force transparency if current brush is null.
   * current pen must be a LineStylePen.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawText(java.lang.String, int, int )
   */
  public void drawText(String text, int _x, int _y ) 
  {
    if( brush_==null ) {
      Unsupported("drawText() without brush");
      return ;
    }
    int x = gd_.devX(_x), y = gd_.devY(_y);
    if( x >= width_ || y >= height_ ) return ; //outside;
    ISize size = adapter_.textExtent( text );
    if( x+gd_.devX(size.getW()) < 0 || y+gd_.devY(size.getH()) < 0 ) return ;//outside;
    drawTextImage( adapter_.getTextImage( text, size ), x, y, true );    
  }

  /**
   * Support any angles.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawText(java.lang.String, int, int, double)
   */
  public void drawText(String text, int x, int y, double angle) 
  {
    double a = Radian.normalize( angle );
    int tx=gd_.devX(x), ty=gd_.devY(y);
    if( tx >= width_ || ty >= height_ ) return ;
    ISize size = adapter_.textExtent( text );
    RGBAImage s = adapter_.getTextImage( text, size );
    RGBAImage r = s.rotateImage( a, RGBA.WHITE, true );
    Polygon rte = RGBAImage.rotateImageBounds( tx, ty, size.getW(), size.getH(), a ); 
    IRect rb = rte.getBounds(); //but at (0,0)     
    drawTextImage( r, tx+rb.getX()-rte.getPolyX(0), 
                      ty+rb.getY()-rte.getPolyY(0), true );
  }

  /**
   * Draw a dashed rectangle with the rectangle coordinates with XOR mode
   */
  public void drawFocus(int x, int y, int w, int h)
  {
      
      IPen npen = new LineStylePen(LineStylePen.DOT);
      IBrush nbrush = new SolidBrush(IGC.C_LIST_SELECTION_TEXT);
      IPen oldpen = setPen(npen); 
      IBrush oldbrush = setBrush(nbrush);
      drawRect(x, y, w, h);

      setPen(oldpen);
      setBrush(oldbrush);
  }
  
  /**
   * Draw the focus with rectangle object. Call drawFocus(x,y,w,h)
   * @see #drawFocus(int, int, int, int)
   */
  public void drawFocus(IRect rect)
  {
      drawFocus(rect.getX(), rect.getY(), rect.getW(), rect.getH());
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getFont()
   */
  public IFont getFont() {
    if( font_==null )
    {      
      font_ = adapter_.getFont();
    }
    return font_.copy();
  }

  /** support all IFont.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setFont(org.eclipse.tptp.platform.report.igc.internal.IFont)
   */
  public IFont setFont(IFont font) 
  { 
    IFont old = getFont();
    adapter_.setFont( font );
    font_ = font;
    return old;   
  }
  
  public IFontMetrics getFontMetrics()
  {
    return adapter_.getFontMetrics();
  }
  
  public IShape getClipping()
  {
    //TODO: support for more shaped clipping ...
//TODO: at least support one-rect clipping.    
    return new Rect( 0, 0, gd_.pixX(width_), gd_.pixY(height_) );
  }
  
  public IShape setClipping( IShape s ) 
  {
    //TODO: support others clipping ..
   /* if( s instanceof IRect )
    {
      IShape old = getClipping();
      IRect r  = (IRect)s;
//TODO: support clipping ...      
      gc_.setClipping( devX(r.getX()), devY(r.getY()), devX(r.getW()), devY(r.getH()) );
      return old;
    }*/
    Unsupported("setClipping() TODO");
    return null;
  }

  public boolean fillShapeInternal( IShape shape )
  {
    if( shape instanceof IRect )
    {
      fillRect( (IRect)shape );
      return true;
    }
    if( shape instanceof IPolygon )
    {
      fillPoly( (IPolygon)shape );
      return true;
    }
    //note: Oval implements IShapeFiller.    
    if( brush_ instanceof IShapeFiller )
    {
      if (( (IShapeFiller)brush_).fillShape( this, gd_, brush_, shape ) ) return true;
    }
    if ( shape instanceof IShapeFiller )
    {      
      if( ((IShapeFiller)shape).fillShape( this, gd_, getBrushInternal(), shape ) ) return true;
    }
    return false;
  }
  
  /**
   * If shape is an IRect, IOval call fillRect/fillOval instead.
   * If current brush implements IShapeFiller, call brush_.fillShape().
   * If shape implements IShapeFiller, call shape.fillShape().
   * Otherwise fill nothing.
   */
  public void fillShape( IShape shape )
  {
    if( ! fillShapeInternal( shape ) )
      Unsupported("fillShape() for shape="+shape+" and brush="+brush_);
  }
  
  /**
   * If path is a IPolygon, use drawPoly().
   * If pen implements IGCHelperDrawPath try to use it.
   * If path implements IGCHelperDrawPath try to use it.
   * Otherwise draw nothing.
   */
  public void drawPath( IPath path )
  {
    if( path instanceof IPolygon )
    {
      drawPoly( (IPolygon)path );
    } else {
      if( pen_==null )
      {
        Unsupported("drawPath() without pen");
        return;
      }
      pen_.drawPath( this, gd_, path );
    }
  }

  
  protected void Unsupported( String what )
  {
    System.err.println("OGC: Unsupported method: "+what);
  }
  

  public int getSystemColor( int id )
  {
    return adapter_.getSystemColor(id);
  }
  
  private class GCDirect implements IGCDirect
  {
    float KdpiX_, KdpiY_;
    public GCDirect() 
    {
      KdpiX_ = adapter_.getKdpiX();
      KdpiY_ = adapter_.getKdpiY();      
    }
    public IGC getIGC() { 
      return OGC.this; 
    }
    
    public boolean usePixelCoordinates()
    {
      return KdpiX_==1.0f && KdpiY_==1.0f;
    }
    
    /** convert from pixel to current device */
    public int devX(int x )
    {
      return (int)(KdpiX_*x);
    }
    /** convert from pixel to current device */
    public int devY(int y )
    {
      return (int)(KdpiY_*y);
    }
    /** convert from current device to pixel */
    public int pixX(int x )
    {
      return (int)(x/KdpiX_);
    }
    /** convert from current device to pixel */
    public int pixY(int y )
    {
      return (int)(y/KdpiY_);
    }
    public void drawVLineDirect(int x, int y1, int y2) 
    {
      if( x < 0 || x>=width_ ) return ;//ouside;
      int yn=y1, yx=y2;
      if( yx<yn ) { int t=yn; yn=yx; yx=yn; }
      if( yx < 0 || yn >= width_  ) return ;//outside;
      int YN = Math.max( yn, 0 ); //clip to buffer
      int YX = Math.min( yx, height_-1 );
      if( brush_ instanceof SolidBrush )
      {
        int rgba = ((SolidBrush)brush_).getRGBA();
        int index=YN*width_+x;
        if( RGBA.IsOpaque(rgba) )
        {
          for( int y=YN; y<=YX; y++ )
          {
            buf_[index] = rgba;
            index += width_;
          }
        } else {
          for( int y=YN; y<=YX; y++ )
          {
            buf_[index] = RGBA.Combine( rgba, buf_[index] );
            index += width_;
          }
        }
      } 
      else if( brush_ !=null )
      {
        brush_.brushBegin( OGC.this, this );
        int index = YN*width_+x;
        for( int y=YN; x<=YX; x++ )
        {
          buf_[index] = brush_.getBrushColor( x,y, buf_[index]);
          index += width_;
        }
        brush_.brushEnd();
      }
      else Unsupported("drawHLineDirect() without brush.");
    }
    public void drawLineDirect(int x1, int y1, int x2, int y2) 
    {
      //TODO: line clipping...    
      if( brush_ instanceof SolidBrush )
      {
        int rgba = ((SolidBrush)brush_).getRGBA();
        line_alg_.setLine( x1, y1, x2, y2 );
        line_alg_.clipTo( 0,0,width_,height_);
        if( RGBA.IsOpaque(rgba) )
        {
          while( line_alg_.nextPoint(null))
          {
            int x = line_alg_.getX();
            //if( x < 0 || x >= width_ ) continue;
            int y = line_alg_.getY();
            //if( y < 0 || y >= height_ ) continue;
            buf_[ y*width_ + x ] = rgba;
          }
        } else {
          while( line_alg_.nextPoint(null))
          {
            int px = line_alg_.getX();
            int py = line_alg_.getY();
            int index = py*width_+px;
            buf_[index] = RGBA.Combine( rgba, buf_[index] ); 
          }
        }
      } 
      else if( brush_ !=null )
      {
        line_alg_.setLine( x1, y1, x2, y2 );
        brush_.brushBegin( OGC.this, this );
        while( line_alg_.nextPoint(null))
        {
          int px = line_alg_.getX();
          int py = line_alg_.getY();
          int index = py*width_+px;
          buf_[index] = brush_.getBrushColor(px,py, buf_[index]); 
        }
        brush_.brushEnd();
      }
      else Unsupported("drawHLineDirect() without brush.");
    }
    /* (non-Javadoc)
     * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawHLineDirect(int, int, int)
     */
    public void drawHLineDirect(int x1, int x2, int y) 
    {
      if( y < 0 || y>=height_ ) return ;//ouside;
      int xn=x1, xx=x2;
      if( xx<xn ) { int t=xn; xn=xx; xx=xn; }
      if( xx < 0 || xn >= width_  ) return ;//outside;
      int XN = Math.max( xn, 0 );
      int XX = Math.min( xx, width_-1 );
      if( brush_ instanceof SolidBrush )
      {
        int rgba = ((SolidBrush)brush_).getRGBA();
        int index=y*width_+XN;
        if( RGBA.IsOpaque(rgba) )
        {
          for( int x=XN; x<=XX; x++ )
          {
            buf_[index++] = rgba;
          }
        } else {
          for( int x=XN; x<=XX; x++,index++ )
          {
            buf_[index] = RGBA.Combine( rgba, buf_[index] ); 
          }
        }
      } 
      else if( brush_ !=null )
      {
        brush_.brushBegin( OGC.this, this );
        int index = y*width_+XN;
        for( int x=XN; x<=XX; x++,index++ )
        {
          buf_[index] = brush_.getBrushColor( x,y, buf_[index]); 
        }
        brush_.brushEnd();
      }
      else Unsupported("drawHLineDirect() without brush.");
    }
    public void drawPointDirect(int x, int y, int _rgba) 
    {
      if( x<0 || y<0 || x>=width_ || y>=height_ ) return ; //outside
      int index = y*width_+x;
      buf_[index] = RGBA.Combine( _rgba, buf_[index] );
    }
    
    public int getPointDirect( int x, int y )
    {
      if( x<0 || y<0 || x>=width_ || y>=height_ ) return RGBA.TRANSPARENT;
      return buf_[y*width_+x];
    }
    
    /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillRect(int, int, int, int)
   */
  public void fillRectDirect(int x, int y, int w, int h) 
  {
    int x1 = x, y1=y, x2=x1+w, y2=y1+h;          
    if( x2<x1) { int i=x1; x1=x2; x2=i; }
    if( y2<y1) { int i=y1; y1=y2; y2=i; }
    if( x2 < 0 || x1 >= width_ || y2 <0 || y1 >= height_ ) return ; //outside
    if( x1 < 0 ) x1=0; 
    if( x2 >= width_ ) x2 = width_-1;
    if( y1 < 0 ) y1=0;
    if( y2 >= height_ ) y2 = height_-1;
    if ( brush_ instanceof SolidBrush ) 
    {
      int bclr = ((SolidBrush)brush_).getRGBA();
      if( RGBA.IsOpaque(bclr) )
      {
        int index0 = y1*width_+x1;
        int index1 = index0+x2-x1+1;
        for( int Y=y1; Y<=y2; Y++ )
        {
          int index=index0;
          for( int X=x1; X<=x2; X++ )
          {
            buf_[index++] = bclr;
          }
          index0 +=width_;
          index1 +=width_;
        }        
      } else {
        int index0 = y1*width_+x1;
        for( int Y=y1; Y<=y2; Y++ )
        {
          int index=index0;
          for( int X=x1; X<=x2; X++,index++ )
          {
            buf_[index] = RGBA.Combine( bclr, buf_[index] );
          }
          index0 +=width_;
        }         
      }
    }
    else if ( brush_ !=null ) 
    {     
      brush_.brushBegin( OGC.this, this );
      int index0 = y1*width_+x1;
      for( int Y=y1; Y<=y2; Y++ )
      {
        int index=index0;
        for( int X=x1; X<=x2; X++,index++ )
        {
          int rgba = buf_[index];
          int bclr = brush_.getBrushColor( X,Y, rgba );
          buf_[index] = RGBA.Combine( bclr, rgba );
        }
        index0 +=width_;
      }        
      brush_.brushEnd();
    }
    else Unsupported("fillRect() without brush_");
  }
  public void drawRectDirect( int x, int y, int w, int h )
  {
    brush_.brushBegin( OGC.this, this );
    int x2=x+w, y2=y+h;
    drawHLineDirect( x, x2, y);
    drawHLineDirect( x, x2, y2);
    y++; y2--;
    drawVLineDirect( x, y, y2 );
    drawVLineDirect( x2, y, y2 );
    brush_.brushEnd();
  }
  }
  
  
}

