/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGGC.java,v 1.4 2008/05/23 14:12:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.svg.internal;



import java.io.IOException;
import java.io.OutputStream;

import org.eclipse.tptp.platform.report.igc.brushes.internal.FullGradientBrush;
import org.eclipse.tptp.platform.report.igc.brushes.internal.GradientBrush;
import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IFontMetrics;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IImage;
import org.eclipse.tptp.platform.report.igc.internal.IPath;
import org.eclipse.tptp.platform.report.igc.internal.IPen;
import org.eclipse.tptp.platform.report.igc.internal.IPolygon;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.internal.IShapeFiller;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.ImageProxy;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Oval;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBAImage;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Size;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;




/**
 * Test for SVG implementation on IGC....
 * 
 * Special feature added on SVG:
 * - fillRect(int,int,int,int) and fillRect(Rect) support GradientBrush.
 * 
 * <u>Note:</u> SVG use clockwise angle whereas IGC uses counter-clockwise angle !.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class SVGGC implements IGC 
{
  /**used when no font are set in gc but one of textExtent() is called */
  private static final int DEFAULT_FONT_SIZE=10;
  
  private OutputStream os_; // svg code generated into os
  private int width_, height_; // SVG canvas size
  protected IBrush brush_; //current brush;
  protected IPen   pen_; //current pen;
  protected IFont  font_; //current font
  protected int fill_ = 0xFF;
  protected int stroke_ = 0xFF;
  protected int strokedasharray_;
  protected int stroke_width_ = 1;
  protected int stroke_opacity_;
  protected int gradient_number_;
  protected GCDirect gd_;
  
  /**
   * Builds a new SVG GC 
   */
  public SVGGC( OutputStream output, int w, int h )
  {
    os_=output;
    width_=w;
    height_=h;
    gd_ = new GCDirect();
  }
  
  
  public IGCDirect getIGCDirect() { return gd_; }
  
  public int getSystemColor( int id )
  {
    return 0;
  }
  
  /**
   * starts a drawing session
   */
  public void begin() {
  	 svgWrite("<?xml version=\"1.0\" standalone=\"no\"?>"
              + "<!DOCTYPE svg PUBLIC \" -//W3C//DTD SVG 1.0//EN\" \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">"
              + "<svg width=\""+ width_ + "\" height=\"" + height_ + "\" >");
  }
  
  /**
   * terminates a drawing session
   */
  public void end() {
  	 svgWrite("</svg>");
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getBrush()
   */
  public IBrush getBrush() 
  {
    return brush_;
  }
  

  /** Support:
   *  -SolidBrush.
   *  -GradientBrush for fillRect 
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setBrush(org.eclipse.tptp.platform.report.igc.internal.IBrush)
   */
  public IBrush setBrush(IBrush _brush) 
  {
  	IBrush old = brush_;
  	brush_ = _brush;
    if( _brush instanceof SolidBrush )
    {   
      //report color as gc's background
      fill_ =   ((SolidBrush)brush_).getRGBA(); 
    }
    else if ( _brush instanceof GradientBrush ) 
    {
      GradientBrush b = (GradientBrush)brush_;
      fill_ =  b.getRGBA2();
      if (b.isVertical())
 	     svgWrite( svgGradient(b, b.getBounds().centerX(), b.getBounds().getY(),
 	     		        b.getBounds().centerX(), b.getBounds().bottom())) ;
 	  else
 	  	svgWrite( svgGradient(b, b.getBounds().getX(), b.getBounds().centerY(),
  		                b.getBounds().right(), b.getBounds().centerY()));
    } 
  	else 
  	if( brush_ instanceof FullGradientBrush )
  	{
  		FullGradientBrush b = (FullGradientBrush)brush_;
  		fill_ =  b.getBrushColor(b.getPoint1().getX(), b.getPoint1().getY(), 0);
  	 
  		svgWrite( svgGradient(b, b.getPoint1().getX(), b.getPoint1().getY(),
  	  	  	             b.getPoint2().getX(), b.getPoint2().getY()));
  	}
    
    return old;
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getPen()
   */
  public IPen getPen() 
  {
    return pen_;
  }
  
  public int getPoint(int x, int y) {
    //convert to
    return RGBA.TRANSPARENT;
  }

  /** Supports LineStylePen, but extract only color for it.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setPen(org.eclipse.tptp.platform.report.igc.internal.IPen)
   */
  public IPen setPen(IPen _pen) 
  {
  	IPen old = pen_;
  	pen_ = _pen;
    if( _pen instanceof LineStylePen )
    {    
      //report color as gc's background
      stroke_ = ((LineStylePen)_pen).getRGBA();
      strokedasharray_ =  ((LineStylePen)_pen).getLineStyle() ;
      stroke_width_ =  ((LineStylePen)_pen).getLineWidth() ;
    }    
    return old; 
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getFont()
   */
  public IFont getFont() 
  {
    return font_;
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setFont(org.eclipse.tptp.platform.report.igc.internal.IFont)
   */
  public IFont setFont(IFont font) 
  {
    font_=font;
    return font_.copy();
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawHLine(int, int, int)
   */
  public void drawHLine(int x1, int x2, int y)
  {
  	drawLine(x1, y, x2, y);
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawVLine(int, int, int)
   */
  public void drawVLine(int x, int y1, int y2) {
  	drawLine(x, y1, x, y2);
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawLine(int, int, int, int)
   */
  public void drawLine(int x1, int y1, int x2, int y2) 
  {
    StringBuffer b = new StringBuffer(128);
    b.append( "<line style=\"" );
    svgStroke( b );
    b.append( "\" " );
    svg2Coord(x1, y1, x2, y2, b);
    b.append(" />");
  	svgWrite( b.toString() );
  }


  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPixelDirect(int, int, int)
   */
  public void drawPoint(int x, int y) {
  	drawLine(x,y,x+1,y);
  }

  private void svgCircle( int cx, int cy, int radius, boolean stroke )
  {
    StringBuffer b = new StringBuffer( 128 );
	b.append( "<circle cx=\"" ); b.append( cx );
	b.append( "\" cy=\""); b.append(cy);
	b.append( "\" r=\""); b.append(radius);
	b.append( "\" style=\"");
	if( stroke )
	  svgStroke( b );
	else
	  svgFill( b );
	b.append( "\" />");
	svgWrite(  b.toString() );
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawCircle(int, int, int)
   */
  public void drawCircle(int cx, int cy, int radius) 
  {
    svgCircle( cx, cy, radius, true );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillCircle(int, int, int)
   */
  public void fillCircle(int cx, int cy, int radius) {
	svgCircle( cx, cy, radius, false );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawOval(int, int, int, int)
   */
  public void drawOval(int x, int y, int w, int h) {
    int rx = w/2, ry = h/2;
    if( rx==0 || ry == 0 ) return ;
  	drawEllipse(x+rx, y+ry, 0.0, rx, ry);
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillOval(int, int, int, int)
   */
  public void fillOval(int x, int y, int w, int h) {
    int rx = w/2, ry = h/2;
    if( rx==0 || ry == 0 ) return ;
  	fillEllipse(x+rx, y+ry, 0.0, rx, ry);
  }

  private String svgArcPoints( int r1, int r2, double _start_angle, double _arc_length )
  {
    StringBuffer sb = new StringBuffer(512);
    sb.append("0,0");
    
    double da = Radian._2PI/100;
    int npt = (int)Math.round( _arc_length / da );
    da = _arc_length / npt;
    double a=_start_angle;
    for( int i=0; i<=npt; i++ )
    {
      double c = Math.cos(a);
      double s = Math.sin(a);
      //double r = (r1*r2)/Math.sqrt( r2*c*c + r1*s*s );
      int x = (int)Math.round( r1*c );
      int y = -(int)Math.round( r2*s );
      
      sb.append(' ');
      sb.append( x );
      sb.append(',' );
      sb.append( y );
      
      a += da;
    }
    
    sb.append(" 0,0");
    return sb.toString();
  }
  
  private void svgArc( int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length, boolean stroke )
  {
    StringBuffer b = new StringBuffer(1024);
    b.append("<g transform=\"translate(" );
    b.append( xc ); b.append(',' ); b.append( yc ); b.append(")\" >");
    b.append("<g transform=\"rotate(");
    b.append( -Radian.iR2D( _r1_angle ) );
    b.append( ")\" >\n<polyline points=\"" );
    b.append( svgArcPoints(r1,r2,_start_angle,_arc_length) );
    b.append( "\" style=\"" );
    if( stroke )
      svgStroke( b );
    else
      svgFill( b );
    b.append( "\" /></g></g>" );
    
    svgWrite( b.toString() );
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawArc(int, int, double, int, int, double, double)
   */
  public void drawArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length) 
  {
    svgArc( xc,yc,_r1_angle,r1,r2, _start_angle, _arc_length, true );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillArc(int, int, double, int, int, double, double)
   */
  public void fillArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length) 
  {
    svgArc( xc,yc,_r1_angle,r1,r2, _start_angle, _arc_length, false );
  }

  private void svgEllipse( int xc, int yc, int rx, int ry, boolean stroke )
  {
    StringBuffer b = new StringBuffer(128);
    b.append("<ellipse ");
    svgEllipse( xc,yc,rx,ry, b );
    b.append( " style=\"" );
    if( stroke )
      svgStroke( b );
    else
      svgFill( b );
    b.append("\" />");
    
    svgWrite( b.toString() );
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawEllipse(int, int, double, int, int)
   */
  public void drawEllipse(int xc, int yc, double _r1_angle, int r1, int r2) {
  	
  	double a = Radian.normalize( _r1_angle );
  	int ia = Radian.iR2D( a );
  	
    if( ia==0 )
    {
      svgEllipse( xc,yc,r1,r2, true );
    }
    else
    if ( ia==90 || ia==270 )
    {
      svgEllipse( xc,yc,r2,r1, true );
    }
    else
    {
      StringBuffer b = new StringBuffer(512);
      b.append("<g transform=\"translate(" );
      b.append( xc );
      b.append(',');
      b.append( yc );
      b.append(")/><g transform=\"rotate(");
      b.append( -Radian.iR2D( _r1_angle ) );
      b.append(")\">\n<ellipse " );
      svgEllipse( 0,0,r1,r2, b );
      b.append(" style=\"");
      svgStroke( b );
      b.append( "\"/></g></g>");
      svgWrite( b.toString() );
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillEllipse(int, int, double, int, int)
   */
  public void fillEllipse(int xc, int yc, double _r1_angle, int r1, int r2) {
  	
  	double a = Radian.normalize( _r1_angle );
  	int ia = Radian.iR2D( a );
    if( ia==0 )
    {
      svgEllipse( xc,yc,r1,r2, false );
    }
    else
    if( ia==90 || ia==270 )
    {
      svgEllipse( xc,yc,r2,r1, false );
    }
  	else
    {
      StringBuffer b = new StringBuffer(512);
      b.append("<g transform=\"translate(" );
      b.append( xc );
      b.append(',');
      b.append( yc );
      b.append(")/><g transform=\"rotate(");
      b.append( -Radian.iR2D( _r1_angle ) );
      b.append(")\">\n<ellipse " );
      svgEllipse( 0,0,r1,r2, b );
      b.append(" style=\"");
      svgFill(b);
      b.append( "\"/></g></g>");
      svgWrite( b.toString() );
    }
  }

  private void svgRect( int x, int y, int w, int h, boolean stroke )
  {
    StringBuffer b = new StringBuffer(128);
    b.append("<rect ");
    svgRect(x, y, w, h, 0, 0, b);
    b.append(" style=\"");
    if( stroke )
      svgStroke( b );
    else
      svgFill( b );
    b.append("\" />");
    svgWrite( b.toString() );
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillRect(int, int, int, int)
   */
  public void fillRect(int x, int y, int w, int h) 
  {    
    svgRect( x, y, w, h, false );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawRect(int, int, int, int)
   */
  public void drawRect(int x, int y, int w, int h) {
    svgRect( x, y, w, h, true );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillRect(org.eclipse.tptp.platform.report.igc.internal.util.Rect)
   */
  public void fillRect(IRect r) {
    fillRect( r.getX(), r.getY(), r.getW(), r.getH() );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawRect(org.eclipse.tptp.platform.report.igc.internal.util.Rect)
   */
  public void drawRect(IRect r) {
    drawRect(r.getX(), r.getY(), r.getW(), r.getH());
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(int, int, org.eclipse.tptp.platform.report.igc.internal.IImage)
   */
  public void drawImage(IImage image, int x, int y ) {
  	drawImage(image, x, y, image.getWidth(), image.getHeight());
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(org.eclipse.tptp.platform.report.igc.internal.IImage, int, int, int, int, int, int, int, int)
   */
  public void drawImage(IImage image, int srcX, int srcY, int srcW, int srcH, int dstX, int dstY, int dstW, int dstH) {    
    drawImage(image, dstX, dstY, dstW, dstH);
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(org.eclipse.tptp.platform.report.igc.internal.IImage, int, int, int, int, int, int, int, int)
   */
  public void drawImage(IImage image, int srcX, int srcY, int srcW, int srcH) {    
  	if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    
      StringBuffer b = new StringBuffer(128);
      b.append( "<image " );
      svgRect(srcX, srcY, srcW, srcH,0,0, b);
      b.append(" xlink:href=\"" );
      b.append( proxy.getURL() );
      b.append( "\" />");
      svgWrite( b.toString() );
    }
    else 
       Unsupported("drawImage(IImage, int,int) where IImage isn't an instance of SWTImage");
  }

  private void svgPoly( IPolygon poly, boolean stroke )
  {
    int size=poly.getPolySize();
    int x[] = new int[size];
    int y[] = new int[size];
    for( int i=0; i<size; ++i )
    {
      x[i] = poly.getPolyX(i);
      y[i] = poly.getPolyY(i);
    }
    
    StringBuffer b = new StringBuffer(1024);
    if( stroke )
      b.append("<polyline ");
    else
      b.append("<polygon ");
    
    svgPoints(x, y, b);
    b.append(" style=\"");
    if( stroke )
      svgStroke(b);
    else
      svgFill(b);
    b.append("\" />");
    
    svgWrite( b.toString() );
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void drawPoly(IPolygon poly) 
  {
    svgPoly( poly, true );
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void fillPoly(IPolygon poly) 
  {
    svgPoly( poly, false );
  }

  /**
   * Draw text at given location, x,y is the upper left corner.
   * if not transparent, background of text is filled using current brush.
   */
  public void drawText( String text, int x, int y)
  {
  	drawText(text, x, y, 0.0);
  }

  
  /**
   * Draw rotated text at given location (uppper left corner). 
   * if not transparent, background of text is filled using current brush.
   */ 
  public void drawText( String text, int x, int y, double angle )
  {
  	double a = Radian.normalize(angle);
  	int ia = Radian.iR2D( a );
  	boolean no_rotation = ia==0 || ia==360;
  	
  	int dy=0;
  	if (font_ != null)
  	{
  		dy = font_.getFontSize();
  		y += font_.getFontSize();
  	}
  	
  	StringBuffer b = new StringBuffer(128);
  	if( no_rotation )
  	{
  		b.append( "<text>" );
  	}
  	else
  	{
  		b.append("<g transform=\"translate(" );
  		b.append( x ); b.append(','); b.append(y-dy); 
  		b.append( ")\" ><g transform=\"rotate( -" ); b.append( ia );
  		b.append( ")\" ><text>" );
  	}
  	
  	b.append( "<tspan ");
  	if( no_rotation )
  	{
  	  svgCoord(x,y,b) ;
  	  b.append(' ');
  	}
  	
  	svgFont( b );
  	b.append( "  style=\"stroke:none;" ); svgFill(b); b.append( "\" >" );
  	b.append( text );
  	b.append( "</tspan>" );
  	
  	if( no_rotation )
  	{
  		b.append( "</text>" );
  	}
  	else
  	{
  		b.append( "</text></g></g>" );
  	}
  	
  	svgWrite( b.toString() );
  }
  
  /**
   * Not implemented in SVG do nothing
   */
  public void drawFocus(int x, int y, int w, int h)
  {      
  }
  
  /**
   * Draw the focus with rectangle object. Call drawFocus(x,y,w,h)
   * @see #drawFocus(int, int, int, int)
   */
  public void drawFocus(IRect rect)
  {
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String)
   */
  public IFontMetrics getFontMetrics() 
  {
    final int fh ;
    if( font_!=null ) 
    {
      fh = font_.getFontSize();
    } else {
      fh = DEFAULT_FONT_SIZE;
    }
	return new IFontMetrics()
	{
	  public int getAscent() { return gd_.pixY(fh); }
      public int getDescent() { return gd_.pixY(0); } 
      public int getHeight() { return gd_.pixY(fh); }
      public int getLeading() { return gd_.pixY(0); }
	};
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String)
   */
  public IPolygon textExtent(String text, int x, int y, double angle) 
  {    
    double a = Radian.normalize( angle );
    ISize size = textExtent( text );
    return RGBAImage.rotateImageBounds( x, y, size.getW(), size.getH(), a );
  }
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String)
   */
  public ISize textExtent(String text) 
  {
	if (font_ != null)
  	{
	  return new Size(text.length()*font_.getFontSize(), font_.getFontSize());
  	}
	else
	  return new Size(text.length()*DEFAULT_FONT_SIZE, DEFAULT_FONT_SIZE);
  }

  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String, double)
   */
  public ISize textExtent(String text, double angle)
  {
    double a = Radian.normalize(angle);
    int ia = Radian.iR2D( a );
    switch( ia )
    {
    case 0:
    case 360:
      return textExtent( text );
    case 90:
    case 270:{
      Size p =(Size)textExtent( text );
      p.setSize( p.getH(), p.getW() );
      return p;
    }
    default: {
      ISize size = textExtent( text );
      IPolygon p= RGBAImage.rotateImageBounds( 0, 0, size.getW(), size.getH(), a );
      IRect r = p.getBounds();
      ((Size)size).setSize(r.getW(),r.getH());
      return size;
    }
    }
  }
  
  public IShape getClipping()
  {
    return null;//new Rect(0,0,0,0);// r.x, r.y, r.width, r.height );
  }
  
  public IShape setClipping( IShape s ) 
  {
    IShape old = getClipping();
    if( s == null )
    {
      //reset clipping to no-clipping
    }
    else if( s instanceof IRect )
    {
      IRect r = (IRect)s;
      //gc_.setClip( r.getX(), r.getY(), r.getW(), r.getH());
    } 
    //TODO: support more shapes..
    else Unsupported("setClipping() with shape="+s);
    return old;
  }
  
  /**
   * Supported shapes: IRect, Oval.
   */
  public void fillShape( IShape shape )
  {
    if( shape instanceof IRect )
    {
      fillRect( (IRect)shape );
    }
    else if ( shape instanceof Oval )
    {
      Oval o = (Oval)shape;
      int cx= o.getCenterX(), cy=o.getCenterY();
      int rx= o.getRadiusX(), ry=o.getRadiusY();
      fillOval( cx-rx, cy-ry, 2*rx, 2*ry );
    }
    else if ( shape instanceof IShapeFiller )
    {
      ((IShapeFiller)shape).fillShape(this, getIGCDirect(), getBrush(), shape);
    }
    else Unsupported("fillShape() for shape="+shape);
  }
  
  /**
   * Supported path: IPolygon.
   */
  public void drawPath( IPath path )
  {
  	if( path instanceof IPolygon )
    {
      drawPoly( (IPolygon)path );
    } else {
      pen_.drawPath( this, gd_, path );
    }
  }

  protected void Unsupported( String what )
  {
    System.err.println("SVGGC: Unsupported method: "+what);
  }
  
  /**
   * Generates a String into the stream format.
   */
  protected void svgWrite(String string) {
		try {
			os_.write(string.getBytes("UTF-8"));
			os_.write( '\n' );
		} catch (IOException e) {
			e.printStackTrace();
		} catch (NullPointerException e) {}
     
  }
  
  protected void svgColor( int rgba, StringBuffer b )
  {
    int rgb = rgba>>8;
    String hex = Long.toString( ((long)rgb)&0xFFFFFFL, 16 );
    int miss = 6-hex.length();
    b.append('#');
    for(int i=0;i<miss; i++) b.append('0');
    b.append( hex );
  }
  
  protected void svgFill( StringBuffer b )
  {
    b.append("stroke:none;");
  	if( brush_ instanceof GradientBrush )
    { 
  	  b.append( "fill:url(#Gradient_" );
  	  b.append( gradient_number_ );
  	  b.append( ')');
    }
  	else 
  	if( brush_ instanceof FullGradientBrush )
  	{
   	  b.append( "fill:url(#Gradient_" ); b.append( gradient_number_ ); b.append(')');
  	}
    else //fill using solid color (already set in gc_) 
    if( brush_ instanceof SolidBrush ) 
    {
      b.append( "fill:" ); svgColor( fill_, b );
    }
    else
    {
      b.append( "fill:none" );
    }
  	
  	if (!RGBA.IsOpaque(stroke_))
  	{
      	//s += ";fill-opacity:" + RGBA.GetA(fill_)/255.0;
	  b.append( ";opacity:" );
	  b.append( RGBA.GetA(fill_)/255.0 );
  	}
  }
  
  protected void svgStroke( StringBuffer b )
  { 	
  	int stroke = stroke_;
  	if( brush_ instanceof GradientBrush )
    {
  	  GradientBrush brush = (GradientBrush)brush_;
  	  stroke = brush.getRGBA1();
    }
  	
  	b.append( "fill:none;stroke:" ); svgColor( stroke, b );
    if (!RGBA.IsOpaque(stroke))
    {
      b.append( ";stroke-opacity:" );
      b.append( RGBA.GetA(stroke)/255.0 );
    }
    
    if (stroke_width_==0) {
      b.append( ";stroke-width:1" );
    } else {
      b.append( ";stroke-width:" ); b.append( stroke_width_ );
    }
    
    switch (strokedasharray_)
	{
      case LineStylePen.DASH:
      	   b.append( ";stroke-dasharray:10 10" );
           break;
      case LineStylePen.DOT:
      	   b.append( ";stroke-dasharray:2 2" );
           break;
      case LineStylePen.SOLID:
      	   break;
	}
  }
  
  protected void svgFont( StringBuffer b )
  {
  	if (font_ != null)
  	{
  		if (font_.getFontName()!=null)
  		{
  		  b.append( " font-family=\"" );
  		  b.append( font_.getFontName());
  		  b.append('\"');
  		}
  		b.append( " font-size=\"" ); 
  		b.append( font_.getFontSize() );
  		b.append( '\"' );
  		
  		if ((font_.getFontStyle() & IFont.BOLD)!=0)
  		{
  		  b.append( " font-weight=\"bold\"" );
  		}
  		if ((font_.getFontStyle() & IFont.ITALIC)!=0)
  		{
  		  b.append( " font-style=\"italic\"" );
  		}
  	}
  }
  
  protected String svgGradient(GradientBrush b, int x1, int y1, int x2, int y2)
  {
  	gradient_number_++;
  	
  	StringBuffer s = new StringBuffer(512);  	
  	s.append( "<linearGradient id=\"Gradient_" ); s.append( gradient_number_ ); s.append( "\" " );
	s.append( "gradientUnits=\"userSpaceOnUse\" "); svg2Coord(x1, y1, x2, y2, s); 
    s.append( '>' );
	s.append( "<stop offset=\"0\" style=\"stop-color:" ); svgColor(b.getRGBA1(), s); s.append("\" />");
  	s.append( "<stop offset=\"1\" style=\"stop-color:" ); svgColor(b.getRGBA2(), s); s.append("\" />");
	s.append( "</linearGradient>");
  	return s.toString();
  }
  
  protected String svgGradient(FullGradientBrush b, int x1, int y1, int x2, int y2)
  {
  	gradient_number_++;

  	StringBuffer s = new StringBuffer(512);  	
  	s.append( "<linearGradient id=\"Gradient_" ); s.append( gradient_number_ ); s.append( "\" " );
	s.append( "gradientUnits=\"userSpaceOnUse\" "); svg2Coord(x1, y1, x2, y2, s); 
    s.append( '>' );
	s.append( "<stop offset=\"0\" style=\"stop-color:" ); svgColor(b.getBrushColor(x1, y1,0), s); s.append("\" />");
  	s.append( "<stop offset=\"1\" style=\"stop-color:" ); svgColor(b.getBrushColor(x2, y2,0), s); s.append("\" />");
	s.append( "</linearGradient>");
  	return s.toString();
  }
  
  protected void svgCoord( int x, int y, StringBuffer b )
  {
    b.append( "x=\""); b.append(x);
    b.append("\" y=\""); b.append(y);
    b.append('\"');
  }
  
  protected void svg2Coord(int x1, int y1, int x2, int y2, StringBuffer b )
  {
    b.append( "x1=\""); b.append(x1);
    b.append( "\" y1=\""); b.append(y1);
    b.append( "\" x2=\""); b.append(x2);
    b.append( "\" y2=\""); b.append(y2);
    b.append( '\"');
  }
  
  protected void svgRect(int x, int y, int w, int h, int rx, int ry, StringBuffer b )
  {
    b.append("x=\""); b.append(x);
    b.append("\" y=\""); b.append(y);
    b.append("\" width=\""); b.append(w);
    b.append("\" height=\""); b.append(h);
    b.append('\"');
  	if (rx>0) {
  	  b.append(" rx=\"" ); b.append(rx); b.append('\"');
  	}
  	if (ry>0) {
  	  b.append(" ry=\"" ); b.append(ry); b.append('\"');
  	}
  }
  
  protected void svgEllipse(int cx, int cy, int rx, int ry, StringBuffer b )
  {
    b.append("cx=\""); b.append(cx);
    b.append("\" cy=\""); b.append(cy);
    b.append("\" rx=\""); b.append(rx);
    b.append("\" ry=\""); b.append(ry);
    b.append('\"');
  }
  
  protected void svgPoints(int [] x, int [] y, StringBuffer b )
  {
  	b.append( "points=\"" );
  	for (int i=0; (i<x.length && i<y.length); i++)
  	{
  	  b.append( x[i] );
  	  b.append( ',' );
  	  b.append( y[i] );
  	  b.append( ' ' );
  	}
  	b.append('\"');
  }
  
  public OutputStream getOutputStream() {
      return os_;
  }
  
  public void setOutputStream(OutputStream os_) {
      this.os_ = os_;
  }
  
  private class GCDirect implements IGCDirect
  {
    public IGC getIGC() { return SVGGC.this; }
    
    public boolean usePixelCoordinates()
    {
      return true;
    }
    
    /** convert from pixel to current device */
    public int devX(int x )
    {
      return x;
    }
    /** convert from pixel to current device */
    public int devY(int y )
    {
      return y;
    }
    /** convert from current device to pixel */
    public int pixX(int x )
    {
      return x;
    }
    /** convert from current device to pixel */
    public int pixY(int y )
    {
      return y;
    }
    public void drawRectDirect(int x, int y, int w, int h) 
    {
    	getIGC().drawRect(x,y, w, h);
    }
    
    public void drawPointDirect(int x, int y, int _rgba) 
    {
    	getIGC().drawPoint(x,y);
    }
    
    public int getPointDirect( int x, int y )
    {
      return RGBA.TRANSPARENT;
    }
    public void drawHLineDirect(int x1, int x2, int y) 
    {
    	getIGC().drawLine(x1,y, x2,y);
    }
    
    public void drawVLineDirect(int x, int y1, int y2) 
    {
    	getIGC().drawLine(x, y1, x, y2);
    }

    public void fillRectDirect( int x, int y, int w, int h )
    {
    	getIGC().fillRect(x,y,w,h);
    }

    public void drawLineDirect(int x1, int y1, int x2, int y2) 
    {
    	getIGC().drawLine(x1,y1,x2,y2);
    }
  }

}
