/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SWTGC.java,v 1.5 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.swt.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.report.igc.alg.internal.LineAlg;
import org.eclipse.tptp.platform.report.igc.internal.*;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.ImageProxy;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Oval;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBAImage;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.Segment;
import org.eclipse.tptp.platform.report.igc.util.internal.Size;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;


/**
 * implementation of IGC for SWT ... for TEST ONLY NEED TO BE IMPROVED.
 * Does not support transparency in any way...
 * Support clipping using IRect only.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class SWTGC implements IGC
{
  /** swt's device used */
  protected Device device_;
  /** swt's GC used to draw */
  protected GC gc_; 
  /**true is gc can be dispose on dispose() call */
  protected boolean dispose_gc_;  
  /** current brush used, null if not already created. */
  protected IBrush brush_;
  /** current pen used, null if not already created. */
  protected IPen pen_;
  /** current font used, null if not created */
  protected IFont font_;
  /** current IGCDirect used */
  protected IGCDirect gd_; 
  /** current used swt font (null for none) */
  protected org.eclipse.swt.graphics.Font swtfont_ ;
  
  /** true if current pen is null or a LineStylePen compliant with SWT line style definition.
   * meaning swt.GC methods can be used.
   */
  protected boolean line_style_pen_swt_compliant_;
  /** true if current brush is null or a SolidBrush compliant with SWT, so I can use swt.GC methods. */
  protected boolean brush_swt_compliant_;
  
  /** dpiX/Y factor used to convert from pixel to device */
  protected float KdpiX_, KdpiY_;
  
  /** Create an IGC with an existing GC, this one is not disposed on dispose() call */
  public SWTGC( Device device, GC gc )
  {
    device_ = device;
    gc_=gc;
    dispose_gc_=false;
    gd_ = new GCDirect();
    fixDPI();
    createPenFromGC();
    createBrushFromGC();
  }
  
  private void fixDPI()
  {
    if( device_ instanceof Display )
    {
      KdpiX_ = KdpiY_ = 1.0f;
    } else {
      Point dev_dpi = device_.getDPI();
      Point dsp_dpi = Display.getDefault().getDPI();     
      KdpiX_ = dev_dpi.x / (float)dsp_dpi.x;
      KdpiY_ = dev_dpi.y / (float)dsp_dpi.y;
    }
  }
  /**@return factor to convert pixel to current device coordinate */  
  //private float getKdpiX() { return KdpiX_; }
  /**@return factor to convert pixel to current device coordinate */  
  //private float getKdpiY() { return KdpiY_; }
  
  public IGCDirect getIGCDirect() { return gd_; }
  
  /** dispose any resource used */
  public void dispose()
  {
    if( gc_!=null && dispose_gc_ ) gc_.dispose();
    if( swtfont_!=null) { swtfont_.dispose(); swtfont_=null; }
    gc_=null;
  }
  
  public Color createColor( int _rgba )
  {
     return new Color( device_, RGBA.GetR( _rgba ), RGBA.GetG( _rgba ), RGBA.GetB( _rgba ) );
  }
  public int createRGBA( Color c )
  {
     return RGBA.Get( c.getRed(), c.getGreen(), c.getBlue() ); 
  }

  /** @return no null brush but do not copy it */
  private IBrush getBrushInternal() {
    if( brush_==null )
    {
      //get brush from GC
      Color c = gc_.getBackground();
      brush_ = new SolidBrush( RGBA.Get(c.getRed(),c.getGreen(),c.getBlue(), RGBA.OPAQUE));
    }    
    return brush_;      
  }
  
  public IBrush getBrush() {
    //return a copy as caller might modify it.
    return getBrushInternal().copyBrush();      
  }

  /**
   * Change current brush, special support for SolidBrush and SWTFillGradientRectangleBrush,
   * others will be used/or not depending on shape to fill.
   * GradientBrush is used for fillRect() method only
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setBrush(org.eclipse.tptp.platform.report.igc.internal.IBrush)
   */
  public IBrush setBrush(IBrush _brush) 
  {
    if( brush_==null ) brush_=getBrush(); //force brush creation
    if( _brush instanceof SolidBrush )
    {   
      IBrush old = brush_;
      SolidBrush brush = new SolidBrush( ((SolidBrush)_brush) );
      brush_= brush;
      //report color as gc's background
      gc_.setBackground( createColor( brush.getRGBA() ) );
      brush_swt_compliant_ = true;
      return old ; //no copy need here
    }
    else if ( _brush instanceof SWTFillGradientRectangleBrush ) 
    {
      IBrush old = brush_;
      SWTFillGradientRectangleBrush b = new SWTFillGradientRectangleBrush( ((SWTFillGradientRectangleBrush)_brush));
      brush_=b;
      gc_.setBackground( createColor( b.getRGBA2() ));
      brush_swt_compliant_=false;//as it's used only for fillRect...
      return old;
    } 
    brush_swt_compliant_=(brush_==null);
    IBrush old = brush_;
    brush_ = _brush; //but doesn't changed gc's color...    
    return old; //as brush isn't changed.
  }

   /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#pen()
   */
  public IPen getPen() {
    //return a copy, caller can do what he want without ennoying me.
    return getPenInternal().copyPen();
  }
  
  /** @return not null pen, but doesn't copy it . */
  private IPen getPenInternal() {
    if( pen_==null )
    {
      //get brush from GC
      Color c = gc_.getBackground();
      int swtsty = gc_.getLineStyle();
      int sty = LineStylePen.SOLID;
      switch( swtsty )
      {      
        default:
        case SWT.LINE_SOLID: sty = LineStylePen.SOLID; break;
        case SWT.LINE_DASH : sty = LineStylePen.DASH; break;
        case SWT.LINE_DOT  : sty = LineStylePen.DOT; break;
      } 
      pen_ = new LineStylePen( RGBA.Get(c.getRed(),c.getGreen(),c.getBlue(), RGBA.OPAQUE),
                               sty, gc_.getLineWidth() );
    }
    return pen_; 
  }

  /**
   * Change current pen, special supports for:<BR>
   *  -LineStylePen: use directly swt.GC. methods.<BR>
   * Others are taken into account depending on path to draw.<BR>
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setPen(org.eclipse.tptp.platform.report.igc.internal.IPen)
   */
  public IPen setPen(IPen _pen) {
    if( pen_==null ) pen_=getPen(); //force pen creation
    if( _pen instanceof LineStylePen )
    {    
      IPen old = pen_;
      LineStylePen pen = new LineStylePen( ((LineStylePen)_pen) );
      pen_ = pen;
      //report color as gc's background
      gc_.setForeground( createColor( pen.getRGBA() ) );
      int swtsty = SWT.LINE_SOLID;
      line_style_pen_swt_compliant_=true;
      switch( pen.getLineStyle() )
      {
      case LineStylePen.SOLID        : swtsty = SWT.LINE_SOLID; break;
      case LineStylePen.DASH         : swtsty = SWT.LINE_DASH; break;
      case LineStylePen.DOT          : swtsty = SWT.LINE_DOT; break;
      case LineStylePen.DASH_DOT     : swtsty = SWT.LINE_DASHDOT; break;
      case LineStylePen.DASH_DOT_DOT : swtsty = SWT.LINE_DASHDOTDOT; break;
      default: line_style_pen_swt_compliant_=false;
      }
      if( line_style_pen_swt_compliant_ )
      {
        gc_.setLineStyle( swtsty );
        gc_.setLineWidth( gd_.devY(pen.getLineWidth()) );
      }
      return old;
    }
    line_style_pen_swt_compliant_= (pen_==null);
    IPen old =pen_;
    pen_ = _pen;
    return old;
  }

  private void createPenFromGC()
  {
    LineStylePen pen = new LineStylePen();
    //report color as gc's background
    pen.setRGBA( createRGBA( gc_.getForeground() ) );
    pen.setLineWidth( gc_.getLineWidth() );
    line_style_pen_swt_compliant_=true;
    switch( gc_.getLineStyle() )
    {
    case SWT.LINE_SOLID     : pen.setLineStyle( LineStylePen.SOLID ); break;
    case SWT.LINE_DASH      : pen.setLineStyle( LineStylePen.DASH ); break;
    case SWT.LINE_DOT       : pen.setLineStyle(  LineStylePen.DOT ); break;
    case SWT.LINE_DASHDOT   : pen.setLineStyle(  LineStylePen.DASH_DOT ); break;
    case SWT.LINE_DASHDOTDOT: pen.setLineStyle(  LineStylePen.DASH_DOT_DOT ); break;
    }
    pen_ = pen;
  }
  private void createBrushFromGC()
  {
    brush_ = new SolidBrush( createRGBA( gc_.getBackground() ));
    brush_swt_compliant_ = true;
  }
    
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawLine(int, int, int, int)
   */
  public void drawLine(int x1, int y1, int x2, int y2) 
  {
    if( line_style_pen_swt_compliant_ )
    {
      gc_.drawLine( gd_.devX(x1), gd_.devY(y1), gd_.devX(x2), gd_.devY(y2) );
    } else if( pen_!=null ) {
      pen_.drawPath( this,gd_,getSegment( x1, y1, x2, y2 ) );
    }
  }

  
  
  
  private Segment segment_; //see drawLine(), internal use only
  private Segment getSegment( int x1, int y1, int x2, int y2 ) { 
    if(segment_==null) 
    {
      return new Segment( x1,y1,x2,y2); 
    }
    segment_.setLine(x1,y1,x2,y2);
    return segment_;
  }

  /**
   * @return RGBA.TRANSPARENT as this is not supported by swt's GC class.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getPoint(int, int)
   */
  public int getPoint(int x, int y) {
    //convert to
    return RGBA.TRANSPARENT;
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPixel(int, int)
   */
  public void drawPoint(int x, int y) 
  {
    gc_.drawPoint( gd_.devX(x), gd_.devY(y) );    
  }

  

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawCircle(int, int, int)
   */
  public void drawCircle(int cx, int cy, int radius) 
  {
    if( line_style_pen_swt_compliant_ )
    {
      int rx = gd_.devX(2*radius);
      int ry = gd_.devY(2*radius);
      gc_.drawOval( gd_.devX(cx-radius), gd_.devY(cy-radius), rx, ry );    
    } else {
//TODO      
      Unsupported("drawOval() using not swt compliant pen");
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillCircle(int, int, int)
   */
  public void fillCircle(int cx, int cy, int radius) 
  {
    int rx=gd_.devX(2*radius);
    int ry=gd_.devY(2*radius);
    gc_.fillOval( gd_.devX(cx-radius), gd_.devY(cy-radius), rx+1, ry+1 );
//TODO:add support for other brush...    
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawOval(int, int, int, int)
   */
  public void drawOval(int x, int y, int w, int h) 
  {
    if( line_style_pen_swt_compliant_ )
    {
      gc_.drawOval( gd_.devX(x), gd_.devY(y), gd_.devX(w), gd_.devY(h) );
    } else {
//TODO: using Oval class...  
      Unsupported("drawOval() using not swt compliant pen: "+pen_);
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillOval(int, int, int, int)
   */
  public void fillOval(int x, int y, int w, int h) 
  {
    if( brush_==null || brush_ instanceof SolidBrush )
    {
      gc_.fillOval( gd_.devX(x), gd_.devY(y), gd_.devX(w)+1, gd_.devY(h)+1 );
    } else {
      //user directly oval shape instead of trying brush first.
      int rx=w/2, ry=h/2;
      Oval oval = new Oval( x+rx, y+ry, rx, ry );
      /*
      if( brush_ instanceof IShapeFiller )
      {
        if( ((IShapeFiller)brush_).fillShape( this, brush_ oval ) ) return ;
      }*/
      oval.fillShape( this, gd_, getBrushInternal(), oval );      
    }
  }

  public void drawArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length) 
  {
    double a = Radian.normalize( _r1_angle );
    if( a==0.0 )
    {
      int deg_len = Radian.iR2D(_arc_length); //swt bark if arc length is 0 ! 
      if( deg_len != 0 ) {        
        gc_.drawArc( gd_.devX(xc-r1), gd_.devY(yc-r2), gd_.devX(2*r1), gd_.devY(2*r2), Radian.iR2D(_start_angle), deg_len );
      }
      return ;
    }
    //as isn't directly gc_.drawArc() one ...
    Unsupported("drawArc with _r1_angle!=0");
  }

  public void fillArc(int xc, int yc, double _r1_angle, int r1, int r2, double _start_angle, double _arc_length) 
  {
    double a = Radian.normalize( _r1_angle );
    if( a==0.0 )
    {
      if( brush_swt_compliant_ )
      {
        int deg_len = Radian.iR2D(_arc_length); //swt bark if arc length is 0 !
        if( deg_len != 0 ) {        
          // +1 come from swt's difference between drawArc and fillArc    
          gc_.fillArc( gd_.devX(xc-r1), gd_.devY(yc-r2), gd_.devX(2*r1)+1, gd_.devY(2*r2)+1, Radian.iR2D(_start_angle), deg_len );
        }
        return ;
      } 
      /* I don't know ... because SWT drawPoint it's really too slow for this ...
       else {
        Oval oval = new Oval(xc,yc, r1,r2, _start_angle, _arc_length );
        if( brush_ instanceof IShapeFiller )
        {
          if(((IShapeFiller)brush_).fillShape( this, gd_, brush_, oval )) return ;
        }
        if( oval.fillShape( this, gd_, getBrushInternal(), oval ) ) return ;
      }*/
      Unsupported("fillARc with brush "+brush_);
      return ;
    }
    //as isn't directly gc_.fillArc() one ...
    Unsupported("fillArc with _r1_angle!=0");
  }

  public void drawEllipse(int xc, int yc, double _r1_angle, int r1, int r2) 
  {
    double a = Radian.normalize(_r1_angle );
    if( a==0.0 )
    {
      gc_.drawOval( gd_.devX(xc-r1), gd_.devY(yc-r2), gd_.devX(2*r1), gd_.devY(2*r2) );
      return ;
    }
    //TODO: others ellipses.
    Unsupported("drawEllipse with _r1_angle !=0 ");
  }

  public void fillEllipse(int xc, int yc, double _r1_angle, int r1, int r2) 
  {
    double a = Radian.normalize( _r1_angle );
    if( a==0 )
    {
      gc_.fillOval( gd_.devX(xc-r1), gd_.devY(yc-r2), gd_.devX(2*r1)+1, gd_.devY(2*r2)+1 );
      return ;
    }
    //TODO: others ellipses
    Unsupported("fillEllipse() with _r1_angle !=0 ");
  }

  public void fillRect(int x, int y, int w, int h) 
  {
    if ( brush_ instanceof SolidBrush ) 
    {
      gc_.fillRectangle( gd_.devX(x), gd_.devY(y), gd_.devX(w), gd_.devY(h) );
      return;
    }
    if( brush_ instanceof SWTFillGradientRectangleBrush )
    {
      //special brush.
      Color fc = gc_.getForeground();
      SWTFillGradientRectangleBrush b = (SWTFillGradientRectangleBrush)brush_;
      gc_.setForeground( createColor( b.getRGBA1() ));
      //background already set.
      gc_.fillGradientRectangle( gd_.devX(x),gd_.devY(y),gd_.devX(w),gd_.devY(h), b.isVertical() );
      gc_.setForeground( fc );
      return ;
    }
    if ( brush_ instanceof IShapeFiller ) 
    {
      if( (((IShapeFiller)brush_).fillShape( this, gd_, brush_, new Rect(x,y,w,h) ))) return;
    }
    if( brush_ !=null )
    {
      //apply brush for each point of rect.
      Color old = gc_.getForeground();
      brush_.brushBegin( this, gd_ );
      int dx=gd_.devX(x), dy=gd_.devY(y), X=dx+gd_.devX(w), Y=dy+gd_.devY(h);
      for( int j=dy; j<=Y; j++ )
      {
        for( int i=dx; i<=X; i++)
        {
          int rgba = brush_.getBrushColor( i,j, RGBA.TRANSPARENT );
          if( rgba != RGBA.TRANSPARENT )
          {
            gc_.setForeground( createColor(rgba) );
            gc_.drawPoint( i,j );
          }
        }
      }
      brush_.brushEnd();
      gc_.setForeground(old);
    }    
  }

  /**
   * If current pen is null or a LineStylePen, use swt.GC.drawRectangle().
   * Otherwise use pen's methods.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawRect(int, int, int, int)
   */
  public void drawRect(int x, int y, int w, int h) 
  {
    if( pen_==null || pen_ instanceof LineStylePen )
    {
      gc_.drawRectangle( gd_.devX(x), gd_.devY(y), gd_.devX(w), gd_.devY(h) );
    }
    else 
    {
      pen_.drawPath( this, gd_,new Rect(  x, y, w, h ) );
    }
  }
  
  public void fillRect( IRect r )
  {
    fillRect( r.getX(), r.getY(), r.getW(), r.getH() );
  }
  public void drawRect( IRect r )
  {
    drawRect( r.getX(), r.getY(), r.getW(), r.getH() );
  }


  /** 
   * Support for ImageProxy, SWTImage and RGBAImage, others are ignored.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(int, int, org.eclipse.tptp.platform.report.igc.internal.IImage)
   */
  public void drawImage( IImage image, int x, int y) 
  {
    if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    }
    if( image instanceof SWTImage )
    {
      SWTImage img = (SWTImage)image;
      gc_.drawImage( img.getImage(), gd_.devX(x), gd_.devY(y) );
      return ;
    }
    if( image instanceof RGBAImage )
    {
      Image img = null;
      try {
        img = new Image( device_, SWTImage.ToImageData( (RGBAImage)image) );
        gc_.drawImage( img, gd_.devX(x), gd_.devY(y) );
      } finally {
        img.dispose();
      }
      return ;
    }
    Unsupported("drawImage(int,int,IImage) where IImage isn't an instance of SWTImage");
  }

  /** 
   * Support for ImageProxy, SWTImage and RGBAImage, others are ignored.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(int, int, org.eclipse.tptp.platform.report.igc.internal.IImage)
   */
  public void drawImage( IImage image, int x, int y, int w, int h )
  {
    if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    }
    if( image instanceof SWTImage )
    {
      SWTImage img = (SWTImage)image;
      Image swtimg = img.getImage();
      Rectangle r = swtimg.getBounds();
      gc_.drawImage( swtimg, 0,0,r.width,r.height,  gd_.devX(x),gd_.devY(y),gd_.devX(w),gd_.devY(h) );
      return ;
    }
    if( image instanceof RGBAImage )
    {
      RGBAImage rimg = (RGBAImage)image;
      Image img = null;
      try {
        img = new Image( device_, SWTImage.ToImageData( rimg ) );
        gc_.drawImage( img, 0,0,rimg.getWidth(),rimg.getHeight(), gd_.devX(x), gd_.devY(y),gd_.devX(w), gd_.devY(h) );
      } finally {
        img.dispose();
      }
      return ;
    }
    Unsupported("drawImage(IImage,...) where IImage isn't an instance of SWTImage");
  }
  
  /** 
   * Support for ImageProxy, SWTImage and RGBAImage others are ignored.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawImage(int, int, org.eclipse.tptp.platform.report.igc.internal.IImage)
   */
  public void drawImage( IImage image, int srcX, int srcY, int srcW, int srcH, int dstX, int dstY, int dstW, int dstH )
  {
    if( image instanceof ImageProxy )
    {
      ImageProxy proxy = (ImageProxy)image;
      if( !proxy.isLoaded() ) proxy.loadImage();
      image = proxy.getLoaded();
    }
    if( image instanceof SWTImage )
    {
      SWTImage img = (SWTImage)image;
      gc_.drawImage( img.getImage(), srcX,srcY,srcW,srcH,
                                     gd_.devX(dstX),gd_.devY(dstY),gd_.devX(dstW),gd_.devY(dstH) );
      return ;
    }
    if( image instanceof RGBAImage )
    {
      RGBAImage rimg = (RGBAImage)image;
      Image img = null;
      try {
        img = new Image( device_, SWTImage.ToImageData( rimg ));
        gc_.drawImage( img, srcX,srcY,srcW,srcH, gd_.devX(dstX),gd_.devY(dstY),gd_.devX(dstW),gd_.devY(dstH) );
      } finally {
        img.dispose();
      }
      return ;
    }
    Unsupported("drawImage(IImage,...) where IImage isn't an instance of SWTImage");
  }

  /**
   * If current pen is a LineStylePen use swt.gc.drawPolyline.
   * If polygon is a IPath:
   *  - If pen implements IGCHelperDrawPath, try to use this drawPath().
   *  - If polygon implements IGCHelperDrawPath, try to use this drawPath().
   *  
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void drawPoly(IPolygon poly) 
  {
    //use gc_.drawPolyline()
    if( pen_ ==null || pen_ instanceof LineStylePen )
    {
      Polygon p=null;
      if( poly instanceof Polygon )
      {
        p = (Polygon)poly;
      } else {
        p = new Polygon( poly );
      }
      if( gd_.usePixelCoordinates() )
      {
        gc_.drawPolyline( p.getPoints() );
      } else {
        int xy[] = p.getPoints();
        int s=xy.length;
        int devxy[] = new int[s];
        for( int i=0; i<s; )
        {
          devxy[i] = gd_.devX(xy[i]); i++;
          devxy[i] = gd_.devY(xy[i]); i++;
        }
        gc_.drawPolyline( devxy );
      }
      return ;
    }
    if( poly instanceof IPath )
    {
      pen_.drawPath( this, gd_, (IPath)poly );
      return ;
    }
/*TODO: deprecated    
    if( poly instanceof IPath )
    {
      if (pen_ instanceof IGCHelperDrawPath )
      {
        if( ((IGCHelperDrawPath)pen_).drawPath( this, pen_, (IPath)poly ) ) return ;
      }
      if ( poly instanceof IGCHelperDrawPath ) 
      {
        if(((IGCHelperDrawPath)poly).drawPath( this, pen_, (IPath)poly ) ) return ;
      }
    }*/
    //try it my self.
   /*TODO: {
      pen_.penStart( this );
//TODO: convert to device coordinate ?
      if( KdpiX_==1.0f & KdpiY_==1.0f )
      {
        //no conversion need
        
      } else {
        //convert polygon to device coordinate..
      }
      pen_.penEnd( this );
    }*/
    Unsupported("drawPoly() with pen="+pen_);
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#fillPoly(org.eclipse.tptp.platform.report.igc.internal.IPolygon)
   */
  public void fillPoly(IPolygon poly) 
  {
    if( brush_swt_compliant_ )
    {
      Polygon p = null;
      if( poly instanceof Polygon )
      {      
        p = (Polygon)poly;
      } else {
        p = new Polygon( poly );
      }
      if( gd_.usePixelCoordinates() )
      {
        //no scaling is needed
        gc_.fillPolygon( p.getPoints() );
      } else {
        int xy[] = p.getPoints();
        int s=xy.length;
        int devxy[] = new int[s];
        int i=0;
        for( i=0; i<s; )
        {
          devxy[i] = gd_.devX(xy[i]); i++;
          devxy[i] = gd_.devY(xy[i]); i++;
        }
        gc_.fillPolygon( devxy );
      }
      return ;
    }
    if( poly instanceof IShape )
    {
      if( brush_ instanceof IShapeFiller )
      {
        if( ((IShapeFiller)brush_).fillShape( this, gd_, brush_, (IShape)poly ) ) return ;
      }
      if( poly instanceof IShapeFiller )
      {
        if( ((IShapeFiller)poly).fillShape( this, gd_, brush_, (IShape)poly ) ) return ;
      }
    }
    Unsupported("fillPoly() with brush="+brush_);
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String)
   */
  public ISize textExtent(String text) 
  {
    org.eclipse.swt.graphics.Point te = gc_.textExtent( text );
    //as font is created using right device, convert result to pixel coordinate space
    return new Size( gd_.pixX(te.x), gd_.pixY(te.y) );
  }

  /**
   * Support only 0,90,180 and 270 degree angles
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#textExtent(java.lang.String, double)
   */
  public ISize textExtent(String text, double _angle) 
  {
    double a = Radian.normalize( _angle );
    if( a==0 || a==Radian._PI )
    {
      org.eclipse.swt.graphics.Point te = gc_.textExtent( text );    
      return new Size( gd_.pixX(te.x), gd_.pixY(te.y) );
    }
    else if ( a==Radian._PI2 || a==Radian._3PI2 )
    {
      org.eclipse.swt.graphics.Point te = gc_.textExtent( text );    
      return new Size( gd_.pixX(te.y), gd_.pixY(te.x) );
    }
    Unsupported("textExtent(String,int): angle other than 0,90,180,270? aren't supported");
    return null;
  }
  
  public IPolygon textExtent(String text, int x, int y, double angle) 
  {
    double a = Radian.normalize( angle );
    ISize size = textExtent( text );
    return RGBAImage.rotateImageBounds( x, y, size.getW(), size.getH(), a );
  }
  
  /**
   * Need a SolidBrush to works.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawText(java.lang.String, int, int )
   */
  public void drawText(String text, int x, int y ) 
  {
    if( brush_ instanceof SolidBrush )
    {
      Color c = gc_.getForeground();   
      gc_.setForeground( createColor( ((SolidBrush)brush_).getRGBA() ) );
      int dtx = gd_.devX(x);
      int dty = gd_.devY(y);
      gc_.drawText( text, dtx, dty, true );
      
      if( font_ !=null ) {
        SWTOGCAdapter.DrawFontLineStyle( gc_, dtx, dty, gc_.textExtent(text).x, font_.getFontStyle(), gd_.devX(5));
      }
      gc_.setForeground( c );
    }
    else Unsupported("drawText() without SolidBrush");
  }
  
  /**
   * Support only 0,90,180 and 270 degree angles and SolidBrush
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#drawText(java.lang.String, int, int, double)
   */
  public void drawText(String text, int x, int y, double angle) 
  {
    double a = Radian.normalize( angle );
    if( brush_swt_compliant_ && ( a==0 || a==Radian._PI2 || a==Radian._PI || a==Radian._3PI2) )
    {
      //gift! as swt simple implementation is to do only 0?
      //use "direct to gc" for simple rotation (save time)
      rotatedDrawText( text, x, y, Radian.iR2D(a) );
      return ;
    }
    else      
    {
      //other rotation (brushes) is more complicated (are result is not really beautifull
      //as GC haven't getPixel() method to support antialiased.
      int tx=gd_.devX(x), ty=gd_.devY(y);
      RGB bg = new RGB(0,0,0);
      ISize size = textExtent( text );
      RGBAImage s = SWTImage.ToRGBAImage( getTextImage( text, new Color(null,0,0,0), bg  ) );
      RGBAImage r = s.rotateImage( a, RGBA.WHITE, true );
      Polygon rte = RGBAImage.rotateImageBounds( tx, ty, size.getW(), size.getH(), a ); 
      IRect rb = rte.getBounds(); //but at (0,0)     
      drawTextImage( r, tx+rb.getX()-rte.getPolyX(0), 
                        ty+rb.getY()-rte.getPolyY(0), true );
    }
  }
  
  /** devx,devy is top left corner of image (! be careful) */
  protected void drawTextImage( RGBAImage data, int devx, int devy, boolean transparent )
  {
    int white = RGBA.WHITE;
    int black = RGBA.BLACK;
    //boolean no_transp = !( transparent || brush_==null);
    //boolean use_brush = no_transp && brush_!=null;
    int brush_rgba = RGBA.BLACK; //for solid brush only
    boolean solid_brush  = brush_ instanceof SolidBrush;
    if( solid_brush ) brush_rgba = ((SolidBrush)brush_).getRGBA();
    //int brush_rgb_ = (brush_rgba & 0xFFFFFF00);
    int brush_alpha = brush_rgba&0xFF;     
    Color c_brush_rgba = createColor( brush_rgba );
    //as SWTGC haven't any way to getPixel(), RGBA.Combine() can't be called:
    //this means no alpha is processed correctly, but to convert B/W text image
    //an alpha threshold is used to test if a gray pixel in text image have to be
    //drawn or not.  Value is choosen between text drawn as "bold" (->0) and disappearing
    //text (only black pixel drawn, if threshold is 255).
    final int alpha_threshold = 85;
    
    Color saved = gc_.getForeground();
    brush_.brushBegin( this, gd_ );
    int Y=devy;
    int height = data.getHeight();
    int width  = data.getWidth();
    int image[] = data.getImage();
    int ir0 = 0; //as we start at ix=0 and iy=0, otherwise: iy*width+iy
    for( int iy=0; (iy<height); iy++,Y++, ir0+=width )
    {
      if( Y < 0 ) continue; //replace this with start value in iy, index0,...
      int X=devx;
      for( int ix=0,  ir=ir0; (ix<width); ix++, X++, ir++ )
      {
        if( X < 0 ) continue;
        
        int rgba = image[ ir ];        
        
        if( rgba==black )
        { 
          //fully opaque
          if( solid_brush ) {
            gc_.setForeground( c_brush_rgba );
            gc_.drawPoint( X,Y );
          } else {           
            gc_.setForeground( createColor( brush_.getBrushColor(X,Y,RGBA.TRANSPARENT)));
            gc_.drawPoint( X,Y );
          }
        }
        else if( rgba != white ) //white means fully transparent 
        {
          //extract alpha value from blue channel (but same as red or green channel).
          //remember: text image is black on white background.
          int  alpha = 255-RGBA.GetB(rgba); //255:white:transparent, 0: black: opaque
          if( solid_brush ) {
            alpha = (alpha*brush_alpha)/255; //scale to brush alpha value.  
            //gc_.setForeground( createColor(RGBA.Combine( brush_rgb_+alpha, RGBA.TRANSPARENT )) );
            if(alpha>alpha_threshold) {
              gc_.setForeground( c_brush_rgba );
              gc_.drawPoint( X, Y );
            }
          } else {              
            int brgba = brush_.getBrushColor(X,Y, RGBA.TRANSPARENT );
            alpha = (alpha*(brgba&0xFF))/255; //combine alpha from brush and image           
            if(alpha>alpha_threshold) {
              gc_.setForeground( createColor( brgba | 0xFF ) );
              gc_.drawPoint( X, Y );
            }
          }
        }        
      }
    }
    brush_.brushEnd();
    gc_.setForeground( saved );
  }

  /**
   * Draw the focus with rectangle coordinates
   * Call directly the method drawFocus(x,y,w,h) of the SWT GC
   * @see org.eclipse.swt.GC#drawFocus
   */
  public void drawFocus(int x, int y, int w, int h)
  {
      gc_.drawFocus(gd_.devX(x), gd_.devY(y), gd_.devX(w), gd_.devY(h));
  }
  
  /**
   * Draw the focus with rectangle object. Call drawFocus(x,y,w,h)
   * @see #drawFocus(int, int, int, int)
   */
  public void drawFocus(IRect rect)
  {
      gc_.drawFocus(gd_.devX(rect.getX()), gd_.devX(rect.getY()), gd_.devX(rect.getW()), gd_.devX(rect.getH()));
  }
  
  
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#getFont()
   */
  public IFont getFont() {
    if( font_==null )
    {      
      org.eclipse.swt.graphics.Font sf = gc_.getFont();
      if( sf==null ) return null;
      FontData fd = sf.getFontData()[0];
      if( fd==null ) return null;
      int swtsty = fd.getStyle();
      int sty = IFont.NORMAL;
      if( (swtsty&SWT.BOLD)!=0 ) sty |= IFont.BOLD;
      if( (swtsty&SWT.ITALIC)!=0 ) sty |= IFont.ITALIC;
      font_ = new Font( fd.getLocale(), fd.getName(), fd.getHeight(), sty );
    }
    return font_;
  }

  /** support all IFont.
   * @see org.eclipse.tptp.platform.report.igc.internal.IGC#setFont(org.eclipse.tptp.platform.report.igc.internal.IFont)
   */
  public IFont setFont(IFont font) 
  {    
      IFont old = getFont();
      if( swtfont_!=null ) { swtfont_.dispose(); swtfont_=null; }
      if( font!=null )
      {
        int sty = font.getFontStyle();
        int swtsty= SWT.NORMAL;
        if( (sty & IFont.BOLD)!=0 ) swtsty |= SWT.BOLD;
        if( (sty & IFont.ITALIC)!=0 ) swtsty |= SWT.ITALIC;
        int size = font.getFontSize();
        swtfont_ = new org.eclipse.swt.graphics.Font( device_, font.getFontName(), size, swtsty );
      }
      font_=font;
      gc_.setFont( swtfont_ );
/*FontData fd[]=swtfont.getFontData();      
System.out.print("SWTGC: setFont() = ");
for( int i=0;i<fd.length; i++) System.out.print(" ["+i+"]{name="+fd[i].getName()+",h:"+fd[i].getHeight()+",s:0x"+Integer.toString(fd[i].getStyle(),16)+"}");
System.out.println("\n");*/      
      return old;
  }
  
  public IFontMetrics getFontMetrics()  
  {
    final FontMetrics fm = gc_.getFontMetrics();
    //take care: return value in pixel, so I need to convert them from device coordinates to pixels.
    return new IFontMetrics() {
      public int getAscent() { return gd_.pixY(fm.getAscent()); }
      public int getDescent() { return gd_.pixY(fm.getDescent()); } 
      public int getHeight() { return gd_.pixY(fm.getHeight()); }
      public int getLeading() { return gd_.pixY(fm.getLeading()); }      
    };
  }
  
  public IShape getClipping()
  {
    /*TODO:convert Region to IShape, but choose getClipping() or getClipping(region) ???
    Region region = new Region();
    gc_.getClipping( region );
    */
    org.eclipse.swt.graphics.Rectangle r = gc_.getClipping();
    if( r==null ) return null;
    return new Rect( gd_.pixX(r.x), gd_.pixY(r.y), gd_.pixX(r.width), gd_.pixY(r.height) );
  }
  
  public IShape setClipping( IShape s ) 
  {
    //TODO: support others clipping .. => convert them to region.
    if( s instanceof IRect )
    {
      IShape old = getClipping();
      IRect r  = (IRect)s;
      gc_.setClipping( gd_.devX(r.getX()), gd_.devY(r.getY()), gd_.devX(r.getW()), gd_.devY(r.getH()) );
      return old;
    }
    else if ( s == null ) 
    {
      //reset clipping:
      gc_.setClipping( (Rectangle)null );
    }
      
    return null;
  }

  /**
   * If shape is an IRect, IOval call fillRect/fillOval instead.
   * If current brush implements IShapeFiller, call brush_.fillShape().
   * If shape implements IShapeFiller, call shape.fillShape().
   * Otherwise fill nothing.
   */
  public void fillShape( IShape shape )
  {
    //use swt.GC method
    if( shape instanceof IRect )
    {
      fillRect( (IRect)shape );
      return ;
    }
    if ( shape instanceof Oval )
    {      
      Oval o = (Oval)shape;
      int cx= o.getCenterX(), cy=o.getCenterY();
      int rx= o.getRadiusX(), ry=o.getRadiusY();
      fillOval( cx-rx, cy-ry, 2*rx, 2*ry );
      return ;
    }
    if ( shape instanceof IPolygon)
    {
        fillPoly((IPolygon)shape);
        return;
    }
    if( brush_ instanceof IShapeFiller )
    {
      if (( (IShapeFiller)brush_).fillShape( this, gd_, brush_, shape ) ) return ;
    }
    if ( shape instanceof IShapeFiller )
    {
      if( ((IShapeFiller)shape).fillShape( this, gd_, getBrushInternal(), shape ) ) return ;
    }
    
    Unsupported("fillShape() for shape="+shape+" and brush="+brush_);
  }
  
  /**
   * If path is a IPolygon, use drawPoly().
   * If pen implements IGCHelperDrawPath try to use it.
   * If path implements IGCHelperDrawPath try to use it.
   * Otherwise draw nothing.
   */
  public void drawPath( IPath path )
  {
    if( path instanceof IPolygon )
    {
      drawPoly( (IPolygon)path );
    } else {
      pen_.drawPath( this, gd_, path );
    }
  }

  
  protected static void Unsupported( String what )
  {
    System.err.println("SWTGC: Unsupported method: "+what);
  }
  
  private ImageData getTextImage( String text, Color txt_clr, RGB bg )
  {
    org.eclipse.swt.graphics.Point sz = gc_.textExtent( text );
    ImageData data = null;
    Image img = null;
    GC igc = null;
    try {
      img = new Image( device_, sz.x, sz.y);//not better:gc_.getFontMetrics().getHeight()+20);//sz.y );
      igc = new GC( img );
      igc.setForeground( txt_clr );
      RGB rgb = txt_clr.getRGB();
      bg.red   = 255-rgb.red;
      bg.green = 255-rgb.green;
      bg.blue  = 255-rgb.blue;
      Color cbg = new Color( device_, bg );
      igc.setBackground( cbg );
      igc.fillRectangle( 0,0,sz.x,sz.y);
      //if device is not 24/32 bits, the filled color is not 255,255,255 but a "reduced" one.
      //this mean I must get "background" color from image:
      data = img.getImageData();
      rgb = data.palette.getRGB( data.getPixel(0,0));
      bg.red = rgb.red;
      bg.green= rgb.green;
      bg.blue = rgb.blue;

      //strange but the following doesn't works: I need to create a new font (a copy!)
      //igc.setFont( gc_.getFont() );
      FontData fontdata = gc_.getFont().getFontData()[0]; //might produce NPE ?
      org.eclipse.swt.graphics.Font font = new org.eclipse.swt.graphics.Font( device_, fontdata ); 
      igc.setFont( font );
      igc.drawText( text, 0,0, true );
      if( font_ !=null ) {
        SWTOGCAdapter.DrawFontLineStyle( igc, 0, 0, sz.x, font_.getFontStyle(), gd_.devX(5) );
      }
      font.dispose();
      igc.dispose();
      data = img.getImageData();
    } finally {
      if(img!=null) img.dispose();
      if(igc!=null) igc.dispose();
    }
    return data;
  }
  
  /** tx,ty is top left corner (in pixels) where text is printed, angle in degree.
   * This method use current gc's background as font color. */
  protected void rotatedDrawText( String text, int tx, int ty, int angle )
  {
     Color txt_clr = gc_.getBackground(); //as text color comes from brush
     tx = gd_.devX( tx );
     ty = gd_.devY( ty );
     switch( angle%360 )
     {
      case 270: 
      {
          RGB bg = new RGB(0,0,0);
          ImageData s = getTextImage( text, txt_clr, bg );
          int dx= tx;//s.height-1;
          for( int sy=0; sy<s.height; sy++,dx-- )
          {
            int dy=ty;
            for( int sx=0; sx<s.width; sx++,dy++ )
            {
              int pixel = s.getPixel( sx,sy );
              RGB rgb = s.palette.getRGB( pixel );
              if(rgb.red!=bg.red || rgb.green!=bg.green || rgb.blue!=bg.blue )
              {
                gc_.setForeground( new Color(device_,rgb));
                gc_.drawPoint( dx, dy );
              }
            }
          }  
        break;
      }
      case 90:
      {        
          RGB bg = new RGB(0,0,0);
          ImageData s = getTextImage( text, txt_clr, bg );
          int dx=tx;
          for( int sy=0; sy<s.height; sy++,dx++ )
          {
            int dy=ty;//s.width-1;
            for( int sx=0; sx<s.width; sx++,dy-- )
            {
              int pixel = s.getPixel( sx,sy );
              RGB rgb = s.palette.getRGB( pixel );
              if(rgb.red!=bg.red || rgb.green!=bg.green || rgb.blue!=bg.blue )
              {
                gc_.setForeground( new Color(device_,rgb));
                gc_.drawPoint( dx, dy); 
              }
            }
          }  
        break;
      }
      case 180:
      {
          RGB bg = new RGB(0,0,0);
          ImageData s = getTextImage( text, txt_clr, bg );
          int dy = ty;
          for( int sy=0; sy<s.height; sy++,dy--)
          {
            int dx=tx;//s.width-1;
            for( int sx=0; sx<s.width; sx++,dx-- )
            {
              int pixel = s.getPixel( sx,sy );
              RGB rgb = s.palette.getRGB( pixel );
              if(rgb.red!=bg.red || rgb.green!=bg.green || rgb.blue!=bg.blue )
              {
                gc_.setForeground( new Color(device_,rgb));
                gc_.drawPoint( dx,dy);
              }
            }
          }  
        break;
      }
      case 0: {
        Color c = gc_.getForeground();
        gc_.setForeground( txt_clr );
        gc_.drawText( text, tx, ty, true ); //easy
        if( font_ !=null ) {
          SWTOGCAdapter.DrawFontLineStyle( gc_, tx, ty, gc_.textExtent(text).x, font_.getFontStyle(), gd_.devX(5) );
        }
        gc_.setForeground( c );
        break;
      }
      default:
        throw new IllegalArgumentException("Unsupported angle "+angle+"?");
     }
  }
  
  public int getSystemColor( int id )
  {
    return SWTSystemColor.GetSystemColor( device_, id );
  }
   
  private class GCDirect implements IGCDirect
  {
    public IGC getIGC() { return SWTGC.this; }
    
    public boolean usePixelCoordinates()
    {
      return KdpiX_==1.0f && KdpiY_==1.0f;
    }
    
    /** convert from pixel to current device */
    public int devX(int x )
    {
      return (int)(KdpiX_*x);
    }
    /** convert from pixel to current device */
    public int devY(int y )
    {
      return (int)(KdpiY_*y);
    }
    /** convert from current device to pixel */
    public int pixX(int x )
    {
      return (int)(x/KdpiX_);
    }
    /** convert from current device to pixel */
    public int pixY(int y )
    {
      return (int)(y/KdpiY_);
    }
    public void drawPointDirect(int x, int y, int _rgba) 
    {
      Color c = gc_.getForeground();
      int crgba = RGBA.Get( c.getRed(), c.getGreen(), c.getBlue() );
      boolean chg = _rgba!=crgba;
      if( chg )
      {
        gc_.setForeground( createColor( _rgba) );
        gc_.drawPoint( x, y );
        gc_.setForeground( c );
      } else {
        gc_.drawPoint( x, y );
      }
    }
    
    public int getPointDirect( int x, int y )
    {
      return RGBA.TRANSPARENT;
    }
    public void drawHLineDirect(int x1, int x2, int y) 
    {
      if( brush_swt_compliant_ )
      {
        gc_.drawLine( x1, y, x2, y );
      } 
      else if( brush_ !=null )
      {
        Color c = gc_.getForeground();
        for( int x=x1; x<=x2; x++ )
        {
          int rgba = brush_.getBrushColor( x,y, RGBA.TRANSPARENT );
          if( rgba!=RGBA.TRANSPARENT) 
          {
            gc_.setForeground( createColor(rgba) );
            gc_.drawPoint( x, y );
          }
        }
        gc_.setForeground(c);
      }
      else Unsupported("drawHLineDirect() without brush.");
    }
    
    public void drawVLineDirect(int x, int y1, int y2) 
    {
      if( brush_swt_compliant_ )
      {
        gc_.drawLine( x, y1, x, y2 );
      } 
      else if( brush_!=null)
      {
        Color c = gc_.getForeground();
        for( int y=y1; y<=y2; x++ )
        {
          int rgba = brush_.getBrushColor( x,y, RGBA.TRANSPARENT );
          gc_.setForeground( createColor(rgba) );
          gc_.drawPoint( x, y );
        }
        gc_.setForeground(c);    
      }
      else Unsupported("drawVLineDirect() without brush.");
    }

    public void fillRectDirect( int x, int y, int w, int h )
    {
      if( brush_ instanceof SolidBrush )
      {
        Color c= gc_.getBackground();
        gc_.setBackground( createColor( ((SolidBrush)brush_).getRGBA() ) );
        gc_.fillRectangle( x, y, w, h );
        gc_.setBackground( c );
      }
      else if( brush_ instanceof SWTFillGradientRectangleBrush )
      {
        //special brush.
        Color fc = gc_.getForeground();
        Color bg = gc_.getBackground();
        SWTFillGradientRectangleBrush b = (SWTFillGradientRectangleBrush)brush_;
        gc_.setForeground( createColor( b.getRGBA1() ));
        gc_.setBackground( createColor( b.getRGBA2() ));
        //background already set.
        gc_.fillGradientRectangle( gd_.devX(x),gd_.devY(y),gd_.devX(w),gd_.devY(h), b.isVertical() );
        gc_.setForeground( fc );
        gc_.setBackground( bg );
      }
      else Unsupported("fillRectDirect() with brush = "+brush_ );
    }
    public void drawRectDirect( int x, int y, int w, int h )
    {
      brush_.brushBegin( SWTGC.this, this );
      int x2=x+w-1, y2=y+h-1;
      drawHLineDirect( x, x2, y);
      drawHLineDirect( x, x2, y2);
      y++; y2--;
      drawVLineDirect( x, y, y2 );
      drawVLineDirect( x2, y, y2 );
      brush_.brushEnd();
    }
    public void drawLineDirect(int x1, int y1, int x2, int y2) 
    {
      if( brush_swt_compliant_ )
      {
        Color c = gc_.getForeground();
        gc_.setForeground( gc_.getBackground() );
        gc_.drawLine( x1, y1, x2, y2 );
        gc_.setForeground( c );
      } 
      else if( brush_ != null ) 
      {
        //no pen draw path here.
        LineAlg line = new LineAlg( x1, y1, x2, y2 );
        Color c = gc_.getForeground();
        gc_.setForeground( gc_.getBackground() );
        int last_rgba = 0;
        boolean first=true;
        while( line.nextPoint(null ) )
        {
          int x = line.getX(), y = line.getY();
          int rgba = brush_.getBrushColor( x, y, RGBA.TRANSPARENT );
          if( first || last_rgba!=rgba ) {
            gc_.setForeground( createColor( rgba ) );
            first=false;
            last_rgba=rgba;
          }
          gc_.drawPoint( x, y );
        }
        gc_.setForeground( c );
        //mmmh ! drawPath will convert segment to device coordinate 
        //so segment will be twice converted ! bug here !
        //but convert x1 to pixel not the solution too we lost precision!
        pen_.drawPath( SWTGC.this,gd_,getSegment( x1, y1, x2, y2 ) );
      }
      else
      {
        Unsupported("drawLineDirect() without brush ");
      }
    }
  }
}
