/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SWTOGCAdapter.java,v 1.2 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.swt.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IFontMetrics;
import org.eclipse.tptp.platform.report.igc.internal.IImage;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.ogc.internal.IOGCAdapter;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBAImage;
import org.eclipse.tptp.platform.report.igc.util.internal.Size;


/**
 * IOGCAdapter for SWT graphic system.
 *
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class SWTOGCAdapter implements IOGCAdapter
{
  protected Device device_;
  /** dpiX/Y factor used to convert from pixel to device */
  protected float KdpiX_, KdpiY_;
  /** gc used for font operation */
  private GC font_gc_;
  /** font current used, created at setFont(), must be disposed */
  private  org.eclipse.swt.graphics.Font swtfont_;
  /** keep IFont style for getTextImage() */
  private int ifont_style_;

  public SWTOGCAdapter( Device _device )
  {
    device_=_device;
    fixDPI();
    ifont_style_ = IFont.NORMAL;
  }
  
  private void fixDPI()
  {
    if( device_ instanceof Display )
    {
      KdpiX_ = KdpiY_ = 1.0f;
    } else {
      Point dev_dpi = device_.getDPI();
      Point dsp_dpi = Display.getDefault().getDPI();     
      KdpiX_ = dev_dpi.x / (float)dsp_dpi.x;
      KdpiY_ = dev_dpi.y / (float)dsp_dpi.y;
    }
  }
  /**@return factor to convert pixel to current device coordinate */  
  public float getKdpiX() { return KdpiX_; }
  /**@return factor to convert pixel to current device coordinate */  
  public float getKdpiY() { return KdpiY_; }
  
  public void restart()
  {
    if( font_gc_!=null ) font_gc_.setFont(null);
    if( swtfont_!=null ) { swtfont_.dispose(); swtfont_=null; }
  }
  
  public void dispose()
  {    
    if( font_gc_ !=null ) { font_gc_.dispose(); font_gc_=null; }
    if( swtfont_ != null) { swtfont_.dispose(); swtfont_=null; }
  }
  
  public Size textExtent(String text) 
  {
    org.eclipse.swt.graphics.Point te = getFontGC().textExtent( text );
    return new Size( te.x, te.y );
  }
  
  public int getSystemColor( int id ) 
  {
    return SWTSystemColor.GetSystemColor( device_, id );
  }

  public RGBAImage getTextImage( String text, ISize size )
  {
    //TODO: +1 for underline font style but might be not enough ... have to work more for this.
    Image img= new Image( device_, size.getW()+1, size.getH()+1 );
    GC gc = new GC( img );
    try{      
      gc.setForeground( device_.getSystemColor(SWT.COLOR_BLACK));
      gc.setBackground( device_.getSystemColor(SWT.COLOR_WHITE));
      gc.setFont( getFontGC().getFont() );          
      gc.drawText( text, 0,0, false ); 
      DrawFontLineStyle( gc, 0,0,size.getW(), ifont_style_, devX(5) );
      return SWTImage.ToRGBAImage( img.getImageData() );
    } finally {
      gc.dispose();
      img.dispose();
    }    
  }
  
  private GC getFontGC()
  {
    if( font_gc_==null )
    {
      font_gc_ = new GC( device_ );
      if( swtfont_!=null) 
        font_gc_.setFont( swtfont_ );
    }
    return font_gc_;
  }
  
  public IFont getFont()
  {
    org.eclipse.swt.graphics.Font sf = getFontGC().getFont();
    if( sf==null ) return null;
    FontData fd = sf.getFontData()[0];      
    if( fd==null ) return null;
    int swtsty = fd.getStyle();
    int sty = IFont.NORMAL;
    if( (swtsty&SWT.BOLD)!=0 ) sty |= IFont.BOLD;
    if( (swtsty&SWT.ITALIC)!=0 ) sty |= IFont.ITALIC;
    return new Font( fd.getLocale(), fd.getName(), fd.getHeight(), sty );
  }
  
   
  public IFontMetrics getFontMetrics()  
  {
    final FontMetrics fm = getFontGC().getFontMetrics();
    return new IFontMetrics() {
      public int getAscent() { return pixY(fm.getAscent()); }
      public int getDescent() { return pixY(fm.getDescent()); } 
      public int getHeight() { return pixY(fm.getHeight()); }
      public int getLeading() { return pixY(fm.getLeading()); }      
    };
  }
  
  public void setFont( IFont font )
  {
    //dispose current font
    if( swtfont_!=null )
    {
      swtfont_.dispose();
      swtfont_=null;
    }
    if( font==null )
    {
      //reset font_gc_ too
      if( font_gc_!=null) 
      {
        font_gc_.dispose();
        font_gc_=null;
      }
      ifont_style_ = IFont.NORMAL;
    }
    else
    {      
      ifont_style_ = font.getFontStyle();
      int sty = font.getFontStyle();
      int swtsty= SWT.NORMAL;
      if( (sty & IFont.BOLD)!=0 ) swtsty |= SWT.BOLD;
      if( (sty & IFont.ITALIC)!=0 ) swtsty |= SWT.ITALIC;
      int size = font.getFontSize();
      swtfont_ = new org.eclipse.swt.graphics.Font( device_, font.getFontName(), size, swtsty );
      if( font_gc_!=null ) {
        font_gc_.setFont( swtfont_ );
      }
      //else: font will be applyed at font_gc_ creation time.
    }
  }
  
  /** _x,_y in device coord */
  protected void drawImageData( int []buf, int width, int height, int _x, int _y, ImageData data )
  {
    PaletteData pal = data.palette;
    int iw=data.width, ih=data.height;
    if( _x+iw-1 < 0 || _y+ih-1 < 0 ) return ; //outside
    
    int X1=Math.max( 0, -_x), Y1=Math.max(0,-_y);
    int X2=Math.min( iw, width-_x), Y2=Math.min( ih, height-_y);
    
    //global alpha value...
    int alpha = data.alpha < 0 ? 255 : (data.alpha&0xFF);//limit it to 255
    //or by pixel alpha value
    final byte []pixel_alpha = data.alpha < 0 ? data.alphaData : null;

    int tt = data.getTransparencyType();
    if( tt==SWT.TRANSPARENCY_ALPHA && data.alpha >= 0 )
    {
      if( data.alpha==0 ) return ;//general alpha is 0: full transparent
      if( data.alpha==255 || data.alpha<0 && data.alphaData==null ) {
        tt = SWT.TRANSPARENCY_NONE; //image is fully opaque, of alphaData is missed
      }
    }
    else if ( tt == SWT.TRANSPARENCY_MASK && data.maskData==null )
    {
      tt = SWT.TRANSPARENCY_NONE;
    }
    
    RGB rgb=new RGB(0,0,0);
    int index0 =(_y+Y1)*width+_x+X1;
    switch( data.getTransparencyType() )
    {
    case SWT.TRANSPARENCY_NONE:
      for( int Y=Y1; Y<Y2; Y++ )
      {
        int index =index0;
        for( int X=X1; X<X2; X++,index++ )
        {
          rgb = pal.getRGB( data.getPixel( X, Y ) );
          int rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue );
          buf[index] = rgba; //no RGBA.Combine needed
        }      
        index0+=width;
      } 
    break;
    case SWT.TRANSPARENCY_ALPHA:
      if( pixel_alpha==null) //general alpha
      {
        for( int Y=Y1; Y<Y2; Y++ )
        {
          int index =index0;
          for( int X=X1; X<X2; X++,index++ )
          {
            rgb = pal.getRGB( data.getPixel( X, Y ) );
            int rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue, alpha );
            buf[index] = RGBA.Combine( rgba, buf[index] );
          }      
          index0+=width;
        } 
      } else {
        int indexA0 = Y1*iw+X1;
        for( int Y=Y1; Y<Y2; Y++ )
        {
          int index =index0;
          int indexA=indexA0;
          for( int X=X1; X<X2; X++,index++,indexA++ )
          {
            alpha = pixel_alpha[indexA];
            if( alpha==0 ) continue;
            rgb = pal.getRGB( data.getPixel( X, Y ) );
            int rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue, alpha );
            buf[index] = RGBA.Combine( rgba, buf[index] );
          }      
          index0+=width;
          indexA0+=iw;
        } 
      }
    break;
    case SWT.TRANSPARENCY_PIXEL:
    {
      rgb = pal.getRGB( data.transparentPixel );
      int transparent_rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue );
      for( int Y=Y1; Y<Y2; Y++ )
      {
        int index =index0;
        for( int X=X1; X<X2; X++,index++ )
        {
          rgb = pal.getRGB( data.getPixel( X, Y ) );
          int rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue );
          if( rgba != transparent_rgba ){
            buf[index] = rgba; //as rgba is opaque
          }
        }      
        index0+=width;
      } 
    }
    break;
    case SWT.TRANSPARENCY_MASK:
    {
//TODO:this have to be checked because I never tested it ... so if it produce unexpected resulst
// including exception you know what you have to do ...      
      //this have to be checked
      int byte_per_scanline = (((iw + 7) / 8 + (data.maskPad - 1)) / data.maskPad) * data.maskPad;
      byte bit=(byte)0x80;
      if( (X1%8)>0) bit >>= (X1%8);
      int indexM0=Y1*(byte_per_scanline)+(X1/8);
      byte msk=data.maskData[indexM0];
      for( int Y=Y1; Y<Y2; Y++ )
      {
        int index =index0;
        int indexM=indexM0;
        for( int X=X1; X<X2; X++,index++ )
        {
          if( (bit&msk)!=0 )
          {
            rgb = pal.getRGB( data.getPixel( X, Y ) );
            int rgba = RGBA.Get( rgb.red, rgb.green, rgb.blue );
            buf[index] = rgba; //as rgba is opaque
          }
          bit >>= 1;
          if( bit==0 ) 
          {
           indexM++;
           msk = data.maskData[indexM];
           bit=(byte)0x80;
          }
        }      
        index0+=width;
        indexM0+=byte_per_scanline;
      }
    }
    break;
    }//swtich

  }
  
  public boolean drawImage( int buf[], int width, int height, IImage image, int x, int y  )
  {
    if( image instanceof SWTImage )
    {
      SWTImage img = (SWTImage)image;
      drawImageData( buf,width,height, x, y, img.getImage().getImageData()  );
      return true;
    }
    return false;
  }
  
  public boolean drawImage( int buf[], int width, int height, IImage image, int x, int y,int w, int h  )
  {
    if( image instanceof SWTImage )
    {
      SWTImage img = (SWTImage)image;
      Rectangle r = img.getImage().getBounds();
      ImageData data = img.getImage().getImageData();
      ImageData scaled = data;      
//TODO:?convert _w/_h to device coordinates ?      
      if( w!=data.width || h !=data.height)
      {
        scaled = data.scaledTo( w, h );
      }
      drawImageData( buf,width,height,  x, y, scaled );
      return true;
    }
    return false;
  }
  
  public boolean drawImage( int buf[], int width, int height, IImage image, int srcX, int srcY, int srcW, int srcH, int dstX, int dstY, int dstW, int dstH )
  {
    if( !(image instanceof SWTImage) ) return false;
    
    SWTImage img = (SWTImage)image;
    ImageData data = img.getImage().getImageData();
    PaletteData pal = data.palette;
    //reduce source image
    ImageData reduced = data;
    if( srcX!=0 || srcY !=0 || srcW != data.width || srcH != data.height )
    {
      reduced = new ImageData( srcW, srcH, data.depth, pal );
      int Y=srcY+srcH;
      int X=srcX+srcW;
      int ry=0;
      for( int sy=srcY; sy<Y; sy++,ry++)
      {
        int rx=0;
        for( int sx=srcX; sx<X; sx++,rx++ )
        {
          reduced.setPixel( rx,ry, data.getPixel( sx,sy ));
        }
      }
    }
    ImageData scaled = reduced;
    int x= devX(dstX), y=devY(dstY);
    int dw = devX(dstW), dh = devX(dstH);
    if( dw!=srcW || srcH!=dh )
    {
      scaled = data.scaledTo( dw, dh );
    }
    drawImageData( buf,width,height, x, y, scaled );
    return true;
  }
  
  /** convert from pixel to current device */
  public int devX(int x )
  {
    return (int)(KdpiX_*x);
  }
  /** convert from pixel to current device */
  public int devY(int y )
  {
    return (int)(KdpiY_*y);
  }
  /** convert from current device to pixel */
  public int pixX(int x )
  {
    return (int)(x/KdpiX_);
  }
  /** convert from current device to pixel */
  public int pixY(int y )
  {
    return (int)(y/KdpiY_);
  }
  
  /** 
   * dtx,dty is top left corner of text in device coordinates.
   * gc foreground must be already set with text color
   * dev_dash_size must be devX(5).
   * This is public as it is shared code with SWTGC. 
   */
  public static void DrawFontLineStyle( GC _gc, int dtx, int dty, int dtw, int ifont_style, int dev_dash_size )
  {
    if( ifont_style==IFont.NORMAL ) return ;
    if( (ifont_style & IFont.LINE_STYLES)== 0 ) return ; //nothing to draw.
    
    FontMetrics fm = _gc.getFontMetrics();
    int descent = fm.getDescent();
    //calcul 
    int psize = fm.getHeight();
    int linesize = psize/10;
    if(linesize == 0) linesize=1;
    int linesize2 = linesize/2;
        
    if( (ifont_style & IFont.BOLD) !=0 ) //bold ?
    {
      linesize += linesize2;
      if(linesize==1) { linesize = 2; linesize2=1; }
    }
    int space= linesize2 + 2;
    int olw = -1;
    
    int save_lw = _gc.getLineWidth();
    int save_ls = _gc.getLineStyle();
    
    _gc.setLineWidth( linesize );
    _gc.setLineStyle( SWT.LINE_SOLID );

    int _txth = fm.getHeight();
    
    int ly = dty+_txth-descent+space;//-linesize2 ;//??
        
    if( Font.HaveStyle( ifont_style, Font.UNDERLINE ))
    {
      _gc.drawLine( dtx, ly, dtx+dtw, ly );
    }
    //underline hide any dashed style
    else if ( Font.HaveStyle( ifont_style, Font.DASHED ))
    {
      //use modulo for text that use different style on several sequential text.
      final int size=dev_dash_size;
      final int size2=2*size;
      int xx = size*(int)((dtx+dtw)/size);
      int x = size*(int)Math.ceil(dtx/size);
      if( x < dtx ) x+=size2;
      for( ; x<xx; x+= size2 )
      {
        _gc.drawLine( x, ly, x+size, ly );
      }
    }
    if( Font.HaveStyle( ifont_style, Font.OVERLINE ))
    {
      ly = dty+linesize/2;
      _gc.drawLine( dtx,ly,dtx+dtw,ly);
    }
    if( Font.HaveStyle( ifont_style, Font.STRIKE ))
    {
      ly = dty+fm.getLeading()+fm.getAscent()/2 +linesize2;
      _gc.drawLine( dtx,ly,dtx+dtw,ly);
    }
    
    //restore gc settings
    _gc.setLineWidth( save_lw );
    _gc.setLineStyle( save_ls );
  }
}
