/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Circle.java,v 1.2 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

import org.eclipse.tptp.platform.report.igc.alg.internal.CircleAlg;
import org.eclipse.tptp.platform.report.igc.alg.internal.OvalAlg;
import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPath;
import org.eclipse.tptp.platform.report.igc.internal.IPathElement;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.internal.IShapeFiller;
import org.eclipse.tptp.platform.report.igc.internal.IVector;


/**
 * IPath and IShape implementation for a circle, or circle arc.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class Circle implements IPath, IPathElement, IShape, IShapeFiller
{  
  protected int cx_,cy_, radius_; // center of circle and radius
  protected double arc_start_, arc_length_; //angle of arc start, and arc length.

  /** Create empty cirle, must call setCircle or setCircle arc to define circle. */
  public Circle()
  {
    setCircle(0,0,0);
  }
  public Circle( int cx, int cy, int radius )
  {
    setCircle( cx,cy,radius);
  }
  public Circle( int cx, int cy, int radius, double arc_start, double arc_length )
  {
    setArc( cx,cy,radius, arc_start, arc_length );
  }
  public Circle( Circle c ) 
  {
    setCircle( c );
  }
  
  public void setCircle( int cx, int cy, int radius )
  {
    cx_=cx; cy_=cy;
    radius_=radius;
    arc_start_=0.0;
    arc_length_=Radian._2PI;    
  }
  public void setArc( int cx, int cy, int radius, double arc_start, double arc_length )
  {
    cx_=cx; cy_=cy;
    radius_=radius;
    arc_start_ = Radian.normalize(arc_start);
    arc_length_ = arc_length;
  }
  public void setCircle( Circle c )
  {
    cx_=c.cx_; 
    cy_=c.cy_;
    arc_start_ = c.arc_start_;
    arc_length_= c.arc_length_;
    radius_    = c.radius_;
  }
  
  public int getCenterX() { return cx_; }
  public int getCenterY() { return cy_; }
  public int getRadius() { return radius_; }
  public double getArcStart() { return arc_start_; }
  public double getArcLength() { return arc_length_; }
  
  protected boolean path_element_is_circle_;
  public boolean pathBegin( IGC gc, IGCDirect gd ) 
  {    
    path_element_is_circle_ = gd.usePixelCoordinates()
       || gd.devX(radius_)==gd.devY(radius_) ;//as device coord might change circle to oval
    return true; //circle is converted to device coord at startPathElement().
  }

  public void pathEnd() { path_element_returned_=false; }

  protected boolean path_element_returned_;
  public IPathElement nextPathElement() 
  {
    if( path_element_returned_ ) return null;
    path_element_returned_ = true; //as circle have only one path element.
    if( path_element_is_circle_ ) return this;    
    //will be converted to device coord at startPathElement...
    return new Oval( cx_,cy_, radius_,radius_, arc_start_, arc_length_ );
  }

  public IPathElement copyPathElement() { return new Circle(this); }

  protected CircleAlg path_alg_; //available between startPathElement/endPathElement
  protected int curr_index_, curr_octant_;
  protected Vector vector_ = new Vector();
  protected boolean finished_;
  public boolean pathElementBegin(IGC gc, IGCDirect gd)
  {
//TODO: convert to device coordinates ????    
    path_alg_ = new CircleAlg( cx_,cy_,radius_, arc_start_, arc_length_ );
    path_alg_.restart();    
    return !finished_;
  }
  public void pathElementEnd() { path_alg_ = null; }

  public boolean nextPoint(IPoint point) 
  { 
    return path_alg_.nextPoint( point );    
  }

  public void backTangent(IVector vector) { path_alg_.getCurrentTangent( vector ); }
  public void frontTangent(IVector vector) {path_alg_.getCurrentTangent( vector ); }

  public IShape copyShape() { return new Circle(this); }

  public boolean contains(int x, int y) 
  {
    return CircleAlg.Contains( x,y, cx_,cy_,radius_,arc_start_,arc_length_ );
  }

  public boolean contains(IPoint p) 
  {
    return CircleAlg.Contains( p.getX(), p.getY(), cx_,cy_,radius_,arc_start_,arc_length_ );
  }

  public IRect getBounds() 
  {
    int r = radius_ < 0 ? -radius_ : radius_;
    int r2 = r <<1; //2*r;
    return new Rect(cx_-r, cy_-r, r2, r2 );
  }

  public boolean fillShape(IGC gc, IGCDirect gd, IBrush brush, IShape shape) 
  {
    if( shape!=this ) return false;
    
    int drx = gd.devX(radius_);
    int dry = gd.devY(radius_);
    
    if( arc_length_ >= Radian._2PI || arc_length_ <= -Radian._2PI )
    {
      if( drx==dry ) { //still a circle in this IGC
        CircleAlg.FillCircle( gd, cx_,cy_, drx );
      } else {
        OvalAlg.FillOval( gd, cx_, cy_, drx, dry );
      }
    } else {
      if( drx==dry ) {               
        CircleAlg.FillArc( gd, cx_, cy_, radius_, arc_start_, arc_length_ );
      } else {
        OvalAlg.FillArc( gd, cx_,cy_, drx,dry, arc_start_, arc_length_ );
      }
    }
    return true;
  }
  
  /** @return true if rectangle is contained in this circle */
  public boolean contains( IRect r )
  {
    int x=r.getX(), y=r.getY();
    if( !contains( x, y )) return false;
    int xr=x+r.getW()-1;
    if( !contains( xr, y)) return false;
    int yb=y+r.getH()-1;
    return contains( x, yb ) && contains( xr, yb );
  }
  
  /** works only if circle is a full circle (not and arc) */
  public boolean contains( Circle c )
  {
    if( !contains( c.cx_,c.cy_)) return false;
    int dx = c.cx_-cx_, dy=c.cy_-cy_;
    long r02 = dx*dx + dy*dy;
    long rc2 = c.radius_*c.radius_;
    long r2 = radius_*radius_;
    return (r02+rc2) <= r2 ;
  }
}
