/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ComplexPath.java,v 1.4 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

import org.eclipse.tptp.platform.report.igc.alg.internal.IAlg;
import org.eclipse.tptp.platform.report.igc.alg.internal.LineAlg;
import org.eclipse.tptp.platform.report.igc.alg.internal.Spline3Alg;
import org.eclipse.tptp.platform.report.igc.alg.internal.Spline4Alg;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPath;
import org.eclipse.tptp.platform.report.igc.internal.IPathElement;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IVector;


/**
 * Complex path might be composed by the following path operator:
 * - move to a point<br>
 * - line to a point (from last entered point)<br>
 * - conic to, or draw spline with 3 controls points, last entered points plus two new<br>
 * - cubic to, or draw spline with 4 controls points, last entered points plus three new<br>
 * <p>
 * the first operator of complex path must be moveTo() one.<br>
 * Event if ComplexPath is a IPath it's made of only one IPathElement.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class ComplexPath implements IPath, IPathElement
{
  public static final int MOVE_TO=0;
  public static final int LINE_TO=1;
  public static final int CONIC_TO=2;
  public static final int CUBIC_TO=3;
  
  protected int []points_; //trio: xx_TO,x,y xxx_TO can be -1 for intermediary points
  protected int len_;
  protected boolean hull_path_; //in this mode all path are MOVE_TO/LINE_TO  
  
  /**
   * Create an empty complex path. use moveTo() to start path, and any path operator
   * to create path.
   */
  public ComplexPath() {}
  /**
   * Create a full copy of the given complex path.
   */
  public ComplexPath( ComplexPath p )
  {
    if( p.points_!=null )
    {
      points_ = new int[p.points_.length];
      System.arraycopy( p.points_,0, points_,0, p.points_.length );
      len_=points_.length;
    }
  }
  
  /** 
   * @return true if complex path is in "hull path" mode.
   * @see setHullPath()
   */
  public boolean isHullPath() { return hull_path_; }
  /**
   * Change "hull path" mode of this complex path.<br>
   * When "hull path" mode is on the path is generated as a polygon joining all control points
   * (except MOVE_TO operator which is preserved).
   */
  public void setHullPath( boolean b ) 
  {
    hull_path_=b;
  }
  
  /**
   * Reset current path, all operator are lost.<br>
   * You must redefine path starting with MOVE_TO operator.
   */
  public void reset()
  {
    len_=0;
  }  
  /**
   * Append MOVE_TO operator to current path.
   */
  public void moveTo( int x, int y )
  {
    checkSize( len_+3 );
    points_[len_++] = MOVE_TO;
    points_[len_++] = x;
    points_[len_++] = y;
  }
  /**
   * Append LINE_TO operator to current path.<br>
   * First point of line is the last point of path.
   */
  public void lineTo( int x, int y )
  {
    if( len_==0 )throw new IllegalStateException("ComplexPath must be started by moveTo()");
    checkSize( len_+3 );
    points_[len_++] = LINE_TO;
    points_[len_++] = x;
    points_[len_++] = y;
  }
  /**
   * Append CONIC_TO operator to current path.<BR>
   * First point of operator is the last point of path. 
   * This define 3 controls points that will be used to generate a spline path.
   */
  public void conicTo( int x1, int y1, int x2, int y2 )
  {
    if( len_==0 )throw new IllegalStateException("ComplexPath must be started by moveTo()");
    checkSize( len_+6 );
    points_[len_++] = CONIC_TO;
    points_[len_++] = x1;
    points_[len_++] = y1;
    points_[len_++] = -1;
    points_[len_++] = x2;
    points_[len_++] = y2;
  }
  /**
   * Append CUBIC_TO operator to current path.<BR>
   * First point of operator is the last point of path. 
   * This define 4 controls points that will be used to generate a spline path.
   */
  public void cubicTo( int x1, int y1, int x2, int y2, int x3, int y3 )
  {
    if( len_==0 )throw new IllegalStateException("ComplexPath must be started by moveTo()");
    checkSize( len_+9 );
    points_[len_++] = CUBIC_TO;
    points_[len_++] = x1;
    points_[len_++] = y1;
    points_[len_++] = -1;
    points_[len_++] = x2;
    points_[len_++] = y2;
    points_[len_++] = -1;
    points_[len_++] = x3;
    points_[len_++] = y3;
  }
  
  /** resize points_ array if its size is not enough to reach given size */
  protected void checkSize( int nsize )
  {
    if( points_==null )
    {
      points_ = new int[ Math.max(21,nsize) ];
      len_=0;
    }
    else if ( nsize >= points_.length )
    {
      int np[] = new int[nsize+21];
      System.arraycopy( points_,0, np,0, points_.length);
      points_=np;
    }
  }
  
  protected boolean give_path_element_;

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPath#pathBegin(org.eclipse.tptp.platform.report.igc.internal.IGC, org.eclipse.tptp.platform.report.igc.internal.IGCDirect)
   */
  public boolean pathBegin(IGC gc, IGCDirect gd) 
  {
    give_path_element_=true;
    return true;
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPath#pathEnd()
   */
  public void pathEnd() {}
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPath#nextPathElement()
   */
  public IPathElement nextPathElement() 
  {
    if( give_path_element_ ) {
      give_path_element_=false;
      return this;
    }
    return null;
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#copyPathElement()
   */
  public IPathElement copyPathElement() 
  {
    return new ComplexPath(this);
  }
  
  protected int curr_; //index of operator currently generating.
  protected IAlg alg_; //algorithm used to generate path of current path operator. 
  protected IGCDirect gd_; //null if gd.usePixelCoordinates() return true; otherwise must be used to convert to device coordinates
  
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#pathElementBegin(org.eclipse.tptp.platform.report.igc.internal.IGC, org.eclipse.tptp.platform.report.igc.internal.IGCDirect)
   */
  public boolean pathElementBegin(IGC gc, IGCDirect gd) 
  {
    if( points_==null || len_<6 ) return false;
    curr_=3;
    for( ;curr_<len_&&points_[curr_]==MOVE_TO; curr_+=3 );
    if( gd.usePixelCoordinates() )
    {
      gd_=null;
      switch( hull_path_ ? LINE_TO : points_[curr_] )
      {
      case LINE_TO:
        alg_ = new LineAlg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2]);
        break;
      case CONIC_TO:
        alg_ = new Spline3Alg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2], points_[curr_+4], points_[curr_+5] );
        break;
      case CUBIC_TO:
        alg_ = new Spline4Alg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2], points_[curr_+4], points_[curr_+5], points_[curr_+7], points_[curr_+8] );
        break;
      default:
        return false;
      }
    } else {
      gd_=gd;
      switch( hull_path_ ? LINE_TO : points_[curr_] )
      {
      case LINE_TO:
        alg_ = new LineAlg( gd.devX(points_[curr_-2]),gd.devY(points_[curr_-1]), gd.devX(points_[curr_+1]), gd.devY(points_[curr_+2]));
        break;
      case CONIC_TO:
        alg_ = new Spline3Alg( gd.devX(points_[curr_-2]),gd.devY(points_[curr_-1]), gd.devX(points_[curr_+1]), gd.devY(points_[curr_+2]), gd.devX(points_[curr_+4]), gd.devY(points_[curr_+5]) );
        break;
      case CUBIC_TO:
        alg_ = new Spline4Alg(gd.devX(points_[curr_-2]),gd.devY(points_[curr_-1]), gd.devX(points_[curr_+1]), gd.devY(points_[curr_+2]), gd.devX(points_[curr_+4]), gd.devY(points_[curr_+5]), gd.devX(points_[curr_+7]), gd.devY(points_[curr_+8]) );
        break;
      default:
        return false;
      }
    }
    return true;
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#pathElementEnd()
   */
  public void pathElementEnd() { gd_=null; }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#nextPoint(org.eclipse.tptp.platform.report.igc.internal.IPoint)
   */
  public boolean nextPoint(IPoint point) 
  {
    if( alg_.nextPoint( point ) ) return true;
    //next path part... 
    next:
    for(;;)
    {
      switch( hull_path_ ? LINE_TO : points_[curr_] )
      {      
      case LINE_TO : curr_ += 3; break;
      case CONIC_TO: curr_ += 6; break;
      case CUBIC_TO: curr_ += 9; break;
      }
      boolean was_moveto = curr_>=len_ ? false : (points_[curr_]==MOVE_TO);
      for( ;curr_<len_&&points_[curr_]==MOVE_TO; curr_+=3 );
      if( curr_ >= len_ ) return false;
      switch( hull_path_ ? LINE_TO : points_[curr_] )
      {
      case LINE_TO:
        if( curr_+2 >= len_ ) return false;
        if( gd_==null ) {
          alg_ = new LineAlg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2]);
        } else {
          alg_ = new LineAlg( gd_.devX(points_[curr_-2]),gd_.devY(points_[curr_-1]), gd_.devX(points_[curr_+1]), gd_.devY(points_[curr_+2]));
        }
        break;
      case CONIC_TO:
        if( curr_+5 >= len_ ) return false;
        if( gd_==null ) {
          alg_ = new Spline3Alg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2], points_[curr_+4], points_[curr_+5] );
        } else {
          alg_ = new Spline3Alg( gd_.devX(points_[curr_-2]),gd_.devY(points_[curr_-1]), gd_.devX(points_[curr_+1]), gd_.devY(points_[curr_+2]), gd_.devX(points_[curr_+4]), gd_.devY(points_[curr_+5]) );        
        }
        break;
      case CUBIC_TO:
        if( curr_+8 >= len_ ) return false;
        if( gd_==null ) {
          alg_ = new Spline4Alg(points_[curr_-2],points_[curr_-1], points_[curr_+1], points_[curr_+2], points_[curr_+4], points_[curr_+5], points_[curr_+7], points_[curr_+8] );
        } else {
          alg_ = new Spline4Alg(gd_.devX(points_[curr_-2]),gd_.devY(points_[curr_-1]), gd_.devX(points_[curr_+1]), gd_.devY(points_[curr_+2]), gd_.devX(points_[curr_+4]), gd_.devY(points_[curr_+5]), gd_.devX(points_[curr_+7]), gd_.devY(points_[curr_+8]) );
        }
        break;
      default:
        return false;
      }//switch
      //consume first point except on MOVE_TO
      if( !was_moveto ) {
        if( !alg_.nextPoint( point ) ) continue next;
      }
      if( alg_.nextPoint(point) ) return true;
      //else: next path  
    }
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#backTangent(org.eclipse.tptp.platform.report.igc.internal.IVector)
   */
  public void backTangent(IVector vector) 
  {
    alg_.backTangent( vector );
  }
  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.IPathElement#frontTangent(org.eclipse.tptp.platform.report.igc.internal.IVector)
   */
  public void frontTangent(IVector vector)
  {
    alg_.frontTangent( vector );
  }
}
