/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GradientGenerator.java,v 1.4 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;



/**
 * Gradient generator is a RGBA generator using a linear interpolation from a coeficient 'k'
 * which valid range is 0..1.
 * "color" method do the work.<br>
 * A Gradient is defined with  discrete values of color (2,3, or more), each color (RGBA)
 * are associated to a value of 'k'. Colors and range of color are defined using setColor()
 * method (or directly constructor).
 * <br><br>
 * For example "<code>new GradientColor( 0x0000FFFF, 0xFFFFFFFF, 0xFF0000FF );</code>" define a gradient
 * made of Red,White and Blue colors.<br>
 * Flat colors can also be used, for example:<br>
 * <code>setGradient( new double[] {0.0,0.3,0.3,0.7,0.7,1.0}, new int[]{0x0000FFFF, 0x0000FFFF,0xFFFFFFFF,0xFFFFFFFF, 0xFF0000FF,0xFF0000FF});</code><br>
 * is the french flag.
 * <br>
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class GradientGenerator implements IGradientGenerator
{
  private int[] rgba_;
  private double[] k_;
  
  /** Create a blank gradient, use setColor() of setGradient() methods to define model */
  public GradientGenerator()
  {    
  }
  
  /** Create a gradient with two colors. @see setGradient(int,int) */
  public GradientGenerator( int _rgba0, int _rgba1 )
  {
    setGradient( _rgba0, _rgba1 );
  }
  
  /** Create a gradient with three colors., @see setGradient(int,int,int) */
  public GradientGenerator( int _rgba0, int _rgbaM, int _rgba1 )
  {
    setGradient( _rgba0, _rgbaM, _rgba1 );
  }
  
  /** Create a gradient using array, @see setGradient(double[],int[]) */
  public GradientGenerator( double _k[], int _rgba[] )
  {
    setGradient( _k, _rgba );
  }
  
  /** Create a gradient copying an other */
  public GradientGenerator( GradientGenerator _gradient )
  {
    setGradient( _gradient );
  }
  
  /** Create a gradient using equidistant spaced colors */
  public GradientGenerator( int _rgba[] )
  {
    setGradient( _rgba );
  }
  
  /**
   * Reset color model.
   */
  public void reset()
  {
    rgba_=null;
    k_=null;
  }
  
  /**
   * Change all color model of grandient, to used only two colors.
   */
  public void setGradient( int _rgba0, int _rgba1)
  {
    rgba_ =new int[]{ _rgba0, _rgba1};
    k_    =new double[]{0.0,1.0};
  }
  
  /**
   * Change all color model of grandient, to used only three colors at coeficient 0,0.5 and 1.
   */
  public void setGradient( int _rgba0, int _rgbaM, int _rgba1)
  {
    rgba_ =new int[]{ _rgba0, _rgbaM, _rgba1};
    k_    =new double[]{0.0,0.5,1.0};
  }
  
  /**
   * Change all color model of gradient using array for definition of coeficient and colors.
   * Arrays are checked, and result is a validated gradient (no more than twice same-value coeficient
   * are retrieved from arrays).
   */
  public void setGradient( double _k[], int _rgba[] )
  {
    if( _k==null ||_rgba==null ) return ;
    int len = Math.min( _k.length, _rgba.length );
    if( len==0 )
    {
      rgba_=null;
      k_=null;
      return ;
    }
    
    rgba_ = new int[len];
    k_ = new double[len];
    
    rgba_[0] = _rgba[0];
    k_[0] = _k[0];
    int l=0;
    
    int i=1;
    int def=0;

    for( ; i<len; i++ )
    {
      if( k_[i]==k_[l] )
      {
        //no more than twice same value of k please.
        if( def==0 ) def=1; else continue;           
      }
      else def=0;
      ++l;      
      k_[l] = _k[i];
      rgba_[l] = _rgba[i];
    }
    //bad gradient ?
    l++; //l was on last element, now is real length.
    if( l!=len )
    {
      int c[] = new int[l];
      double k[] = new double[l];
      System.arraycopy( k_, 0, k, 0, l );
      System.arraycopy( rgba_, 0, c, 0, l );
      k_=k;
      rgba_=c;
    }
  }
  
  /**
   * Reset this gradient using equaldistant colors defined in array.
   * Only reset this gradient if _rgba is null or have size less than 2.
   */
  public void setGradient( int _rgba[] )
  {
    rgba_=null;
    k_=null;
    
    if( _rgba==null || _rgba.length<2 ) return ;
    
    rgba_= new int[_rgba.length];
    k_ = new double[_rgba.length];
    
    double dk = 1.0/(double)(_rgba.length-1);
    double ck=0.0;
    for( int i=0; i<_rgba.length; ++i)
    {
      rgba_[i]=_rgba[i];
      k_[i] = ck;
      ck+=dk;
    }
  }
  
  /**
   * Reset this gradient using a full copy of parameter.
   */
  public void setGradient( GradientGenerator _gradient )
  {
    if( _gradient==null )return ;
    
    int size = _gradient.size();
    if( size==0 )
    {
      k_=null;
      rgba_=null;
    }
    else
    {
      k_ = new double[size];
      rgba_ = new int[size];
      System.arraycopy( _gradient.k_, 0, k_, 0, size );
      System.arraycopy( _gradient.rgba_, 0, rgba_, 0, size );
    }
  }
  
  /**
   * Change color value at given index.
   * @param _index index of color to change [0..size(
   * @param _rgba new color
   */
  public void setColorAt( int _index, int _rgba )
  {
    rgba_[_index] = _rgba ;
  }
  
  /** 
   * Replace all colors defined at value _k  by _rgba color 
   * @return index of inserted color.
   */
  public int setColor( double _k, int _rgba )
  {
    if( rgba_==null )
    {
      rgba_ = new int[] {_rgba};
      k_ = new double[] {_k};
      return 0;
    }
    int already=0;
/*TODO:remove doublon: keep them, I need for editor    
    for( int i=0; i<k_.length; ++i)
    {
      if( k_[i]==_k ) already++;
      if( k_[i]>_k) break; //sorted.
    }*/
    int c[] = new int[rgba_.length-already+1];
    double k[] = new double[k_.length-already+1];
    int i=0;
    for( ; i<rgba_.length; ++i )
    {
      if( k_[i] >= _k ) break;
      k[i] = k_[i];
      c[i] = rgba_[i];
    }
    int j=i;
    k[j] = _k;
    c[j]=_rgba;
    int index=j;
    j++;
//TODO:doublon:   for( ; i<rgba_.length && k_[i]==_k; ++i );
    for( ; i<rgba_.length; ++i,++j )
    {
      k[j] = k_[i];
      c[j] = rgba_[i];  
    }
    k_=k;
    rgba_=c;  
    return index;
  }
  
  /**
   * Replace all color in range _k1.._k2 by this range 
   */
  public void setColor( double _k1, int _rgba1, double _k2, int _rgba2 )
  {  
    if( rgba_==null)
    {
      rgba_ = new int[] { _rgba1, _rgba2 };
      k_ = new double[] { _k1, _k2 };
      return ;
    }
    
    if( _k1==_k2) 
    {
      setColor( _k1, _rgba1 );
      return ;      
    }
    
    if( _k2<_k1 )
    {
      double k=_k1; _k1=_k2; _k2=_k1;
      int c=_rgba1; _rgba1=_rgba2; _rgba2=c;
    }
    
    int count=0;
    int i;
    int len=k_.length;
    boolean first=true;
    for( i=0; i<len; ++i )
    {
      if( k_[i]==_k1)
      {
        if( first) first=false; else break;
      }
      if( k_[i] > _k1 ) break;
    }
    first=true;
    for( ; i<len && k_[i]<=_k2; ++i )
    {
      if( k_[i]  > _k2 ) break;
      if( k_[i] == _k2 ) 
      {
        //count only if there are two _k2 (end, first)
        if( i==len-1 ) 
        {
          count++; //only one: replace it
        }
        else if ( k_[i+1]==_k2 )
        {
          count++; //two, replace first.
        }
        break;
      }
      count++;
    }
    //I know number of color removed..
    double k[] = new double[len-count+2];
    int c[] = new int[len-count+2];
    //now copy/replace colors..
    first=true;
    for( i=0; i<len; ++i )
    {
      if( k_[i] >_k1) break;
      if( k_[i]==_k1)
      {
        if( first) first=false; else break;
      }
      k[i] = k_[i];
      c[i] = rgba_[i];
    }
    //insert new range:
    int j=i;
    k[j] = _k1; c[j] = _rgba1; j++;
    k[j] = _k2; c[j] = _rgba2; j++;
    
    //skip those colors.
    first=true;
    for( ; i<len && k_[i]<=_k2; ++i )
    {
      if( k_[i]  > _k2 ) break;
      if( k_[i] == _k2 ) 
      {
        //count only if there are two _k2 (end, first)
        if( i==len-1 ) 
        {
          i++; //only one: replace it
        }
        else if ( k_[i+1]==_k2 )
        {
          i++; //two, replace first.
        }
        break;
      }
    }
    //finish copy
    for( ;i<len; ++i, ++j )
    {
      k[j] = k_[i];
      c[j] = rgba_[i];
    }
    
    rgba_=c;
    k_=k;  
  }
  
  /**
   * Remove color defined at given index.
   */
  public void removeColorAt( int index )
  {
    int size=size();
    if( index <0 || index >= size) return ;
    if( size==1)
    {
      k_ = null;
      rgba_=null;
      return ;
    }
    int nc[] = new int[size-1];
    double nk[] = new double[size-1];
    int iw=0;
    for(int i=0; i<size; i++)
    {
      if( i==index) continue;
      nc[iw]=rgba_[i];
      nk[iw]=k_[i];
      iw++;
    }
    k_=nk;
    rgba_=nc;
  }
  
  /** 
   * Remove all color defined at value _k, have no effet for _k<=0.0 and _k>=1.0
   */
  public void removeColor( double _k )
  {
    if( rgba_==null || rgba_.length<3 ) return ;
    if( _k<=0.0 || _k>1.0 ) return ;
    int count=0;
    int i;
    for( i=0; i<k_.length; ++i ) 
    {
      if( k_[i]==_k) count++;
      if( k_[i] >_k) break;
    }
    if( count==0) return ; //no color defined 
    double k[] = new double[k_.length-count];
    int c[] = new int[k_.length-count];
    
    for( i=0; i<k_.length; ++i)
    {
      if( k_[i] >= _k ) break;
      k[i] = k_[i];
      c[i] = rgba_[i];
    }
    int j=i;
    for( ; i<k_.length && k_[i]==_k; ++i );
    for( ; i<k_.length; ++i, ++j )
    {
      k[j] = k_[i];
      c[j] = rgba_[i];
    }
    
    k_ = k;
    rgba_=c;
  }
  
  /** @return number of color used to define gradient model */
  public int size()
  {
    return rgba_==null ? 0 : rgba_.length;
  }
  /**@return value of gradient coeficient defined in model at given index ([0..size[) */
  public double getValueAt( int index )
  {
    return k_[index];
  }
  
  /**@return RGBA color of gradient model for given index ([0..size[) */
  public int getColorAt( int index )
  {
    return rgba_[index];
  }
  
  /**
   * @return gradient color defined at value _k using linear extrapolation.
   * return transparent color if _k &lt;0 or &gt;1.0.
   */
  public int getRGBA( double _k )
  {
    if( rgba_==null || rgba_.length<2 || _k<0 || _k > 1.0 || Double.isNaN(_k) ) return 0; //transparent.
    int i=rgba_.length-1;
    while( k_[i] > _k  && i >0 ) i--;
    if( i<0 ) return 0;
    
    if( k_[i]==_k ) return rgba_[i];
    
    int c1 = rgba_[i];
    int c2 = rgba_[i+1];
    return RGBA.Gradient( c1,c2,(_k-k_[i])/(k_[i+1]-k_[i]));
  }
  
  public String toString()
  {
    String s = super.toString();
    s += "[" + (k_==null ? 0 : k_.length) + "]";
    if( k_!=null && k_.length > 0 )
    {
      s += "{";
      for( int i=0; i<k_.length; ++i )
      {
        if(i>0) s+=", ";
        String c = Long.toString(((long)rgba_[i])&0xFFFFFFFFL,16);
        while( c.length()<8) c = "0"+c;
        s += Double.toString(k_[i]) +":"+ c;
      }
      s+="}";
    }
    return s;
  }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.igc.internal.util.IGradientGenerator#containsTransparency()
   */
  public boolean containsTransparency() 
  {
    for( int i=0; i<rgba_.length; ++i )
    {
      if( !RGBA.IsOpaque( rgba_[i] ) ) return true;
    }
    return false;
  }
  
  /** @return Rainbow gradient generator */
  public static GradientGenerator Rainbow()
  {
    return new GradientGenerator(new int[]{RGBA.RED,RGBA.ORANGE,RGBA.YELLOW,RGBA.GREEN,RGBA.BLUE,RGBA.MAGENTA});
  }
}
