/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LineStylePen.java,v 1.4 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPath;
import org.eclipse.tptp.platform.report.igc.internal.IPathElement;
import org.eclipse.tptp.platform.report.igc.internal.IPen;

/**
 * Implements a IPen using RGBA color, line style and line width.
 * IGC implementors could use pen's parameter and call directly graphics native methods.
 * 
 * This pen provide also drawPath method, but this drawing is done using 1 pixel line width.
 * (no support of other line width ).
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class LineStylePen implements IPen 
{
  //swt's definition compliant (measured on win32 platform).
  public static final int SOLID=0;
  public static final int DASH=1; //18 pixels on, 6 off.
  public static final int DOT=2; //3 pixels on, 3 off.
  public static final int DASH_DOT=3; //9 pixels on, 6 off, 3 on, 6 off
  public static final int DASH_DOT_DOT=4; //9 pixels on, 3 off, 3 on, 3 off, 3 on, 3 off
  //not swt's compliant:
  public static final int PIXEL=5; //one pixel on, one pixel off...
  
  /** first byte is length of pattern in pixels, second one is 1 if pattern start drawing pixel,
   * 0 if not, the following value are successive pixels count with same "drawing" property.
   */
  public static final int DEF_DOT[] = new int[]{6,1,3,3}; //in pixels
  public static final int DEF_DASH[] = new int[]{24,1,18,6};
  public static final int DEF_DASH_DOT[] = new int[]{24,1,9,6,3,6};
  public static final int DEF_DASH_DOT_DOT[] = new int[]{24,1,9,3,3,3,3,3};
  public static final int DEF_PIXEL[] = new int[]{2,1,1,1};
  
  protected int rgba_;
  protected int line_style_;
  protected int line_width_;
//TODO: add cap style and join style ???.. if it's a real pen with drawing methods...  
  protected IGC gc_; //available between penBegin/penEnd methods calls
    
  /** Create pen using RGBA color, line style and line width */
  public LineStylePen( int _rgba, int line_style, int line_width )
  {
    rgba_=_rgba;
    line_style_ = line_style;
    line_width_ = line_width;
  }
  /** Create pen using RGBA color, line style and 0 (zero) line width */
  public LineStylePen( int _rgba, int line_style )
  {
    rgba_=_rgba;
    line_style_ = line_style;
    line_width_ = 0;
  }
  /** Create a SOLID pen with 0 line width and given RGBA color */
  public LineStylePen( int _rgba )
  {
    rgba_=_rgba;
    line_style_ = SOLID;
    line_width_ = 0;
  }
  /** Create a copy of pen */
  public LineStylePen( LineStylePen _pen )
  {
    rgba_=_pen.rgba_;
    line_width_=_pen.line_width_;
    line_style_=_pen.line_style_;
  }
  /** Create black solid pen with 0 pixel line width */
  public LineStylePen()
  {
    rgba_=RGBA.BLACK;
    line_width_=0;
    line_style_=SOLID;
  }
  
  public IPen copyPen() { return new LineStylePen(this); }
  
  public String toString()
  {
    String s = super.toString();
    s = s+"{rgba="+RGBA.Str(rgba_)+", style="+StrStyle(line_style_)+", width="+line_width_+" }";
    return s;
  }
  
  public static String StrStyle( int style )
  {
    switch( style )
    {
      case SOLID: return "SOLID";
      case DASH : return "DASH";
      case DOT  : return "DOT";
      case DASH_DOT  : return "DASH_DOT";
      case DASH_DOT_DOT  : return "DASH_DOT_DOT";
    }
    return Integer.toString(style);
  }
  /**@return current RGBA color of pen */
  public int getRGBA() { return rgba_; }
  /**@return current line style (@see SWT constants)*/
  public int getLineStyle() { return line_style_; }
  /**@return current line width */
  public int getLineWidth() { return line_width_; }
  
  /**Change current RGBA color of pen */
  public void setRGBA( int _rgba )
  {
    rgba_=_rgba;
  }
  /** Change current line style, use SWT.LINE_xxxx constants */
  public void setLineStyle( int line_style )
  {
    line_style_=line_style;
  }
  /** Change current line width */
  public void setLineWidth( int line_width )
  {
    line_width_=line_width;
  }

  protected void drawPath( IGC gc, IGCDirect gd, IPath path, int def[] )
  {
//System.out.println("LSP.drawPath() path="+path);    
    int rdef[] = def ;
    if( def[0] == 0 ) return ;
    boolean draw = def[1]==1;
    if( !gd.usePixelCoordinates() )
    {
      // definition need to be converted to pixels.
      int len = def.length;      
      int nlen = gd.devX( def[0] ); //why not devY ?
      int ni=0;  
      rdef = new int[def.length];
      rdef[0] = nlen;
      rdef[1] = def[1];
      double k = nlen/(double)def[0];
      for( int i=2; i<len; ++i )
      {
        int npix= (int)Math.round(k*def[i]); 
        rdef[i] = npix;
      }
    }
    
    int lx=-1, ly=-1;
    int len = rdef[0];
    int curr=0;
    boolean on=rdef[1]==1;
    int icurr=2;
    Point p = new Point();
    int def_len = rdef.length;
    boolean transp = !RGBA.IsOpaque(rgba_);    
    for( IPathElement e=path.nextPathElement(); e!=null; e=path.nextPathElement() )
    {
      if(!e.pathElementBegin(gc,gd)) continue;
      while( e.nextPoint(p) ) {
        int px = p.getX(), py=p.getY();
        if( lx!=px || ly!=py ) {
          if( on ) {
            int rgba = rgba_;
            if(transp) rgba = RGBA.Combine( rgba_, gd.getPointDirect(px,py));
            gd.drawPointDirect( px, py, rgba );
          }
          curr++;
          if ( curr >= rdef[icurr]) {
            on=!on; 
            curr=0;
            icurr++;            
            if( icurr >= def_len ) {            
              icurr=2;
              on=rdef[1]==1;
            }
          }  
          lx=px;
          ly=py;
        }
      }
      e.pathElementEnd();
    }   
  }
  /**
   * Support only 0 and 1 pixel line width.
   * @see org.eclipse.tptp.platform.report.igc.internal.IPen#drawPath(org.eclipse.tptp.platform.report.igc.internal.IGC, org.eclipse.tptp.platform.report.igc.internal.IPath)
   */
  public void drawPath(IGC gc, IGCDirect gd, IPath path) 
  {
    //if (line_width_!=0 && line_width_!=1 ) return ; //unsupported.
    
    path.pathBegin( gc, gd );
    Point p = new Point();
    int lx=-1,ly=-1;
    boolean transp = !RGBA.IsOpaque(rgba_);
    switch( line_style_ )
    {
    case SOLID:{
      for( IPathElement e=path.nextPathElement(); e!=null; e=path.nextPathElement() )
      {
        if(!e.pathElementBegin(gc,gd)) continue;
        while( e.nextPoint(p) ) {
          int px = p.getX(), py=p.getY();
          if( lx!=px || ly!=py ) {
            int rgba = rgba_;
            if(transp) rgba = RGBA.Combine( rgba_, gd.getPointDirect(px,py));
            gd.drawPointDirect( px, py, rgba );
            lx=px;
            ly=py;
          }
        }
        e.pathElementEnd();
      }
      break;
    }
    case DOT: drawPath( gc, gd, path, DEF_DOT ); break;
    case DASH: drawPath( gc, gd, path, DEF_DASH ); break;
    case DASH_DOT: drawPath( gc, gd, path, DEF_DASH_DOT ); break;
    case DASH_DOT_DOT: drawPath( gc, gd, path, DEF_DASH_DOT_DOT ); break;
    case PIXEL: drawPath( gc, gd, path, DEF_PIXEL); break;
    }
    path.pathEnd();
  }

}

