/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Polygon.java,v 1.2 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPath;
import org.eclipse.tptp.platform.report.igc.internal.IPathElement;
import org.eclipse.tptp.platform.report.igc.internal.IPoint;
import org.eclipse.tptp.platform.report.igc.internal.IPolygon;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;



/**
 * Implementation of IPolygon using SWT GC.drawPoly() compliant definition.
 * I means one dimension integer array where x and y coodinate are stored "xyxyxy..".
 * Note: if this is compliant to SWT definition, this class doesn't depends on any SWT classes.
 *
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class Polygon implements IPolygon,IPath
{
  protected int points_xy_[];
  protected int size_;
  
  /** Create empty polygon, use resize() of setPoints() to define it */
  public Polygon()
  {
    size_=0;    
  }
  
  /** Create a sized polygon, all points are (0,0), use setPoint() to define them */
  public Polygon( int _alloc_size )
  {
    size_ = _alloc_size;
    points_xy_ = new int[2*size_];
  }
  
  /**!no copy of points, use reference to given array only. */
  public Polygon( int points[] )
  {
    points_xy_ = points;
    size_ = points==null ? 0 : points.length/2;
  }
  
  /**Create SWTPolygon from IPolygon interface, do a deep-copy in case of Polygon */
  public Polygon( IPolygon poly )
  {
    int size = poly==null ? 0 : poly.getPolySize();
    if( size==0 ) return ;
    if( size==1 )
    {
      points_xy_ = new int[2];
      points_xy_[0] = poly.getPolyX(0);
      points_xy_[1] = poly.getPolyY(0);
    } else {
      boolean closed = poly.isPolyClosed();
      int x0 = poly.getPolyX(0), xS= poly.getPolyX(size-1);
      int y0 = poly.getPolyY(0), yS= poly.getPolyY(size-1);
      boolean swt_closed = closed && (x0==xS)&&(y0==yS);
      if( swt_closed ) {  
        size_=size;
        points_xy_ = new int[2*size_];
        points_xy_[0] = x0;
        points_xy_[1] = y0;
        int w=2;
        for( int i=1; i<size; i++ )
        {
          points_xy_[w++] = poly.getPolyX(i);
          points_xy_[w++] = poly.getPolyY(i);
        }
      } else { 
        size_ = size + (closed?1:0);
        points_xy_ = new int[2*size_];
        points_xy_[0] = x0;
        points_xy_[1] = y0;
        int w=2;
        for( int i=1; i<size; i++ )
        {
          points_xy_[w++] = poly.getPolyX(i);
          points_xy_[w++] = poly.getPolyY(i);
        }
        if( closed ) //do a swt closure.
        {
          points_xy_[w++] = x0;
          points_xy_[w++] = y0;
        }
      }
    }  
  }
  
  public IShape copyShape() { return new Polygon(this); }
  
  public void setPolySize( int _size )
  {
    if( _size < size_ ) 
    {
      size_=_size;
    } else {
      resize( _size );
    }
  }
  
  /** change number of points only if size is reduced */
  public void resize( int _num_points )
  {
    //same size 
    if( points_xy_.length/2==_num_points ) return ;
    int np [] = new int[_num_points*2];
    //number of points have been reduced ?
    int new_size = Math.min( size_, _num_points);
    //copy available points:
    int min = 2*Math.min( size_, _num_points );
    for( int i=0; i<min; ++i ) np[i]=points_xy_[i];
    size_ = new_size;
    points_xy_=np;
  }
  /** reduce internal array to size needed */
  public void compact()
  {
    //compact need ?
    if( size_ == points_xy_.length/2 ) return ;
    int np[] = new int[2*size_];
    System.arraycopy( points_xy_,0, np,0, np.length );
    points_xy_=np;
  }    
  /** replace or resize polygon up to given index */
  public void setPoint( int index, int x, int y )
  {
    int idx = 2*index;
    if( (idx+1) >= points_xy_.length )
    {
      //expand array
      int xy[] = new int[idx+21]; //10 points more
      System.arraycopy( points_xy_,0,xy,0,points_xy_.length);
      points_xy_ = xy;
    } 
    points_xy_[  idx]=x;
    points_xy_[++idx]=y;
    
    if( index >= size_ ) size_=index+1;
  }

  public int getPolySize() { return size_; }
  public int getPolyX(int index) { return points_xy_[2*index  ]; }
  public int getPolyY(int index) { return points_xy_[2*index+1]; }
  
  public boolean isPolyClosed() {
    return size_>=2 && points_xy_[0]==points_xy_[2*size_-2] 
                    && points_xy_[1]==points_xy_[2*size_-1] ;
    }
  
  public int[] getPoints() { return points_xy_; }
  
  public void translate( int _tx, int _ty )
  {
    for(int i=0; i<points_xy_.length;)
    {
      points_xy_[i++] += _tx;
      points_xy_[i++] += _ty;
    } 
  }
  
  /** @return String representation of polygon including first 20 points coordinates */
  public String toString()
  {
    String s=super.toString()+"{";
    int idx=0;
    int cnt = size_ < 20 ? size_ : 20 ;
    for( int i=0;i<cnt; ++i)
    {
      s += Integer.toString(points_xy_[idx++])+",";
      s += Integer.toString(points_xy_[idx++])+", ";
    }
    if( size_ >= 20 ) s+="...}"; else s+="}";
    return  s;
  }
  
  public boolean contains( IPoint p )
  {
    return contains( p.getX(), p.getY() );
  }
  
  public boolean contains( int px, int py )
  {
    if( size_ < 3 ) return false; //must have at least 3 points
    int length = points_xy_.length;
    if( (length&0x01)!=0) length--; //have extra int (array end is [...x,y,x,y,x], skip last x)
    int last_ip = length-2;
    int lx = points_xy_[last_ip];
    int ly = points_xy_[last_ip+1];
    int segment_count=0;

    for( int i=0; i<length;  )
    {
      int x = points_xy_[i++];
      int y = points_xy_[i++];
      //vertical line py cut the segment [lx,ly]-[x,y] above py ?
      int xn,xx,yxn,yxx;
      if( lx < x ) //sort X
      {
        xn = lx; yxn = ly; 
        xx =  x; yxx =  y; 
      } else { 
        xn =  x; yxn =  y; 
        xx = lx; yxx = ly; 
      }
      if( px >= xn && px <= xx && ( py >= ((y<ly)?y:ly)))
      {
        if( xn == xx ) //vertical segment
        {
          if( py <= ((y<ly)?ly:y)) // point inside segment ?
          {
            return true;
          }
          return false;
        }
        float sy = yxn +(px-xn)*(yxx-yxn)/(xx-xn);
        if( sy == py ) //exactly on a point
        {
          return true;
        }
        //half line [py,-oo) cut segment ?  
        if( sy < py ) {
          //take care: do not count twice segment end point (which is also end point for next/prev
          //segment, if it's the case count only for one of both point (choose lx,ly, so reject x,y)
          if( sy!=y || px!=x ) {
            if( sy==ly && px==lx ) {
              //but another care: do not count end point if prev/next segment of (lx,ly) are on
              //same part of <px or px>
              //i is next point X
              int llx = points_xy_[(i-6+length)%length];//pref (lx,ly) point x coordinate
              if( (x<px && llx>px) || (x>px && llx<px) ) {
                segment_count++;
              }
            } else {
              segment_count ++;
            }
          }
        }
      }
      
      lx=x; 
      ly=y;
    }
    //point is inside polygon if number of segment cut is not pair.
    return ((segment_count & 0x01) != 0);
  }
  
  public IRect getBounds()
  {
    if( size_==0 ) return new Rect();    
    int xn = points_xy_[0], xx=xn;
    int yn = points_xy_[1], yx=yn;
    int w=2;
    for( int i=1; i<size_; i++ )
    {
      int x = points_xy_[w++];
      int y = points_xy_[w++];
      if( x < xn ) xn=x; else if ( x> xx ) xx=x;
      if( y < yn ) yn=y; else if ( y> yx ) yx=y;
    }
    
    return new Rect( xn,yn, xx-xn, yx-yn );
  }

  protected IGCDirect gd_; //current gc used. available between startPath/endPath calls
  protected Segment path_segment_;
  protected int current_index_;
  
  public boolean pathBegin(IGC gc, IGCDirect gd) {
    if( size_==0 ) return false;
    gd_=gd;
    current_index_=-1;
    path_segment_ = new Segment();
    return true;
  }

  public void pathEnd() {
    gd_=null;    
    path_segment_=null;
  }

  public IPathElement nextPathElement() 
  {
    if( current_index_ >= size_ ) return null; //no more elements
    current_index_++;
    if( current_index_+1 >= size_ ) return null; //end reached.
    int ci = current_index_;
    path_segment_.setLine( gd_.devX(points_xy_[2*ci  ]), gd_.devY(points_xy_[2*ci+1]), 
                           gd_.devX(points_xy_[2*ci+2]), gd_.devY(points_xy_[2*ci+3]) ); 
    return path_segment_;
  }
  
  /** rotate all points with given angle (radian), (cx,cy) is center of rotation */
  public void rotate( int cx, int cy, double _angle )
  {
    double c = Math.cos(_angle);
    double s = Math.sin(_angle);
    for(int i=0,j=0; i<points_xy_.length;)
    {
      int px = points_xy_[i++] -cx ;
      int py = points_xy_[i++] -cy;
      int rx = (int)(  px*c + py*s );
      int ry = (int)( -px*s + py*c );
      points_xy_[j++] = cx+rx;
      points_xy_[j++] = cy+ry;
    } 
  }
 
  /** Change current polygon points !no copy of points, use reference to given array. */
  public void setPoints( int points[] )
  {
    points_xy_ = points;
    size_ = points==null ? 0 : points.length/2;
  }
  
  /** @return center of polygon */
  public Point getCenter( Point ctr )
  {
    return GetCenter( this, ctr );
  }
  
  /** @return center of polygon */
  public static Point GetCenter( IPolygon poly, Point ctr )
  {
    if( poly==null ) return null;
    int size = poly.getPolySize();
    if( size==0 ) return ctr;
    if( ctr==null ) ctr = new Point();
    int x=0, y=0;
    for( int i=0; i<size; i++ )
    {
      x += poly.getPolyX( i );
      y += poly.getPolyY( i );
    }
    x /= size;
    y /= size;
    ctr.setPoint( x, y );
    return ctr;
  }
  
  public static boolean Contains( IPolygon p, int px, int py )
  {
    int size = p.getPolySize();
    if( size < 3 ) return false; //must have at least 3 points    
    int last_ip = size-1;
    int lx = p.getPolyX( last_ip );
    int ly = p.getPolyY( last_ip );
    int segment_count=0;

    for( int i=0; i<size; i++  )
    {
      int x = p.getPolyX( i );
      int y = p.getPolyY( i );
      //vertical line py cut the segment [lx,ly]-[x,y] above py ?
      int xn,xx,yxn,yxx;
      if( lx < x ) //sort X
      {
        xn = lx; yxn = ly; 
        xx =  x; yxx =  y; 
      } else { 
        xn =  x; yxn =  y; 
        xx = lx; yxx = ly; 
      }
      if( px >= xn && px <= xx && ( py >= ((y<ly)?y:ly)))
      {
        if( xn == xx ) //vertical segment
        {
          if( py <= ((y<ly)?ly:y)) // point inside segment ?
          {
            return true;
          }
          return false;
        }
        float sy = yxn +(px-xn)*(yxx-yxn)/(xx-xn);
        if( sy == py ) //exactly on a point
        {
          return true;
        }
        //half line [py,-oo) cut segment ?  
        if( sy < py ) {
          //take care: do not count twice segment end point (which is also end point for next/prev
          //segment, if it's the case count only for one of both point (choose lx,ly, so reject x,y)
          if( sy!=y || px!=x ) {
            if( sy==ly && px==lx ) {
              //but another care: do not count end point if prev/next segment of (lx,ly) are on
              //same part of <px or px>
              //i is next point X
              int llx = p.getPolyX((i-2+size)%size );//pref (lx,ly) point x coordinate
              if( (x<px && llx>px) || (x>px && llx<px) ) {
                segment_count++;
              }
            } else {
              segment_count ++;
            }
          }
        }
      }
      
      lx=x; 
      ly=y;
    }
    //point is inside polygon if number of segment cut is not pair.
    return ((segment_count & 0x01) != 0);
  }
}
