/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RGBA.java,v 1.2 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

/**
 * Store RGBA color.<br>
 * RGBA is a 32 bits colors packed in a java int, including alpha channel,
 * all value are in range 0..255.<br>
 * Alpha of 0 means fully transparent, 255 is opaque.<br>
 * Red mask is 0xFF000000 (take care of sign shifting, please)<br>
 * Green mask is 0x00FF0000<br>
 * Blue mask is 0x0000FF00<br>
 * Alpha mask if 0x000000FF<br>
 *
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class RGBA 
{
  /* Define opaque colors */
  public static final int WHITE=0xFFFFFFFF;
  public static final int BLACK=0x000000FF;
  public static final int RED  =0xFF0000FF;
  public static final int GREEN=0x00FF00FF;
  public static final int BLUE =0x0000FFFF;
  public static final int YELLOW=0xFFFF00FF;
  public static final int CYAN  =0x00FFFFFF;
  public static final int MAGENTA=0xFF00FFFF;
  public static final int LIGHT_GRAY=0xC0C0C0FF;
  public static final int DARK_GRAY=0x808080FF;
  public static final int LIGHT_YELLOW=0xFFFFC0;
  public static final int ORANGE =0xFFA000FF;
  public static final int DARK_RED   =0x800000FF;
  public static final int DARK_GREEN =0x008000FF;
  public static final int DARK_BLUE  =0x000080FF;
  
  /** opaque value for A(lpha) field. */
  public static final int OPAQUE=255;
  /** transparent value for A(lpha), this is value for transparent color too. */
  public static final int TRANSPARENT=0;
  
  
  protected int r_,g_,b_,a_;
  
  /** Create opaque black */
  public RGBA()
  {
    a_=OPAQUE;
  }
  /** Create opaque RGB color */
  public RGBA( int _r, int _g, int _b )
  {
    r_=_r; g_=_g; b_=_b; a_=OPAQUE;
  }
  /** Create alpha'ed RGB colr */
  public RGBA( int _r, int _g, int _b, int _a )
  {
    r_=_r; g_=_g; b_=_b; a_=_a;
  }
  /** Create alpha'ed RGB color */
  public RGBA( int _rgba )
  {
    set(_rgba);
  }
  
  public boolean isOpaque() { return a_==OPAQUE; }
  public boolean isInvisible() { return a_==0; }
  
  public int getR() { return r_; }
  public int getG() { return g_; }
  public int getB() { return b_; }
  public int getA() { return a_; }
  
  public int get() 
  {
    return (r_<<24) | (g_<<16) | (b_<<8) | a_;
  }

  public void setR(int v) { r_=v; }
  public void setG(int v) { g_=v; }
  public void setB(int v) { b_=v; }
  public void setA(int v) { a_=v; }
  
  public void set( int _r, int _g, int _b, int _a )
  {
    r_=_r; g_=_g; b_=_b; a_=_a;
  }  
  
  public void set( int _rgba )
  {
    a_ = (_rgba&0xFF); _rgba>>=8;
    b_ = (_rgba&0xFF); _rgba>>=8;
    g_ = (_rgba&0xFF); _rgba>>=8;
    r_ = (_rgba&0xFF);
  }
  
  /** @return encoded RGBA color using OPAQUE as value for alpha channel */
  public static int Get( int _r, int _g, int _b )
  {
    return ((_r&0xFF)<<24) | ((_g&0xFF)<<16) | ((_b&0xFF)<<8) | OPAQUE;
  }
   
  /** @return encoded RGBA color */
  public static int Get( int _r, int _g, int _b, int _a )
  {
    return ((_r&0xFF)<<24) | ((_g&0xFF)<<16) | ((_b&0xFF)<<8) | (_a&0xFF);
  } 
  
  public static boolean IsOpaque( int _rgba )
  {
    return (_rgba&0xFF)==0xFF;
  }
  
  public static boolean IsInvisible( int _rgba )
  {
    return (_rgba&0xFF)==0x00;
  }
  
  /** @return red value of given RGBA color */
  public static int GetR( int _rgba )
  {
    return ((_rgba & 0xFF000000) >>24 )&0xFF; //a: due to sign bit copy by >>
  }
  /** @return green value of given RGBA color */
  public static int GetG( int _rgba )
  {
    return (_rgba & 0x00FF0000) >> 16 ;
  }
  /** @return blue value of given RGBA color */
  public static int GetB( int _rgba )
  {
    return (_rgba & 0x0000FF00) >> 8 ;
  }
  /** @return alpha value of given RGBA color */
  public static int GetA( int _rgba )
  {
    return _rgba & 0xFF ;
  }
  
  /**
   * @return combined color if _front is viewed in front of _back.
   * @param _front front color
   * @param _back back color
   * @return combined color.
   */
  public static int Combine( int _front, int _back )
  {
    int _af = (_front&0xFF); 
    if( _af== OPAQUE ) return _front;
    if( _af== 0   ) return _back;
    int _ab = (_back&0xFF); 
    if( _ab == 0 ) return _front ;

    _front >>=8;
    double af = (_af)/255.0;
    double bf = (_front&0xFF)/255.0; _front>>=8;
    double gf = (_front&0xFF)/255.0; _front>>=8;
    double rf = (_front&0xFF)/255.0;
    
    _back  >>=8;
    double ab = (_ab)/255.0;
    double bb = (_back&0xFF)/255.0; _back>>=8;
    double gb = (_back&0xFF)/255.0; _back>>=8;
    double rb = (_back&0xFF)/255.0;
    
    double k = (1.0-af)*ab;
    int r = (int)(255*( af*rf + k*rb ));
    int g = (int)(255*( af*gf + k*gb ));
    int b = (int)(255*( af*bf + k*bb ));
    int a = (int)(255*( af    + k    ));
    
/*if(r>255||g>255||b>255||a>255)
{
System.out.println( "B:"+Long.toString(_back&0xFFFFFFFF,16)+" F:"+Integer.toString(_front,16)+" => "+r+","+g+","+b+" a="+a);
}*/


    int rgba = (r<<24) | (g<<16) | (b<<8) | a;
//System.out.println( "B:"+Integer.toString(_back,16)+" F:"+Integer.toString(_front,16)+" => "+Integer.toString(rgba));
    return rgba;
  }
  
  
  /**
   * Return gradient color between _c1 and _c2 using _k value: if _k&lt;= 0 color is _c1;
   * if _k&gt;=1 color is _c2; otherwise compute linear approximation.
   * @param _rgba1 color for _k value less or equal to  0
   * @param _rgba2 color for _k value grater or equal to 1
   * @param _k value for linear approximation.
   */
  public static int Gradient( int _rgba1, int _rgba2, double _k )
  {
    return Gradient( _rgba1, _rgba2, _k, 0 );
  }
  
  /**
   * Return gradient color between _c1 and _c2 using _k value: if _k&lt;= 0 color is _c1;
   * if _k&gt;=1 color is _c2; otherwise compute linear approximation.
   * @param _rgba1 color for _k value less or equal to  0
   * @param _rgba2 color for _k value grater or equal to 1
   * @param _k value for linear approximation.
   * @param _ncolor if &gt;0 and &lt;256 reduce number of different color.
   */
private static int smp=0;  
  public static int Gradient( int _rgba1, int _rgba2, double _k, int _ncolors )
  {
    if( _k <=0.0 ) return _rgba1;
    if( _k >=1.0 ) return _rgba2;
    
    int a1 = (_rgba1&0xFF); _rgba1>>=8;
    int b1 = (_rgba1&0xFF); _rgba1>>=8;
    int g1 = (_rgba1&0xFF); _rgba1>>=8;
    int r1 = (_rgba1&0xFF);
    
    int a2 = (_rgba2&0xFF); _rgba2>>=8;
    int b2 = (_rgba2&0xFF); _rgba2>>=8;
    int g2 = (_rgba2&0xFF); _rgba2>>=8;
    int r2 = (_rgba2&0xFF);
    
    if( _ncolors > 0 && _ncolors < 256 )
    {
      if( _ncolors==1 )
      {
        _k=0.5; //by choice
      }
      else if ( _k!=0.0 && _k!=1.0 )
      {
        double range = 1.0/(double)(_ncolors);
        int cnt=0;
        for(  ;_k>=range; cnt++, _k-=range);
        _k = cnt*range;
        if( _k>1.0 ) _k=1.0;
      }
    }
    
    int r = (int)(r1 + (r2-r1)*_k);
    int g = (int)(g1 + (g2-g1)*_k);
    int b = (int)(b1 + (b2-b1)*_k);
    int a = (int)(a1 + (a2-a1)*_k);
    
    int rgba = (r<<24) | (g<<16) | (b<<8) | a ;
    return rgba;
  }
  
  /**
   * @return RGBA color applying k_shadow and k_light on _rgba parameter to create lighter and darker color,
   * then apply cos_angle value as a gradient coefficient.
   * @param _rgba base color to shadowing/lighting
   * @param k_shadow 0..1, 0 means shadow is _rgba, 1 shadow is _black
   * @param k_light 0..1, 0 means light is _rgba, 1 light is white
   * @param cos_angle 0..1, 0 to 0.5 return shadow color, 0.5..1 return light color
   */
  public static int Shadow( int _rgba, float k_shadow, float k_light, float cos_angle )
  {
    if( k_shadow<0 ) k_shadow=0.0f; else if (k_shadow>1 ) k_shadow=1.0f;
    if( cos_angle<0 ) cos_angle=0.0f; else if (cos_angle>1 ) cos_angle=1.0f;
    int r = GetR( _rgba );
    int g = GetG( _rgba );
    int b = GetB( _rgba );
    cos_angle = Math.abs(cos_angle);
    if( cos_angle < 0.5 )
    {
      //dark:
      int dr = (int)(k_shadow*0 + (1-k_shadow)*r);
      int dg = (int)(k_shadow*0 + (1-k_shadow)*g);
      int db = (int)(k_shadow*0 + (1-k_shadow)*b);
      cos_angle *=2;
      r = (int)( cos_angle*r + (1-cos_angle)*dr );
      g = (int)( cos_angle*g + (1-cos_angle)*dg );
      b = (int)( cos_angle*b + (1-cos_angle)*db );    
    } else {
      //light:
      int lr = (int)(k_light*255 + (1-k_light)*r);
      int lg = (int)(k_light*255 + (1-k_light)*g);
      int lb = (int)(k_light*255 + (1-k_light)*b);
      cos_angle -= 0.5;
      cos_angle *=2;      
      r = (int)( cos_angle*lr + (1-cos_angle)*r );
      g = (int)( cos_angle*lg + (1-cos_angle)*g );
      b = (int)( cos_angle*lb + (1-cos_angle)*b );
    }
    return RGBA.Get(  r,g,b,GetA(_rgba));
  }
  
  /** @return hexa decimal string representing RGBA fields "RRGGBBAA" */
  public static String Str( int _rgba )
  {
    return Long.toString( ((long)_rgba)&0xFFFFFFFFL, 16 );
  }
  
  /** transform current color to black and white preversing Alpha channel */
  public void toBW()
  {
    int bw = ( 299*r_ + 587*g_ + 114*b_ ) / 1000;
    r_ = bw;
    g_ = bw;
    b_ = bw;
  }
  
  /** Transform given RGBA color to a Black and White, preserving Alpha channel */
  public static int ToBW( int _rgba )
  {
    int r = (_rgba>>24)&0xFF;
    int g = (_rgba>>16)&0xFF;
    int b = (_rgba>> 8)&0xFF;
    int a = (_rgba    )&0xFF;
    int bw = ( 299*r + 587*g + 114*b ) / 1000;
    r = bw;
    g = bw;
    b = bw;
    return (r<<24) | (g<<16) | (b<<8) | a;
  }
  /** transform currnet color to negative black and white, presreve alpha channel */
  public void toNegativeBW()
  {
    int bw = 255-( 299*r_ + 587*g_ + 114*b_ ) / 1000;
    r_ = bw;
    g_ = bw;
    b_ = bw;
  }
  /** Transform given RGBA color to a Negative Black and White, preserving Alpha channel */
  public static int ToNegativeBW( int _rgba )
  {
    int r = (_rgba>>24)&0xFF;
    int g = (_rgba>>16)&0xFF;
    int b = (_rgba>> 8)&0xFF;
    int a = (_rgba    )&0xFF;
    int bw = 255-( 299*r + 587*g + 114*b ) / 1000;
    r = bw;
    g = bw;
    b = bw;
    return (r<<24) | (g<<16) | (b<<8) | a;
  }
  
  /** transform current color to negative one, preserve alpha channel */
  public void toNegative()
  {
    r_ = 255-r_;
    g_ = 255-g_;
    b_ = 255-b_;
  }
  /** Transform given color to negative one, preserve alpha channel */
  public static int ToNegative( int _rgba )
  {
    int r = (_rgba>>24)&0xFF;
    int g = (_rgba>>16)&0xFF;
    int b = (_rgba>> 8)&0xFF;
    int a = (_rgba    )&0xFF;
    r = 255-r;
    g = 255-g;
    b = 255-b;
    return (r<<24) | (g<<16) | (b<<8) | a;
  }
  
  /**
   * Change current color to a lighter one, using coefficient 0..1, transparency is preserved.
   * Value of 0 set same color, value of 1 set white.
   */
  public void lighter( float _k )
  {
    if( _k<0 ) _k=0.0f; else if (_k>1 ) _k=1.0f;
    r_ = (int)(r_ + _k*(255-r_));
    g_ = (int)(g_ + _k*(255-g_));
    b_ = (int)(b_ + _k*(255-b_));
  }
  
  /**
   * @return lighter color than given one, using coefficient 0..1, transparency is preserved.
   * Value of 0 return same color, value of 1 return white.
   */
  public static int Lighter( int _rgba, float _k )
  {
    if( _k<0 ) _k=0.0f; else if (_k>1 ) _k=1.0f;
    int r = GetR( _rgba );
    int g = GetG( _rgba );
    int b = GetB( _rgba );
    r = (int)(r + _k*(255-r));
    g = (int)(g + _k*(255-g));
    b = (int)(b + _k*(255-b));
    return RGBA.Get( r,g,b,GetA(_rgba));
  }
  
  /**
   * Change current color to a darker one, using coefficient 0..1.
   * 0 set same color, 1 set black color.
   */
  public void darker( int _clr, float _k )
  {
    if( _k<0 ) _k=0.0f; else if (_k>1 ) _k=1.0f;
    r_ = (int)( r_-_k*r_);
    g_ = (int)( g_-_k*g_);
    b_ = (int)( b_-_k*b_);
  }

  /**
   * @return darker color than given one, using coefficient 0..1.
   * 0 return same color, 1 return black color.
   */
  public static int Darker( int _rgba, float _k )
  {
    if( _k<0 ) _k=0.0f; else if (_k>1 ) _k=1.0f;
    int r = GetR( _rgba );
    int g = GetG( _rgba );
    int b = GetB( _rgba );
    r = (int)( r-_k*r);
    g = (int)( g-_k*g);
    b = (int)( b-_k*b);
    return RGBA.Get( r,g,b,GetA(_rgba));
  }

}
