/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RGBAImage.java,v 1.2 2008/05/23 14:11:54 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.igc.util.internal;

import org.eclipse.tptp.platform.report.igc.internal.IImage;


/**
 * Image stored using RGBA in an integer, use a one dimension array.
 * index of pixel in this array is y*width_+x;
 * 
 * @see RGBA
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class RGBAImage implements IImage
{
  protected int width_, height_;
  protected int image_[];
  
  /** Create an empty image */
  public RGBAImage( int width, int height )
  {
    width_ = width;
    height_= height;
    image_ = new int[width_*height_]; 
  }
  /** Create image using existing array (no copy of this array) */
  public RGBAImage( int width, int height, int []image )
  {
    width_=width;
    height_=height;
    image_=image;
  }
  
  /** @return the width of the image */
  public int getWidth() { return width_; }
  /** @return the height of the image */
  public int getHeight() { return height_; }
  /** @return pixel RGBA color */
  public int getPixel(int x, int y ) 
  {
    if( x<0 || y<0 || x>=width_ || y>=height_) return RGBA.TRANSPARENT;
    return image_[y*width_+x];
  }
  
  /** @return image RGBA buffer, index of point (x,y) is (y*width+h) */
  public int[] getImage() { return image_; }
  
  /** internal used to rotate image with antialiasing effect */
  protected int antialiased( double _rx, double _ry )
  {
    if( _rx < 0 ) _rx=0;
    int sx = (int)_rx;
    int sy = (int)_ry; 
    
    double dx = _rx-sx, dy = _ry-sy;
    double a1 = (1-dx)*(1-dy);
    double a2 = (1-dx)*dy; 
    double a3 = dx*dy;
    double a4 = dx*(1-dy);
    //double a4 = 1-a1-a2-a3;
    
    int index = sy*width_+sx;
    int rgba = image_[ index ]; //point(sx,sy)
    int r1 = RGBA.GetR( rgba );
    int g1 = RGBA.GetG( rgba );
    int b1 = RGBA.GetB( rgba );
//TODO: alpha is taken as opaque ... modify this ?    
    int al1 = RGBA.GetA( rgba );
    
    int r2=0,g2=0,b2=0,al2=0;
    int r3=0,g3=0,b3=0,al3=0;
    boolean incomplete=false;            
    if( sy+1 < height_)
    {
      rgba = image_[ index+width_ ]; // point(sx,sy+1)
      r2 = RGBA.GetR( rgba );
      g2 = RGBA.GetG( rgba );
      b2 = RGBA.GetB( rgba );
      al2 = RGBA.GetA( rgba );
      if( sx+1 < width_ )
      {
        rgba = image_[ index+width_+1 ]; //point( sx+1, sy+1 );
        r3 = RGBA.GetR( rgba );
        g3 = RGBA.GetG( rgba );
        b3 = RGBA.GetB( rgba );
        al3 = RGBA.GetA( rgba );
      } else {
        incomplete=true;
        a3=0.0;
      }
    } else { 
      incomplete=true;
      a2=a3=0.0; 
    }
    int r4=0,g4=0,b4=0,al4=0;            
    if( sx+1 < width_ )
    {
      rgba = image_[ index+1 ]; //point(sx+1,sy);
      r4 = RGBA.GetR( rgba );
      g4 = RGBA.GetG( rgba );
      b4 = RGBA.GetB( rgba );
      al4 = RGBA.GetA( rgba );
    } else { 
      incomplete=true;
      a4=0.0; 
    }
            
    if( incomplete ) //sum must be 1.0
    {
      double sum=(a1+a2+a3+a4);            
      double factor = 1.0/sum;
      a1 *= factor;
      a2 *= factor;
      a3 *= factor;
      a4 *= factor;
    }
//missing alpha ...    
    int R = (int)Math.round(a1*r1 +a2*r2 +a3*r3 +a4*r4 );
    int G = (int)Math.round(a1*g1 +a2*g2 +a3*g3 +a4*g4 );
    int B = (int)Math.round(a1*b1 +a2*b2 +a3*b3 +a4*b4 );
    int A = (int)Math.round(a1*al1 +al2*al2 +a3*al3 +a4*al4 );

//TODO: remove debug code here            
if( R<0||R>255 || G<0||G>255 || B<0||B>255 )//||incomplete
{
  System.out.println("BAD COLOR RGB="+R+","+G+","+B);
  System.out.println(" ry="+_ry+" sy="+sy);
  System.out.println(" rx="+_rx+" sx="+sx);
  System.out.println(" dx="+dx+" dy="+dy);
  System.out.println(" a1="+a1+" a2="+a2+" a3="+a3+" a4="+a4+" sum="+(a1+a2+a3+a4)+" incomplete="+incomplete);
  System.out.println(" r1="+r1+" r2="+r2+" r3="+r3+" r4="+r4);
  System.out.println(" g1="+r1+" g2="+r2+" g3="+r3+" g4="+r4);
  System.out.println(" b1="+r1+" b2="+r2+" b3="+r3+" b4="+r4);
}

     return RGBA.Get( R, G, B, A );
  }
  
  protected static int rotateX( double cos, double sin, int x, int y ) 
  {
    return (int)Math.round(  cos*x +sin*y );          
  }
  protected static int rotateY( double cos, double sin, int x, int y ) 
  {
    return (int)Math.round( -sin*x +cos*y );          
  }
  
  /**
   * Rotate this image using angle.
   * @param angle angle of rotation (<b>degree</b>)
   * @param new_pixel RGBA value of pixel created by rotation, they cannot come from source imag.
   * @param antialiased true if antialiasing must be apply during rotation (consume more time).
   * @return new rotated image.
   */
  protected RGBAImage rotateImageAny( int angle, int new_pixel, boolean antialiased )
  {
    double a = Radian.D2R( angle );
    double c = Math.cos(a);
    double s = Math.sin(a);
    final int X = width_-1;
    final int Y = height_-1;
    //rotate bounds:
    int rx0=rotateX(c,s,0,0), ry0=rotateY(c,s,0,0);
    int rx1=rotateX(c,s,X,0), ry1=rotateY(c,s,X,0);
    int rx2=rotateX(c,s,X,Y), ry2=rotateY(c,s,X,Y);
    int rx3=rotateX(c,s,0,Y), ry3=rotateY(c,s,0,Y);
    //bounds of destination image
    int rxmin = Math.min( rx0, Math.min( rx1, Math.min(rx2, rx3)));
    int rxmax = Math.max( rx0, Math.max( rx1, Math.max(rx2, rx3)));
    int rymin = Math.min( ry0, Math.min( ry1, Math.min(ry2, ry3)));
    int rymax = Math.max( ry0, Math.max( ry1, Math.max(ry2, ry3)));
//System.out.println("Rot bounds:"+rxmin+","+rymin+","+rxmax+","+rymax);          
    final int destX = rxmax-rxmin;
    final int destY = rymax-rymin;
    RGBAImage dest = new RGBAImage( destX+1, destY+1 );
    int dst[] = dest.image_;
//todo use fill polygon algorithm as many pixel are out of source image:
int dx=0, dy=0, ix=0, iy=0;      
try{          
    int y = 0+rymin;
    double sy = s*y, cy = c*y;
//idea compute fix,fiy in a incremental way ?    
    int iw=0;
    //scan rotated image, applying inverse rotation to get source pixel.
    for( dy=0; dy<=destY; dy++,y++,sy+=s,cy+=c )
    {
      int x = 0+rxmin;
      double cx = c*x, sx=s*x;
      for( dx=0; dx<=destX; dx++,x++,cx+=c,sx+=s,iw++ )
      {
        //apply inverse rotation:
        double fix = cx -sy;
        ix = (int)Math.round( fix );
        if( ix >=0 && ix <= X )
        {
          double fiy = +sx + cy ;
          iy = (int)Math.round( fiy );
          if( iy >= 0 && iy <= Y )
          {
            //in source image
            if( antialiased )
            {
              dst[iw] = antialiased( fix, fiy );
            } else {
              dst[iw] = image_[ iy*width_+ix ];
            }
          }
          else dst[iw] = new_pixel;
        } 
        else dst[iw] = new_pixel;              
      }
    }
}catch(IllegalArgumentException exx) {
  System.err.println("rotateImageDataAny/ ix="+ix+" iy="+iy+" dx="+dx+" dy="+dy+" np_index="+new_pixel+" image="+dest.width_+"x"+dest.height_);
  throw exx;
}
   return dest;
  }
  
  /** 
   * @return rotated size of (width,height) using angle (radian).
   * width, height is size at 0 radian. 
   */
  public static Size rotateImageSize( int width, int height, double angle )
  {
    int angle_degree = Radian.iR2D( angle );    
    switch( angle_degree%360 )
    {
      case 0: 
      case 180:
        return new Size( width, height );//easy
      case 90:
      case 270:
        return new Size( height, width );      
      default:
      {       
        double crot = Math.cos( angle );
        double srot = Math.sin( angle );
        final int X = width-1;
        final int Y = height-1;
        //rotate bounds (0,0,W-1,H-1)
        int rx0=rotateX(crot,srot, 0,0 ), ry0=rotateY(crot,srot, 0,0 );
        int rx1=rotateX(crot,srot, X,0 ), ry1=rotateY(crot,srot, X,0 );
        int rx2=rotateX(crot,srot, X,Y ), ry2=rotateY(crot,srot, X,Y );
        int rx3=rotateX(crot,srot, 0,Y ), ry3=rotateY(crot,srot, 0,Y );
        //get X/Y min/max for each rotated corner  
        int rxmin = Math.min( rx0, Math.min( rx1, Math.min(rx2, rx3)));
        int rxmax = Math.max( rx0, Math.max( rx1, Math.max(rx2, rx3)));
        int rymin = Math.min( ry0, Math.min( ry1, Math.min(ry2, ry3)));
        int rymax = Math.max( ry0, Math.max( ry1, Math.max(ry2, ry3)));        
        return new Size( rxmax-rxmin+1, rymax-rymin+1 );
      }
    }
  }
  
  /** 
   * @return rotated bounds of image (width x height) using angle (radian).
   * (x,y) is top left corner of image and rotation center.
   *  Returned polygon have 4 points, first is top left corner of text, second
   * if top right corner of text, third is bottom right and fourth is bottom left.
   * Bounds of polygon which is also size of rotated image can be retrieve using Polygon.getBounds()
   */
  public static Polygon rotateImageBounds( int x, int y, int width, int height, double angle )
  {
    int deg_angle = Radian.iR2D( Radian.normalize(angle));
    switch( deg_angle )
    {
    case 0:{
      int xx = x+width-1;
      int yx = y+height-1;
      return new Polygon( new int [] { x,y,  xx,y, xx,yx, x,yx } );
    }
    case 90:{
      int yt = y-width+1;      
      int xx = x+height-1;
      return new Polygon( new int[] { x,y, x,yt, xx,yt, xx,y } );
    }
    case 180: {
      int xl = x-width+1, yt = y-height+1;      
      return new Polygon( new int [] { x,y, xl,y, xl,yt, x,yt } );
    }
    case 270:{
      int xl = x-height+1;
      int yx = y+width-1;
      return new Polygon( new int [] { x,y, x,yx, xl,yx, xl,y } );
    }
    default:{
      int xx = x+width-1;
      int yx = y+height-1;
      int xy[] = new int [] { x,y,  xx,y, xx,yx, x,yx };
      Polygon p = new Polygon( xy );
      p.rotate( x,y, angle );
      return p;
    }
    }
  }
 
  /** 
   * rotate this image using angle (<b>radian</b>), and antialiased property
   * @return a new RGBA Image, except for 0 radian angle, return this image.
   */
  public RGBAImage rotateImage( double angle, int new_pixel_rgba, boolean antialiased )
  {
    int deg_angle = Radian.iR2D( Radian.normalize(angle) );
    switch( deg_angle%360 )
     {
      case 0: return this;//easy
      case 90:
      {
        RGBAImage r = new RGBAImage( height_, width_ );
        int dst[] = r.image_;
        int dx=0;        
        int ir=0;
        for( int sy=0; sy<height_; sy++,dx++ )
        {
          int iw=(r.height_-1)*r.width_+dx;
          for( int sx=0; sx<width_; sx++,iw-=r.width_,ir++ )
          {
            dst[ iw ] = image_[ir];
          }
        }  
        return r;
      }
      case 270: 
      {        
        //rotate image 
        RGBAImage r = new RGBAImage( height_, width_ );
        int [] dst = r.getImage();
        int dst_width = r.width_;
        int dx=height_-1;
        int ir=0;
        for( int sy=0; sy<height_; sy++,dx-- )
        {
          int iw = dx;
          for( int sx=0; sx<width_; sx++, iw+=dst_width )
          {
            dst[ iw ] = image_[ ir++ ];
          }
        }  
        return r;
      }      
      case 180:
      {
        RGBAImage r = new RGBAImage( width_, height_ );
        int dst[] =r.image_;
        int dy=height_-1;
        int size=width_*height_;
        int iw= size-1;
        for( int ir=0; ir<size; ir++,iw--)
        {
          dst[ iw ] = image_[ ir ];
        }        
        return r;
      }
      default:
        return rotateImageAny( deg_angle, new_pixel_rgba, antialiased );
     }
  }
  
  
}
