/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Signal.java,v 1.3 2008/05/23 14:12:06 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 25 mai 2003
 *
 */
package org.eclipse.tptp.platform.report.signals.internal;

import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Vector;

/**
 * A Signal is made to prevent a variety of java's method at one time:
 * when the signal is <b>emitted</b>.
 * All the methods called (named 'slot' or 'receiver' must be <b>connected</b> (ie: registered) to
 * the signals.
 * <br><br>
 * A Signal have a <b>prototype</b>: at emit time an array of arguments
 * is provided. To be connected method must handle from zero to all
 * arguments class.
 * For documentation purpose and used to display messages, a message
 * have a name. It's recommended this signal's name hold the signals prototype.
 * <br><br>
 * Signature are built using methods name and list of parameters class:
 * "aMethod(int,MyClass,java.lang.String)"
 * All class from package "java.lang.XXXX" can be printed as XXXX (ie String,Object)
 * <br><br>
 * Note for <b>SWT's widget</b>, a dispose listener is added to widget at connection
 * time to disconnect it automatically at dispose time.
 * <br><br>
 * Example defining a signal:
 * <pre>
 * public class MyClass 
 * {
 *   //the signal name holds signature
 *   public Signal a_signal = new Signal("valueChanged(int,String)");
 * 
 *   public void setValue( int _index, String _v )
 *   {
 *     //store and process value....
 *     //emit signal, first pack parameters
 *     // if params have one dimension use emit(Object)
 *     Object[] params = { new Integer(_index), _v };
 *     //second emit signals
 *     a_signal.emit( params );
 *   }
 * }
 * </pre>
 * <P>
 * Example defining a recevier (a fully normal class):
 * <pre>
 * public class AnotherClass
 * {
 *   public void aMethod( int _a1, String _a1 )
 *   {
 *     // ...
 *   }
 *   public void updateData()
 *   {
 *     // ...
 *   }
 * }
 * </pre>
 * <p>
 * Example connecting Signal to slots:
 * <pre>
 * public class AClass
 * {
 *   protected MyClass my_inst ;
 *   protected AnotherClass have_slots;
 * 
 *   public AClass()
 *   {
 *     my_inst = new MyClass();
 *     have_slots = new AnotherClass();
 *     //connecting first method (signature fully compliant)
 *     my_inst.a_signal.connect( have_slots, "aMethod(int,java.lang.String)" );
 *     //connecting an onther method ()
 *     my_inst.a_signal.connect( have_slots, "updateData()");
 * 
 *     //now each time signal is emitted, aMethod() and updateData() will be called
 *   }
 * }
 * </pre>
 * @author ademuyser
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 *
 */
public class Signal
{
  /** signals name */
  private String name_;
  /** list of connected methods */ 
  private Vector receivers_;
  /** true if call to emit() does nothing */
  private boolean blocked_ = false;
  /** true is all signals will be blocked */
  private static boolean s_blocked_ = false;

  
  /** 
   * Create a signal.
   * It's recommended to use as the name of the signal the full signature
   * this signal can emit, and a unic understandable name.
   * Because name will be printed out each time error occurs on signal.
   * example:
   *   public final Signal sgn_valueChanged = new Signal("valueChanged(String)");
   */
  public Signal( String _name )
  {
    name_=_name;
  }

  /**
   * Analyse _str to extract a list of String, [0] is the method name
   * all other indexes are arguments.
   * ie: _str="aMethod(int,java.lang.String)"
   * return { "aMathod", "int", "java.lang.String" }
   */
  static protected String[] DecompileSignature( String _str )
  {
//System.out.println("Decompile("+_str+")");      
    if( _str==null ) return null;
    int len=_str.length();
    if( len==0 ) return null;
    try
    {
      int i = 0;
      char c = _str.charAt(i);
      //skip space
      for (; Character.isSpaceChar(c); ++i, c = _str.charAt(i));
      if (!Character.isJavaIdentifierStart(c))
      {
        System.err.println("method name doesn't start with a valid java identifier character");
        return null;
      }
      int j = i + 1;
      c = _str.charAt(j);
      for (; Character.isJavaIdentifierPart(c); ++j, c=_str.charAt(j));
      String mth_name = _str.substring(i, j);
      i = j;
      //skip space
      for (;Character.isSpaceChar(c); ++i, c = _str.charAt(i) );
      //must have '('
      if (c != '(')
      {
        System.err.println("missing '(' in signature");
        return null;
      }
      ++i;
      c = _str.charAt(i);

      Vector v = new Vector();
      v.add(mth_name);

      //parse arguments...
      for (;;)
      {
        //skip space
        for (; Character.isSpaceChar(c); ++i, c = _str.charAt(i));
        if (c == ')')
          break;
        //class name or arg#i
        j = i + 1;
        c = _str.charAt(j);
        for (;Character.isJavaIdentifierPart(c) || (c == '.') || (c == '$'); ++j, c = _str.charAt(j));
        String arg_class = _str.substring(i, j);
        v.add(arg_class);
        i = j;
        //skip space
        for (; Character.isSpaceChar(c); ++i, c = _str.charAt(i));
        if (c == ',') {	i++; c = _str.charAt(i); continue; } 
        if (c == ')') break; //end

        System.err.println("Unexpected character '" + c + "' in signature");
        return null;
      }

      String r[] = new String[v.size()];
      for (i = 0; i < v.size(); ++i)
      {
        r[i] = (String)v.get(i);
      }
      return r;
    }
    catch (StringIndexOutOfBoundsException e)
    {
      //might thrown in case of malformed signature (truncated)
      return null;
    }
/* * * * * * Same thing using Pattern and Matcher classes ... require jdk 1.4.0   
 *   Pattern p_name = Pattern.compile("\\s*(\\w+)\\s*\\(");
 *   Pattern p_arg  = Pattern.compile("\\s*([\\$\\.\\w]+)\\s*([,)])");
 *   Matcher  m = p_name.matcher( _str );
 *   int matched_len=0;
 *   Vector v=new Vector();
 *   String mth_name=null;
 *   if( m.find() )
 *   {
 *     mth_name = m.group(1);
 *     v.add( mth_name );
 *     matched_len = m.group(0).length();
 *   }
 *   String rest = _str.substring( matched_len );
 *   m = p_arg.matcher( rest );
 *   int i=0;
 *   while( m.find() )
 *   {
 *     String arg = m.group(1);
 *     v.add( arg );
 *   }
 *   String r[] = new String[v.size()];
 *   for( i=0; i<v.size(); ++i )
 *   {
 *     r[i] = (String)v.get(i);
 *   }
 *   return r;
 * * */    
  }
  
    
    /** 
     * @return true if signal contains one or more receivers
     */
    public boolean haveReceiver()
    {
      return receivers_ != null && receivers_.size() > 0;
    }
    
    /**
     * @return true if signal is blocked at instance level 
     */
    public boolean isBlocked()
    {
      return blocked_;
    }
    
    /** @return true if all signals are blocked */
    public static boolean isAllBlocked()
    {
      return s_blocked_;
    }
    
    /** Change blocked status for this signal */
    public void setBlocked( boolean _b )
    {
      blocked_=_b;
    }
    
    /** Change global blocked status */
    public static void setAllBlocked( boolean _b )
    {
      s_blocked_=_b;
    }

    public static interface IConnectionFactory
    {
      public Connection createConnection( Object _o, Method _m, String _method_sgn );
    }
    protected static IConnectionFactory connection_factory_;
    protected static Connection createConnection( Object _o, Method _m, String _method_sgn )
    {
      if( connection_factory_==null ) {
        return new Connection( _o, _m, _method_sgn );
      } else {
        return connection_factory_.createConnection( _o, _m, _method_sgn );
      }
    }
    static
    {
      //try to load SWTConnectionFactory this way as Signal class MUST NOT depend
      //on any SWT's classes 
      try {
        //first try to load DisposeListener SWT class to test if SWT classes are there...
        Class clt = Class.forName("org.eclipse.swt.events.DisposeListener");
        //if ok, try to load the SWT factory for connections ...
        Class cls = Class.forName("org.eclipse.tptp.platform.report.signals.internal.SWTConnectionFactory");
        connection_factory_ = (IConnectionFactory)cls.newInstance();
      } catch( Exception e ) {
        //not an error: e.printStackTrace();
      }
    }

    // internal use only
    protected static class Connection
    {
      public WeakReference r_receiver_;
      public Method method_;
      public String method_sgn_;
      public Connection( Object _o, Method _m, String _method_sgn )
      {
        r_receiver_ = new WeakReference(_o);
        method_ = _m;
        method_sgn_=_method_sgn;
      }
    }
    
    /** Used when an error occurs... */
    public static class MethodNotFoundException extends Exception {
      /**
         * 
         */
        private static final long serialVersionUID = 1L;
    public MethodNotFoundException( String _mth, Class _class) {
        super(
          "method '"
            + _mth
            + "' not found in '"
            + _class.getName()
            + "'");
      }
      public MethodNotFoundException(String _m) {
        super(_m);
      }
    }
    
    /**
     * Return Class where the method is searched in...
     * if _receiver_class is null, return _receiver if is an instance of Class,
     * of receiver.getClass().
     * So _receiver and _receiver_class cannot be null at same time.
     * 
     * @return Class instance to search method.
     */
    protected Class getClassOf( Object _receiver, Class _receiver_class )
    {
      Class co = _receiver_class;
      if( co == null )
      {
        //receiver is already a class (connection to a static method)
        if( _receiver instanceof Class )
        {
          co = (Class)_receiver ;
        }
        else
        {
          co = _receiver.getClass();
        }
      }
      return co;
    }
    
    /**
     * @return method of _receiver as signature ("name(int,java.lang.String)" described.
     *   null if no signature found
     */
    protected Method getMethodFromSignature( Object _receiver, Class _class, String _signature )
    {
      Class co = getClassOf( _receiver, _class );
//System.out.println("getMethodFromSignature receiver's class is "+co);      
      String s[] = DecompileSignature( _signature );
      if( s==null || s.length==0 )
      {
        System.err.println("Signal: Can't decompile signature '"+_signature+"' for receiver '"+_receiver.getClass().getName()+"'");
        return null;
      }      
      Method[] ms = co.getMethods();
      /*try*/ {
        for (int i = 0; i < ms.length; ++i)
        {
          Method m = ms[i];
//System.out.println("connect/checking method '"+m.getName()+"'");   
          //compare method's names
          if ( s[0].equals(m.getName()))
          {
//System.out.println(" - same names");
            Class arg_type[] = m.getParameterTypes();
            //compare numbers of parameters
            boolean match=(s.length-1)==arg_type.length;
//System.out.println(" checking '"+s[0]+"'");                        
//System.out.println(" s.length="+s.length+" at.l="+arg_type.length);            
            if( match )
            {
              //compare parameters types
              for( int ii=0; ii<arg_type.length; ++ii )
              {
                String a = arg_type[ii].getName();
//System.out.println(" checking '"+s[0]+"' arg["+ii+"]="+a);
                if( !a.equals( s[ii+1]) )
                {
                  //second try, add "java.lang." as prefix..
                  String java_lang_s = "java.lang."+s[ii+1];
                  if( !java_lang_s.equals( a ) )
                  {
                    match=false;
                    break;
                  }
                }
              }
            }           
            if (match) //isCompatible(m)) {
            {
              return m;
            }
          }
        }
      }
     
      //method not found
      return null;
    }

    /** 
     * Connect a receiver to the signal, check is a given method name exists.
     * Remember receiver's method must be accessible from Signal class (ie: public).
     * To connect to a static method of a class set _receiver as MyClass.class.
     * 
     * @param _receiver the objet which will receive the call
     * @param _receiver_class the class used to search method, if null receiver.getClass()
     *        is used (or receiver directly if it's an instanceof Class).
     * @param _method_name method name and signature of _receiver to call
     * @throws MethodNotFoundException if the method or the @param _method_name doesn't exist
     */
    public void connectChecked(Object _receiver, Class _receiver_class, String _method_sgn)
      throws MethodNotFoundException
   {
      Method m = getMethodFromSignature( _receiver, _receiver_class, _method_sgn );
      
      if( m!=null )
      {        
         if (receivers_ == null) receivers_ = new Vector();
         receivers_.add( createConnection(_receiver, m, _method_sgn ) );
      }
      else
      {
        throw new MethodNotFoundException(
          "Method '"
          + _method_sgn
          + "' not found in '"
          + getClassOf( _receiver, _receiver_class )
          + "'");
      }
    }
    
    /** 
     * Check connect to object _receiver, using receiver (if instance of Class)
     * or receiver's class.
     * This method is redefined to facilitate usage of Signal class.
     * @see #connectChecked( Object, Class, String )
     */
    public final void connectChecked( Object _receiver, String _method_sgn )
          throws MethodNotFoundException
    {
       connectChecked( _receiver, null, _method_sgn );
    }

    /**
     * Add a receiver, if the connection failed, a message is printed to System.err,
     * no exception thrown.
     * Remember receiver's method must be accessible from Signal class (ie: public).
     * To connect signal to a static method, pass MyClass.class as _receiver parameter.
     * If method is not visible in _receiver (ie inner anonymous class), you can
     * set _class with a visible interface or class.
     */
    public void connect(Object _receiver, Class _class,  String _method_name)
    {
      try {
        connectChecked( _receiver, _class, _method_name);
      } catch (Signal.MethodNotFoundException _e) {
        System.err.println(_e.getMessage());
      }
    }
    
    /**
     * Redefined to facilitate Signal usage.
     * same as connect( _receiver, null, _method_name )
     */
    public final void connect( Object _receiver, String _method_name )
    {
      connect( _receiver, null, _method_name );
    }
    
    /** 
     * Connect a signal to this signals.
     * When this signals will be emitted, connected signals will emit too.
     */
    public void connect( Signal _signal )
    {
      if( _signal==null ) return ;
      if (receivers_ == null) receivers_ = new Vector();
      receivers_.add( _signal );
    }
    
    /**
     * Disconnect a connected signal.
     */
    public void disconnect( Signal _signal )
    {
      if( _signal==null ) return ;
      if( receivers_ == null ) return ;
      receivers_.remove( _signal );
    }

    /**
     * Disconnect a receiver from the signal, all the connections defined
     * for this receiver will be removed (@see disconnect#2) 
     */
    public void disconnect( Object _receiver )
    {
      if( _receiver==null ) return ;
      int n_disconnect=0;
      if (receivers_ != null) 
      {
        for (int i = 0; i < receivers_.size();)
        {
          Object obj = receivers_.get(i);
          if( !(obj instanceof Connection) ) { i++; continue; }
          Connection p = (Connection)obj;
          Object r = p.r_receiver_.get();
          if (r == null) { //clean up lost weak references
            receivers_.remove(i);
          }
          else if (r == _receiver)
          {
            receivers_.remove(i);
            n_disconnect++;
          }
          else
          {
            i++;
          }
        }
      }
      if( n_disconnect==0 )
      {
        System.err.println("Signal:warning: receiver '"+_receiver.getClass().getName()+"' not disconnected, no connection found");
      }
    }
    
    /**
     * Disconnect a particular method for a given receiver
     */
    public void disconnect( Object _receiver, String _signature )
    {
      if( _receiver==null ) return ;
      Method m = getMethodFromSignature( _receiver, null, _signature );
      if( m==null )
      {
        System.out.println("Signal:warning: disconnect failed can't decompile signature '"+_signature+"'");
        return ;
      }
      int n_disconnect=0;
      if (receivers_ != null) 
      {
        for (int i = 0; i < receivers_.size();)
        {
          Object obj = receivers_.get(i);
          if( !(obj instanceof Connection)) { ++i; continue; }
          Connection p = (Connection)obj;
          Object r = p.r_receiver_.get();
          if (r == null) { //clean up removed weak reference
            receivers_.remove(i);
          }
          else if (r == _receiver && p.method_==m )
          {
            receivers_.remove(i);
            n_disconnect++;
          }
          else
          {
            i++;
          }
        }
      }
      if( n_disconnect==0 )
      {
        System.err.println("Signal:warning: receiver '"+_receiver.getClass().getName()+"' not disconnected, no connection found");
      }
    }

    /**
     * Clear the signal's list of receiver
     */
    public void clear()
    {
      if (receivers_ == null)
        return;
      receivers_.clear();
      receivers_ = null;
    }

    /** 
     * Emit signal, all receiver's connected method will be called.
     * NB: if signal is blocked or all signal are blocked, emit does nothing.
     * @param _parameters actual parameters for the call
     */
    public void emit(Object[] _parameters)
    {
      emitSignal( _parameters );
    }
    
    /**
     * Redefined for usability purpose, strictly equals to
     * <code>Object pm[] = { _param };
     * emit( pm );
     * </code>
     * Usefull when signal have only one parameter.
     * @param _parameters s 
     */
    public final void emit( Object _parameter )
    {
      Object pm[] = { _parameter };
      emitSignal( pm );
    }
    
    /**
     * Redefined or usability purpose, emit signal witout parameter.
     */
    public final void emit()
    {
      emitSignal( null );
    }
    
    /**
     * derivative Signal's class can use this to send signals.
     * @see
     */
    protected void emitSignal( Object[] _parameters )
    {
      if (receivers_ == null) return;
      if (blocked_) return;
      if (s_blocked_) return;
      int parameters_length = _parameters==null ? 0 : _parameters.length;
      for (int i = 0; i < receivers_.size(); ++i)
      {
        Object obj = receivers_.get(i);
        if( obj instanceof Signal )
        {
          Signal sgn = (Signal)obj;
          sgn.emit( _parameters );
          continue;
        }

        Connection pair = (Connection)obj;
        Object receiver = pair.r_receiver_.get();
        if (receiver == null) {
          receivers_.remove(i);
          --i;
          continue;
        }
        Method mth = pair.method_;
        Class  args[] = mth.getParameterTypes();

        Object param[] = null;
        //same parameters count
        if( args.length== parameters_length )
        {
          param = _parameters ;
        }
        //parameters count is reduced
        else if ( args.length < parameters_length )
        {
          param = new Object[args.length];
          for( int a=0; a<args.length; ++a )
          {
            param[a] = _parameters[a];
          }
        }
        invokeConnection( receiver, mth, param, pair.method_sgn_ );
      }
    }

    /** 
     * Derivative Signal's class can reimplements this method to process connected
     * method invokation.
     * @param _receiver object for method invocation.
     * @param _method method to call.
     * @param _params parameters suitable for this _method
     */
    protected void invokeConnection( Object _receiver, Method _method, Object []_params, String _method_signature )
    {
        try {
          _method.invoke( _receiver, _params );
        }
        catch (InvocationTargetException e)
        {
          System.err.println("Signal["+name_+"]: emit catch InvocationTargetException : " + e.getCause().getMessage());
          System.err.println("      receiver class is '"+_receiver.getClass().getName()+"'");
          if( e.getCause()!=null )
            e.getCause().printStackTrace();
        }
        catch (IllegalAccessException _e)
        {
          System.err.print("signal["+name_+"]: emit catch IllegalAccessException : " );
          if( _e.getCause()!=null)
          {
            System.err.println( _e.getCause() );
          } else {
            System.err.println( _e.getMessage() );
          }
          System.err.println("  receiver '"+_receiver.getClass().getName()+"."+_method_signature+"'");
          if( _e.getCause()!=null )
            _e.getCause().printStackTrace();
        }

    }
}
