/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractIImageProvider.java,v 1.4 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.tools.internal;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.*;

import org.eclipse.tptp.platform.report.igc.internal.IImage;


/**
 *
 * Implementation of IDIImageProvider definiting common behaviour for image search path
 * (url, class path for image resources ...).
 * This class must be subclassed to create IImage for the underlaygin graphic system
 * with "real images" (ie: SWT or AWT, ... ).
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public abstract class AbstractIImageProvider implements IDIImageProvider
{
   /** current broken image, can be null before first getBrokenImage() call */
   protected IImage brokenImage;
   /** search path for images in file system */
   private ArrayList search_path;
   /** classes for resources as stream search */
   private ArrayList classes_resources;
   /** current image loaded, can also contains several brokenImage */
   private Hashtable imagePool;

 
   /** 
    * Create and return the broken image, must not return null image in any way
    * (at least create in memory image, a red tiny rectangle for example). 
    */
   protected abstract IImage createBrokenImage();
   /** @return new IImage for the given file name, or null if image can't be loaded.
    * This method must never return broken image. */
   protected abstract IImage createImageFromFile( String filename );
   /** @return new IImage for the given input stream, or null if image can't be loaded.
    * This method must never return broken image. */
   protected abstract IImage createImageFromInputStream( InputStream is );
   /** called to dispose any resource system associated with image */
   protected abstract void disposeImage( IImage image );

   /**
    * @return the image used when other image aren't found.
    */
   public IImage getBrokenImage() 
   {
     if( brokenImage!=null ) return brokenImage;     
     //create broken image.
     brokenImage = createBrokenImage();
     return brokenImage;
   }
   
   

   /**
     * Change current broken image, disposing current one before. After this call broken imag
     * is under the control of image provider, do not dispose it, image provide will do.
    */
   public void setBrokenImage(IImage image)
   {
     brokenImage = image;
   }

   

   /**
    * @return image searched into current search path list, if not found return null.
    * You have to dispose returned image.
    */
   private IImage getImageFromSearchPath( String name)
   {
     IImage img;
     String imgfilename = searchImage(name);
     if (imgfilename != null)
     {
       img = createImageFromFile( imgfilename );
       return img;
     } else {
       return null;
     }
   }
  
   /**
    * @return image from an url, or broken image if nor found.
    */
   private IImage getImageViaHttp( URL url)
   {
     InputStream is;
     try {
       is = url.openStream();
       IImage img = createImageFromInputStream( is );
       is.close();
       if (img != null) {
         return img;
       }
     } catch (Exception e) {
       e.printStackTrace();
     }
     return getBrokenImage();
   }
 
   /**
    * @return image from classes resources, or null if not found.
    */
   private IImage getImageFromClassResources( String res_name ) 
   {
     if( classes_resources==null ) return null;
     for( Iterator it=classes_resources.iterator(); it.hasNext(); )
     {
       Class clazz = (Class)it.next();
       try {
          return createImageFromInputStream( clazz.getResourceAsStream( res_name ) );
       } 
       catch (Throwable t) {}
     }
                
     return null;
   }


   /**
    * Return image, first look into current image pool, then try to get image from an url, or
    * try from search path, file system, in the worst case return the broken image. This means you don't have
    * to <b>dispose</b> returned image, but the creator of DImageProvider must do this calling
    * IDIImageProvider.dispose().
    * If animage is found, put it in current image pool, next time it will be faster...
    */
   public IImage getImage( String name )
   {
     //-0 already have this image ?     
     IImage img = imagePool!=null ? (IImage) imagePool.get(name) : null;
     if (img != null) return img;

     //-1 try from an url
     try {
       URL url = new URL(name);
//TODO: why only "http", why not "file:///c:/tmp/foo.gif" this is a valid url too !       
//       if (url.getProtocol().equals("http"))
//       {         
         InputStream is = url.openStream();
         try{
           img = createImageFromInputStream( is );
         } 
         finally
         {
           is.close();
         }
//       } else {
         // try a "file:" protocol
//         img = createImageFromFile( url.getPath() );
//       }
     }
     catch (Exception e) 
     {       
       //not an url, IOException, FileNotFoundException and other (bad) reason...
     } 
     
     //-2 get image from classes resources:
     if( img==null )
     {
       img = getImageFromClassResources( name );
     }
     
     //-3 get image from search path ...
     if( img == null )
     {
       String real_name = searchImage( name );
       if( real_name !=null )
       {
         img = createImageFromFile( real_name );
       }
     }
     
     //-4 get image directly in file system
     if( img == null )
     {
       try
       {
         img = createImageFromFile( name );         
       }
       catch( Exception e )
       {
         //losts of reason to pass here, but none provide expected image ;-(.
       }
     }
     
     //-X Ok, use broken image.
     if( img == null )
     {
       img = getBrokenImage();
     }

     //remember image for next time ... 
     if( imagePool==null ) imagePool = new Hashtable();
     imagePool.put(name, img);

     return img;
   }   
   
   /**
    * @return Image currently stored in image pool, remove this image from pool.
    * After this the caller have the responsability to <b>dispose</b> taken image.
    * (as image is no longer in image pool image provider can't dispose it).
    * Return null if image isn't in pool...
    */
   public IImage takeImage( String name )
   {
     if( name==null || imagePool==null ) return null;
     IImage img = (IImage)imagePool.remove( name );
     return img;
   }
   
   /**
    * @return real file name for given image name using search path list, file must exist on file system.
    */
   public String searchImage( String ip )
   {
      if( search_path== null ) return null;

      String retval = null;
      for (int i = 0; i<search_path.size(); i++)
      {
        String si = search_path.get(i) + File.separator + ip;
        File f = new File(si);
        if (f.exists())
        {
          retval = si;
          break;
        }
      }
      return retval;
   }
 
   /**
    * Dispose all image resource (including broken image) held by this image provider,
    * do not forget to call this method else you'll become under hand of the "no more handle" crash...
    */
   public void dispose()
   {
     if( imagePool!=null )
     {
       for( Enumeration e=imagePool.elements(); e.hasMoreElements(); )
       {
         IImage img = (IImage)e.nextElement();
         if( img!=null && img!=brokenImage )
         {
           disposeImage( img );
           img=null; 
         }
       }
       imagePool.clear();
       imagePool = null;
     }

     if( brokenImage !=null )
     {
       disposeImage( brokenImage );
       brokenImage=null;
     }
     
   }
   
   /**
    * Clear this object, do a full dispose of contained images, and clear search path and resources classes.
    */
   public void clear()
   {
     dispose();
     search_path.clear();        search_path = null;
     classes_resources.clear();  classes_resources = null;
   }
   
   /**
    * Change current search path for images, used to get image in file system.
    * set paths to null, clear current search path.
    */
   public void setImageSearchPath( String[] paths)
   {
      if( search_path==null ) search_path = new ArrayList();
      search_path.clear();
      if (paths != null )
      {
         for (int i = 0; i<paths.length; i++)
          addImageSearchPath( paths[i] );
      }  
   }
   
   /**
    * Clear current search path.
    */
   public void clearImageSearchPath()
   {
     search_path.clear();
     search_path = null;
   }
   
   /**
    * Add a search path for images for search in file system, only if search path isn't already in.
    */   
   public void addImageSearchPath( String path )
   {
      if (path ==null) return;  
      if( search_path==null ) 
      {
        search_path = new ArrayList();
      }
      else
      {
        for( Iterator it=search_path.iterator(); it.hasNext(); )
        {
          String p = (String)it.next();
          if( path.equals( p ) ) return ; //already in.
        }
      }
      search_path.add(path);  
   }

   /**
    * Remove a search path for images for search in file system.
    */   
   public void removeImageSearchPath( String path )
   {
      if (path ==null  || search_path == null ) return;      
      search_path.remove(path);  
   }
   
   /**
    * Add a new class to get image from it's resources (do not include if class i already in)
    * @see java.lang.Class#getResourceAsStream
    */
   public void addResourceClass( Class c )
   {
     if( c==null ) return ;
     if( classes_resources == null )
     {
       classes_resources = new ArrayList();
     }
     else
     {
       for( Iterator it=classes_resources.iterator(); it.hasNext(); )
       {
         Class clazz = (Class)it.next();
         if( c==clazz ) return ; //already
       }
     }
     classes_resources.add( c );
   }
   
   /**
    * Remove a class from the resources classes.
    * @return true if class was removed, false otherwise.
    */
   public boolean removeResourceClass( Class c )
   {
     if( classes_resources==null ) return false ;
     return classes_resources.remove( c );
   }
   
   /**
    * @see org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider#setImage(java.lang.String, org.eclipse.tptp.platform.report.igc.internal.IImage)
    */
   public void setImage(String name, IImage image) 
   {
       imagePool.put(name, image);
   }   

}
 
   