/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DChartCreator.java,v 1.3 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 15 janv. 2004
 *
 */
package org.eclipse.tptp.platform.report.tools.internal;

import org.eclipse.tptp.platform.report.chart.internal.DXYSurface;
import org.eclipse.tptp.platform.report.chart.internal.IXYSurfaceColorizer;
import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drawutil.internal.DSymbolRegistry;


/**
 * This class provide methods that helps you to create rapidly simple chart.
 * It's recommended to take a look/learn about JScrib's graphic, reading 
 * thoses classes: DGraphic, DAxis, DCurve, DPoint and DCoord.
 * 
 * @author ademuyser
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DChartCreator
{
  /**
   * Create an one-series 3D Histogram from values using default colors.
   * @param title Title of the graphic.
   * @param values datas used to create graphic.
   */  
  public static DGraphic Histogram( String title, double[] values )
  {
    return Histogram( title, values, true, false, Palette.PAL_DEFAULT );
  }

  /**
   * Create an one-series Histogram from values using default colors.
   * @param title Title of the graphic.
   * @param values datas used to create graphic.
   * @param _3d true to use 3D histogram.
   */  
  public static DGraphic Histogram( String title, double[] values, boolean _3d, boolean _thinbar )
  {
    return Histogram( title, values, _3d, _thinbar, Palette.PAL_DEFAULT );
  }
  
  /**
   * Create an one-series Histogram from values.
   * @param title Title of the graphic.
   * @param values datas used to create graphic.
   * @param _3d true to use 3D histogram.
   * @param _thinbar true to use thin bar.
   * @param palette a color generator called for each bar of histogram.
   */  
  public static DGraphic Histogram( String title, double[] values, boolean _3d, boolean _thinbar, IPalette palette )
  {
    DGraphic g= OneAxisChart( DGraphic.T_HISTOGRAM, title, null, new double[][]{values}, palette );
    if(_3d) g.getProperties().store( DGraphic.P_3D, true );
    if(_thinbar) g.getProperties().store( DGraphic.P_THIN_BAR, true );
    return g;
  }
  
  /**
   * Create a multi-series Histogram from values.
   * @param title Title of the graphic.
   * @param values datas used to create graphic, first indice is series, second is data in serie
   * @param _3d true to use 3D histogram.
   * @param _thinbar true to use thin bar.
   * @param palette a color generator called for each bar of histogram.
   */  
  public static DGraphic Histogram( String title, String []names, double[][] values, boolean _3d, boolean _thinbar, IPalette pal )
  {
    DGraphic g= OneAxisChart( DGraphic.T_HISTOGRAM_IM, title, names, values, pal );
    if(_3d) g.getProperties().store( DGraphic.P_3D, true );
    if(_thinbar) g.getProperties().store( DGraphic.P_THIN_BAR, true );
    return g;
  }
  
  /**
   * Create a stack bar graphic.
   * @param title Title of the graphic.
   * @param values datas used to create graphic, first indice is series, second is data in serie
   * @param _3d true to use 3D bar.
   * @param _thinbar true to use thin bar.
   * @param palette a color generator called for each bar of histogram (PAL_DEFAULT if null)
   */  
  public static DGraphic StackBars( String title, String []names, double[][] values, boolean _3d, boolean _thinbar, IPalette pal )
  {
    //very close to OnAxis() method but the computing of min/max require cumulative value
    //bar by bar...
    if( pal==null ) pal=Palette.PAL_DEFAULT ;

    DGraphic g = new DGraphic();
    g.setTitle( title );
    g.setRenderableId( DGraphic.T_STACKBARS );
    if(_3d) g.getProperties().store( DGraphic.P_3D, true );
    if(_thinbar) g.getProperties().store( DGraphic.P_THIN_BAR, true );
    
    DAxis axis = new DAxis("v");
    g.addChild( axis );
    if( names==null )
    {
      g.getProperties().store( DGraphic.P_SHOW_LEGEND, false );
    }

    IDItem last_g_child = axis;
    int nc = pal.getNumColors();

    int nbar=0;
    for( int icrv=0; icrv<values.length; ++icrv )
    {
      int size = values[icrv].length;
      if( size > nbar ) nbar = size;
    }
    //min/max by bar
    double stacked_value[] = new double[nbar]; 
    
    for( int icrv=0; icrv<values.length; ++icrv )
    {
      DCurve crv = new DCurve();
      if( names!=null ) crv.setName( names[icrv] );
    
      DStyle s_crv = new DStyle();
      crv.setStyle( s_crv );    
      g.insertChild( crv, last_g_child );
      last_g_child = crv;
      s_crv.setBackColor( pal.getColor(icrv%nc) );
        
      int size = values[icrv].length;
      DPoint last_p=null;
      for( int i=0; i<size; ++i )
      {
        //obviously negative value will produce problems... use absolute?
        stacked_value[i] += values[icrv][i];
      
        DPoint p = new DPoint();
        crv.insertChild( p, last_p );
        p.addChild( new DCoord( axis, values[icrv][i] ) );
        //set a style to point only if graphic have one curve.
        if( values.length==1)
        {
          DStyle ps = new DStyle();
          ps.setBackColor( pal.getColor(i%nc) );
          p.setStyle( ps );
        }
        last_p=p;
      }
    }
    
    double min=stacked_value[0], max=stacked_value[0];
    for( int ib=1; ib<nbar; ib++ )
    {      
           if( stacked_value[ib] < min ) min = stacked_value[ib];
      else if( stacked_value[ib] > max ) max = stacked_value[ib];
    }

    if(min>=0.0) min=0.0; else min=1.1*min;
    if(max!=0.0) { if(max==min) max=1.1*max; } else max=min+1.0;

    int m=1;
    if( min<0 && max > 0 ) m=2;

    DPropertyStore props = axis.getProperties();
    props.store( DAxis.P_MIN, min );
    props.store( DAxis.P_MAX, max );
    props.store( DAxis.P_STEP_LINE, (max-min)/(4*m));
    props.store( DAxis.P_STEP_UNIT, (max-min)/(4*m));
    props.store( DAxis.P_STEP_DOT,  (max-min)/(10*m));
    
    return g;
  }

  /**
   * Create a pie chart graphic (internally known as a Sectors graphic).
   * @param title the title of the graphic.
   * @param values datas used to create graphic.
   * @param palette colors generator (PAL_DEFAULT is null).
   */
  public static DGraphic PieChart( String title, double values[], boolean _3d, IPalette palette )
  {
    return PieChart( title, null, values, _3d, palette );
  }
  
  /**
   * Create a pie chart graphic (internally known as a Sectors graphic) using default palette.
   * @param title the title of the graphic.
   * @param values datas used to create graphic.
   */
  public static DGraphic PieChart( String title, double values[] )
  {
    return PieChart( title, null, values, true, null );
  }
  
  /**
   * Create a pie chart graphic (internally known as a Sectors graphic) using default palette.
   * @param title the title of the graphic.
   * @param names array of name for each value.
   * @param values datas used to create graphic.
   * @param _3d true to use a 3D pie chart, false to a flat pie chart.
   * @param palette colors generator (PAL_DEFAULT is null).
   */
  public static DGraphic PieChart( String title, String names[], double values[], boolean _3d, IPalette palette )
  {
    double nval[][]=null;
    if( names!=null )
    {    
      nval = new double[values.length][];
      for( int i=0; i<values.length; ++i )
      {
        nval[i] = new double[] { values[i] };
      }
    } else {
      nval = new double[][]{values};
    }
    return OneAxisChart( _3d ? DGraphic.T_SECTORS3D : DGraphic.T_SECTORS, title, names, nval, palette );
  }

  /**
   * Internal Use.
   * Create a One axis chart (histogram,stackbar,pie chart ...).
   */
  protected static DGraphic OneAxisChart( String type, String title, String curve_name[], double[][] values, IPalette pal )
  {

    if( pal==null ) pal=Palette.PAL_DEFAULT ;
    DGraphic g = new DGraphic();
    g.setTitle( title );
    g.setRenderableId( type );
    
    DAxis axis = new DAxis("v");
    g.addChild( axis );
    if( curve_name==null )
    {
      g.getProperties().store( DGraphic.P_SHOW_LEGEND, false );
    }

    IDItem last_g_child = axis;
    int nc = pal.getNumColors();
    double min=0, max=0;
    
    for( int icrv=0; icrv<values.length; ++icrv )
    {
      DCurve crv = new DCurve();
      if( curve_name!=null && icrv<curve_name.length ) crv.setName( curve_name[icrv] );
    
      DStyle s_crv = new DStyle();
      crv.setStyle( s_crv );    
      g.insertChild( crv, last_g_child );
      last_g_child = crv;
      s_crv.setBackColor( pal.getColor(icrv%nc) );
        
      int size = values[icrv].length;
      DPoint last_p=null;
      for( int i=0; i<size; ++i )
      {
        if( i==0 && icrv==0)
        {
          min=max=values[icrv][i];
        }
        else if ( values[icrv][i]<min ) min=values[icrv][i]; 
        else if ( values[icrv][i]>max ) max=values[icrv][i];
      
        DPoint p = new DPoint();
        crv.insertChild( p, last_p );
        p.addChild( new DCoord( axis, values[icrv][i] ) );
        //set a style to point only if graphic have one curve.
        if( values.length==1)
        {
          DStyle ps = new DStyle();
          ps.setBackColor( pal.getColor(i%nc) );
          p.setStyle( ps );
        }
        last_p=p;
      }
    }

    if(min>=0.0) min=0.0; else min=1.1*min;
    if(max!=0.0) max=1.1*max; else max=min+1.0;

    int m=1;
    if( min<0 && max > 0 ) m=2;
    
    DPropertyStore props = axis.getProperties();
    props.store( DAxis.P_MIN, min );
    props.store( DAxis.P_MAX, max );
    props.store( DAxis.P_STEP_LINE, (max-min)/(4*m));
    props.store( DAxis.P_STEP_UNIT, (max-min)/(4*m));
    props.store( DAxis.P_STEP_DOT,  (max-min)/(10*m));
    
    return g;
  }

  /**
   * Create XY curves using only one X coordinates data, no symbol.
   * see {@link #XYCurves} for parameter definition.
   */
  public static DGraphic XYCurvesOneX( String title, String curve_name[], double[][] values, boolean use_symbol, IPalette pal )
  {
    return XYCurves( title, curve_name, values, true, use_symbol, pal );
  }
  
  /**
   * Create a XY chart, the number of curve is done by values array length and one_x parameter.
   * @param title the title of the chart.
   * @param curve_name if not null provide name of the curve.
   * @param values provide curves X and Y data, this array drive the number of curves. see one_x parameter.
   * @param one_x if true first array in data provide X values, and all other array are curves (Y values).
   *              if false first array is X for curve 1, second is Y for curve 1, third is X for curve 2,...
   * @param use_symbol if true curves type are set to T_LINE_POINTS, otherwise type is T_LINE
   * @param pal set palette to set curves colors, if null, PAL_DEFAULT is used.
   */
  public static DGraphic XYCurves( String title, String curve_name[], double[][] values, boolean one_x, boolean use_symbol, IPalette pal )
  {
    return XYCurves( title, curve_name, values, one_x, use_symbol, false, false, pal );
  }
  /**
   * Create a XY chart using only area curves.
   * The number of curve is done by values array length and one_x parameter.
   * @param title the title of the chart.
   * @param curve_name if not null provide name of the curve.
   * @param values provide curves X and Y data, this array drive the number of curves. see one_x parameter.
   * @param one_x if true first array in data provide X values, and all other array are curves (Y values).
   *              if false first array is X for curve 1, second is Y for curve 1, third is X for curve 2,...
   * @param use_symbol if true curves type are set to T_LINE_POINTS, otherwise type is T_LINE
   * @param pal set palette to set curves colors, if null, PAL_DEFAULT is used.
   */
  public static DGraphic XYArea( String title, String curve_name[], double[][] values, boolean one_x, boolean use_symbol, IPalette pal )
  {
    return XYCurves( title, curve_name, values, one_x, use_symbol, true, false, pal );
  }
  /**
   * Create a XY chart using only Stacked Area curves.
   * The number of curve is done by values array length and one_x parameter.
   * @param title the title of the chart.
   * @param curve_name if not null provide name of the curve.
   * @param values provide curves X and Y data, this array drive the number of curves. see one_x parameter.
   * @param one_x if true first array in data provide X values, and all other array are curves (Y values).
   *              if false first array is X for curve 1, second is Y for curve 1, third is X for curve 2,...
   * @param use_symbol if true curves type are set to T_LINE_POINTS, otherwise type is T_LINE
   * @param pal set palette to set curves colors, if null, PAL_DEFAULT is used.
   */
  public static DGraphic XYStackedArea( String title, String curve_name[], double[][] values, boolean one_x, boolean use_symbol, IPalette pal )
  {
    return XYCurves( title, curve_name, values, one_x, use_symbol, true, true, pal );
  }
  
  /**
   * Create a XY chart using only Stacked Area curves.
   * The number of curve is done by values array length and one_x parameter.
   * @param title the title of the chart.
   * @param curve_name if not null provide name of the curve.
   * @param values provide curves X and Y data, this array drive the number of curves. see one_x parameter.
   * @param one_x if true first array in data provide X values, and all other array are curves (Y values).
   *              if false first array is X for curve 1, second is Y for curve 1, third is X for curve 2,...
   * @param use_symbol if true curves type use symbols (false for only lines).
   * @param use_area if true curves type is in AREA ones, if false use LINES or POINTS.
   * @param use_stacked if true curves type is inf STACKED ones, false use lines/area/symbols.
   * @param pal set palette to set curves colors, if null, PAL_DEFAULT is used.
   */
  public static DGraphic XYCurves(String title, String curve_name[], double[][] values, boolean one_x, boolean use_symbol, boolean use_area, boolean use_stacked, IPalette pal )
  {
    if( curve_name==null && values==null ) return null;
    if( pal == null ) pal = Palette.PAL_DEFAULT ;


    DGraphic g = new DGraphic();
    g.setTitle( title );
    g.setRenderableId( DGraphic.T_XY );
    
    DAxis x_axis = new DAxis("x");
    DAxis y_axis = new DAxis("y");
    g.addChild( x_axis );
    g.addChild( y_axis );
    if( curve_name==null )
    {
      g.getProperties().store( DGraphic.P_SHOW_LEGEND, false );
    }
    
    IDItem last_g_child = y_axis;
    int nc = pal.getNumColors();
    double x_min=0, x_max=0, y_min=0, y_max=0;
    boolean init_minmax=true;
    
    int n_curve = n_curve = one_x ? values.length-1 : values.length/2;
    String curve_type = DCurve.T_LINE;
    if( use_stacked )
    {
      if( use_area )
      {
        curve_type = use_symbol ? DCurve.T_STACKED_AREA_LINE_POINTS
                                : DCurve.T_STACKED_AREA_LINE;
      } else {
        curve_type = use_symbol ? DCurve.T_STACKED_LINE_POINTS
                                : DCurve.T_STACKED_LINE;
      }
    }
    else
    {
      if( use_area )
      {
        curve_type = use_symbol ? DCurve.T_AREA_LINE_POINTS
                                : DCurve.T_AREA_LINE ;
      } else {
        curve_type = use_symbol ? DCurve.T_LINE_POINTS
                                : DCurve.T_LINE;
      }
    }
    
    for( int icrv=0; icrv<n_curve; ++icrv )
    {
      int ix = one_x ? 0 : 2*icrv;
      int iy = one_x ? icrv+1 : ix+1;
      
      DCurve crv = new DCurve();
      if( curve_name!=null && icrv<curve_name.length ) crv.setName( curve_name[icrv] );
    
      DStyle s_crv = new DStyle();
      crv.setStyle( s_crv );    
      g.insertChild( crv, last_g_child );
      last_g_child = crv;
      s_crv.setBackColor( pal.getColor(icrv%nc) );
      
      crv.setType( curve_type );
      if( use_symbol )
      {
        crv.getProperties().store( DCurve.P_SYMBOL, GetSymbolId(icrv));
      }
      
      int size_x = values[ix].length;
      int size_y = values[iy].length;
      int size   = size_x < size_y ? size_x : size_y;

      DPoint last_p=null;
      for( int i=0; i<size; ++i )
      {
        double vx = values[ix][i];
        double vy = values[iy][i];
        
        if( init_minmax )
        {
          x_min=x_max=vx;
          y_min=y_max=vy;
          init_minmax=false;
        }
        else 
        {
          if ( vx<x_min ) x_min=vx; else if (vx>x_max) x_max=vx;
          if ( vy<y_min ) y_min=vy; else if (vy>y_max) y_max=vy;
        }
      
        DPoint p = new DPoint();
        crv.insertChild( p, last_p );
        p.addChild( new DCoord( x_axis, vx ) );
        p.addChild( new DCoord( y_axis, vy ) );

        last_p=p;
      }
    }
    
    //force min!=max otherwise default render will bark 'invalid axis'
    if( x_min==x_max )
    {
      if( x_min==0.0 )
      {
        x_min=-1; x_max=+1.0;
      } else {
        x_min = 0.9*x_min;
        x_max = 1.1*x_max;
      }
    }
    
    x_axis.getProperties().store( DAxis.P_MIN, x_min );
    x_axis.getProperties().store( DAxis.P_MAX, x_max );
    x_axis.getProperties().store( DAxis.P_STEP_LINE, 0.125*(x_max-x_min) );
    x_axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(x_max-x_min) );
    //min max is wrong for stacked area/lines (doens't cumulate values ... need to fix this ?
    if( !use_stacked )
    {
      if( y_min==y_max )
      {
        if( y_min==0.0 )
        {
          y_min=-1; y_max=+1.0;
        } else {
          y_min = 0.9*y_min;
          y_max = 1.1*y_max;
        }
      }
      y_axis.getProperties().store( DAxis.P_MIN, y_min );
      y_axis.getProperties().store( DAxis.P_MAX, y_max );
      y_axis.getProperties().store( DAxis.P_STEP_LINE, 0.125*(y_max-y_min) );
      y_axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(y_max-y_min) );
    }
        
    
    return g;
  }
  
  private static String GetSymbolId( int index )
  {
    index = index % 13;
    switch( index )
    {
      case  0 : return DSymbolRegistry.GetSquare   ().getId();
      case  1 : return DSymbolRegistry.GetOval     ().getId();
      case  2 : return DSymbolRegistry.GetUp       ().getId();
      case  3 : return DSymbolRegistry.GetDown     ().getId();
      case  4 : return DSymbolRegistry.GetLeft     ().getId();
      case  5 : return DSymbolRegistry.GetRight    ().getId();
      case  6 : return DSymbolRegistry.GetPlus     ().getId();
      case  7 : return DSymbolRegistry.GetMinus    ().getId();
      case  8 : return DSymbolRegistry.GetDiamond  ().getId();
      case  9 : return DSymbolRegistry.GetMoonDown ().getId();
      case 10 : return DSymbolRegistry.GetMoonUp   ().getId();
      case 11 : return DSymbolRegistry.GetMoonLeft ().getId();
      case 12 : return DSymbolRegistry.GetMoonRight().getId();
    }
    return null;
  }
  
  /**
   * Create a XYZ chart, the number of curve is done by values array length (divided by 3), 
   * first contains X coords for curve 1, second contains Y coords for curve 1, thrid Z for curve 1,
   * fourth X/Curve 2, 5th Y/Curve 2... and so on.
   * @param title the title of the chart.
   * @param curve_name if not null provide name of curve (in order they are defined)
   * @param values provide curves X,Y,Z data, this array drive the number of curves.
   * @param pal set palette to set curves colors, if null, PAL_DEFAULT is used.
   */
  public static DGraphic XYZCurves( String title, String curve_name[], double[][] values, IPalette pal )
  {
    if( curve_name==null && values==null ) return null;
    if( pal == null ) pal = Palette.PAL_DEFAULT ;


    DGraphic g = new DGraphic();
    g.setTitle( title );
    g.setRenderableId( DGraphic.T_XYZ );
    
    DAxis x_axis = new DAxis("x");
    DAxis y_axis = new DAxis("y");
    DAxis z_axis = new DAxis("z");
    g.addChild( x_axis );
    g.addChild( y_axis );
    g.addChild( z_axis );
    if( curve_name==null )
    {
      g.getProperties().store( DGraphic.P_SHOW_LEGEND, false );
    }
    
    IDItem last_g_child = z_axis;
    int nc = pal.getNumColors();
    double x_min=0, x_max=0, y_min=0, y_max=0, z_min=0, z_max=0;
    boolean init_minmax=true;
    
    int n_curve = n_curve = values.length/3;

    for( int icrv=0; icrv<n_curve; ++icrv )
    {
      int ix = 3*icrv;
      int iy = ix+1;
      int iz = iy+1;
      
      DCurve crv = new DCurve();
      if( curve_name!=null && icrv<curve_name.length ) crv.setName( curve_name[icrv] );
    
      DStyle s_crv = new DStyle();
      crv.setStyle( s_crv );    
      g.insertChild( crv, last_g_child );
      last_g_child = crv;
      s_crv.setBackColor( pal.getColor(icrv%nc) );
      
      crv.setType( DCurve.T_LINE );
      
      int size_x = values[ix].length;
      int size_y = values[iy].length;
      int size_z = values[iz].length;
      int size   = Math.min(size_x, Math.min( size_y, size_z ));

      DPoint last_p=null;
      for( int i=0; i<size; ++i )
      {
        double vx = values[ix][i];
        double vy = values[iy][i];
        double vz = values[iz][i];
        
        if( init_minmax )
        {
          x_min=x_max=vx;
          y_min=y_max=vy;
          z_min=z_max=vz;
          init_minmax=false;
        }
        else 
        {
          if ( vx<x_min ) x_min=vx; else if (vx>x_max) x_max=vx;
          if ( vy<y_min ) y_min=vy; else if (vy>y_max) y_max=vy;
          if ( vz<z_min ) z_min=vz; else if (vz>z_max) z_max=vz;
        }
      
        DPoint p = new DPoint();
        crv.insertChild( p, last_p );
        p.addChild( new DCoord( x_axis, vx ) );
        p.addChild( new DCoord( y_axis, vy ) );
        p.addChild( new DCoord( z_axis, vz ) );

        last_p=p;
      }
    }

    if( x_min==x_max )
    {
      if( x_min==0.0 )
      {
        x_min=-1; x_max=+1.0;
      } else {
        x_min = 0.9*x_min;
        x_max = 1.1*x_max;
      }
    }
    if( y_min==y_max )
    {
      if( y_min==0.0 )
      {
        y_min=-1; y_max=+1.0;
      } else {
        y_min = 0.9*y_min;
        y_max = 1.1*y_max;
      }
    }
    if( z_min==z_max )
    {
      if( z_min==0.0 )
      {
        z_min=-1; z_max=+1.0;
      } else {
        z_min = 0.9*z_min;
        z_max = 1.1*z_max;
      }
    }
    x_axis.getProperties().store( DAxis.P_MIN, x_min );
    x_axis.getProperties().store( DAxis.P_MAX, x_max );
    y_axis.getProperties().store( DAxis.P_MIN, y_min );
    y_axis.getProperties().store( DAxis.P_MIN, y_min );
    z_axis.getProperties().store( DAxis.P_MAX, z_max );
    z_axis.getProperties().store( DAxis.P_MAX, z_max );
    
    x_axis.getProperties().store( DAxis.P_STEP_LINE, 0.125*(x_max-x_min) );
    y_axis.getProperties().store( DAxis.P_STEP_LINE, 0.125*(y_max-y_min) );
    z_axis.getProperties().store( DAxis.P_STEP_LINE, 0.125*(z_max-z_min) );
    x_axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(x_max-x_min) );
    y_axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(y_max-y_min) );
    z_axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(z_max-z_min) );
    
    return g;
  }
  
  public static int encodeColor( DColor _color )
  {
    int r = _color.getRed();
    int g = _color.getGreen();
    int b = _color.getBlue();
    return ((r&0xFF)<<16) | ((g&0xFF)<<8) | (b&0xFF);
  }
  
  

  /**
   * Create a simple meter chart using default min/max (0..100) if value is inside range
   * (if outside use 10% less for min, 10% more for max). 
   * @param title the title of the graphic.
   * @param axis_title is the title of axis displayed by meter chart.
   * @param values value of the needle of meter.
   */
  public static DGraphic Meter( String title, String axis_title, double value )
  {
    return Meter( title, axis_title, value, 0.0,0.0, null,null, null );
  }

  /**
   * Create a simple meter chart using given value (needle), min and max.
   * @param title the title of the graphic.
   * @param axis_title is the title of axis displayed by meter chart.
   * @param values value of the needle of meter.
   * @param min the value of minimum displayed by meter.
   * @param max the value of maximum displayed by meter.
   */
  public static DGraphic Meter( String title, String axis_title, double value, double min, double max )
  {
    return Meter( title, axis_title, value, min,max, null,null, null );
  }
  
  /**
   * Create a meter chart using given value (needle), min and max and a "list" of region.
   * For best result, ensure region values are sorted from min to max.
   * First region starts at min value and ends at region[0] value.
   * Second region starts at the end of first region and ends at region[1] value. and so on.
   * Color palette is used to define the fill color of region.
   * @param title the title of the graphic.
   * @param axis_title is the title of axis displayed by meter chart.
   * @param values value of the needle of meter.
   * @param min the value of minimum displayed by meter.
   * @param max the value of maximum displayed by meter.
   * @param regions array defining the region of meter (ie: sectors above the needle).
   * @param regions_name name of regions displayed in legend.
   * @param palette is used to set fill color of regions.
   */
  public static DGraphic Meter( String title, String axis_title, double value, double min, double max, double[]regions, String []regions_names, IPalette palette )
  {
    if(palette==null) palette=Palette.PAL_DEFAULT ;

    DGraphic g = new DGraphic();
    g.setTitle( title );
    g.setRenderableId( DGraphic.T_METER );
    
    DAxis axis = new DAxis("v");
    axis.setTitle( axis_title );
    g.addChild( axis );
    g.getProperties().store( DGraphic.P_SHOW_LEGEND, false );

    DCurve cneedle = new DCurve();
    g.addChild( cneedle );
    DPoint needle = new DPoint();
    cneedle.addChild(needle);
    needle.addChild( new DCoord( axis, value ) );
    
    double max_region_value=value;
    boolean have_max_region_value=false;
    if( regions != null && regions.length>0 )
    {
      IDItem last=cneedle;
      for( int i=0; i<regions.length; ++i )
      {
        DCurve cregion = new DCurve();
        if( regions_names!=null && i < regions_names.length )
        {
          cregion.setName( regions_names[i] );
        }
        DPoint pregion = new DPoint();
        cregion.addChild( pregion );
        pregion.addChild( new DCoord(axis,regions[i]));
        
        g.insertChild( cregion, last );
        last=cregion;
        
        DStyle sty = new DStyle();
        sty.setBackColor( palette.getColor(i) );
        cregion.setStyle(sty);
        
        if( regions[i] > max_region_value ) 
        {
          max_region_value=regions[i];
          have_max_region_value=true;
        }
      }
    }
    
    boolean default_minmax=false;
    if( min==max ) //check if default min/max is ok for value.
    {
      if( value<0.0 || value > 100.0)
      {
        min = 0.9*value;
        max = 1.1*value;
      } else {
        default_minmax=true;
      }
    }
    else
    {
      if( value < min ) { min = 0.9*value; }
      if( value > max ) { max = 1.1*value; }
    }
    if( have_max_region_value && max_region_value > max )
    {
      max = max_region_value;
    }
      
    //default min/max of meter can't cope with that value.
    if( !default_minmax )
    {
      axis.getProperties().store( DAxis.P_MIN, min );
      axis.getProperties().store( DAxis.P_MAX, max );
    } else {
      min=0;
      max=100;
    }
    
    axis.getProperties().store( DAxis.P_STEP_LINE, 0.1*(max-min) );
    axis.getProperties().store( DAxis.P_STEP_UNIT, 0.25*(max-min));
    
    return g;
  }
  
  public static final int W_DEFAULT=0;
  public static final int W_NONE=1;
  public static final int W_TOP_LEFT=2;
  public static final int W_TOP_CENTER=3;
  public static final int W_TOP_RIGHT=4;
  public static final int W_LEFT_TOP=5;
  public static final int W_LEFT_CENTER=6;
  public static final int W_LEFT_BOTTOM=7;
  public static final int W_RIGHT_TOP=8;
  public static final int W_RIGHT_CENTER=9;
  public static final int W_RIGHT_BOTTOM=10;  
  public static final int W_BOTTOM_LEFT=11;
  public static final int W_BOTTOM_CENTER=12;
  public static final int W_BOTTOM_RIGHT=13;
  
  /**
   * Configure the location of graphic's title.
   * Refer to constant W_xxxx to known possible values.
   * W_DEFAULT let render choose for you, W_NONE hide title.
   */
  public static void ConfigureTitle( DGraphic _graphic, int _where )
  {
    if( _graphic==null ) return ;
    _graphic.getProperties().remove( DGraphic.P_SHOW_TITLE );
    _graphic.getProperties().remove( DGraphic.P_TITLE_LAYOUT );
    
    switch( _where )
    {
      case W_DEFAULT: break;
      case W_NONE:
        _graphic.getProperties().store( DGraphic.P_SHOW_TITLE, false );
        break ;
      default:
        _graphic.getProperties().store( DGraphic.P_TITLE_LAYOUT, ToAlignmentPair(_where,true) );
    }
  }
  
  /**
   * Configure the location of graphic's legend.
   * Refer to constant W_xxxx to known possible values.
   * W_DEFAULT let render choose for you, W_NONE hide legend.
   */
  public static void ConfigureLegend( DGraphic _graphic, int _where )
  {
    if( _graphic==null ) return ;
    _graphic.getProperties().remove( DGraphic.P_SHOW_LEGEND );
    _graphic.getProperties().remove( DGraphic.P_LEGEND_LAYOUT );
    
    switch( _where )
    {
      case W_DEFAULT: break;
      case W_NONE:
        _graphic.getProperties().store( DGraphic.P_SHOW_LEGEND, false );
        break ;
      default:
        _graphic.getProperties().store( DGraphic.P_LEGEND_LAYOUT, ToAlignmentPair(_where,false) );
    }
  }
  
  private static DAlignmentPair ToAlignmentPair( int _where, boolean _rot )
  {
    switch( _where )
    {
      default:
      case W_TOP_LEFT:
        return new DAlignmentPair( IDAlignment.TOP, IDAlignment.LEFT );
      case W_TOP_CENTER:
        return new DAlignmentPair( IDAlignment.TOP, IDAlignment.CENTER );
      case W_TOP_RIGHT:
        return new DAlignmentPair( IDAlignment.TOP, IDAlignment.RIGHT );
      case W_LEFT_TOP:
        return new DAlignmentPair( IDAlignment.LEFT, IDAlignment.TOP|(_rot?IDAlignment.ROTCCW90:0) );
      case W_LEFT_CENTER:
        return new DAlignmentPair( IDAlignment.LEFT, IDAlignment.CENTER|(_rot?IDAlignment.ROTCCW90:0) );
      case W_LEFT_BOTTOM:
        return new DAlignmentPair( IDAlignment.LEFT, IDAlignment.BOTTOM|(_rot?IDAlignment.ROTCCW90:0) );
      case W_RIGHT_TOP:
        return new DAlignmentPair( IDAlignment.RIGHT, IDAlignment.TOP|(_rot?IDAlignment.ROTCW90:0) );
      case W_RIGHT_CENTER:
        return new DAlignmentPair( IDAlignment.RIGHT, IDAlignment.CENTER|(_rot?IDAlignment.ROTCW90:0) );
      case  W_RIGHT_BOTTOM:  
        return new DAlignmentPair( IDAlignment.RIGHT, IDAlignment.BOTTOM|(_rot?IDAlignment.ROTCW90:0) );
      case W_BOTTOM_LEFT:
        return new DAlignmentPair( IDAlignment.BOTTOM, IDAlignment.LEFT );
      case W_BOTTOM_CENTER:
        return new DAlignmentPair( IDAlignment.BOTTOM, IDAlignment.CENTER );
      case W_BOTTOM_RIGHT:
        return new DAlignmentPair( IDAlignment.BOTTOM, IDAlignment.RIGHT );
    }
  }

  /**
   * Create a multi-series Series3D graphic from values.
   * All series use same curve type set by curve_type parameter. 
   * @param title Title of the graphic.
   * @param values datas used to create graphic, first indice is series, second is data in serie
   * @param _thinbar true to use thin bar.
   * @param palette a color generator called for each curves.
   * @param curve_type use DCurve.T_xxx constants supported by DGraphic.T_SERIES3D graphic type.
   */ 
  public static DGraphic Series3D( String title, String []names, double[][] values, boolean _thinbar, IPalette pal, String curve_type )
  {
    //do not use palette to set different color for each point of a curve in case of
    //graphic with one curve, as Series3DAddCurve() might be used too. 
    if( pal==null ) pal=Palette.PAL_DEFAULT;
    if( values.length==1 ) {
      pal = new Palette( pal.getColor( 0 ));
    }
    
    DGraphic g= OneAxisChart( DGraphic.T_SERIES3D, title, names, values, pal );
    if(!_thinbar) g.getProperties().store( DGraphic.P_THIN_BAR, false );//as it'default value is "true"
    //change curve type.
    if( curve_type!=null && !DCurve.T_LINE.equals( curve_type ) )
    {
      for( IDItem it = g.getFirstChild(); it!=null; it=it.getNext() )
      {
        if( !(it instanceof DCurve) ) continue;
        ((DCurve)it).setType(curve_type);
      }
    }
    return g;
  }
  
  /**
   * Create a multi-series Series3D graphic with one curve.
   * You can add more curve to this graphic using Series3DAddCurve() method. 
   * @param title Title of the graphic.
   * @param name   name of curve.
   * @param values datas used to create curve,
   * @param _thinbar true to use thin bar.
   * @param palette a color generator called for curve.
   * @param curve_type use DCurve.T_xxx constants supported by DGraphic.T_SERIES3D graphic type.
   */ 
  public static DGraphic Series3D( String title, final String name, final double[] values, boolean _thinbar, IPalette pal, String curve_type )
  {
    //do not use palette to set different color for each point of a curve in case of
    //graphic with one curve, as Series3DAddCurve() might be used too. 
    if( pal==null ) pal=Palette.PAL_DEFAULT;
    pal = new Palette( pal.getColor( 0 ));
    
    DGraphic g= OneAxisChart( DGraphic.T_SERIES3D, title, new String[]{name}, new double[][]{values}, pal );
    if(!_thinbar) g.getProperties().store( DGraphic.P_THIN_BAR, false );//as it'default value is "true"
    //change curve type.
    if( curve_type!=null && !DCurve.T_LINE.equals( curve_type ) )
    {
      for( IDItem it = g.getFirstChild(); it!=null; it=it.getNext() )
      {
        if( !(it instanceof DCurve) ) continue;
        ((DCurve)it).setType(curve_type);
      }
    }
    return g;
  }
  
  /**
   * Add acurvee to an existing T_SERIES3D graphic type.
   * @param graphic existing T_SERIES3D graphic type (see Series3D() method).
   * @param name Name of curve.
   * @param values datas used to create graphic, first indice is series, second is data in serie
   * @param _thinbar true to use thin bar.
   * @param palette a color generator used to choose color for new curve.
   * @param curve_type use DCurve.T_xxx constants supported by DGraphic.T_SERIES3D graphic type.
   */ 
  public static void Series3DAddCurve( DGraphic graphic, final String name, final double[] values, IPalette pal, String curve_type )
  {
    if( graphic==null ) return ;
    
    int ncurve = 0;
    DAxis gaxis = null;
    for( IDItem it=graphic.getFirstChild(); it!=null; it=it.getNext() )
    {
      if( it instanceof DAxis ) {
        gaxis = (DAxis)it;
      }
      else if ( it instanceof DCurve ) { 
        ncurve++;
      }
    }
    
    if( pal==null ) pal=Palette.PAL_DEFAULT;
    
    pal = new Palette( pal.getColor( ncurve ));
    
    DGraphic g= OneAxisChart( DGraphic.T_SERIES3D, null, new String[]{name}, new double[][]{values}, pal );    
    //change curve type.
    DCurve curve = null;
    DAxis axis=null;
    for( IDItem it = g.getFirstChild(); it!=null; it=it.getNext() )
    {
      if( curve==null && (it instanceof DCurve) ) {
        curve = (DCurve)it;
        curve.setType(curve_type);
      } 
      else if( axis==null && (it instanceof DAxis) ) {
        axis =(DAxis)it;
      }
    }

    //add curve to grahpic:
    graphic.addChild( curve );
    curve.setParent( graphic );
    curve.setType( curve_type );
  
    //merge axis values: (are Number)
   
    //all DCoord of DPoint in curve must use gaxis
    for( IDItem it=curve.getFirstChild(); it!=null; it=it.getNext() )
    {
      if( it instanceof DPoint ) {
        for( IDItem ic = it.getFirstChild(); ic!=null; ic=ic.getNext() ) 
        {
          if( ic instanceof IDCoord ) {
            ((IDCoord)ic).setAxis( gaxis );
          }
        }
      }
    }
    
    DPropertyStore p1 = axis.getProperties();
    DPropertyStore p2 = gaxis.getProperties();
    double min = Math.min( p1.get( DAxis.P_MIN, 0.0 ), 1.1*p2.get( DAxis.P_MIN, 0.0 ) );
    double max = Math.max( p1.get( DAxis.P_MAX, 0.0 ), 1.1*p2.get( DAxis.P_MAX, 0.0 ) );
    
    if(min>=0.0) min=0.0; 
    if(max==0.0) max=min+1.0;

    int m=1;
    if( min<0 && max>0 ) m=2;
    
    p2.store( DAxis.P_MIN, min );
    p2.store( DAxis.P_MAX, max );
    p2.store( DAxis.P_STEP_LINE, (max-min)/(4*m));
    p2.store( DAxis.P_STEP_UNIT, (max-min)/(4*m));
    p2.store( DAxis.P_STEP_DOT,  (max-min)/(10*m));    
  }
  
  /**
   * Create a Pie3D graphic.
   * @param title Title of the graphic.
   * @param values datas used to create graphic,
   * @param palette a color generator called for curve.
   */ 
  public static DGraphic Pie3D( String title, double[] values, IPalette pal )
  {
    DGraphic g= OneAxisChart( DGraphic.T_PIE3D, title, null, new double[][]{values}, pal );
    return g;
  }
  
  /**
   * Create a Pie3D graphic.
   * @param title Title of the graphic.
   * @param names name of curve, null for none.
   * @param values datas used to create graphic,
   * @param shifted true if value at same index must stand in a shifted sector, null for no shift
   * @param palette a color generator called for curve.
   */ 
  public static DGraphic Pie3D( String title, String names[], double[] values, boolean shifted[], IPalette pal )
  {
    return Pie3D( title, names, values, shifted, pal, true );
  }
  
  /**
   * Create a Torus3D graphic.
   * @param title Title of the graphic.
   * @param values datas used to create graphic,
   * @param palette a color generator called for curve.
   */ 
  public static DGraphic Torus3D( String title, double[] values, IPalette pal )
  {
    DGraphic g= OneAxisChart( DGraphic.T_TORUS3D, title, null, new double[][]{values}, pal );
    return g;
  }
  
  /**
   * Create a Torus3D graphic.
   * @param title Title of the graphic.
   * @param names name of curve, null for none.
   * @param values datas used to create graphic,
   * @param shifted true if value at same index must stand in a shifted sector, null for no shift
   * @param palette a color generator called for curve.
   */ 
  public static DGraphic Torus3D( String title, String names[], double[] values, boolean shifted[], IPalette pal )
  {
    return Pie3D( title, names, values, shifted, pal, false );
  }
  
  /**
   * Create a Pie3D graphic.
   * @param title Title of the graphic.
   * @param names name of curve, null for none.
   * @param values datas used to create graphic,
   * @param shifted true if value at same index must stand in a shifted sector, null for no shift
   * @param palette a color generator called for curve.
   * @param pie3D true to create a Pie3D, false to create a Torus3D.
   */ 
  private static DGraphic Pie3D( String title, String names[], double[] values, boolean shifted[], IPalette pal, boolean pie3D )
  {
    if( shifted==null && names==null ) return Pie3D( title, values, pal );
    //to shift, must use one curve by sector ... or value.
    double [][]val = new double[values.length][1];
    for( int i=0; i<values.length; i++ )
    {
      val[i][0] = values[i];
    }
    DGraphic g= OneAxisChart( pie3D ? DGraphic.T_PIE3D : DGraphic.T_TORUS3D, title, names, val, pal );
    //nos apply shift property for shifted curves...
    IDItem it = g.getFirstChild();
    for( ;it!=null && !(it instanceof DCurve); it=it.getNext());
    if( it!=null ) 
    {
      for( int i=0; i<shifted.length; ++i )
      {
        if( shifted[i] ) ((DCurve)it).getProperties().store( DCurve.P_SHIFT_SECTORS, 0.3f );
        it=it.getNext();
        for( ;it!=null && !(it instanceof DCurve); it=it.getNext());
        if( it==null) break;
      }
    }
    return g;
  }
  
  /**
   * Create a Pie3D graphic.
   * @param title Title of the graphic.
   * @param sizex number of points along X axis.
   * @param sizey number of points along Y axis.
   * @param values datas used to create graphic, array size must be sizex*sizey
   * @param z_axis true if a "z" axis must be created, false for none.
   * @param clr Surface color generator used.
   */ 
  public static DGraphic XYSurface( String title, int sizex, int sizey, double[] values, boolean z_axis, IXYSurfaceColorizer clr )
  {
    DGraphic gph = new DGraphic();
    gph.setRenderableId( DGraphic.T_XY_SURFACE );
    gph.setTitle( title );
    
    if( clr!=null ) {
      gph.getProperties().store( DGraphic.P_XYS_COLORIZER, clr );
    }
    
    DXYSurface surf = new DXYSurface( sizex, sizey, values, 0,0 );
    surf.updateZMinMax();
    
    gph.addChild( surf );
    
    if( z_axis )
    {
      DAxis axis = new DAxis("z");
      gph.addChild( axis );
      DPropertyStore store = axis.getProperties();
      store.store( DAxis.P_MIN, surf.getZMin() );
      store.store( DAxis.P_MAX, surf.getZMin() );
      store.store( DAxis.P_STEP_UNIT, (surf.getZMax()-surf.getZMin())/4 );
      store.store( DAxis.P_STEP_DOT,  (surf.getZMax()-surf.getZMin())/10 );
    }
    
    return gph;
  }
  
  /**
   * Add properties to draw contour line on a T_XY_SURFACE graphic.
   * @param gph a T_XY_SURFACE graphic
   * @param delta_z distance between two contour line height.
   * @param rgba color used for lines
   * @see DGraphic T_XY_SURFACE type.
   */
  public static void XYSurfaceContour( DGraphic gph, double delta_z, int rgba )
  {
    if( gph==null )return ;
    if(!gph.isRenderableId( DGraphic.T_XY_SURFACE ) ) return; 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE, delta_z ); 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE_COLOR, rgba ); 
  }
  /**
   * Add properties to draw contour line on a T_XY_SURFACE graphic.
   * @param gph a T_XY_SURFACE graphic
   * @param delta_z distance between two contour line height.
   * @param clr color used for lines
   * @see DGraphic T_XY_SURFACE type.
   */
  public static void XYSurfaceContour( DGraphic gph, double delta_z, IDColor clr )
  {
    if( gph==null )return ;
    if(!gph.isRenderableId( DGraphic.T_XY_SURFACE ) ) return; 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE, delta_z ); 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE_COLOR, clr ); 
  }
  /**
   * Add properties to draw contour line on a T_XY_SURFACE graphic.
   * @param gph a T_XY_SURFACE graphic
   * @param delta_z distance between two contour line height.
   * @param clr colorizer used to set line color depending in line position in space.
   * @see DGraphic T_XY_SURFACE type.
   */
  public static void XYSurfaceContour( DGraphic gph, double delta_z, IXYSurfaceColorizer clr )
  {
    if( gph==null )return ;
    if(!gph.isRenderableId( DGraphic.T_XY_SURFACE ) ) return; 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE, delta_z ); 
    gph.getProperties().store( DGraphic.P_XYS_CONTOUR_LINE_COLOR, clr ); 
  }
}
