/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DJavaSourceColorizer.java,v 1.3 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 16 janv. 2004
 *
 */
package org.eclipse.tptp.platform.report.tools.internal;


import org.eclipse.tptp.platform.report.core.internal.DColor;
import org.eclipse.tptp.platform.report.core.internal.DFont;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.IDFont;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;

import com.ibm.icu.util.StringTokenizer;

/**
 * Really tiny simple java source colorizer for jscrib.
 * From a String, extract part of it and build DText children chain with
 * styles...<br>
 * <br>
 * Simple to use:<br>
 * <pre>  DJavaSourceColorizer.Colorize( parent_item, text );
 * </pre>
 * @see colorize
 * @see Colorize.
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DJavaSourceColorizer
{
  /**
   * @see DJavaSourceColorizer#colorize(IDItem,String)
   */
  public static void Colorize( IDItem parent, String source )
  {
    (new DJavaSourceColorizer()).colorize( parent, source );
  }

  /**
   * Colorize source, and appending lots of DText in children chain into parent.
   */
  public void colorize( IDItem parent, String source )
  {
    if( parent==null ) return ;
    
    StringTokenizer st = new StringTokenizer( source, getSeparators(),true);
    StringBuffer sb = new StringBuffer( source.length() );
    IDItem last=parent.getLastChild();
    
    IDStyle model = parent.getAppliedStyle();

    int state=0; //0:none 1:string 2:character
    
    while( st.hasMoreTokens() )
    {
      String token = st.nextToken();   
      switch( state )
      {
        case 0:
          if( isStringStart( token ) )
          {
            if( sb.length()>0 )
            {
              last = createText( new String(sb), null, parent, last );
              sb.setLength(0);
            }
            state=1;
            sb.append( token );
          }
          else if( isCharacterStart( token ) )
          {
            if( sb.length()>0 )
            {
              last = createText( new String(sb), null, parent, last );
              sb.setLength(0);
            }
            state=2;
            sb.append( token );
          }
          else if ( isKeyword( token ) )
          {
            if( sb.length()>0 )
            {
              last = createText( new String(sb), null, parent, last );
              sb.setLength(0);
            }
            last = createText( token, getKeywordStyle( model ), parent, last );
          }
          else
          {
            if( isNumber( token ) )
            {
              if( sb.length()>0 )
              {
                last = createText( new String(sb), null, parent, last );
                sb.setLength(0);
              }
              last = createText( token, getNumberStyle(model), parent, last );
            }
            else
            {
              sb.append( token );
            }
          }
         break;
         
       case 1:
          if( isStringEnd( token ) )
          {
            sb.append( token );
            last = createText( new String(sb), getStringStyle(model), parent, last );
            state=0;
            sb.setLength(0);
          } else {
            sb.append( token );
          }
          break;

       case 2:
          if( isCharacterEnd( token ) )
          {
            sb.append( token );
            last = createText( new String(sb), getCharacterStyle(model), parent, last );
            state=0;
            sb.setLength(0);
          } else {
            sb.append( token );
          }
          break;      }
    }
    
    if( sb.length()>0 )
    {
      parent.insertChild( new DText( new String(sb)), last );
    }    
  }
  
  /**
   * @return string containing all separators of language.
   */
  protected String getSeparators()
  {
    //separator used to cut source String..
    return " \t\n,;.,~|&+-=^*/%:(){}[]\'\"";
  }
  
  /**
   * @return true if str is a keyword of the language.
   */
  protected boolean isKeyword( String str )
  {
    if( str==null || str.length()<2 ) return false;
    switch( str.length() )
    {
      case 2 : if( "if".equals(str) ) return true;
               if( "do".equals(str) ) return true;
               return false;
               
      case 3 : switch( str.charAt(0) )
               {
                 case 'f': return "for".equals(str);
                 case 'i': return "int".equals(str);
                 case 'n': return "new".equals(str);
                 case 't': return "try".equals(str);
               }
               return false;
              
      case 4 : switch( str.charAt(0) )
               {
                 case 'b' : return "byte".equals(str);
                 case 'c' : return "case".equals(str)
                                || "char".equals(str);
                 case 'e' : return "else".equals(str);
                 case 'g' : return "goto".equals(str);
                 case 'l' : return "long".equals(str);
                 case 'n' : return "null".equals(str);
                 case 't' : return "true".equals(str)
                                || "this".equals(str);
                 case 'v' : return "void".equals(str);
               }
               return false;
               
      case 5 : switch( str.charAt(0) )
               {
                 case 'b' : return "break".equals(str);
                 case 'c' : return "catch".equals(str)
                                || "class".equals(str)
                                || "const".equals(str);
                 case 'f' : return "false".equals(str)
                                || "final".equals(str)
                                || "float".equals(str);
                 case 's' : return "short".equals(str)
                                || "super".equals(str);
                 case 't' : return "throw".equals(str);
                 case 'w' : return "while".equals(str);
               }
               return false;
            
      case 6 :
               switch( str.charAt(0) )
               {
                 case 'd' : return "double".equals(str);
                 case 'i' : return "import".equals(str);
                 case 'n' : return "native".equals(str);
                 case 'p' : return "public".equals(str);
                 case 'r' : return "return".equals(str);
                 case 's' : return "static".equals(str)
                                || "switch".equals(str);
                 case 't' : return "throws".equals(str);
               }
               return false;
               
      case 7 : switch( str.charAt(0) )
               {
                 case 'b' : return "boolean".equals(str);
                 case 'e' : return "extends".equals(str);
                 case 'd' : return "default".equals(str);
                 case 'f' : return "finally".equals(str);
                 case 'p' : return "package".equals(str)
                                || "private".equals(str);
               }
               return false;
               
      case 8 : switch( str.charAt(0) )
               {
                 case 'a' : return "abstract".equals(str);
                 case 'c' : return "continue".equals(str);
                 case 's' : return "strictfp".equals(str);
                 case 'v' : return "volatile".equals(str);
               }
               return false;
               
      case 9 : switch( str.charAt(0) )
               {
                 case 'i' : return "interface".equals(str);
                 case 'p' : return "protected".equals(str);
                 case 't' : return "transcient".equals(str);
               }
               return false;

      case 10: return "implements".equals(str);
               
      case 12: return "synchronized".equals(str);
    }
    return false;
  }
  
  /**
   * @return true if str is a number
   */
  protected boolean isNumber( String str )
  {
     //decimal
     try{
         Integer.parseInt( str );
         return true;
     }
     catch( NumberFormatException e ) {}
     //hexadecimal
     try{
       Integer.parseInt( str, 16 );
       return true;
     }
     catch( NumberFormatException ee ) {}
     
     return false;
  }
  
  /**
   * @return true if str begins a string in the language.
   */
  protected boolean isStringStart( String str )
  {
    return "\"".equals( str );
  }
  
  /**
   * @return true is str ends a string in the language.
   */
  protected boolean isStringEnd( String str )
  {
    return "\"".equals( str );
  }
  
  /**
   * @return true if str begins a character in the language.
   */
  protected boolean isCharacterStart( String str )
  {
    return "'".equals(str);
  }
  
  /**
   * @return true is str ends a character in the language.
   */
  protected boolean isCharacterEnd( String str )
  {
    return "'".equals(str);
  }

  private DStyle s_number;
  /**
   * @return style to colorize numbers
   */
  protected IDStyle getNumberStyle( IDStyle model )
  {
    if( s_number!=null ) return s_number;
    s_number = new DStyle();
    s_number.setForeColor( new DColor(57,123,90));
    return s_number;
  }
  
  private DStyle s_string;
  /**
   * @return style to colorize strings.
   */
  protected IDStyle getStringStyle( IDStyle model )
  {
    if( s_string!=null ) return s_string;
    s_string = new DStyle();
    s_string.setForeColor( new DColor(0,0,255) );
    return s_string;
  }
  
  /**
   * @return style to colorize strings.
   */
  protected IDStyle getCharacterStyle( IDStyle model )
  {
    return getStringStyle(model);
  }

  private DStyle s_keyword;
  /**
   * @return style to colorize keywords.
   */
  protected IDStyle getKeywordStyle( IDStyle model )
  {
    if( s_keyword!=null ) return s_keyword;
    
    DStyle sty = new DStyle();
    if( model!=null && model.getFont()!=null )
    {
      IDFont font = model.getFont();
      sty.setFont( new DFont(font.getFamily(),font.getScript(),font.getSize(),DFont.BOLD) );
    } else {
      sty.setFont( new DFont("courier",null,8,DFont.BOLD));
    }
    sty.setForeColor( new DColor(123,0,82) );
    s_keyword=sty;
    return sty;
  }
  
  /**
   * ceate a piece of text colorized with style, insert if into parent after last, and
   * return this piece of text.
   */
  protected IDItem createText( String text, IDStyle style, IDItem parent, IDItem last )
  {
    DText dt = new DText( text );
    dt.setStyle( style );
    parent.insertChild( dt, last );
    return dt;
  }
}
