/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DLinkResolver.java,v 1.2 2008/05/23 14:11:52 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.tools.internal;

import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;


/**
 * A DLinkResolver is able to find tags in the document to resolve links.
 * As DExtensible the DLinkResolver class can be extended using "doMethod" mechanism
 * 
 * @see DExtensible
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class DLinkResolver extends DExtensible
{
   /** 
    * This class is used as the third parameter of doMethod.
    * if link is resolved by an extension, found tagged item (if any),
    * can be set here.
    */
   public static class Arg
   {
     /** returned by doMethod to said, yes (or no) I resolve this link. 
      * a resolved link can also have null taggedItem and null needdRebuild. */
     public boolean      resolved;
     /** returned by doMethod if this method found a document item associated to link */
     public IDItem   document;
     /** returned by doMethod if this method found tagged item associated to link */
     public IDTaggedItem taggedItem;
     /** returned by doMethod if this method update items of current document, and
      * want (have) to declare it to link resolver caller.
      */
     public IDItem       needRebuild;
     /** Progress monitor to indicate status of the search progress */
     public IDProgressMonitor    pmonitor;
     
   }
   
   
   private IDLink link = null;
   private IDItem document;
   private IDTaggedItem taggedItem;
   private IDItem needRebuild;
   private boolean resolved;
   
   /**
    * Creates a DLinkResolver and initializes the DLink to resolve with l.
    * @param l link to be resolved
    */
   public DLinkResolver( IDLink l ) 
   {
      link = l;    
   }
   
   /**
    * Creates a DLinkResolver.
    */
   public DLinkResolver() 
   {}

   /**
	* Resolves the link.
	*/
   public void resolve(IDProgressMonitor pm) 
   {
	  taggedItem = null;

      Arg arg = new Arg();
      arg.pmonitor = pm;
      
      if (pm!=null)	pm.beginTask("Resolve link", IDProgressMonitor.UNKNOWN);
      invokeDoMethod( link, this, arg ); 
      if (pm!=null) pm.done();
      
      taggedItem  = arg.taggedItem;
      document    = arg.document;
      needRebuild = arg.needRebuild;
      resolved    = arg.resolved;
   }
   
   /** 
    * doMethod for IDLink, if no IDExtension handle link, here is the piece of code to
    * make the last try...
    */
   public void doMethod( IDLink link, DExtensible ext, Object arg )
   {
     if( link==null ) return ;
     if (!DLinkUtil.isInternalLink( link )) return;
    
     IDItem d = link.getLinkedItem();
     if (d == null)
     {
     	System.err.println("JSCRIB ERROR: no item associated with link.");
     	return;
     }
     
     d = d.getRoot();
     if (d == null)
     {
     	System.err.println("JSCRIB ERROR: no root element for this item");
     	return;
     }
     
     if (!(d instanceof IDDocument))
     {
     	System.err.println("JSCRIB ERROR: root element should be an IDDocument");
     	return;
     }

     searchTaggedItem(d, DLinkUtil.getPathItems(link), (Arg)arg); //, ((Arg)arg).pmonitor);   
     
     ((Arg)arg).resolved = true ;
   }

   	
   // search a tagged item with the tag \tag in deep before
   private static IDTaggedItem searchChildTaggedItem(IDItem item, String tag, IDProgressMonitor pm)
   { 
   	  for (IDItem c = item.getFirstChild();c!=null;c=c.getNext())
   	  {
   	  	if (pm!=null)
   	    {
   	  		if (pm.isCanceled()) break;
   	  		pm.worked(1);
   	  	}
   	  	
   	  	if (c instanceof IDTaggedItem)
   	  	{
   	  		IDTaggedItem tc = (IDTaggedItem)c;
   	  		if((tc.getTag()!=null)&&(tc.getTag().equals(tag)))
   	  			return tc;
   	  	}
   	  	
   	  	if (c.getFirstChild()!=null)
   	  	{
   	  		IDTaggedItem f = searchChildTaggedItem(c, tag, pm);
   	  		if (f!=null) return f;
   	  	}

   	  }
   	  
   	  return null;
   }
   
   // search a tagged item with path \tags
   private static void searchTaggedItem(IDItem item, String[] tags, Arg arg) //, IDProgressMonitor pm)
   {     
     if (item==null) return;
     
     // test if the link is jscrib-link:tag or jscrib-link:/tag
     boolean abs = (tags.length>1)&&(tags[0].length()==0);
     int idxstart = 0;
    
     IDItem start = item;
     // if the link is an absolute link we search the document
     // with the tag tags[1]
     if (abs)
     {
     	idxstart = 2;
     	for (;;start=start.getNext())
     	{
     		if ((start instanceof IDDocument)
			    && (((IDDocument)start).getTag().equals(tags[1])))
     		{
     		    arg.document = ((IDDocument)start).getDocument();
     			break;
     		}
     		
     		if (start.getNext()==item) break;
     	
     	}
     	// no document found for the tag
     	if (start==item) return;
     	
     	start = arg.document;
     }
     
     // the document is found we search in deep before
     for (int i=idxstart; i<tags.length;i++)
     {
     	if (arg.pmonitor!=null)
     	{
     		if (arg.pmonitor.isCanceled()) break;
     		arg.pmonitor.worked(1);
     	}
     	
     	start = searchChildTaggedItem(start, tags[i], arg.pmonitor);
     	if (start==null) break;
     }
     
     arg.taggedItem = (IDTaggedItem)start;
     
   }
	  
   /**
    * Access method for the link property.
    * @return   the current value of the link property
    */
   public IDLink getLink() 
   {
      return link;    
   }
   
   /**
    * Sets the value of the link property.
    * @param aLink the new value of the link property
    */
   public void setLink( IDLink aLink) 
   {
      link = aLink;    
   }
   
   /**
    * Access method for the taggedItem property.
    * @return   the current value of the taggedItem property
    */
   public IDTaggedItem getTaggedItem() 
   {
      return taggedItem;    
   }
   
   /**
    * Checks if the IDItem is need to rebuild after link resolution
    * @return true is link is resolved and found that document must be rebuild.
    */
   public IDItem getItemToRebuild()
   {
     return needRebuild;
   }
   
   /**
    * Checks if the link is resolved
    * @return true is last resolve() call have resolved link.
    */
   public boolean isResolved()
   {
     return resolved;
   }
   
   /**
    * This method creates a circular linked list with the documents
    * passed in the table docs. Use this method before calling the method
    * resolve of the DLinkResolver object. All documents should be linked 
    * circulary for the DLinkResolver could find a tag in another document
    * where the link is in.
    * Table might are hole made of null IDDocument inside.
    */
   public static void makeDocumentPool(IDDocument[] docs)
   {
   	  if (docs==null || docs.length==0) return;
   	  
   	  //found first not null document in table:
   	  IDDocument first=null;
   	  int i=0;
   	  for( ;i<docs.length; ++i )
   	  {
   	    if( docs[i]!=null) { first=docs[i]; break; }
   	  }   	  
   	  //link not null documents
   	  IDDocument last=first;
   	  for( ;i<docs.length; ++i )
   	  {
   	    if( docs[i]!=null) {
   	      last.setNext( docs[i] );
   	      last=docs[i];
   	    }
   	  }
   	  //link last document to first one:
   	  last.setNext( first );
   }

   public IDItem getDocument() {
      return document;
   }
}
