package org.eclipse.hyades.logging.java;

import java.io.UnsupportedEncodingException;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.logging.ErrorManager;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.LoggingAgent;

/**********************************************************************
 * Copyright (c) 2004, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SingleLoggingAgentHandler.java,v 1.5 2008/01/24 02:26:07 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Extension of the Java Logging <code>java.util.logging.Handler</code>
 * class used by <code>java.util.logging.Logger</code> instances to publish 
 * logged <code>java.util.logging.LogRecord</code> instances to a single, 
 * named Logging Agent.
 * <p>
 * The name of the single Logging Agent is (see <code>DEFAULT_LOGGING_AGENT_NAME</code>:
 * <p>
 * "Logging Agent Handler"
 * <p>
 * This handler is used to permit multiple <code>java.util.logging.Logger</code>
 * instances to publish logged <code>java.util.logging.LogRecord</code> instances
 * to a single, named Logging Agent, as opposed to a logger-specific Logging Agent.
 * <p>
 * An instance of this class will be returned from the <code>getHandlers()</code> 
 * API on an <code>java.util.logging.Logger</code> instance.  
 * <p>
 * The default <code>java.util.logging.LogManager</code> implementation
 * uses the following configuration variable in the default 
 * &lt;JRE&gt;/lib/logging.properties configuration file to load the configuration 
 * for an <code>java.util.logging.Logger</code> instance:
 * <p>
 * handlers = <existing handlers>, org.eclipse.hyades.logging.java.SingleLoggingAgentHandler
 * <p>
 * Alternatively, an instantiation of this handler class may be set directly on
 * <code>java.util.logging.Logger</code> instances by using the 
 * <code>addHandler()</code> API.
 * <p>
 * By default, the handler's logging level is set to <code>ALL</code> if no run-time
 * configuration value is specified.  This handler is implicitly configured 
 * to use the <code>org.eclipse.hyades.logging.java.XmlFormatter</code> 
 * formatter. 
 * <p>
 * NOTE:  The Java Logging classes must be on the CLASSPATH at run-time to 
 * utilize this handler class.
 * <p>
 * 
 * 
 * @author		Paul E. Slauenwhite
 * @version	    July 28, 2005
 * @since       July 19, 2004
 * @see         java.util.logging.Handler
 * @see         java.util.logging.Level
 * @see         java.util.logging.LogManager
 * @see         org.eclipse.hyades.logging.java.XmlFormatter
 */
public class SingleLoggingAgentHandler extends Handler {

    /**
     * Flag denoting if this handler instance has been closed (e.g. see <code>close()</code>).
     */
    protected boolean isClosed = false;

    /**
     * The single Logging Agent to which successfully logged messages are written.
     */
    protected static LoggingAgent loggingAgent = null;

    /**
     * Instance count of this class.
     */
    protected static int instanceCount = 0;

    /**
     * Current thread lock for synchronized operations.
     */
    protected static final Object CURRENT_THREAD_LOCK = new Object();

    /**
     * Default Logging Agent name.
     */
    public static final String DEFAULT_LOGGING_AGENT_NAME = "Logging Agent Handler";

    /**
     * Constructor to create a Logging Agent Handler.
     * <p>
     * The configuration for this type of handler is retrieved from the
     * <code>java.util.logging.LogManager</code> and set on this handler
     * instance.
     * <p>
     * Configuration includes:
     * <p>
     * <ol>
     * <li>Level</li>
     * <li>Filter</li>
     * <li>Encoding</li>
     * </ol>
     * <p>
     * The default values for these configuration properties are:
     * <p>
     * <ol>
     * <li>ALL</li>
     * <li>null</li>
     * <li>null or default platform encoding</li>
     * </ol>
     * <p>
     * NOTE: This handler is implicitly configured to use the 
     * <code>org.eclipse.hyades.logging.java.XmlFormatter</code> 
     * formatter. 
     */
    public SingleLoggingAgentHandler() {

        //Retrieve the LogManager:
        LogManager logManager = LogManager.getLogManager();

        //Retrieve the class' name:
        final String className = this.getClass().getName();

        //Retrieve and set the handler's level based on the
        // org.eclipse.hyades.logging.java.SingleLoggingAgentHandler.level property:
        try {
            setLevel(Level.parse(logManager.getProperty(className.concat(".level")).trim()));
        } 
        catch (Throwable t) {
            //Ignore since the default level of the super class is Level.ALL.
        }

        //Retrieve the handler's filter based on the
        // org.eclipse.hyades.logging.java.SingleLoggingAgentHandler.filter property:
        String defaultProperty = logManager.getProperty(className.concat(".filter"));

        //If the filter is not specified then do no filtering. Otherwise
        // instantiate our filter:
        if (defaultProperty == null) {
            setFilter(null);
        } 
        else {

            try {
                
                final String finalDefaultProperty = defaultProperty.trim();
                
                //Consult the system's class loader to resolve the configured 
                //filter class using privileged security:
                Class filterClass = ((Class) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                    public Object run() throws Exception {
                        return (ClassLoader.getSystemClassLoader().loadClass(finalDefaultProperty));
                    }
                })));

                setFilter((Filter) (filterClass.newInstance()));               
            } 
            catch (Throwable t) {
                
                //Ignore exception since insufficient security privileges
                //for accessing the system's class loader or cannot
                //instantiate the configured filter class.
                setFilter(null);
            }
        }

        //Retrieve the handler's encoding based on the
        // org.eclipse.hyades.logging.java.SingleLoggingAgentHandler.encoding
        // property:
        defaultProperty = logManager.getProperty(className.concat(".encoding"));

        //If the encoding is not specified we set it to be null:
        if (defaultProperty == null) {
            
            try {
                setEncoding(null);
            } 
            catch (UnsupportedEncodingException u) {
                //Ignore since cannot set the handler's encoding to default platform encoding or 'null'.
            }
        } else {
            
            try {
                setEncoding(defaultProperty.trim());
            } 
            catch (Throwable t) {
                
                try {
                    setEncoding(null);
                } 
                catch (UnsupportedEncodingException u) {
                    //Ignore since cannot set the handler's encoding to default platform encoding or 'null'.
                }
            }
        }

        super.setFormatter(new XmlFormatter());
        
        synchronized (CURRENT_THREAD_LOCK) {
            instanceCount++;
        }
    }

    /**
     * Publishing a <code>java.util.logging.LogRecord</code> instance to a single, 
     * named Logging Agent.
     * <p>
     * The <code>java.util.logging.LogRecord</code> is converted using
     * the handler's implicit formatter (e.g.
     * <code>org.eclipse.hyades.logging.java.XmlFormatter</code>) and
     * propagated on to the handler's single, named Logging Agent.
     * <p>
     * This method quietly returns if the handler has been closed 
     * (e.g. see <code>close()</code>) and/or the parameter 
     * <code>java.util.logging.LogRecord</code> is not loggable 
     * (e.g. see <code>isLoggable()</code>).
     * <p>
     * 
     * @param logRecord
     *            The <code>java.util.logging.LogRecord</code> to be published
     *            to a single, named Logging Agent.
     */
    public void publish(LogRecord logRecord) {

        //Checks if the handler has not been closed, the LogRecord has an 
        //appropriate level and whether it satisfies all filter(s):
        if ((!isClosed) && (isLoggable(logRecord))) {
           
            String formattedLogRecord = null;

            try {
                formattedLogRecord = getFormatter().format(logRecord).trim();
            } 
            catch (Exception e) {
            
                writeError(null, e, ErrorManager.FORMAT_FAILURE);
                
                return;
            }

            synchronized (CURRENT_THREAD_LOCK) {
                
                if(loggingAgent == null){
                    loggingAgent = new LoggingAgent(DEFAULT_LOGGING_AGENT_NAME);
                }

                loggingAgent.write(formattedLogRecord);
            }
        }
    }

    /**
     * Flush any waiting messages to the single, named Logging Agent.
     * <p>
     * This method quietly returns if the handler has been closed 
     * (e.g. see <code>close()</code>).
     */
    public void flush() {

        if(!isClosed){

            synchronized (CURRENT_THREAD_LOCK) {
                
                if((loggingAgent != null) && (loggingAgent.isLogging())){
                    loggingAgent.flush();
                }
            }
        }
    }

    /**
     * Flush any waiting messages and de-registering the single, named Logging Agent.
     * <p>
     * This method quietly returns if the handler has been closed 
     * (e.g. see <code>close()</code>).
     */
    public void close() {
        
        if(!isClosed){
            
            flush();
            
	        synchronized (CURRENT_THREAD_LOCK) {
	                    
                if((--instanceCount == 0) && (loggingAgent != null)){
                    loggingAgent.deregister();
	            }
	        }
	        
	        isClosed = true;
        }
    }

    /**
     * @see java.lang.Object#finalize()
     */
    protected void finalize() throws Throwable {
        
        super.finalize();
        
        close();
    }
    
    /**
     * This method quietly returns since the handler's formatter is implicitly configured.
     * <p>
     * 
     * @see java.util.logging.Handler#setFormatter(java.util.logging.Formatter)
     */
    public void setFormatter(Formatter newFormatter) throws SecurityException {
         //No-operation since the handler's formatter is implicitly configured.
    }
    
    /**
     * Private utility method for retrieving the
     * <code>java.util.logging.ErrorManager</code> and writing an error
     * message, exception and error code.
     * <p>
     * If the <code>java.util.logging.ErrorManager</code> can not be
     * retrieved, the resultant exception and stack trace are written to
     * standard error.
     * <p>
     * @param message
     *            The error message to be written to the
     *            <code>java.util.logging.ErrorManager</code>.
     * @param exception
     *            The error exception to be written to the
     *            <code>java.util.logging.ErrorManager</code>.
     * @param code
     *            The error code to be written to the
     *            <code>java.util.logging.ErrorManager</code>.
     */
    protected void writeError(String message, Exception exception, int code) {

        try {
            getErrorManager().error(message, exception, code);
        } 
        catch (Throwable t) {
            
            System.err.println("Error: " + t.getMessage());
            t.printStackTrace();
        }
    }
}