package org.eclipse.hyades.logging.java;

import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XmlFormatter.java,v 1.12 2008/04/10 01:53:54 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Java Logging <code>java.util.logging.Formatter</code>
 * class used by <code>java.util.logging.Handler</code> instances to format
 * <code>java.util.logging.LogRecord</code> instances to XML.
 * <p>
 * If the <code>java.util.logging.LogRecord</code> instance is an
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code> (e.g.
 * instance of the
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
 * interface), the encapsulated Common Base Event is serialized to XML based on
 * the Common Base Event v.1.0.1 specification. Otherwise, the
 * <code>java.util.logging.LogRecord</code> is first converted to a Common
 * Base Event (see below for mapping details) and serialized to XML based on the 
 * Common Base Event v.1.0.1 specification.
 * <p>
 * The <code>java.util.logging.LogRecord</code> instance is converted to a Common
 * Base Event by mapping the <code>java.util.logging.LogRecord</code> properties 
 * to Common Base Event properties using the following mapping:
 * <p>
 * <table border="1" cellpadding="10">
 * <tr>
 * <th><code>java.util.logging.LogRecord</code> Property</th>
 * <th><code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> Property</th>
 * </tr>
 * <tr>
 * <td>level</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:level" <br>
 * ExtendedDataElement[0...n].Type = "noValue" <br>
 * ExtendedDataElement[0...n].Children[0].Name = "name" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's name&gt; <br>
 * ExtendedDataElement[0...n].Children[1].Name = "value" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "int" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's numerical
 * value&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>loggerName</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:loggerName"
 * <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;logger's name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>message#</td>
 * <td>
 * Unresolved (e.g. not localized or formatted) messages:<br>
 * MsgDataElement.MsgCatalogId = &lt;message&gt;<br>
 * - or - <br>
 * Resolved (e.g. localized and formatted) messages:<br>
 * msg = &lt;message&gt;<br>
 * </td>
 * </tr>
 * <tr>
 * <td>millis</td>
 * <td>creationTime = &lt;millis&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>parameters*</td>
 * <td>MsgDataElement.MsgCatalogTokens = &lt;string representation of
 * parameters&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundle*#</td>
 * <td>[not mapped] <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundleName</td>
 * <td>MsgDataElement.MsgCatalog = &lt;resource bundle name&gt; <br>
 * MsgDataElement.MsgCatalogType = "Java" <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sequenceNumber</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sequenceNumber" <br>
 * ExtendedDataElement[0...n].Type = "long" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;sequence number&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceClassName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceClassName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source class name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceMethodName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceMethodName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source method name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>threadID</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:threadID"
 * <br>
 * ExtendedDataElement[0...n].Type = "int" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;thread ID&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>thrown*</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:stackTrace" <br>
 * ExtendedDataElement[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Values[0...n] =
 * [&lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
 * <code>java.lang.Throwable</code>'s localized message&gt;]]<br>
 * ExtendedDataElement[0...n].Values[(n + 1)...m] =
 * &lt; <code>java.lang.Throwable</code>'s stackTraceElement[0...(m - (n + 1))]&gt;]<br>
 * <br>
 * ExtendedDataElement[0...n].Children[0...m].Name = "Cause" <br>
 * ExtendedDataElement[0...n].Children[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Children[0...m].Values[0...n] =
 * &lt;cause's class name&gt;[: &lt;cause's localized message&gt;]<br>
 * ExtendedDataElement[0...n].Children[0...m].Values[(n + 1)...m] =
 * &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
 * <li>[children[0] = &lt;cause's cause&gt;]<br>
 * </td>
 * </tr>
 * </table>
 * <p>
 * <b>* </b> <code>java.util.logging.LogRecord</code> property is stored in
 * <code>java.util.logging.LogRecord</code> super class to eliminate loss of
 * data.
 * <p>
 * <b># </b>When the <code>java.util.logging.LogRecord.resourceBundleName</code> 
 * (e.g <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>) 
 * is set to a non-null resource bundle name, the non-null <code>java.util.logging.LogRecord.message</code> property
 * is assumed to be resolved (e.g. localized and formatted) and set to the 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalogId<code>
 * property.  Conversely, when the <code>java.util.logging.LogRecord.resourceBundleName</code> 
 * (e.g <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>) 
 * is not set (e.g. null resource bundle name), the non-null <code>java.util.logging.LogRecord.message</code> property
 * is assumed to be unresolved (e.g. no localized or formatted) and set to the 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msg</code>
 * property.  Furthermore, when the <code>java.util.logging.LogRecord.resourceBundleName</code> 
 * (e.g <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>) 
 * is set to a non-null resource bundle name, the <code>java.util.logging.LogRecord.message</code> 
 * property is assumed to be resolved (e.g. localized and formatted) and moved from the 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msg</code> property to the 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalogId<code>
 * property.  Conversely, when the <code>java.util.logging.LogRecord.resourceBundleName</code> 
 * (e.g <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>) 
 * is set to a null resource bundle name, the <code>java.util.logging.LogRecord.message</code> property 
 * is assumed to be unresolved (e.g. no localized or formatted) and moved from the
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalogId<code>
 * property to the <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msg</code>
 * property.  Noteworthy, the <code>java.util.logging.LogRecord.resourceBundle</code> is not used to 
 * determine if the <code>java.util.logging.LogRecord.message</code> property is resolved 
 * (e.g. localized and formatted) since the resource bundle name 
 * (e.g <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>) 
 * cannot be resolved from the <code>java.util.logging.LogRecord.resourceBundle</code> thereby generating 
 * an invalid Common Base Event (e.g. <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalogId<code>
 * without the required <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent.msgDataElement.MsgCatalog<code>).
 * <p>
 * If the Extended Data Element's <code>values</code> property is
 * larger than 1024 characters, the string is segmented into a String array of
 * 1024-character elements and set on the Extended Data Element. However, if the
 * Extended Data Element's <code>values</code> property is 1024 or less
 * characters, the string is set directly on the first element a String array
 * and set on the Extended Data Element. As such, the <code>type</code>
 * property is set to <code>stringArray</code>.
 * <p>
 * Names identifying the associated
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
 * properties in the encapsulated Common Base Event for the
 * <code>java.util.logging.LogRecord</code> properties are provided as static
 * string constants in the <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code> 
 * class. These name properties are prefixed by the
 * <code>CommonBaseEventLogRecord:</code> namespace (see
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
 * <p>
 * NOTE: To ensure a well-formed Common Base Event is generated, a default <code>Situation</code> 
 * and <code>SourceComponentId</code> properties are constructed.
 * <p>
 * An instance of this class will be returned from the
 * <code>getFormatter()</code> API on <code>java.util.logging.Handler</code>
 * instances.
 * <p>
 * The default <code>java.util.logging.LogManager</code> implementation uses
 * the following configuration variable in the default
 * &lt;JRE&gt;/lib/logging.properties configuration file to load configuration
 * for <code>java.util.logging.Logger</code> instances:
 * <p>
 * <handler name>.formatter = org.eclipse.hyades.logging.java.XmlFormatter
 * <p>
 * Alternatively, an instantiation of this formatter class may be set directly
 * on <code>java.util.logging.Handler</code> instances by using the
 * <code>setFormatter()</code> API.
 * <p>
 * Callers may provide a Common Base Event XML file template (see 
 * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
 * newly created Common Base Event before serialization to XML.  The template 
 * Event XML file is a well-formed XML document conforming to a predefined XML schema 
 * (e.g. templateEvent.xsd).  The naming convention used for the template Event XML file 
 * is:
 * <p>
 * &lt;logger name&gt;.event.xml
 * <p>
 * where &lt;logger name&gt; is the name of the logger specified in the parameter 
 * <code>java.util.logging.LogRecord</code>.  For example, the template event XML file 
 * for the <code>myLogger</code> logger factory would be named <code>myLogger.event.xml</code>.
 * <p>
 * The desired event configuration template XML file must by on the classpath
 * and accessible by this class. The event configuration template XML file is
 * loaded using the following look-up sequence:
 * <p>
 * <ol>
 * <li>The <code>XmlFormatter</code> class' class loader
 * is queried for the event configuration template XML file.</li>
 * <li>The system's class loader is queried for the event configuration
 * template XML file.</li>
 * <li>The current thread's context class loader is queried for the event
 * configuration template XML file.</li>
 * </ol>
 * <p>
 * NOTE: The Java Logging classes must be on the CLASSPATH at run-time to
 * utilize this formatter class.
 * <p>
 * 
 * @author   Paul E. Slauenwhite
 * @version  July 28, 2006
 * @since    April 1, 2003
 * @see      java.util.logging.Formatter
 * @see      java.util.logging.LogRecord
 * @see      org.eclipse.hyades.logging.java.CommonBaseEventLogRecord
 * @see      org.eclipse.hyades.logging.core.IExternalizableToXml
 * @see      org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl
 */
public class XmlFormatter extends Formatter {

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for
     * generating event instances.
     */
    protected final static EventFactory EVENT_FACTORY = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactoryHome</code> for
     * generating pre-configured event instances.
     */
    protected final static EventFactoryHome EVENT_FACTORY_HOME = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl", XmlFormatter.class.getClassLoader());
    
    /**
     * Generates the XML representation of the parameter
     * <code>java.util.logging.LogRecord</code> instance.
     * <p>
     * If the parameter <code>java.util.logging.LogRecord</code> instance is
     * an <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * (e.g. instance of the
     * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
     * interface), the encapsulated Common Base Event is serialized to XML based
     * on the Common Base Event v.1.0.1 specification. Otherwise, the
     * <code>java.util.logging.LogRecord</code> is first converted to a Common
     * Base Event (see class comment header for mapping details) and serialized to 
     * XML based on the Common Base Event v.1.0.1 specification.
     * <p>
     * Callers may provide a Common Base Event XML file template (see 
     * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
     * newly created Common Base Event before serialization to XML.  See class comment 
     * header for more details.
     * <p>
     * 
     * @param logRecord
     *            The <code>java.util.logging.LogRecord</code> instance to be
     *            formatted to XML.
     * @return The XML representation of the parameter
     *         <code>java.util.logging.LogRecord</code> instance.
     * @see java.util.logging.Formatter#format(LogRecord)
     */
    public String format(LogRecord logRecord) {

        //Determine if the parameter LogRecord subclass implements IExternalizableToXml (e.g. CommonBaseEventLogRecord):
        try {
            return (((IExternalizableToXml)(logRecord)).externalizeCanonicalXmlString().concat(Constants.LINE_SEPARATOR));
        } 
        catch (Exception e) {
            //Ignore since LogRecord subclass does not implement IExternalizableToXml or a serialization error.
        }

        //Map the parameter LogRecord's properties to Common Base Event properties:
        CommonBaseEvent commonBaseEvent = null;
        String loggerName = logRecord.getLoggerName();
        
        if((loggerName != null) && (loggerName.trim().length() > 0)){

            //Retrieve an EventFactory instance based on the logger name and create a pre-configured Common Base Event based 
            //on the an XML configuration template file that exists on the class loader's or context class loader's classpath.
            commonBaseEvent = EVENT_FACTORY_HOME.getEventFactory(loggerName).createCommonBaseEvent();
            
            try {
                commonBaseEvent.complete();
            } 
            catch (CompletionException c) {
                //Ignore since content completion is based on 'best-effort'.
            }
        }
        else{
            commonBaseEvent = EVENT_FACTORY.createCommonBaseEvent();
        }
        
        //Create and populate a default report situation:
        if(commonBaseEvent.getSituation() == null){

            //Create a new instance of a report situation:
            ReportSituation reportSituation = EVENT_FACTORY.createReportSituation();
            reportSituation.setReasoningScope("INTERNAL");
            reportSituation.setReportCategory("LOG");
    
            //Create a new instance of a situation:
            Situation situation = EVENT_FACTORY.createSituation();
            situation.setCategoryName(Situation.REPORT_SITUATION_CATEGORY);
            situation.setSituationType(reportSituation);

            commonBaseEvent.setSituation(situation);
        }
        
        //Create and populate a default source component ID:
        if(commonBaseEvent.getSourceComponentId() == null){

            //Create a new instance of a component ID:
            ComponentIdentification sourceComponentId = EVENT_FACTORY.createComponentIdentification();
            sourceComponentId.setLocation(Constants.LOCAL_HOST_IP_ADDRESS);
            sourceComponentId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
            sourceComponentId.setExecutionEnvironment("Java");
            sourceComponentId.setComponent("Logging");
            sourceComponentId.setSubComponent("Logger");
            sourceComponentId.setComponentIdType("Application");
            sourceComponentId.setComponentType("Logging_Application");

            commonBaseEvent.setSourceComponentId(sourceComponentId);
        }
        
        //Map the sequence number property:
        ExtendedDataElement sequenceNumberExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
        sequenceNumberExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);
        sequenceNumberExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_LONG_VALUE);
        sequenceNumberExtendedDataElement.setValues(new String[] { Long.toString(logRecord.getSequenceNumber())});

        commonBaseEvent.addExtendedDataElement(sequenceNumberExtendedDataElement);

        //Map the thread ID property:
        ExtendedDataElement threadIDExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
        threadIDExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
        threadIDExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
        threadIDExtendedDataElement.setValues(new String[] { Integer.toString(logRecord.getThreadID())});

        commonBaseEvent.addExtendedDataElement(threadIDExtendedDataElement);

        //Map the millis property:
        commonBaseEvent.setCreationTimeAsLong(logRecord.getMillis());

        //Map the level property:
        Level level = logRecord.getLevel();
        
        if(level != null){
            
            ExtendedDataElement levelNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            levelNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
            levelNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            levelNameExtendedDataElement.setValues(new String[] { level.getName()});
    
            ExtendedDataElement levelValueExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            levelValueExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);
            levelValueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
            levelValueExtendedDataElement.setValues(new String[] { Integer.toString(level.intValue())});
    
            ExtendedDataElement levelExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            levelExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL);
            levelExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
            levelExtendedDataElement.addChild(levelNameExtendedDataElement);
            levelExtendedDataElement.addChild(levelValueExtendedDataElement);
    
            commonBaseEvent.addExtendedDataElement(levelExtendedDataElement);
            
            if (Level.SEVERE.intValue() >= level.intValue()) {
                commonBaseEvent.setSeverity((short) 50);
            } else if (Level.WARNING.intValue() >= level.intValue()) {
                commonBaseEvent.setSeverity((short) 30);
            } else {
                commonBaseEvent.setSeverity((short) 10);
            }
        }

        //Map the logger name property:
        if(loggerName != null){
            
            ExtendedDataElement loggerNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            loggerNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
            loggerNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            loggerNameExtendedDataElement.setValues(new String[] {loggerName});
    
            commonBaseEvent.addExtendedDataElement(loggerNameExtendedDataElement);
        }
                
        //Map the parameters property:
        Object[] parameters = logRecord.getParameters();
        
        if (parameters != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            } 

            MsgCatalogToken msgCatalogToken = null;

            for (int counter = 0; counter < parameters.length; counter++) {

                msgCatalogToken = EVENT_FACTORY.createMsgCatalogToken();
                msgCatalogToken.setValue(String.valueOf(parameters[counter]));

                msgDataElement.addMsgCatalogToken(msgCatalogToken);
            }
        }

        //Map the resource bundle name property:
        String resourceBundleName = logRecord.getResourceBundleName();
        
        if(resourceBundleName != null){
            
            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            }

            msgDataElement.setMsgCatalogType("Java");
            msgDataElement.setMsgCatalog(resourceBundleName);
        }
                
        //Map the message property:
        String message = logRecord.getMessage();
        
        if(message != null){
        
            //ASSUMPTION:  The message property is assumed to be resolved (e.g. localized and 
            //formatted) if the resource bundle name has not been set, otherwise the 
            //message property is assumed to be unresolved (e.g. unlocalized and unformatted).
            if (resourceBundleName != null) {
                commonBaseEvent.getMsgDataElement().setMsgCatalogId(message);
            }
            else{
                commonBaseEvent.setMsg(formatMessage(logRecord));
            }
        }

        //Map the source class name property:
        String sourceClassName = logRecord.getSourceClassName();
        
        if(sourceClassName != null){
            
            ExtendedDataElement sourceClassNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            sourceClassNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);
            sourceClassNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            sourceClassNameExtendedDataElement.setValues(new String[] { sourceClassName});

            commonBaseEvent.addExtendedDataElement(sourceClassNameExtendedDataElement);
        }

        //Map the source method name property:
        String sourceMethodName = logRecord.getSourceMethodName();
        
        if(sourceMethodName != null){
            
            ExtendedDataElement sourceMethodNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            sourceMethodNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);
            sourceMethodNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            sourceMethodNameExtendedDataElement.setValues(new String[] { sourceMethodName});

            commonBaseEvent.addExtendedDataElement(sourceMethodNameExtendedDataElement);
        }

        //Map the thrown property:
        Throwable thrown = logRecord.getThrown();
        
        if(thrown != null){
            commonBaseEvent.addExtendedDataElement(EventHelpers.convertToExtendedDataElement(thrown,CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN));
        }
        
        return (((IExternalizableToXml)(commonBaseEvent)).externalizeCanonicalXmlString().concat(Constants.LINE_SEPARATOR));
    }
}