package org.eclipse.hyades.logging.core;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoggingAgent.java,v 1.6 2008/01/24 02:26:07 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

import java.net.InetAddress;

import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.AgentConfiguration;
import org.eclipse.hyades.internal.execution.remote.AgentControllerListener;
import org.eclipse.hyades.internal.execution.remote.AgentNotRegisteredException;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.MonitorListener;
import org.eclipse.hyades.internal.execution.remote.QueueManager;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;

/**
 * An agent used to queue logged records until the Agent Controller
 * is available and the Logging Agent is being monitored at which time all queued and
 * future logged records are written to the Agent Controller. 
 * The default type of the agent is 'logging'. 
 * <p>
 * A Logging Agent instance is created via the constructor.
 * For example,
 * <p>
 * <code>
 * LoggingAgent loggingAgent = new LoggingAgent("My Logging Agent");
 *
 * loggingAgent.write("My log message.");
 * </code>
 * 
 * A Logging Agent instance which type is other than 'logging' can be created
 * via the constructor.  
 * 
 * <code>
 * public LoggingAgent(String name, String type)
 * </code>
 *
 * @author      CindyJ
 * @author		Paul E. Slauenwhite
 * @author		Richard K. Duggan
 * @version	    September 4, 2006
 * @since	    March 7, 2003
 */
public class LoggingAgent {

    /**
     * The type of the Logging Agent.
     */
    public static final String LOGGING_AGENT_TYPE = "Logging";

    /**
     * The skeleton to which to write log records.
     */
    protected RemoteComponentSkeleton delegate;

    /**
     * The queue for log records until the Agent Controller is available and the Logging Agent is being monitored.
     */
    protected QueueManager queue; 

    /**
     * If the Agent Controller is available.
     */
    protected boolean controllerAvailable = false;

    /**
     * If the Logging Agent is being monitored.
     */
    protected boolean isMonitored = false;

    /**
     * Current thread lock.
     */
    protected Object currentThreadLock = new Object();

    /**
     * Constructor to create and initialize a Logging Agent instance using
     * the parameter name.
     *
     * @param name The name of the newly created Logging Agent.
     */
    public LoggingAgent(String name) {
    	this(name,100,LOGGING_AGENT_TYPE); 
    }
    
    /**
     * Constructor to create and initialize a Logging Agent instance using
     * the parameter name and type.
     *
     * @param name The name of the newly created Logging Agent.
     * @param type The type of the newly created Logging Agent.
     */
    public LoggingAgent(String name, String type) {
    	this(name,100, type); 
    }
    
    /**
     * Constructor to create and initialize a Logging Agent instance.
     *
     * @param name The name of the newly created Logging Agent.
     * @param bufferSize the size of the buffer to use
     * @param type The type of the newly created Logging Agent.
     */
    public LoggingAgent(String name, int bufferSize,String type) {

    	queue = new QueueManager(bufferSize);
        try {
            delegate = new RemoteComponentSkeleton(name, type);

            delegate.addAgentControllerListener(new AgentControllerListener() {

                public void agentControllerActive() {
                    controllerAvailable = true;
                }

                public void agentControllerInactive() {
                    controllerAvailable = false;
                }
            });

			delegate.addMonitorListener(new MonitorListener() {

				public void monitorActive() {

					synchronized (currentThreadLock) {
						isMonitored = true;
						currentThreadLock.notify();
					}

					flush();
				}

				public void monitorInactive() {
					isMonitored = false;
					flush();
				}
			});

			/**
			 * Bugzilla 113173: This is for the handshake with the Agent Controller
			 * a better chance of the Agent Controller active response being received before LoggingAgent 
			 * constructor returns.
			 */
            delegate.initialize();
        }
        catch (Throwable e) {
            controllerAvailable = false;
        }
    }
    /**
     * Constructor to create and initialize a Logging Agent instance using
     * the parameter name.
     *
     * @param name The name of the newly created Logging Agent.
     * @param bufferSize the size of the buffer to use
     */
    public LoggingAgent(String name, int bufferSize) {

    	this(name,bufferSize,LOGGING_AGENT_TYPE);
    }

    /**
     * Retrieves the default configuration for the Logging Agent.
     * <p>
     * 
     * @return The Logging Agent's default configuration.
     * @deprecated As of TPTP V4.2.0, this method is deprecated since the default configuration for the Logging Agent (e.g. <code>AgentConfiguration</code>) is internal.
     */
    public AgentConfiguration getLoggingAgentDefaultConfiguration() {
        return (delegate.getDefaultConfiguration());
    }

    /**
     * Checks if the Logging Agent is logging (e.g. Agent Controller is available and the Logging Agent is being monitored).
     *
     * @return True if the Logging Agent is logging, otherwise false.
     */
    public boolean isLogging() {
        return ((controllerAvailable) && (isMonitored));
    }

    /**
     * Writes the parameter log record to the Agent Controller
     *
     * If the Agent Controller is available and  the Logging Agent is being
     * monitored, otherwise the parameter log record is added to the Logging
     * Agent's queue of log records waiting to be written to the Agent Controller.
     *
     * @param msg The log record to be written to the Agent Controller.
     */
    public void write(String msg) {

        if (isLogging())
            delegate.logMessageUTF8(msg);

        else
            enQueue(msg);
    }

    /**
     * Adds the parameter log record to the Logging Agent's queue of log
     * records waiting to be written to the Agent Controller.
     *
     * @param msg The log record to be added Logging Agent's queue of log records.
     */
    public void enQueue(String msg) {
        queue.insertInQueue(msg);

    }

    /**
     * Writes any queued log record to the Agent Controller.
     */
    public void flush() {

        if (queue.isFlushNeeded())
            queue.flushCurrentQueue(delegate);
    }

    /**
     * Deregisters this Logging Agent instance from the Agent Controller.
     */
    public void deregister() {
        delegate.deregister();
    }

    /**
     * Retrieves the name of this Logging Agent.
     *
     * @return The name of this Logging Agent.
     */
    public String getName() {
        return delegate.getName();
    }
    
    public String getType() {
        return delegate.getType();
    }

    /**
     * Suspends the current thread until the logging agent is monitored or the
     * the parameter maximum time (milliseconds) has expired.
     *
     * @param maxTime the maximum amount of time (milliseconds) to suspend the current thread
     * @return true if the logger is being monitored
     */
    public boolean waitUntilLogging(long maxTime) {

        synchronized (currentThreadLock) {

            try {
                if (!isLogging())
                    currentThreadLock.wait(maxTime);
            }
            catch (InterruptedException e) {
            }
        }

        return isLogging();
    }



	/**
	 * The following method is used to only log error messages to an active
	 * logging agent.
	 *
	 * @param message	-	The message to be logged
	 *		  messageId -	The message ID corresponding to the error message
	 *		  severity	-	The level of severity of the message (e.g. default = DEBUG, 1 = INFORMATION, 2 = WARNING, 3 = SEVERE, 4 = MAX ERROR LEVEL)
	 */
    public void logErrorMessage(String message, String messageId, int severity) {    	
    	delegate.logErrMsg(message, messageId, severity);
    }
    
    /**
	 * Request for this agent to be monitored by the same client that is
	 * monitoring another agent.
	 * 
	 * @param addr  java.net.InetAddress
	 * @param port The port number
	 * @param agentUUID The agent's UUID
	 * @param timeout milliseconds to wait for the client to start monitoring.
	 *            
	 */
	public void requestMonitorThroughPeer_p(InetAddress addr, long port, String agentUUID, long timeout) {
		delegate.requestMonitorThroughPeer_p(addr, port, agentUUID, timeout);
	}

	/**
	 * Request for this agent to be monitored by the same client that
	 * is monitoring another agent. Using the default port.
	 *
	 * @param addr  java.net.InetAddress
	 * @param agentUUID The agent's UUID
	 * @param timeout Milliseconds to wait for the client to start monitoring.
	 *            
	 */
	public void requestMonitorThroughPeer(InetAddress addr, String agentUUID, long timeout) {
		delegate.requestMonitorThroughPeer(addr, agentUUID, timeout);
	}

	
	/**
	 * Request for this agent to be monitored by the same client that is
	 * monitoring another agent.
	 * 
	 * @param addr java.net.InetAddress
	 * @param port The port number
	 * @param pid The process ID
	 * @param agentName The angent's name
	 * @param timeout Milliseconds to wait for the client to start monitoring.
	 */
	public void requestMonitorThroughPeer_p(InetAddress addr, long port, long pid, String agentName, long timeout) {
		delegate.requestMonitorThroughPeer_p(addr, port, pid, agentName, timeout);
		
	}

	/**
	 * Request for this agent to be monitored by the same client that
	 * is monitoring another agent. Using the default port.
	 * 
	 * @param addr java.net.InetAddress
     * @param pid The process ID
	 * @param agentName The angent's name
	 * @param timeout Milliseconds to wait for the client to start monitoring.
	 */
	public void requestMonitorThroughPeer(InetAddress addr, long pid, String agentName, long timeout) {
		delegate.requestMonitorThroughPeer(addr, pid, agentName, timeout);
	}

	/**
	 * Get the process ID
	 * 
	 * @return The process ID.
	 */
	public long getPID() {
		return delegate.getPID();
	}
	
	/**
	 * Get the agent's UUID
	 * 
	 * @return The agent's UUID
	 */
	
	public String getAgentUUID() {
		return delegate.getAgentUUID();
		
	}
	/**
	 * Generate UUID
	 * @return UUID String
	 */
	public String generateUUID(){
		return delegate.generateUUID();
	}
	
	/**
	 * Get the node's UUID
	 * 
	 * @return The node's UUID
	 */
	public String getNodeUUID() throws AgentNotRegisteredException {
		return delegate.getNodeUUID();
	}
	
	/**
	 * Get the JVM's UUID
	 * 
	 * @return The JVM's UUID
	 */
	public String getJVMUUID() throws AgentNotRegisteredException {
		return delegate.getJVMUUID();
	}
	
	/**
	 * Broadcast messages to attched clients.
	 * @param message The message string to broadcast
	 * @param contextId 
	 */
	
	public void broadcastMessage(String message, long contextId) {
		delegate.broadcastMessage(message, contextId);
	}
	
	/**
	 * Broadcast byte messages to attched clients.
	 * @param message
	 * @param offset
	 * @param length
	 * @param contextId
	 */
	public void broadcastMessage(byte[] message, int offset, int length, long contextId) {
		delegate.broadcastMessage(message, offset, length, contextId);
	}
	
	/**
	 * 
	 * Register the monitor listener.
	 * 
	 * @param listener The LoggingAgentMonitorListener instance.
	 */
	public void addMonitorListener(final LoggingAgentMonitorListener listener) {
			
			delegate.addMonitorListener(new MonitorListener(){
					 
					public void monitorActive(){
						listener.loggingAgentMonitorActive();
					}
					   
					public void monitorInactive(){
						listener.loggingAgentMonitorInactive();
					}
			});
			}
	
	/**
	 * Register the command listener.
	 * 
	 * @param listener The LoggingAgentCustomCommandHandler instance
	 */
	public void addCommandListener(final LoggingAgentCustomCommandHandler listener){
		delegate.addCommandListener(new CustomCommandHandler(){

			public void handleCommand(CustomCommand command){
				listener.handleLoggingAgentCustomCommand(new LoggingAgentCustomCommand(command.getData()));
			}
		});
	}
	/**
     * Checks if the Agent Controller is available.
     *
     * @return True if the Agent Controller is available, otherwise false.
     */
	public boolean isAgentControllerAvailable ()
	{
		return controllerAvailable;
	}
}
