package org.eclipse.hyades.logging.log4j;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.Layout;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.hyades.logging.core.LoggingAgent;

/**********************************************************************
 * Copyright (c) 2004, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SingleLoggingAgentAppender.java,v 1.4 2008/01/24 02:26:07 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Log4J <code>org.apache.log4j.AppenderSkeleton</code> abstract class
 * used by <code>org.apache.log4j.Logger</code> instances to publish logged 
 * <code>org.apache.log4j.spi.LoggingEvent</code> instances to a single, named Logging 
 * Agent.
 * <p>
 * The name of the single Logging Agent is (see <code>DEFAULT_LOGGING_AGENT_NAME</code>:
 * <p>
 * "Logging Agent Appender"
 * <p>
 * This appender is used to permit multiple <code>org.apache.log4j.Logger</code>
 * instances to publish logged <code>org.apache.log4j.spi.LoggingEvent</code> instances
 * to a single, named Logging Agent, as opposed to a logger-specific Logging Agent.
 * <p>
 * An instance of this class will be returned from the
 * <code>getAllAppenders()</code> API on an <code>org.apache.log4j.Logger</code>
 * instance.
 * <p>
 * Users may configure a <code>org.apache.log4j.Logger</code> instance to use the 
 * <code>SingleLoggingAgentAppender</code> by adding the following the entry to the 
 * <code>&lt;log4j:configuration&gt;</code&gt; element in the <code>log4j.xml</code> 
 * properties file:
 * <p>
 * <code>
 * &lt;!-- SingleLoggingAgentAppender --&gt;
 * &lt;!-- Outputs events to a single, named Logging Agent using possible filter(s).  Layout is performed implicitly. --&gt;
 * &lt;appender name="SingleLoggingAgent" class="org.eclipse.hyades.logging.log4j.SingleLoggingAgentAppender" /&gt;
 * </code>
 * <p>
 * Also, the following the entry or appender reference must be added to the <code>&lt;logger&gt;</code> 
 * and/or <code>&lt;root&gt;</code> elements as desired in the <code>log4j.xml</code> properties file:
 * <p>
 * <code>
 * &lt;!-- SingleLoggingAgentAppender for this logger: --&gt;
 * &lt;appender-ref ref="SingleLoggingAgent" /&gt; 
 * </code>
 * <p>
 * Alternatively, an instantiation of this appender class may be set directly on
 * <code>org.apache.log4j.Logger</code> instances by using the
 * <code>addAppender()</code> API.
 * <p>
 * This appender is implicitly configured to use the 
 * <code>org.eclipse.hyades.logging.log4j.XmlLayout</code> formatter. 
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this appender class.
 * <p>
 *  
 * 
 * @author    Paul E. Slauenwhite
 * @version   July 28, 2006
 * @since     July 20, 2004
 * @see       org.apache.log4j.AppenderSkeleton
 * @see       org.apache.log4j.Logger
 * @see       org.eclipse.hyades.logging.log4j.XmlLayout
 */
public class SingleLoggingAgentAppender extends AppenderSkeleton {

    /**
     * The single Logging Agent to which successfully logged messages are
     * written.
     */
    protected static LoggingAgent loggingAgent = null;

    /**
     * Instance count of this class.
     */
    protected static int instanceCount = 0;

    /**
     * Current thread lock for synchronized operations.
     */
    protected static final Object CURRENT_THREAD_LOCK = new Object();

    /**
     * Default Logging Agent name.
     */
    public static final String DEFAULT_LOGGING_AGENT_NAME = "Logging Agent Appender";

    /**
     * Constructor to create a Logging Agent Appender.
     * <p>
     * The layout for this type of appender is defaulted to 
     * a <code>org.eclipse.hyades.logging.log4j.XMLLayout</code> 
     * instance.
     */
    public SingleLoggingAgentAppender() {

        super.setLayout(new XmlLayout());

        synchronized (CURRENT_THREAD_LOCK) {
            instanceCount++;
        }
    }

    /**
     * Publishing a <code>org.apache.log4j.spi.LoggingEvent</code> instance to
     * a single, named Logging Agent.
     * <p>
     * The <code>org.apache.log4j.spi.LoggingEvent</code> is converted using
     * the appender's implicit layout (e.g.
     * <code>org.eclipse.hyades.logging.log4j.XmlLayout</code>) and
     * propagated on to the single, named Logging Agent.
     * <p>
     * This method quietly returns if the appender has been closed (e.g. see
     * <code>close()</code>) and/or the parameter
     * <code>org.apache.log4j.spi.LoggingEvent</code> is <code>null</code>.
     * <p>
     * 
     * @param loggingEvent
     *            The <code>org.apache.log4j.spi.LoggingEvent</code> to be
     *            published to a single, named Logging Agent.
     */
    protected void append(LoggingEvent loggingEvent) {

        //Checks if the appender has not been closed and if the LoggingEvent is
        // not null:
        if ((!closed) && (loggingEvent != null)) {

            String formattedLoggingEvent = null;

            try {
                formattedLoggingEvent = getLayout().format(loggingEvent);
            } 
            catch (Exception e) {

                getErrorHandler().error(null, e, 1, loggingEvent);

                return;
            }

            synchronized (CURRENT_THREAD_LOCK) {

                if (loggingAgent == null) {
                    loggingAgent = new LoggingAgent(DEFAULT_LOGGING_AGENT_NAME);
                }

                loggingAgent.write(formattedLoggingEvent);
            }
        }
    }

    /**
     * Flush any waiting messages to the single, named Logging Agent.
     * <p>
     * This method quietly returns if the appender has been closed (e.g. see
     * <code>close()</code>).
     */
    public void flush() {

        if (!closed) {

            synchronized (CURRENT_THREAD_LOCK) {

                if ((loggingAgent != null) && (loggingAgent.isLogging())) {
                    loggingAgent.flush();
                }
            }
        }
    }

    /**
     * Flush any waiting messages and de-registering the single, named Logging
     * Agent.
     * <p>
     * This method quietly returns if the appender has been closed (e.g. see
     * <code>close()</code>).
     */
    public void close() {

        if (!closed) {

            flush();

            synchronized (CURRENT_THREAD_LOCK) {

                if ((--instanceCount == 0) && (loggingAgent != null)) {
                    loggingAgent.deregister();
                }
            }

            closed = true;
        }
    }

    /**
     * @see org.apache.log4j.Appender#requiresLayout()
     */
    public boolean requiresLayout() {
        return false;
    }

    /**
     * This method quietly returns since the appender's layout is implicitly
     * configured.
     * <p>
     * 
     * @see org.apache.log4j.Appender#setLayout(org.apache.log4j.Layout)
     */
    public void setLayout(Layout layout) {
        //No-operation since the appender's layout is implicitly configured.
    }
}