package org.eclipse.hyades.logging.log4j;



import org.apache.log4j.Layout;
import org.apache.log4j.spi.LocationInfo;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.log4j.spi.ThrowableInformation;
import org.apache.log4j.Level;
import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.core.LoggingCoreUtilities;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;



/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XmlLayout.java,v 1.10 2008/04/10 01:53:54 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Extension of the Log4J <code>org.apache.log4j.Layout</code> abstract
 * class used by <code>org.apache.log4j.Appender</code> instances to format 
 * <code>org.apache.log4j.spi.LoggingEvent</code> instances to XML.
 * <p>
 * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
 * <code>message</code> (e.g. instance of the 
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
 * the encapsulated Common Base Event is serialized to XML based on the Common Base 
 * Event v.1.0.1 specification.  Otherwise, the
 * <code>org.apache.log4j.spi.LoggingEvent</code> is first converted to a Common
 * Base Event (see below for mapping details) and serialized to XML based on the 
 * Common Base Event v.1.0.1 specification.
 * <p>
 * The <code>org.apache.log4j.spi.LoggingEvent</code> instance is converted to a Common
 * Base Event by mapping the <code>org.apache.log4j.spi.LoggingEvent</code> properties 
 * to Common Base Event properties using the following mapping:
 * <p> 
 * <table border="1" cellpadding="10"> 
 * 
 * <tr>
 * <th><code>org.apache.log4j.spi.LoggingEvent</code> Property</th>
 * <th><code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> Property</th>
 * </tr> 
 * <tr>
 * <td>level</td>
 * <td>ExtendedDataElement[0...n].Name = "LoggingEvent:level" <br>
 * ExtendedDataElement[0...n].Type = "noValue" <br>
 * ExtendedDataElement[0...n].Children[0].Name = "name" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's name&gt; #<br>
 * ExtendedDataElement[0...n].Children[1].Name = "value" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "int" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's numerical
 * value&gt; #<br>
 * </td>
 * </tr>
 * <tr>
 * <td>loggerName</td>
 * <td>ExtendedDataElement[0...n].Name = "LoggingEvent:loggerName"
 * <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;logger's name&gt; <br>
 * </td>
 * </tr> 
 * <tr>
 * <td>message</td>
 * <td>
 * ExtendedDataElement[0...n].Name = "LoggingEvent: message" <br>
 * ExtendedDataElement[0...n].Type = "noValue"<br>
 * ExtendedDataElement[0...n].Children[0].Name = &lt;The package and class name of the object&gt; <br>
 * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;objet string value&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>renderedMessage</td>
 * <td>msg = &lt;renderedMessage&gt; <br>
 * </td>
 * </tr> 
 * <tr>
 * <td>millis</td>
 * <td>creationTime = &lt;millis&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>NDC</td>
 * <td>ExtendedDataElement[0...n].Name = "LoggingEvent:NDC "
 * <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;the nested diagnostic contexts of the logging event&gt; <br>
 * </td>
 * </tr> 
 * <tr>
 * <td>locationInfo</td>
 * <td>
 * ExtendedDataElement[0...n].Name = "LoggingEvent:locationInfo" <br>
 * ExtendedDataElement[0...n].Type = "noValue" <br>
 * ExtendedDataElement[0...n].Children[0].Name = "fileName" <br> 
 * ExtendedDataElement[0...n].Children[0].Type = "string"<br> 
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;file's name><br> 
 * ExtendedDataElement[0...n].Children[1].Name = "className"<br> 
 * ExtendedDataElement[0...n].Children[1].Type = "string"<br> 
 * ExtendedDataElement[0...n].Children[1].Values[0] = &lt;class name&gt;<br> 
 * ExtendedDataElement[0...n].Children[2].Name = "methodName"<br> 
 * ExtendedDataElement[0...n].Children[2].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[2].Values[0] =&lt;method name&gt;<br>
 * ExtendedDataElement[0...n].Children[3].Name = "lineNumber" <br>
 * ExtendedDataElement[0...n].Children[3].Type = "int" <br>
 * ExtendedDataElement[0...n].Children[3].Values[0] = &lt;line number&gt;<br>
 * </td>
 * </tr> 
 * <tr>
 * <td>startTime</td>
 * <td>
 * ExtendedDataElement[0...n].Name = "LoggingEvent:startTime" <br>
 * ExtendedDataElement[0...n].Type = "dateTime" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;the time when the application started&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>threadName</td>
 * <td>
 * ExtendedDataElement[0...n].Name = "LoggingEvent:threadName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;thread name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>throwable*</td>
 * <td>ExtendedDataElement[0...n].Name = "LoggingEvent:stackTrace" <br>
 * ExtendedDataElement[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Values[0...n] =
 * [&lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
 * <code>java.lang.Throwable</code>'s localized message&gt;]]<br>
 * ExtendedDataElement[0...n].Values[(n + 1)...m] =
 * &lt; <code>java.lang.Throwable</code>'s stackTraceElement[0...(m - (n + 1))]&gt;]<br>
 * <br>
 * ExtendedDataElement[0...n].Children[0...m].Name = "Cause" <br>
 * ExtendedDataElement[0...n].Children[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Children[0...m].Values[0...n] =
 * &lt;cause's class name&gt;[: &lt;cause's localized message&gt;]<br>
 * ExtendedDataElement[0...n].Children[0...m].Values[(n + 1)...m] =
 * &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
 * <li>[children[0] = &lt;cause's cause&gt;]<br>
 * </td>
 * </tr>
 * </table> 
 * <p>
 * *throwable is the property of ThrowableInformation property <br>
 * #the name and values properties of the level property are get with level.toStirng() and level.toInt() methods.<br>
 * <p>
 * Callers may provide a Common Base Event XML file template (see 
 * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
 * newly created Common Base Event before serialization to XML.  The template 
 * Event XML file is a well-formed XML document conforming to a predefined XML schema 
 * (e.g. templateEvent.xsd).  The naming convention used for the template Event XML file 
 * is:
 * <p>
 * &lt;logger name&gt;.event.xml
 * <p>
 * where &lt;logger name&gt; is the name of the logger specified in the parameter 
 * <code>org.apache.log4j.spi.LoggingEvent</code>.  For example, the template event XML file 
 * for the <code>myLogger</code> logger factory would be named <code>myLogger.event.xml</code>.
 * <p>
 * The desired event configuration template XML file must by on the classpath
 * and accessible by this class. The event configuration template XML file is
 * loaded using the following look-up sequence:
 * <p>
 * <ol>
 * <li>The <code>XmlLayout</code> class' class loader
 * is queried for the event configuration template XML file.</li>
 * <li>The system's class loader is queried for the event configuration
 * template XML file.</li>
 * <li>The current thread's context class loader is queried for the event
 * configuration template XML file.</li>
 * </ol>
 * <p>
 * An instance of this class will be returned from the <code>getLayout()</code> 
 * API on <code>org.apache.log4j.Appender</code> instances.
 * <p>  
 * Users may configure a <code>org.apache.log4j.Appender</code> to use the 
 * <code>XmlLayout</code> by adding the following the entry to the 
 * <code>&lt;appender&gt;</code> element in the <code>log4j.xml</code> 
 * properties file:
 * <p>
 * <code>
 * &lt;!-- XmlLayout --&gt;
 * &lt;!-- Serializes events to XML: --&gt;
 * &lt;layout class="org.eclipse.hyades.logging.log4j.XmlLayout" /&gt;
 * </code>
 * <p>
 * Alternatively, an instantiation of this layout class may be set directly on
 * <code>org.apache.log4j.Appender</code> instances by using the 
 * <code>setLayout()</code> API.
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this appender class.
 * <p>
 *  
 * 
 * @author    Cindy Jin
 * @author    Paul E. Slauenwhite
 * @version   Janauary 08, 2006
 * @since     July 20, 2004
 * @see       org.apache.log4j.Layout
 * @see       org.apache.log4j.spi.LoggingEvent
 * @see       org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl
 */
public class XmlLayout extends Layout {

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for
     * generating event instances.
     */
    protected final static EventFactory EVENT_FACTORY = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactoryHome</code> for
     * generating pre-configured event instances.
     */
    protected final static EventFactoryHome EVENT_FACTORY_HOME = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl", XmlLayout.class.getClassLoader());
    
    /**
     * Instance variable which holds the name space property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * s.
     * <p>
     * This namespace is intended to easily identify
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * s associated with <code>org.apache.log4j.spi.LoggingEvent</code> properties.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_NAMESPACE ="LoggingEvent:";
   
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s level.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>oorg.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("level");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s
     * level's name.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("name");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s
     * level's value.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("value");
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s logger
     * name.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("loggerName");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s message
     * object.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_MESSAGE_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("message");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s NDC
     * name.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_NDC_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("NDC");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s locaiton informaiton
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LOCATION_INFO_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("locationInfo");
   
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the fully qualified class name of the caller making 
     * the logging request.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_CLASS_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("className");
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the fully qualified file name of the caller making 
     * the logging request.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_FILE_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("fileName");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the fully qualified line name of the caller making 
     * the logging request.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LINE_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("lineName");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the fully qualified method name of the caller making 
     * the logging request.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_METHOD_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("methodName");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s fqnOfCategoryClass
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_FQN_OF_CATEGORY_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("fqnOfCategoryClass");
    
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s start time
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_START_TIME_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("startTime");
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent/code>'s thrown.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THREAD_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("threadName");
    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>org.apache.log4j.spi.LoggingEvent</code>'s throwableInformation.
     * <p>
     * This name property is prefixed by the
     * <code>LoggingEvent:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.apache.log4j.spi.LoggingEvent</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THROWN = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("stackTrace");
    /**
     * Generates the XML representation of the parameter 
     * <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * <p>
     * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
     * <code>message</code> (e.g. instance of the 
     * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
     * the encapsulated Common Base Event is serialized to XML based on the Common Base 
     * Event v.1.0.1 specification.  Otherwise, the <code>org.apache.log4j.spi.LoggingEvent</code>
     * is serialized to Common Base Event XML based on a generic serialization algorithm as 
     * implemented in <code>org.eclipse.hyades.logging.events.cbe.util.EventHelpers#convertObjectToCommonBaseEvent(Object,int)</code>.
     * <p>
     * Callers may provide a Common Base Event XML file template (see 
     * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
     * newly created Common Base Event before serialization to XML.  See class comment 
     * header for more details.
     * <p>
     * 
     * @param loggingEvent The <code>org.apache.log4j.spi.LoggingEvent</code> instance to be formatted to XML.
     * @return The XML representation of the parameter <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * @see org.apache.log4j.Layout#format(LoggingEvent)
     */
    public String format(LoggingEvent loggingEvent) {
    	
    	if(loggingEvent == null)
    	{
    		return "";
    	}


    	Object message = loggingEvent.getMessage();

    	if (message instanceof IExternalizableToXml){
    		return ((((IExternalizableToXml) message).externalizeCanonicalXmlString()).concat(Constants.LINE_SEPARATOR));
    	}                        


    	//Map the parameter LoggingEvent's properties to Common Base Event properties:
    	CommonBaseEvent commonBaseEvent = null;
    	String loggerName = loggingEvent.getLoggerName();

    	if((loggerName != null) && (loggerName.trim().length() > 0)){

    		//Retrieve an EventFactory instance based on the logger name and create a pre-configured Common Base Event based 
    		//on the an XML configuration template file that exists on the class loader's or context class loader's classpath.
    		commonBaseEvent = EVENT_FACTORY_HOME.getEventFactory(loggerName).createCommonBaseEvent();

    		try {
    			commonBaseEvent.complete();
    		} 
    		catch (CompletionException c) {
    			//Ignore since content completion is based on 'best-effort'.
    		}
    	}
    	else{
    		commonBaseEvent = EVENT_FACTORY.createCommonBaseEvent();
    	}
    	
    	//bugzilla 91891
    	//Create and populate a default report situation:
    	if(commonBaseEvent.getSituation() == null){

    		//Create a new instance of a report situation:
    		ReportSituation reportSituation = EVENT_FACTORY.createReportSituation();
    		reportSituation.setReasoningScope("INTERNAL");
    		reportSituation.setReportCategory("LOG");

    		//Create a new instance of a situation:
    		Situation situation = EVENT_FACTORY.createSituation();
    		situation.setCategoryName(Situation.REPORT_SITUATION_CATEGORY);
    		situation.setSituationType(reportSituation);

    		commonBaseEvent.setSituation(situation);
    	}

    	//Create and populate a default source component ID:
    	if(commonBaseEvent.getSourceComponentId() == null){

    		//Create a new instance of a component ID:
    		ComponentIdentification sourceComponentId = EVENT_FACTORY.createComponentIdentification();
    		sourceComponentId.setLocation(Constants.LOCAL_HOST_IP_ADDRESS);
    		sourceComponentId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
    		sourceComponentId.setExecutionEnvironment("Java");
    		sourceComponentId.setComponent("Logging");
    		sourceComponentId.setSubComponent("Logger");
    		sourceComponentId.setComponentIdType("Application");
    		sourceComponentId.setComponentType("Logging_Application");

    		commonBaseEvent.setSourceComponentId(sourceComponentId);
    	}
    	
    	//Map the level property
    	Level level = loggingEvent.getLevel();

    	if(level != null){

    		ExtendedDataElement levelNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		levelNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
    		levelNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		levelNameExtendedDataElement.setValues(new String[] { level.toString()});

    		ExtendedDataElement levelValueExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		levelValueExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);
    		levelValueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
    		levelValueExtendedDataElement.setValues(new String[] { Integer.toString(level.toInt())});

    		ExtendedDataElement levelExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		levelExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL);
    		levelExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    		levelExtendedDataElement.addChild(levelNameExtendedDataElement);
    		levelExtendedDataElement.addChild(levelValueExtendedDataElement);

    		commonBaseEvent.addExtendedDataElement(levelExtendedDataElement);
    		
    		if (Level.FATAL.toInt() >= level.toInt()) {
                commonBaseEvent.setSeverity((short) 60);
            }else if (Level.ERROR.toInt() >= level.toInt()) {
                commonBaseEvent.setSeverity((short) 50);
            } else if (Level.WARN.toInt() >= level.toInt()) {
                commonBaseEvent.setSeverity((short) 30);
            } else {
                commonBaseEvent.setSeverity((short) 10);
            }
    	}

    	//Map the logger name property
    	if(loggerName != null){

    		ExtendedDataElement loggerNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		loggerNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
    		loggerNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		loggerNameExtendedDataElement.setValues(new String[] {loggerName});
    		commonBaseEvent.addExtendedDataElement(loggerNameExtendedDataElement);
    	}
    	//Map the message property
    	if(message != null)
    	{
    		ExtendedDataElement messageExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		messageExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_MESSAGE_NAME);
    		messageExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    		ExtendedDataElement messageExtendedDataElementChild = EventHelpers.convertObjectToExtendedDataElement(message, null, 4, 0);
    		messageExtendedDataElement.addChild(messageExtendedDataElementChild);
    		commonBaseEvent.addExtendedDataElement(messageExtendedDataElement);
    	}
    	//Map the rendered message property
    	String messageStr = loggingEvent.getRenderedMessage();

    	if(messageStr != null ){

    		commonBaseEvent.setMsg(messageStr);

    	}

    	//Map the time stamp property
    	commonBaseEvent.setCreationTimeAsLong(loggingEvent.timeStamp);
    	//Map the NDC
    	String ndc = loggingEvent.getNDC();

    	if(ndc != null)
    	{
    		ExtendedDataElement ndcExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		ndcExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_NDC_NAME );
    		ndcExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		ndcExtendedDataElement.setValues(new String[] {ndc});
    		commonBaseEvent.addExtendedDataElement(ndcExtendedDataElement);

    	}
    	//Map the locationInfo
    	LocationInfo locationInfo = loggingEvent.getLocationInformation();
    	if(locationInfo != null)
    	{
    		String fileName = locationInfo.getFileName();
    		ExtendedDataElement fileNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		fileNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_FILE_NAME);
    		fileNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		fileNameExtendedDataElement.setValues(new String[] {fileName});

    		String className = locationInfo.getClassName();
    		ExtendedDataElement classNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		classNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_CLASS_NAME);
    		classNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		classNameExtendedDataElement.setValues(new String[] {className});

    		String methodName = locationInfo.getMethodName();
    		ExtendedDataElement methodNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		methodNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_METHOD_NAME);
    		methodNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		methodNameExtendedDataElement.setValues(new String[] {methodName});

    		String lineNumber = locationInfo.getLineNumber();
    		ExtendedDataElement lineNumberExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		lineNumberExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LINE_NAME);
    		lineNumberExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
    		lineNumberExtendedDataElement.setValues(new String[] {lineNumber});

    		ExtendedDataElement locationInfoExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		locationInfoExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LOCATION_INFO_NAME);
    		locationInfoExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    		locationInfoExtendedDataElement.addChild(fileNameExtendedDataElement);
    		locationInfoExtendedDataElement.addChild(classNameExtendedDataElement);
    		locationInfoExtendedDataElement.addChild(methodNameExtendedDataElement);
    		locationInfoExtendedDataElement.addChild(lineNumberExtendedDataElement);

    		commonBaseEvent.addExtendedDataElement(locationInfoExtendedDataElement);

    	}
    	
    	//Map the applicate start time
    	long startTime = LoggingEvent.getStartTime();
    	ExtendedDataElement startTimeExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    	startTimeExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_START_TIME_NAME);
    	startTimeExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);
    	String startTimeStr = LoggingCoreUtilities.convertMillisecondsToXsdDateTime(startTime);
    	startTimeExtendedDataElement.setValues(new String[] {startTimeStr});
    	commonBaseEvent.addExtendedDataElement(startTimeExtendedDataElement);


    	//Map the thread name
    	String threadName = loggingEvent.getThreadName();

    	if(threadName != null )
    	{
    		ExtendedDataElement threadNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
    		threadNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_THREAD_NAME);
    		threadNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
    		threadNameExtendedDataElement.setValues(new String[] {threadName});
    		commonBaseEvent.addExtendedDataElement(threadNameExtendedDataElement);

    	}
    	//Map the throwable 
    	ThrowableInformation thrownInfo = loggingEvent.getThrowableInformation();
    	if(thrownInfo != null)
    	{
    		Throwable thrown  = thrownInfo.getThrowable();

    		if(thrown != null){
    			commonBaseEvent.addExtendedDataElement(EventHelpers.convertToExtendedDataElement(thrown,EXTENDED_DATA_ELEMENT_NAME_THROWN));
    		}
    	}

    	return (EventFormatter.toCanonicalXMLString(commonBaseEvent, true).concat(Constants.LINE_SEPARATOR));
    }

    /**
     * If the layout handles the throwable object contained within LoggingEvent, 
     * then the layout should return false. Otherwise, if the layout ignores 
     * throwable object, then the layout should return true. 
     * 
     * @return boolean true, if the layout handles the throwable object contained within LoggingEvent, other false.
     * @see org.apache.log4j.Layout#ignoresThrowable()
     */
    public boolean ignoresThrowable() {
        return true;
    }

    /**
     * Activate the options that were previously set with calls to option setters. 
     * This allows to defer activiation of the options until all options have been set. 
     * This is required for components which have related options that remain ambigous 
     * until all are set.  For example, the FileAppender has the File and Append options 
     * both of which are ambigous until the other is also set.
     * 
     * @see org.apache.log4j.spi.OptionHandler#activateOptions()
     */
    public void activateOptions() {
    }
}
