/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileUIManager.java,v 1.4 2006/12/29 19:16:46 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import java.io.File;

import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;

/**
 * Provides common UI functions for profiling, such as to start and stop
 * monitoring a profiling agent. These operations are equivalent to the UI
 * actions to start and stop monitoring an agent in the Profiling Monitor.
 * 
 * @author Adriana Viman
 * @since 4.0
 */
public class ProfileUIManager {

	/**
	 * The singleton instance.
	 */
	private static ProfileUIManager _instance;

	/**
	 * Singleton; not intended to be instantiated via the constructor.
	 */
	private ProfileUIManager() {
	}

	/**
	 * Returns the singleton instance of the <code>ProfileUIManager</code>.
	 * 
	 * @return the singleton instance.
	 * @since 4.0
	 */
	public static synchronized ProfileUIManager getInstance() {
		if (_instance == null) {
			_instance = new ProfileUIManager();
		}
		return _instance;
	}

	/**
	 * Starts monitoring an agent. Any data sent by the agent will be processed
	 * and loaded into the model (when the agent is attached, but not monitoring,
	 * the data is ignored). Does nothing if the agent is already being monitored.
	 * All other errors are logged to the eclipse log.
	 * 
	 * <p>
	 * Note: Must be called from the UI thread.
	 * </p>
	 * 
	 * @param agent the agent to start monitoring.
	 * @since 4.0
	 */
	public void startMonitoring(TRCAgentProxy agent) {
		TRCProcessProxy process = agent.getProcessProxy();
		String host = process.getNode().getName();

		try {
			Node node = PDCoreUtil.profileConnect(host, String.valueOf(process.getNode().getPort()));
			if (node == null) {
				return;
			}

			Object instance = LoadersUtils.locateAgentInstance(agent);
			if (instance != null && instance instanceof Agent) {
				Agent a = (Agent) instance;
				if (a.isActive()) {
					if (agent.isToProfileFile()) {
						a.setProfileFile(agent.getProfileFile());
					} else {
						a.setProfileFile(null);
					}

					PDCoreUtil.setAgentConfiguration(agent, a);
					a.publishConfiguration();

		            CustomCommand applyFilterCommand = new CustomCommand();
		            CustomCommand resumeCommand = new CustomCommand();

		            applyFilterCommand.setData("APPLYFILTERS");
					a.invokeCustomCommand(applyFilterCommand);

					XMLTraceDataProcessor processor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(agent);
					if (processor == null) {
						processor = new XMLTraceDataProcessor(agent);
						LoadersUtils.registerDataProcessor(agent, processor);
					}
					if (processor != null) {
						String processorProfileFileName = processor.getProfileFileName();
						File processorProfilingFile = processorProfileFileName == null ? null : new File(processorProfileFileName);
						boolean newFile = true;

						if (agent.isToProfileFile()) {
							File testFile = new File(agent.getProfileFile());
							if (testFile.exists()) {
								newFile = false;
							}
						}

						//just attach to agent, or profile to model before
						if ((processorProfilingFile == null) && (agent.isToProfileFile())) {
							processor.setProfileFileName(agent.getProfileFile());
							processor.createWriter();
							newFile = processor.isNewProfileFile();
							if (newFile) {
								processor.writeXMLVersion(processor.getWriter());
								processor.startTrace(processor.getWriter());
							}
						}

						//profile file name changed while stop monitor/detach,
						//or file deleted
						else if ((processorProfileFileName != null) && (!processorProfileFileName.equals(agent.getProfileFile()) || !processorProfilingFile.exists())) {
							if (processorProfilingFile.exists()) {
								processor.endTrace(processor.getWriter());
							}
							processor.setProfileFileName(agent.getProfileFile());
							processor.createWriter();
							newFile = processor.isNewProfileFile();
							if (newFile) {
								processor.writeXMLVersion(processor.getWriter());
								processor.startTrace(processor.getWriter());
							}
						}
					}

					a.startMonitoring(processor);
					agent.setCollectionData(true);

					/*
					 * RKD: The profiling agent has a custom command to resume the VM. I assume
					 * there should be code here to ensure that this command is only sent to the
					 * profiler.
					 */
					resumeCommand.setData("RESUME");
					a.invokeCustomCommand(resumeCommand);
					agent.setMonitored(true);
					agent.setActive(true);
					agent.setAttached(true);

					ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
					event.setSource(agent);
					event.setType(ProfileEvent.START_MONITOR);
					UIPlugin.getDefault().notifyProfileEventListener(event);
				}
			}
		}
		catch (Throwable e) {
			CommonPlugin.logError(e);
		}
	}

	/**
	 * Stops monitoring an agent. Any data sent by the agent will be no longer
	 * be processed by the XML loaders and will be discarded. However, the agent
	 * will still be attached, and can start monitoring again at any time if
	 * needed. Does nothing if the agent is not currently being monitored. All
	 * other errors are logged to the eclipse log.
	 * 
	 * <p>
	 * Note: Must be called from the UI thread.
	 * </p>
	 * 
	 * @param agent the agent to stop monitoring.
	 * @since 4.0
	 */
	public void stopMonitoring(TRCAgentProxy agent) {
		TRCProcessProxy process = agent.getProcessProxy();
		String host = process.getNode().getName();
		try {
			Node node = PDCoreUtil.profileConnect(host, String.valueOf(process.getNode().getPort()));
			if (node == null) {
				return;
			}

			Object instance = LoadersUtils.locateAgentInstance(agent);
			if (instance != null && instance instanceof Agent) {
				Agent a = (Agent)instance;
				if (a.isMonitored()) {
					a.stopMonitoring();
					agent.setMonitored(false);
					agent.setAttached(true);
					agent.setActive(true);

					ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
					event.setSource(agent);
					event.setType(ProfileEvent.STOP_MONITOR);
					UIPlugin.getDefault().notifyProfileEventListener(event);
				}
			}
		}
		catch (Throwable e) {
			CommonPlugin.logError(e);
		}
	}
}
