/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: UIPlugin.java,v 1.40 2008/04/30 12:42:17 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import java.net.URL;
import java.util.ArrayList;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.internal.execution.local.control.Application;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.internal.ui.TraceUIManager;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import com.ibm.icu.util.StringTokenizer;

/**
 * 
 * Plugin class that provides way to access plugin instance, resource 
 * bundle and debug enablement.
 * <p>
 * <b>Note:</b>This class is not intended to be extended and use externally, and 
 * it is subject to be changed without notice.
 *
 */
public class UIPlugin
	extends AbstractUIPlugin
    implements Application	
{
	protected static UIPlugin inst;
	protected ResourceBundle fResourceBundle;
	public static final String PLUGIN_ID = "org.eclipse.hyades.trace.ui"; //$NON-NLS-1$

    /**
     * The constructor 
     */
	public UIPlugin() {
		
		if (inst == null)
			inst = this;

	}

	/**
	 * Adds an <code>ISelectionListener</code> to the trace selection manager.
	 */
	public void addSelectionListener(ISelectionListener l) {
		TraceUIManager.getTraceUIManager().addSelectionListener(l);
	}

	/**
	 * Remove an <code>ISelectionListener</code> to the trace selection manager.
	 */
	public void removeSelectionListener(ISelectionListener l) {
		TraceUIManager.getTraceUIManager().removeSelectionListener(l);
	}

	/**
	 * Adds the selection provider for the debug UI.
	 */
	public void addSelectionProvider(
		ISelectionProvider provider,
		IWorkbenchPart part) {
		TraceUIManager.getTraceUIManager().addSelectionProvider(provider, part);
	}

	/**
	 * Remove the console for this process
	 * @param process
	 */
 public void deregisterLaunchProcess(IProcess process) {
  TraceUIManager.deregisterLaunchProcess(process);
 }
 
 /**
	 * Remove the console for this process
	 * @param process
	 */
public void deleteLaunch(IProcess process) {
TraceUIManager.deleteLaunch(process);
}
 
 /**
  * adds the console for this process
  * @param process
  */
 public void registerLaunchProcess(IProcess process) {
  TraceUIManager.registerLaunchProcess(process);
	}

	public static IWorkbenchPage getActivePage() {
		if (getActiveWorkbenchWindow() == null) {
			return null;
		}
		IWorkbenchPage page = getActiveWorkbenchWindow().getActivePage();

		if (page == null
			|| !page.getPerspective().getId().equals(
				PDPerspective.ID_TRACE_PERSPECTIVE)) {
			final IWorkbenchWindow dwindow = getActiveWorkbenchWindow();
			page = null;

			try {

				IWorkbenchPage[] persps = dwindow.getPages();
				for (int idx = 0; idx < persps.length; idx++) {
					if (persps[idx]
						.getPerspective()
						.getId()
						.equals(PDPerspective.ID_TRACE_PERSPECTIVE)) {
						//trace perspective
						page = persps[idx];

						dwindow.setActivePage(page);
						break;
					}
				}

				if (page == null) {
					IAdaptable element =
						UIPlugin.getPluginWorkbench().getRoot();
					IWorkbench workBench = dwindow.getWorkbench();
					if (workBench != null && element != null)
						page =
							workBench.showPerspective(
								PDPerspective.ID_TRACE_PERSPECTIVE,
								getActiveWorkbenchWindow(),
								element);
				}
			} catch (Exception exc) {
				exc.printStackTrace();
			}
		}

		return page;
	}
	
	/**
	 * Returns the active workbench shell.
	 * @return a Shell instance, the active shell
	 */
	public static Shell getActiveWorkbenchShell() {
		return getActiveWorkbenchWindow().getShell();
	}
	
	/**
	 * Returns the active workbench window.
	 * @return a IWorkbenchWindow instnace, the active workbench window
	 */
	public static IWorkbenchWindow getActiveWorkbenchWindow() {
		return getDefault().getWorkbench().getActiveWorkbenchWindow();
	}

	/**
	 * Returns the default UIPlugin instance.
	 * @return a UIPlugin instance
	 */
	public static UIPlugin getDefault() {
		return inst;
	}
	/**
	 * Returns the image descriptor for the given name.
	 * @param name java.lang.String, the name
	 * @return org.eclipse.jface.resource.ImageDescriptor, the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String name) {
		return PDPluginImages.getImageDescriptor(name);
	}

	public HierarchyFactory getPerftraceFactory() {
		return TraceUIManager.getTraceUIManager().getPerftraceFactory();
	} 
	
	/**
	 * Returns the plugin ID for this plugin.
	 * @return String, the plugin ID
	 */
	public static String getPluginId() {
		return PLUGIN_ID;
	}
	
	/**
	 * Returns the workspace for this plugin.
	 * @return IWorkspace, the workspace
	 */
	public static IWorkspace getPluginWorkbench() {
		return ResourcesPlugin.getWorkspace();
	}

    /**
     * Returns the resource bundle for this plugin.
     * @return ResourceBundle
     */
	public ResourceBundle getResourceBundle() {
		
		if(fResourceBundle == null)
		{
			try {
				fResourceBundle = Platform.getResourceBundle(Platform.getBundle(PLUGIN_ID));
			} catch (Exception e) {
				fResourceBundle = null;
			}
		}
		
		return fResourceBundle;
	}
	
	/**
	 * Returns the resource set for this plugin.
	 * @return ResourceSet
	 */
	public ResourceSet getResourceSet() {
		
		return TraceUIManager.getTraceUIManager().getResourceSet();
		
	}
	
	/**
	 * Returns the resource string for the given key
	 * @param key String, the key
	 * @return String, the resource String
	 */
	public static String getResourceString(String key) {
		
		try {
			ResourceBundle bundle = UIPlugin.getDefault().getResourceBundle();
			return bundle.getString(key);
		} catch (Exception e) {
			return key;
		}
	}

	/**
	 * Returns the viewer, the project explorer. 
	 * @return PDProjectExplorer
	 */
	public PDProjectExplorer getViewer() {
		return PDProjectExplorer.getFromActivePerspective();
	}

	/**
	 * Sets default preference values. These values will be used
	 * until some preferences are actually set using Preference dialog.
	 * @param store IPreferenceStore, the preference store where the default values are stored.
	 * @deprecated use {@link CommonUITracePlugin#initializeDefaultPreferences}
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {
		
		//hosts options
		store.setDefault(HyadesConstants.HOST_KEY, "localhost[10002]");
		
		//trace options
		store.setDefault(CommonUITraceConstants.TRACE_PROJECT_NAME, UIPlugin.getResourceString(TraceMessages.TRC_LOCD));
		store.setDefault(CommonUITraceConstants.LOG_PROJECT_NAME, "LogAnalyzerProject");
		store.setDefault(CommonUITraceConstants.LOG_MONITOR_NAME, "DefaultMonitor");
		
		store.setDefault(CommonUITraceConstants.TRACE_MONITOR_NAME, UIPlugin.getResourceString(TraceMessages.MON_LOCD));
		store.setDefault(TraceConstants.LOG_OPTION_KEY, true);
		store.setDefault(TraceConstants.PROF_OPTION_KEY, true);
		store.setDefault(TraceConstants.PROFILE_TIPS, true);
		store.setDefault(CommonUITraceConstants.RELOAD_INFO, true);
		store.setDefault(TraceConstants.NEW_SNAPSHOT_TIPS,true);
		store.setDefault(TraceConstants.PROF_EXEC_STAT, true);
		store.setDefault(TraceConstants.PROF_HEAP_STAT, true);
		store.setDefault(TraceConstants.PROF_HEAP_INST, false);
		store.setDefault(TraceConstants.PROF_AUTO_MONITOR, true);
		store.setDefault(TraceConstants.REFRESH_TYPE, 1);
		store.setDefault(TraceConstants.REFRESH_INTERVAL, 10);
		store.setDefault(CommonUITraceConstants.PROFILING_MONITOR_LAYOUT, CommonUITraceConstants.LAYOUT_FLAT);
		store.setDefault(CommonUITraceConstants.LOGN_LAYOUT, CommonUITraceConstants.LAYOUT_FLAT);

		String def = UIPlugin.getResourceString(TraceMessages.DEF_SETNM);
		store.setDefault(TraceConstants.ACTIVE_FILTER_NAME, def);

		String filters = TraceConstants.DEFAULT_FILTER_SET;

		//default profiling options
		store.setDefault(TraceConstants.FILTERS_SET_KEY, filters);
		
		store.setDefault(TraceConstants.PROF_HEAP_SIZE, true);
		store.setDefault(TraceConstants.PROF_EXECUTION, false);
		store.setDefault(TraceConstants.PROF_EXECUTION_BOUNDARY, false);

		store.setDefault(TraceConstants.LIMIT_TRACE_INVOC_OPTION, false);
		store.setDefault(TraceConstants.LIMIT_TRACE_TIME_OPTION, false);
		store.setDefault(TraceConstants.LIMIT_INVOC_NB, "10000");
		store.setDefault(TraceConstants.LIMIT_TIME_NB, "5");
		store.setDefault(TraceConstants.COLLECT_DEPTH, "1");

		store.setDefault(TraceConstants.AUTO_FILTER_CRITERIA_OPTION, true);
		store.setDefault(TraceConstants.DEBUG_OPTION,getDebugOption());
		
		store.setDefault(CommonUITraceConstants.HIDE_LOGT, false);
		
		store.setDefault(CommonUITraceConstants.TRACE_IMPORT_FILTER_CONTEXT,org.eclipse.hyades.ui.provisional.context.ContextManager.getDefautContext());
	}
	
	/**
	 * system property hyades.trace.ui.debug to enable debug functions in trace.ui plugins
	 * @return dubug option specified in system property, false if none exist.
	 */
	private boolean getDebugOption() {
		boolean debugOption = false;
		String value = System.getProperty("hyades.trace.ui.debug");
		if (value!=null) {
			debugOption = (value.equalsIgnoreCase("true")? true:false);
		}
		return debugOption;
	}


	/**
	 * Removes the selection provider from the debug UI.
	 */
	public void removeSelectionProvider(
		ISelectionProvider provider,
		IWorkbenchPart part) {
			
	    TraceUIManager.getTraceUIManager().removeSelectionProvider(provider, part);
	}
	
	/**
	 * Selection has changed in the debug selection provider.
	 * Notify the listeners.
	 */
	public void selectionChanged(SelectionChangedEvent event) {

		TraceUIManager.getTraceUIManager().selectionChanged(event);
	}
	/**
	 * @see ISelectionListener
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection sel) {
  TraceUIManager.getTraceUIManager().selectionChanged(part, sel);
	}

	/**
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		
		savePluginPreferences();
		PDPluginImages.shutdown();
		
		TraceUIManager.getTraceUIManager().dispose();
		fResourceBundle = null;		
	}
	
	/**
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		
        Runnable runnable = new Runnable() {
			public void run() {
				try	
				{
					TraceUIImages.INSTANCE.initialize(new URL(getBundle().getEntry("/"), "icons/full/"), getImageRegistry());                    
				}
				catch(Exception e) 
				{
					e.printStackTrace();
				}
			}
		};
		
		if (Display.getDefault() == Display.getCurrent()) {
            // We are in the main thread
			runnable.run();
		} else {
            // Post the initialization to the main thread
			Display.getDefault().asyncExec(runnable);
		}
		
		updateHostPreference();
	}
	
	/**
	/** Update the host preference value to the new format in order
	/** to support the IPv6 format.
	/** @Since TPTP 4.5
	**/
	protected void updateHostPreference() {
		
		IPreferenceStore store = getPreferenceStore();
		String hostList = store.getString(CommonUIConstants.HOST_KEY);

		if (hostList.equals("localhost[10002]")
				|| store.getBoolean("HOSTPREF_UPDATED")) {
			store.setValue("HOSTPREF_UPDATED", true);
			return;
		}
		
		StringBuffer buff = new StringBuffer();
		
		StringTokenizer tokens = new StringTokenizer(hostList,",");
		while (tokens.hasMoreElements()) {
			String hostPort = (String) tokens.nextElement();
			int idx = hostPort.lastIndexOf(":");
			if (idx>0) {
				buff.append(hostPort.substring(0, idx)).append("[")
				.append(hostPort.subSequence(idx+1, hostPort.length())).append("]");
			}
			else {
				buff.append(hostPort);
			}
			buff.append(",");
		}
		
		String newPrefValue = buff.toString();
		
		
		store.setValue(CommonUIConstants.HOST_KEY, newPrefValue.substring(0, newPrefValue.length()-1));
		store.setValue("HOSTPREF_UPDATED", true);
	}
	/**
	 * Adds the given listener to the collection of registered profile
	 * event listeners. Has no effect if an identical listener is already
	 * registered.
	 *
	 * @param listener the listener to add
	 */
	public void addProfileEventListener(IProfileEventListener listener) {
		TraceUIManager.getTraceUIManager().addProfileEventListener(listener);
	}
	
	/**
	 * Removes the given listener from the collection of registered profile
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 *
	 * @param listener the listener to remove
	 */
	public void removeProfileEventListener(IProfileEventListener listener) {
		TraceUIManager.getTraceUIManager().removeProfileEventListener(listener);
	}

	
	/**
	 * Add a delete listener to the collection of delete listeners.
	 */
	public void addDeleteListener(IDeleteListener listener) {
		TraceUIManager.getTraceUIManager().addDeleteListener(listener);
	}


	/**
	 * Adds the given listener to the collection of registered pd view
	 * event listeners. Has no effect if an identical listener is already
	 * registered.
	 *
	 * @param listener the listener to add
	 */
	public void addViewSelectionChangedListener(IViewSelectionChangedListener listener) {
		TraceUIManager.getTraceUIManager().addViewSelectionChangedListener(listener);
	}


	/**
	 * Add a trace selection model for an EObject to the collection of selection models.
	 * @param mofObject EObject, the object
	 * @param model ITraceSelection, the trace selection model
	 */
	public void addSelectionModel(EObject mofObject, ITraceSelection model) {
		TraceUIManager.getTraceUIManager().addSelectionModel(mofObject, model);
	}


	/**
	 * Return a list of the delete listeners.
	 * @return java.util.ArrayList
	 */
	public ArrayList getDeleteListeners() {
		return TraceUIManager.getTraceUIManager().getDeleteListeners();
	}

	/**
	 * Return the view selection changed event.
	 * @return ViewSelectionChangedEvent
	 */
	public ViewSelectionChangedEvent getViewSelectionChangedEvent() {
		return TraceUIManager.getTraceUIManager().getViewSelectionChangedEvent();
	}

	/**
	 * Return the profile event.
	 * @return ProfileEvent
	 */
	public ProfileEvent getProfileEvent() {
		return TraceUIManager.getTraceUIManager().getProfileEvent();
	}


	/**
	 * Return the refresh view event, a type of ProfileEvent.
	 * @return ProfileEvent
	 */
	public ProfileEvent getRefreshViewEvent(Object source) {
		return TraceUIManager.getTraceUIManager().getRefreshViewEvent(source);
	}
	
	/**
	 * Return the refresh option event, a type of ProfileEvent.
	 * @author Navid Mehregani - bugzilla 153453
	 * @return ProfileEvent
	 */
	public ProfileEvent getRefreshOptionEvent(Object source) {
		return TraceUIManager.getTraceUIManager().getRefreshOptionEvent(source);
	}
	
	/**
	 * Return the context handler selection changed event, a type of ProfileEvent.
	 * @return ProfileEvent
	 */
	public ProfileEvent getContextHandlerSelectionChangedEvent(Object source) {
		return TraceUIManager.getTraceUIManager().getContextHandlerSelectionChangedEvent(source);
	}


	/**
	 * Return the trace selection model for a given object instance of EObject.
	 * @param mofObject EObject, the given object
	 * @return ITraceSelection the trace selection model
	 */
	public ITraceSelection getSelectionModel(EObject mofObject) {

		return TraceUIManager.getTraceUIManager().getSelectionModel(mofObject);
	}


	/**
	 * Return the update model event, a type of ProfileEvent.
	 * @return ProfileEvent
	 */
	public ProfileEvent getUpdateModelEvent(Object source) {
		return TraceUIManager.getTraceUIManager().getUpdateModelEvent(source);
	}


	/**
	 * Notify the view selection changed listener with the given ViewSelectionChangedEvent event instance.
	 * @param event ViewSelectionChangedEvent
	 */
	public void notifyViewSelectionChangedListener(ViewSelectionChangedEvent event) {
		
		TraceUIManager.getTraceUIManager().notifyViewSelectionChangedListener(event);
	}


	/**
	 * Notify the profile event listener with the given ProfileEvent event instance.
	 * @param event ProfileEvent
	 */
	public void notifyProfileEventListener(ProfileEvent event) {
		
		TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
	}


	/**
	 * Remove the given delete listener from the delete listener collection.
	 */
	public void removeDeleteListener(IDeleteListener listener) {
		
		TraceUIManager.getTraceUIManager().removeDeleteListener(listener);
	}


	/**
	 * Removes the given listener from the collection of registered pd view
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 *
	 * @param listener the listener to remove
	 */
	public void removeViewSelectionChangedListener(IViewSelectionChangedListener listener) {
		TraceUIManager.getTraceUIManager().removeViewSelectionChangedListener(listener);
	}


	/**
	 * Remove the selection model for the given object instance of EObject.
	 * @param mofObject EObject
	 */
	public void removeSelectionModel(EObject mofObject) {
		TraceUIManager.getTraceUIManager().removeSelectionModel(mofObject);
	}

    /**
     * Returns the plugin name.
     * @return String
     */
	public String getName()
	{
		return getPluginId();
	}
	    
	/**
	 * Logs the given status.
	 * @param status IStatus
	 */
	public void log(IStatus status) {
		getLog().log(status);
	}
	
	/**
	 * Logs the given throwable.
	 * @param e Throwable
	 */
	public void log(Throwable e) 
	{
		log(new Status(IStatus.ERROR, PLUGIN_ID, IStatus.ERROR, e.getLocalizedMessage() == null ? e.getClass().getName() : e.getLocalizedMessage(), e)); 
	}
	
	private IConfigurationElement newSymptomWizardElement = null;
	private boolean newSymptomWizardElementInit;
	
	/**
	 * Returns a new symptom db wizard.
	 * @return INewWizard
	 */
	public INewWizard getNewSymDBWizard() {
		try {
			if (!newSymptomWizardElementInit) {
				newSymptomWizardElementInit = true;
				loadWizardExtention();

			}

			if (newSymptomWizardElement != null)
				return (INewWizard) newSymptomWizardElement
						.createExecutableExtension("class");
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Loads the wizard configuration elements, from the extention.
	 *
	 */
	public void loadWizardExtention() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IConfigurationElement[] configurationElements = registry.getConfigurationElementsFor("org.eclipse.ui.newWizards");//$NON-NLS-1$		
		if (configurationElements!=null) {
			for (int i = 0; i < configurationElements.length; i++) {
				IConfigurationElement configurationElement = configurationElements[i];
				if (configurationElement.getName().equals("wizard") //$NON-NLS-1$
					&& configurationElement.getAttribute("id").endsWith("wizard.new.symptomdb")) {//$NON-NLS-1$
					newSymptomWizardElement = configurationElement;
				}
			}
		}
	}
	
	/**
	 * Returns the label with any accelerators removed.
	 * @see org.eclipse.debug.internal.ui.DebugUIPlugin.removeAccelerators(String)
	 * @return label without accelerators
	 */
    public static String removeAccelerators(String label) {
        String title = label;
        if (title != null) {
            // strip out any '&' (accelerators)
            int index = title.indexOf('&');
            if (index == 0) {
                title = title.substring(1);
            } else if (index > 0) {
                //DBCS languages use "(&X)" format
                if (title.charAt(index - 1) == '(' && title.length() >= index + 3 && title.charAt(index + 2) == ')') {
                    String first = title.substring(0, index - 1);
                    String last = title.substring(index + 3);
                    title = first + last;
                } else if (index < (title.length() - 1)) {
                    String first = title.substring(0, index);
                    String last = title.substring(index + 1);
                    title = first + last;
                }
            }
        }
        return title;
    }
}
