/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceNavigatorActionGroup.java,v 1.22 2009/09/30 18:23:52 ewchan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.ui.internal.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.resources.IProject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.log.ui.internal.LogMessages;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.actions.OpenAssociatedTraceViewAction;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.navigator.DistributedLayout;
import org.eclipse.hyades.trace.ui.internal.navigator.ProfileDetailItem;
import org.eclipse.hyades.trace.ui.internal.navigator.SimpleLayout;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenMonitorWizard;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.action.LayoutAction;
import org.eclipse.hyades.ui.internal.extension.INavigatorLayout;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.control.provider.ControlUtility;
import org.eclipse.tptp.trace.ui.internal.control.provider.TraceControlItemManager;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.dialogs.PropertyDialogAction;

/**
 * Group action for the Trace Navigator.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.MainActionGroup} class.
 * 
 */
public class TraceNavigatorActionGroup extends TraceLogActionGroup
{	
	protected OpenWizardAction fOpenMonitorAction;		
	protected PropertyDialogAction fPropertyDialogAction;
	private LayoutAction fFlatLayout;
	private LayoutAction fHierarchicalLayout;

	private TraceControlItemManager controlItemManager;
	
	/**
	 * Constructor for NavigatorActionGroup
	 * @param testNavigator
	 */
	public TraceNavigatorActionGroup(INavigator navigator)
	{
		this(navigator, null);
	}
	
	public TraceNavigatorActionGroup(INavigator navigator, TraceControlItemManager controlItemManager)
	{
		super(navigator);
		this.controlItemManager = controlItemManager;
	}
	
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{	
		deleteActions();
		super.dispose();
	}
		
	protected void deleteActions()
	{
		if (fOpenMonitorAction != null)
		{
			fOpenMonitorAction.dispose();
			fOpenMonitorAction = null;
		}		
		if (fPropertyDialogAction != null)
		{
			fPropertyDialogAction.dispose();
			fPropertyDialogAction = null;
		}

		fFlatLayout = null;
		fHierarchicalLayout = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		super.createActions();
				
		fNewSnapshotGrp = new Separator(CommonUITraceConstants.PROFILE_NEW_SNAPSHOT_GROUP);
		fGCGroup = new Separator(CommonUITraceConstants.PROFILE_GC_GROUP);
		
		fTerminateGrp = new Separator(CommonUITraceConstants.PROFILE_TERMINATE_GROUP);
		
		fSaveToFile = new SaveAction(getNavigator());
	
		fOpenMonitorAction =
			new OpenWizardAction(
		(PDProjectExplorer)getNavigator(),
				OpenMonitorWizard.class,
				UIPlugin.getResourceString(TraceMessages.PROP));

		
		INavigatorLayout layout = getNavigator().getLayout();
		fFlatLayout = new LayoutAction(getNavigator(), new SimpleLayout((PDProjectExplorer)getNavigator()), UIPlugin.getResourceString(TraceMessages.LY_FLAT), PDPluginImages.DESC_IMG_UI_LAYOUT_FLAT, UIPlugin.getDefault().getPreferenceStore(), CommonUITraceConstants.PROFILING_MONITOR_LAYOUT, CommonUITraceConstants.LAYOUT_FLAT);
		fFlatLayout.setChecked(layout instanceof SimpleLayout);
		fHierarchicalLayout = new LayoutAction(getNavigator(), new DistributedLayout((PDProjectExplorer)getNavigator()), UIPlugin.getResourceString(TraceMessages.LY_HR), PDPluginImages.DESC_IMG_UI_LAYOUT_HIERARCHICAL, UIPlugin.getDefault().getPreferenceStore(), CommonUITraceConstants.PROFILING_MONITOR_LAYOUT, CommonUITraceConstants.LAYOUT_HIERARCHICAL);
		fHierarchicalLayout.setChecked(layout instanceof DistributedLayout);
		fDeleteAction = new DeleteAction(getNavigator());
		fPropertyDialogAction = new PropertyDialogAction(getNavigator().getViewSite(), getNavigator().getViewer());
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{		
		IStructuredSelection selection = (IStructuredSelection)getContext().getSelection();
		boolean sameClass = false;		
		Object sel = selection.isEmpty() ? null : selection.iterator().next();
		boolean controllableEntity = sel instanceof TRCProcessProxy || sel instanceof TRCAgentProxy || (sel instanceof ProfileDetailItem && ((ProfileDetailItem)sel).isAnalysisType());

		if (!controllableEntity)
			super.fillContextMenu(menu);
		else
			super.fillContextMenuWithGroups(menu);
		
		boolean multiSelection = (selection.size() > 1);
		if (multiSelection)
			sameClass = areSameClass(selection);
				
		/* In case nothing is in the profiling monitor view */
		if (selection.isEmpty()) 
		{
			fImportActionGroup.fillContextMenu(menu);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshTree);
			return;
		}
		updateActions(selection);
		
		
		
		if(sel instanceof TRCMonitor)
		{
			menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
		}
		else if(sel instanceof TRCNode)
		{
			menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel); 
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
		}
		else if(sel instanceof TRCProcessProxy)
		{			
			if (!multiSelection) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fTerminateGrp);
				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);				
			}
			
			/* Add the control items that are affiliated with a specific type of an agent */
			ControlUtility.addContributedItems(menu, controlItemManager.getContextMenuItems());				
	    	
		}
		else if(sel instanceof TRCAgentProxy)
		{			
			if (!multiSelection || sameClass) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fMonitorGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fAttachGrp);				
			}
			
			if(!multiSelection)
			{
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fNewSnapshotGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fTerminateGrp);
				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
			}
	    						
			/* Add the control items that are affiliated with a specific type of an agent */
			ControlUtility.addContributedItems(menu, controlItemManager.getContextMenuItems());	
		}
		else if (sel instanceof ProfileDetailItem && ((ProfileDetailItem)sel).isAnalysisType())
		{
			if (!multiSelection) 
			{
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
			}

			/* Add the control items that are affiliated with a specific type of an agent */
			ControlUtility.addContributedItems(menu, controlItemManager.getContextMenuItems());
		}
		else if (sel instanceof INavigatorItem )
		{
			if (!multiSelection) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fMonitorGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fAttachGrp);				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);				
			}
		}

		/* Ali M. -- Only add the common delete, save, refresh items if the selection is not a controllable
		 * entity */
		if (!controllableEntity)
		{
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE,fDeleteAction);
			menu.appendToGroup(INavigatorContribution.GROUP_SAVE,fSaveToFile);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshTree);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshLocal);
		}

		if (!multiSelection && !controllableEntity) 
		{	
			if (sel instanceof TRCMonitor) 
			{
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenMonitorAction);
			} 
			else if (sel instanceof TRCNode) 
			{
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenNodeAction);
			} 
			/* Ali M. - https://bugs.eclipse.org/bugs/show_bug.cgi?id=154937 
			 * Add the general properties menu item only if it's not a recognizable item */
			else
			{
				if (fPropertyDialogAction.isApplicableForSelection())
					menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fPropertyDialogAction);				
			}
		}

		if (!controllableEntity)
			fImportActionGroup.fillContextMenu(menu);		
	}
	

	/**
	 * Adds the layouts to the view menu.
	 * 
	 * @see org.eclipse.tptp.platform.common.ui.trace.internal.actions.TraceLogActionGroup#fillActionBars(org.eclipse.ui.IActionBars)
	 */
	public void fillActionBars(IActionBars actionBars) {
		IMenuManager menu = actionBars.getMenuManager();
		menu.add(fFlatLayout);
		menu.add(fHierarchicalLayout);		
		super.fillActionBars(actionBars);
	}

	/**
	 * Convenience method for adding the given views to
	 * the menu.
	 */
	private void addActions(IMenuManager menu, Collection views) {
		menu.add(new Separator());
		Iterator iter = views.iterator();
		while (iter.hasNext()) {
			Object next = iter.next();
			if (next != null && next instanceof IAction && ((IAction)next).isEnabled()) {
				menu.add((IAction)next);
			}
		}
	}

	/**
	 * Adds all appropriate views to this object's menu.
	 */
	private void addViews(IMenuManager menu, Object obj) {
		if (obj != null && ((obj instanceof EObject) || (obj instanceof INavigatorItem))) {
			MenuManager popup = new MenuManager(LogMessages.MENU_MANAGER);
			menu.appendToGroup(GROUP_ADD_VIEWS,popup);
						
			// sorted set
			Set views = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					return ((IAction)o1).getText().compareTo(((IAction)o2).getText());
				}
			});
			
			if (obj instanceof TRCMonitor || obj instanceof TRCNode) {
				getViews(obj, views, true, false);
				addActions(popup, views);
				views.clear();
				getViews(obj, views, false, true);
				addActions(popup, views);
			}
			else {
				getViews(obj, views, true, true);
				addActions(popup, views);
			}
		}
	}
	
	/**
	 * Adds all views to the set for the specified object. If
	 * includeSelf is false, it will omit the views for obj. If
	 * includeChildren is true, it will recursively add all views
	 * for all descendants.
	 */
	private void getViews(Object obj, Set set, boolean includeSelf, boolean includeChildren) {
		
		if (obj==null) return;
		
		String type = null;
		Collection children = new ArrayList();
		if (obj instanceof TRCMonitor) {
			type = CommonUITraceConstants.MONITOR_TYPE;
			children.addAll(((TRCMonitor)obj).getNodes());
		}
		else if (obj instanceof TRCNode) {
			type = CommonUITraceConstants.HOST_TYPE;
			children.addAll(((TRCNode)obj).getProcessProxies());
		}
		else if (obj instanceof TRCProcessProxy) {
			type = null;
			children.addAll(((TRCProcessProxy)obj).getAgentProxies());
		}
		else if (obj instanceof TRCAgentProxy) {
			type = ((TRCAgentProxy)obj).getType();
			
			/* If there are no profiling types under this agent,
			 * add the views for ALL profiling types. */
			if (type.equals(CommonUIConstants.PROFILE_AGENT_TYPE) && !hasProfilingTypes(obj)) {
				Map typesMap = ProfilingSetsManager.instance().getProfilingTypes();
				children = new ArrayList();
				Iterator iter = typesMap.values().iterator();
				while (iter.hasNext()) {
					IProfilingSetType next = (IProfilingSetType)iter.next();
					children.add(new ProfileDetailItem(obj, next.getName(), next.getId(), null));
				}
			}
		}
		else if (obj instanceof INavigatorItem) {
			type = ((INavigatorItem)obj).getType();
		}

		/*
		 * Add all the navigator contribution children..
		 */
		Collection items = null;
		items = NavigatorExtensionUtil.getAllChildren(obj, PDProjectExplorer.ID);
		if (items != null) {
			if (children == null) {
				children = items;
			} else {
				children.addAll(items);
			}
		}
		
		if (includeSelf && type != null) {
			IAction[] actions = getViewMappingForType(type);
			for (int i=0;i<actions.length;++i) {
				if (actions[i] !=null) {
					if (actions[i] instanceof OpenAssociatedTraceViewAction)
					{
						if (!set.contains(actions[i]) || !actions[i].isEnabled())
							actions[i].setEnabled(((OpenAssociatedTraceViewAction)actions[i]).isEnabledForObject(obj));
					}
					set.add(actions[i]);
				}
			}
		}
		if (includeChildren && children != null) {
			Iterator iter = children.iterator();
			while (iter.hasNext()) {
				getViews(iter.next(), set, true, true);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#handleKeyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void handleKeyPressed(KeyEvent event) {
		if(event.character == SWT.DEL && (event.stateMask == 0)) {
			updateActions(HyadesUtil.getActiveNavigator().getStructuredSelection());
			if(fDeleteAction.isEnabled()) {
				fDeleteAction.run();
			}
		}
		
		/*
		 * Open the properties dialog if Alt-Enter is pressed (standard behavior
		 * for eclipse).
		 */
		if(event.keyCode == 0x0D && ((event.stateMask & SWT.ALT) != 0)) {
			IStructuredSelection selection = HyadesUtil.getActiveNavigator().getStructuredSelection();
			if (selection.size() == 1) {
				Object sel = selection.iterator().next();
				updateActions(selection);
				if (sel instanceof IProject) {
					fPropertyDialogAction.run();
				}
				else if (sel instanceof TRCMonitor) {
					fOpenMonitorAction.run();
				}
				else if (sel instanceof TRCNode) {
					fOpenNodeAction.run();
				}
				/*if (sel instanceof TRCProcessProxy) {
					fOpenProcessAction.run();
				}
				if (sel instanceof TRCAgentProxy) {				
					fOpenProfAgentAction.run();
				}*/
			}
		}
	}
}
