/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LaunchProcessUI.java,v 1.11 2008/05/30 23:39:46 aalexeev Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.io.File;
import java.util.jar.JarFile;

import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.core.util.IClassFileReader;
import org.eclipse.jdt.core.util.IMethodInfo;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

/*
 * CONTEXT_ID lpui0000 for classname entry field in launch process ui
 * CONTEXT_ID lpui0001 for classpath entry field in launch process ui
 * CONTEXT_ID lpui0005 for parameters entry field in launch process ui
 * CONTEXT_ID lpui0006 for vm arguments entry field in launch process ui
 */
public class LaunchProcessUI extends AbstractChangeable implements SelectionListener {

	private Button _browseClass;
	private Button _browseDirPath;
	private Button _browseJarPath;
	private Composite _classGroup;
	private Composite _paramGroup;
	private Text _classname;
	private Text _classpath;
	
	private static String MAIN_METHOD_NAME = "main";
	private static String MAIN_METHOD_DESCRIPTOR = "([Ljava/lang/String;)V";	

	private Listener _listener = new Listener() {
		public void handleEvent(Event event) {
			changed();
		}
	};

	public Composite createArgumentsControl(Composite parent) {
		GridData gd;

		_paramGroup = new Composite(parent, SWT.NONE);
		_paramGroup.setLayoutData(GridUtil.createHorizontalFill());
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 3;
		gd = GridUtil.createFill();
		_paramGroup.setLayoutData(gd);
		_paramGroup.setLayout(layout);

		Label label = new Label(_paramGroup, SWT.NONE);
		gd = new GridData();
		gd.verticalAlignment = GridData.BEGINNING;
		label.setText(TraceMessages.CLASSPATH);
		label.setLayoutData(gd);

		Composite pathGroup = new Composite(_paramGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 5;
		layout.marginHeight = 0;
		gd = GridUtil.createHorizontalFill();
		pathGroup.setLayoutData(gd);
		pathGroup.setLayout(layout);

		_classpath = new Text(pathGroup, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
		gd = GridUtil.createFill();
		gd.heightHint = 70;
		gd.widthHint = 90;
		_classpath.setLayoutData(gd);
		_classpath.addListener(SWT.Modify, _listener);

		Composite btnGroup = new Composite(pathGroup, SWT.NONE);
		gd = GridUtil.createVerticalFill();
		gd.horizontalIndent = 5;
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.verticalSpacing = 7;
		btnGroup.setLayoutData(gd);
		btnGroup.setLayout(layout);

		_browseDirPath = new Button(btnGroup, SWT.CENTER | SWT.PUSH);

		gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.verticalAlignment = GridData.FILL;
		_browseDirPath.setLayoutData(gd);

		_browseDirPath.setText(TraceMessages.AD_DIR);
		_browseJarPath = new Button(btnGroup, SWT.CENTER | SWT.PUSH);

		gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;

		_browseJarPath.setLayoutData(gd);
		_browseJarPath.setText(TraceMessages.ADD_JAR);

		label = new Label(btnGroup, SWT.NONE);

		//add listeners
		_browseDirPath.addSelectionListener(this);
		_browseJarPath.addSelectionListener(this);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(_classpath, UIPlugin.getPluginId() + ".lpui0001");
		return _paramGroup;
	}

	public Composite createClassControl(Composite parent) {
		_classGroup = new Composite(parent, SWT.NONE);
		_classGroup.setLayoutData(GridUtil.createHorizontalFill());
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 2;
		layout.marginHeight = 0;
		_classGroup.setLayout(layout);
		
		Label label = new Label(_classGroup, SWT.NONE);
		label.setText(TraceMessages.CLASS_NAME);
		_classname = new Text(_classGroup, SWT.SINGLE | SWT.BORDER);
		_classname.setLayoutData(GridUtil.createHorizontalFill());
		_classname.addListener(SWT.Modify, _listener);

		Composite btnGroup = new Composite(_classGroup, SWT.NONE);
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		btnGroup.setLayoutData(gd);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.marginHeight = 0;
		btnGroup.setLayout(layout);
		
		_browseClass = new Button(btnGroup, SWT.CENTER | SWT.PUSH);
		_browseClass.setLayoutData(GridUtil.createFill());
		_browseClass.setText(TraceMessages.BROWSE);

		//add listeners
		_browseClass.addSelectionListener(this);
		_classname.setFocus();

		PlatformUI.getWorkbench().getHelpSystem().setHelp(_classname, UIPlugin.getPluginId() + ".lpui0000");
		return _classGroup;
	}

	public Composite createControl(Composite parent) {
		Composite content = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 10;
		content.setLayout(layout);
		GridData gd = GridUtil.createFill();
		content.setLayoutData(gd);

		createClassControl(content);
		createArgumentsControl(content);

		_classname.setFocus();

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UIPlugin.getPluginId() + ".lpui0000");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(content, UIPlugin.getPluginId() + ".lpui0000");
		return content;
	}

	/**
	 * sets all buttons and text fields to enable
	 */
	public void enable(boolean enable) {
		enableButtons(enable);
		enableText(enable);
	}

	/**
	 * sets all buttons to be enabled if enable = true, false otherwise
	 */
	private void enableButtons(boolean enable) {
		_browseClass.setEnabled(enable);
		_browseDirPath.setEnabled(enable);
		_browseJarPath.setEnabled(enable);
	}

	/**
	 * sets all text fields to be enabled and editable if enable = true, false
	 * otherwise
	 */
	private void enableText(boolean enable) {
		_classname.setEnabled(enable);
		_classname.setEditable(enable);
		_classpath.setEnabled(enable);
		_classpath.setEditable(enable);
	}

	public Control getBrowseClassButton() {
		return _browseClass;
	}

	public Control getBrowseDirButton() {
		return _browseDirPath;
	}

	public Control getBrowseJarButton() {
		return _browseJarPath;
	}

	public Composite getClassGroup() {
		return _classGroup;
	}

	public String getClassName() {
		return _classname.getText().trim();
	}

	public String getClassPath() {
		return _classpath.getText().trim();
	}

	public Text getClassPathUI() {
		return _classpath;
	}

	public Text getClassUI() {
		return _classname;
	}

	public void setClassName(String name) {
		_classname.setText(name);
	}

	public void setClassPath(String path) {
		_classpath.setText(path);
	}

	/**
	 * if the host is a local host, than the user is allowed to browse if not,
	 * then the user isn't, set them to false;
	 */
	public void setLocalButtons(boolean local) {
		getBrowseClassButton().setEnabled(local);
		getBrowseDirButton().setEnabled(local);
		getBrowseJarButton().setEnabled(local);
	}

	public void widgetDefaultSelected(SelectionEvent event) {
	}

	private boolean hasMainMethod(IMethodInfo [] mi)
	{
		for (int i = 0; i < mi.length; i++)
		{ 
			mi[i].getAttributes();
			if (MAIN_METHOD_NAME.equals(new String(mi[i].getName()))
					&& Flags.isPublic(mi[i].getAccessFlags())
					&& MAIN_METHOD_DESCRIPTOR.equals(new String(mi[i].getDescriptor())))
			{
				return true;
			}
		}
		return false;
	}

	public void widgetSelected(SelectionEvent e) {
		Shell shell = _browseClass.getShell();
		String osName = System.getProperty("os.name");

		if (e.widget == _browseClass) {
			FileDialog dlg = new FileDialog(shell);
			dlg.setFilterExtensions(new String[] { "*.class" });
			dlg.open();

			String fileName = dlg.getFileName();
			if (fileName != null && !fileName.equals("")) {
				String classpath = dlg.getFilterPath();
				
				IClassFileReader cfr = ToolFactory.createDefaultClassFileReader(classpath + File.separator + fileName, IClassFileReader.METHOD_INFOS);
				String qualifiedClassName = null;
				if (cfr == null){
					qualifiedClassName = dlg.getFileName();
					if (qualifiedClassName.endsWith(".class"))
						qualifiedClassName = qualifiedClassName.substring(0, qualifiedClassName.length() - 6);					
					
					MessageDialog.openWarning(_browseClass.getShell(),
							TraceMessages.CRD_CFWT,
							TraceMessages.CRD_CFW);
				}
				else
				{
					if (!hasMainMethod(cfr.getMethodInfos()))
					{
						MessageDialog.openError(_browseClass.getShell(),
								TraceMessages._9,
								TraceMessages._8);
					}
					
					qualifiedClassName = new String(cfr.getClassName());
					 
					int endOfPackageIndex = qualifiedClassName.lastIndexOf("/");
					if (endOfPackageIndex >= 0)
					{
						String packageDir = qualifiedClassName.substring(0, qualifiedClassName.lastIndexOf("/"));
						packageDir = packageDir.replace('/', File.separatorChar);
						int beginningOfPackageIndex = classpath.lastIndexOf(packageDir);
						if (beginningOfPackageIndex >= 0)
						{
							classpath = classpath.substring(0, beginningOfPackageIndex);
							if (classpath.endsWith(File.separator))
								classpath = classpath.substring(0, classpath.length()-1);
						}
					}
				}
				

				if ((osName != null) && (osName.startsWith("Windows")) && (classpath.endsWith(":"))) {
					classpath += "\\";
				}
				
				_classname.setText(qualifiedClassName.replace('/', '.'));
				
				String oldClassPath = _classpath.getText().trim();
				oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
				_classpath.setText(oldClassPath + classpath);

			}
		} else if (e.widget == _browseDirPath) {
			DirectoryDialog dlg = new DirectoryDialog(shell);
			String classpath = dlg.open();

			if (classpath != null && !classpath.equals("")) {

				if ((osName != null) && (osName.startsWith("Windows")) && (classpath.endsWith(":"))) {
					classpath += "\\";
				}

				String oldClassPath = _classpath.getText().trim();
				oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
				//if(!oldClassPath.equals("")) oldClassPath+="\n";
				classpath = oldClassPath + classpath;
				_classpath.setText(classpath);
				_classpath.setFocus();
			}

		} else if (e.widget == _browseJarPath) {
			FileDialog dlg = new FileDialog(shell);
			dlg.setFilterExtensions(new String[] { "*.jar;*.zip" });
			dlg.open();

			String classpath = dlg.getFileName();

			if (classpath != null && !classpath.equals("")) {
				classpath = dlg.getFilterPath() + File.separator + classpath;

				String oldClassPath = _classpath.getText().trim();
				oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
				classpath = oldClassPath + classpath;
				_classpath.setText(classpath);
			}
			
			if (_classname.getText().length() == 0) {
				try {
					JarFile jar = new JarFile(new File(dlg.getFilterPath() + File.separator + dlg.getFileName()));
					_classname.setText(jar.getManifest().getMainAttributes().getValue("Main-Class"));
				} catch (Exception e1) {
					// leave as is in case of problems
				}
			}
		}
		changed();
	}
}