/**********************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.core;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetType;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeItem;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociationData;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LightConfigurationLoader;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.ParentChildNode;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;

/**
 * This utility class is used for the profile launch configuration
 * 
 * @author Ali Mehregani
 */
public class ProfileLaunchUtil
{
	
	/**
	 * Given a child element that has been checked/unchecked, this
	 * method will determine if the gray state of its parent needs to
	 * be changed.
	 * 
	 * @param checkBoxTreeViewer The check box tree viewer
	 * @param event The event that has caused the notification
	 * @return true if at least one of the parent's children is checked 
	 */
	public static boolean changeParentGrayState (CheckboxTreeViewer checkBoxTreeViewer, CheckStateChangedEvent event)
	{
		ITreeContentProvider contentProvider = (ITreeContentProvider)checkBoxTreeViewer.getContentProvider();		
		Object parent = contentProvider.getParent(event.getElement());
		Object[] siblings = contentProvider.getChildren(parent);
		boolean foundChecked = event.getChecked();
		boolean foundUnchecked = false;
		
		if (siblings != null)
		{
			for(int i = 0; i < siblings.length && (!foundChecked || !foundUnchecked); i ++) 
			{
				if (checkBoxTreeViewer.getChecked(siblings[i]) == true) 
					foundChecked = true;				
				else 
					foundUnchecked = true;			
			}
		}		
		
		checkBoxTreeViewer.setChecked(parent, foundChecked);
		checkBoxTreeViewer.setGrayed(parent, foundChecked && foundUnchecked);
		
		return foundChecked;
	}
	
	
	/**
	 * Checks or unchecks the passed tree item and fires off a check state change
	 * event.
	 * @param item The item to be selected/deselected.
	 * @param checked Indicates whether the item should be checked or not
	 */
	public static void changeItemCheckedState (CheckboxTreeViewer checkBoxTreeViewer, ICheckStateListener listener, Object item, boolean checked)
	{
		checkBoxTreeViewer.setChecked(item, checked);
		CheckStateChangedEvent checkStateEvent = new CheckStateChangedEvent(checkBoxTreeViewer, item, checked);
		listener.checkStateChanged(checkStateEvent);
	}
	
	
	/**
	 * Based on the selection of a data collector/analysis type in the profile
	 * launch configuration, this method will update the state of the buttons passed in
	 */
	public static Object updateButtonStatus(ILaunchConfiguration configuration, SelectionChangedEvent event, Button editOptions, 
											Button testAvailability, CheckboxTreeViewer checkboxTreeViewer, ICheckStateListener checkStateListener)
	{
		Object selectedElementObj = event.getSelection();
		Object selectedElement = null;
		if (!(selectedElementObj instanceof StructuredSelection))
			return null;
		
		selectedElement = ((StructuredSelection)selectedElementObj).getFirstElement();
		
		
		LightConfigurationLoader configurationLoader = null;
		boolean isConfigurationPreset = false;
		boolean isButtonStatusSet = false;
		IProfilingSetType profilingSetType = null;
		boolean editOptionState = false;
		boolean testAvailabilityState = false;
		
		DataCollectorAssociationData dcAssociationData = LauncherUtility.findDataCollectorAssociationData (configuration, selectedElement);
		Object childNode = null;
		if (dcAssociationData != null)
		{
			configurationLoader = dcAssociationData.getConfigurationLoader();			
		}							
		else if (selectedElement instanceof ParentChildNode)
		{
			childNode = ((ParentChildNode)selectedElement).child;
		}
		else if (selectedElement instanceof AgentTreeItem)
		{
			childNode = ((AgentTreeItem)selectedElement).getAnalysisType();
			childNode = childNode == null ? ((AgentTreeItem)selectedElement).getProfilingType() : childNode;
		}
		
		if (childNode != null)
		{
			if (childNode instanceof AnalysisType)
				configurationLoader = ((AnalysisType)childNode).getConfigurationLoader();
			
			/* For backward compatibility */
			else if (childNode instanceof IProfilingSetType)
			{
				profilingSetType = (IProfilingSetType)childNode;
				editOptionState = true;				
				isButtonStatusSet = true;
			}
		}				
		
		if (!isButtonStatusSet)
		{
			isConfigurationPreset = configurationLoader != null; ;
			editOptionState = isConfigurationPreset && configurationLoader.isEditable();
			testAvailabilityState = isConfigurationPreset && configurationLoader.getAvailabilityTester() != null;
		}
		
		if (editOptions != null)
		{
			editOptions.setEnabled(editOptionState);		
			injectDataInButton(editOptions, new Object[][]
			                                {
												new Object[]{LauncherConstants.LAUNCH_CONFIGURATION, configuration},
												new Object[]{LauncherConstants.CONFIGURATION_LOADER, configurationLoader},
												new Object[]{LauncherConstants.PROFILING_SET_TYPE, profilingSetType},
												new Object[]{LauncherConstants.CHECKBOX_TREEVIEWER, checkboxTreeViewer},
												new Object[]{LauncherConstants.CHECKSTATE_LISTENER, checkStateListener}
			                                });
		}
		if (testAvailability != null)
		{
			testAvailability.setEnabled(testAvailabilityState);
			injectDataInButton(testAvailability, new Object[][]
			 			                                {
			 												new Object[]{LauncherConstants.LAUNCH_CONFIGURATION, configuration},
			 												new Object[]{LauncherConstants.CONFIGURATION_LOADER, configurationLoader},
			 			                                });
		}		
		
		return selectedElement;
	}


	/**
	 * Populates the data fields of the button passed in
	 * 
	 * @param button The button
	 * @param data An array of Object[2], where object[0] is expected to be
	 * a string corresponding to the key of the data and object[1] is the value
	 */
	private static void injectDataInButton(Button button, Object[][] data)
	{
		for (int i = 0; i < data.length; i++)
		{
			button.setData((String)data[i][0], data[i][1]);
		}
	}
	
	
	/**
	 * Handles the user double clicking an a data collector or an analysis type.
	 * 
	 * @param editOptions The edit option button
	 * @param listener The listener of the edit option
	 */
	public static void handleDoubleClick(Button editOptions, SelectionListener listener)
	{
		if (!editOptions.isEnabled())
			return;
	
		Event constructedEvent =  new Event();
		constructedEvent.widget = editOptions;
		listener.widgetSelected(new SelectionEvent(constructedEvent));
	}


	/**
	 * This method is used to check the validity of the tab that contains the
	 * data collectors and analysis types
	 * 
	 * @param conf The configuration
	 * @param checkBoxTreeViewer The checkbox tree viewer
	 * @return The status of the validity
	 */
	public static IStatus checkValidity(ILaunchConfiguration conf, CheckboxTreeViewer checkBoxTreeViewer, ProfilingSetsManagerCopy profilingSetMgrCopy)
	{
		/* Make sure that at least one data collector is selected */
		Object[] selection = checkBoxTreeViewer.getCheckedElements();
		boolean isValid = true;
		
		if (selection.length <= 0)
		{
			return new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID, LauncherMessages.ERROR_NO_DCM_SELECTED);
		}
		
				
		/* Walk through each selected data collector and analysis
		 * type and confirm that they return an OK status*/
		ILaunchValidator launchValidator;
		for (int i = 0; isValid && i < selection.length; i++)
		{
			launchValidator = null;
			boolean parentChildNode = selection[i] instanceof ParentChildNode;
			boolean agentTreeItem = selection[i] instanceof AgentTreeItem; 
			
			/* Find the validator */
			/* The checked item is a data collector */
			DataCollectorAssociationData dcAssociationData =  LauncherUtility.findDataCollectorAssociationData (conf, selection[i]);
			if (dcAssociationData != null)
				launchValidator = dcAssociationData.getValidator();
							
			/* The checked item is an analysis type */
			else if (parentChildNode || agentTreeItem)
			{
				AgentTreeItem currentAgentTreeItem = agentTreeItem ? (AgentTreeItem)selection[i] : null;
				Object child = parentChildNode ? ((ParentChildNode)selection[i]).child : 
								currentAgentTreeItem.getAnalysisType() == null ? currentAgentTreeItem.getProfilingType() : currentAgentTreeItem.getAnalysisType();
				
				if (child instanceof AnalysisType)
					launchValidator = ((AnalysisType)child).getValidator();				
				
				/* For backward compatibility */
				else if (child instanceof ProfilingSetType)
				{
					IProfilingType profilingType = ((ProfilingSetType)child).getProfilingType();
					
					/* Show an error if the configuration is invalid */
					Control control = createProfilingTypeControls(profilingType, profilingSetMgrCopy);
					String error = profilingType.validateConfiguration(profilingSetMgrCopy);
					destroyControl(control);
					
					if (error != null && error.length() > 0)
					{
						return new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID, error);
					}					
				}
			}
			
			
			if (launchValidator != null)
			{
				IStatus status = launchValidator.validateConfiguration(conf);
				if (status != null && status.getSeverity() != IStatus.OK)
				{
					return status;
				}
			}
		}
		
		return Status.OK_STATUS;
	}
	
	
	/**
	 * This method is used to create the controls of the profiling type before it is
	 * used.  It can be a very expensive operation, but it is needed for backward compatibility.
	 * The destroyControl method should be invoked as soon as no other operations are needed
	 * on the profiling type.
	 * 
	 * @param profilingType The profiling type.
	 * @return The control of the profiling type
	 */
	public static Control createProfilingTypeControls(IProfilingType profilingType, ProfilingSetsManagerCopy profilingSetMgrCopy)
	{
		Composite composite = new Composite (UIPlugin.getActiveWorkbenchShell(), SWT.NONE);
		profilingType.createControl(composite, profilingSetMgrCopy);
		
		return composite;
	}
	
	
	/**
	 * Commonly used to destroy the controls of a profiling type.
	 * 
	 * @param control
	 */
	public static void destroyControl (Control control)
	{
		if (control != null && !control.isDisposed())
			control.dispose();
	}
}
