/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.core;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


/**
 * The purpose of this class is to avoid having multiple classes keeping 
 * unsynchronized copies of the filters.  
 * 
 * @author Ali Mehregani
 */
public class TraceFilterManager
{
	/* The tags and attributes used to serialize the filters in XML */	
	private static final String OPEN_FILTERS_TAG = "<filters>";
	private static final String CLOSE_FILTERS_TAG  = "</filters>";
	
	private static final String OPEN_FILTER_TAG = "<filter";
	private static final String CLOSE_FILTER_TAG  = "</filter>";
	
	private static final String FILTER_ID_ATTRIBUTE="id";
	private static final String FILTER_KEY_ATTRIBUTE="key";
	private static final String FILTER_NAME_ATTRIBUTE="name";
	
	/** Introduced for Bug 323330 */
	private static final String FILTER_IS_AUTOPOP_ATTRIBUTE="autopopulated";
	
	private static final String OPEN_CONTENTS_TAG = "<contents>";
	private static final String CLOSE_CONTENTS_TAG = "</contents>";
	
	private static final String OPEN_CONTENT_TAG = "<content";
	private static final String CLOSE_CONTENT_TAG = "/>";
	
	private static final String CONTENT_TEXT_ATTRIBUTE="text";
	private static final String CONTENT_METHOD_ATTRIBUTE="method";
	private static final String CONTENT_VISIBILITY_ATTRIBUTE="visibility";
	
	
	/* The filter set list 
	 * KEY = filter set ID
	 * VALUE = filter set object */
	private Hashtable filterSets;
	
	/* The instance of this class */
	private static TraceFilterManager instance;
	
	
	private TraceFilterManager()
	{
		filterSets = new Hashtable();
		initialize();
	}
	
	private void initialize()
	{	
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		Element fitlerElements = XMLUtil.loadDom(store.getString(TraceConstants.FILTERS_SET_KEY), "filters");
		ArrayList list = parseFiltersSet(fitlerElements);
		
		storeFilterListInCache (list);
	}

	
	/**
	 * Return the instance of this singleton class
	 * 
	 * @return An instance of this class
	 */
	public static TraceFilterManager getInstance()
	{
		if (instance == null)
			instance = new TraceFilterManager();
		
		return instance;
	}
	
	private void storeFilterListInCache (ArrayList filterList)
	{
		filterSets.clear();
		FilterSetElement filterElement;
		for (int idx = 0, listSize = filterList.size(); idx < listSize; idx++)
		{
			filterElement = (FilterSetElement)filterList.get(idx);
			filterSets.put (filterElement.getId(), filterElement);
		}
	}
	
	
	/**
	 * Update what is stored in the preference and the cached
	 * version of the filter set.
	 * 
	 * @param items The filter set items
	 */
	public void storeFilterSetList(ArrayList items)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		store.setValue(TraceConstants.FILTERS_SET_KEY, OPEN_FILTERS_TAG + serializeFiltersToXML(items) + CLOSE_FILTERS_TAG);
		
		storeFilterListInCache (items);

		// Bug 203414  Persist the filters as soon as they are stored
		UIPlugin.getDefault().savePluginPreferences();
	}

	
	/**
	 * Serializes the filter set passed in into XML
	 * 
	 * @param list The list of filter sets 
	 * @return An XML serialized version of the filter set
	 */
	protected static String serializeFiltersToXML(ArrayList list) 
	{
		StringBuffer temp = new StringBuffer();
		for (int i = 0; i < list.size(); i++)
		{
			temp.append(OPEN_FILTER_TAG);
			FilterSetElement elem = (FilterSetElement)list.get(i);
			temp.append(" " + FILTER_ID_ATTRIBUTE + " = '" + elem.getId() + "'");
			if (elem.getKey() != null) {
				temp.append(" " + FILTER_KEY_ATTRIBUTE + " = '" + elem.getKey() + "'");
			}
			if (elem.getName() != null) {
				temp.append(" " + FILTER_NAME_ATTRIBUTE + " = '" + elem.getName() + "'");
			}
			
			// Introduced for bug 323330
			if(elem.isFilterAutoPopulated()) {
				temp.append(" " + FILTER_IS_AUTOPOP_ATTRIBUTE + " = 'true'");
			}
			
			temp.append(">");
			
			ArrayList children = elem.getChildren();
			temp.append(OPEN_CONTENTS_TAG);
			for (int j = 0; j < children.size(); j++)
			{
				temp.append(OPEN_CONTENT_TAG);
				FilterTableElement childElem = (FilterTableElement)children.get(j);
				temp.append(" " + CONTENT_TEXT_ATTRIBUTE + " = '" + childElem.getText() + "'");
				temp.append(" " + CONTENT_METHOD_ATTRIBUTE + " = '" + childElem.getMethod() + "'");
				temp.append(" " + CONTENT_VISIBILITY_ATTRIBUTE + " = '" + childElem.getVisibility() + "'");
				temp.append(CLOSE_CONTENT_TAG);
			}

			temp.append(CLOSE_CONTENTS_TAG);
			temp.append(CLOSE_FILTER_TAG);
			
			
		}
		return temp.toString();
	}

	public String serializeFiltersToXML() 
	{
		return serializeFiltersToXML(new ArrayList(filterSets.values()));
	}
	
	
	/**
	 * Parses through the element argument to return its respective filters
	 * 
	 * @param filters The filter element
	 * @return A filter set list based on the filters argument
	 */
	protected static ArrayList parseFiltersSet(Element filters)
	{
		//expects the filters element
		ArrayList list = new ArrayList();
	
		if (filters == null)
			return list;
		
		
		//list of filter elements
		NodeList nodeList = filters.getChildNodes();
		ArrayList children;
		
		for (int i = 0; i < nodeList.getLength(); i++)
		{
			Element elem = (Element)nodeList.item(i);
			FilterSetElement fse = new FilterSetElement(elem.getAttribute("id"));
			fse.setName(elem.getAttribute("name"));
			fse.setKey(elem.getAttribute("key"));
			
			String autoPopVal = elem.getAttribute(FILTER_IS_AUTOPOP_ATTRIBUTE);
			if(autoPopVal != null && autoPopVal.trim().equalsIgnoreCase("true")) {
				fse.setFilterAutoPopulated(true);
			}
			
			//set children
			children = getFilter(elem);
			fse.setChildren(children);

			list.add(fse);			
		}
		return list;
	}
	
	private static ArrayList getFilter(Element elem)
	{
		ArrayList list = new ArrayList();
		Element contents, temp;
		//contents tag
		contents = (Element)elem.getFirstChild();
		
		//not parse though content tags
		NodeList contentList = contents.getChildNodes();
		for (int i = 0; i < contentList.getLength(); i++)
		{
			FilterTableElement fte;
			temp = (Element)contentList.item(i);

			if ((temp.getAttribute(CONTENT_VISIBILITY_ATTRIBUTE).equals("0")) || (temp.getAttribute("visibility").equals(TraceMessages.EXCLUDE)))
				fte = new FilterTableElement(temp.getAttribute("text"), temp.getAttribute("method"), TraceMessages.EXCLUDE);
			else
				fte = new FilterTableElement(temp.getAttribute(CONTENT_TEXT_ATTRIBUTE), temp.getAttribute(CONTENT_METHOD_ATTRIBUTE), TraceMessages.INCLUDE);
			
			list.add(fte);
		}
		return list;
	}
	
	
	/**
	 * Returns the hash map version of the filter set.  Modifications to the contents
	 * the returned object will result in changes to the cached structure
	 * that this manager keeps track of.  It's recommended to use 
	 * storeFilterSet to change the content of the filter set.
	 * 
	 * @return A filter set hash map
	 */
	public Map getFilterSet()
	{
		// Even though the object itself is cloned, the contents are not. Modifications to the 
		// contents of the map will have side-effects in this class.
		return (Map)filterSets.clone();
	}
	
	
	/**
	 * Returns a copied list of the filters.
	 * 
	 * @return A filter set list
	 */
	public ArrayList getFilterSetCopy()
	{
		FilterSetElement fse;
		ArrayList filterList = new ArrayList();
		for (Iterator iter = filterSets.values().iterator(); iter.hasNext(); )
		{
			fse = clone ((FilterSetElement)iter.next());
			filterList.add(fse);
		}
		return filterList;
	}
	
	
	/**
	 * Returns the filter element with the id passed in
	 * 
	 * @param id The id of the filter set element
	 */
	public FilterSetElement getFilterSetElement(String id)
	{
		return clone((FilterSetElement)filterSets.get(id));
	}

	
	/**
	 * Returns a cloned version of the filter set element
	 * @param element The filter set element
	 * @return A cloned version of the filter set element
	 */
	private FilterSetElement clone(FilterSetElement element)
	{
		if (element == null)
			return null;
		
		FilterSetElement fse = new FilterSetElement(element.getId());
		
		/* Copy the children of the filter set element */
		ArrayList filterTableElements = element.getChildren();
		ArrayList filterElementsCopy = new ArrayList();
		FilterTableElement currentElement;
		for (int i = 0, filterChildrenCount = filterTableElements.size(); i < filterChildrenCount; i++)
		{
			currentElement = (FilterTableElement)filterTableElements.get(i);
			filterElementsCopy.add(new FilterTableElement(currentElement.getText(), currentElement.getMethod(), currentElement.getVisibility()));
		}
		fse.setChildren(filterElementsCopy);		
		fse.setKey(element.getKey());
		fse.setName(element.getName());
		fse.setFilterAutoPopulated(element.isFilterAutoPopulated());
		
		return fse;
	}
}
