/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceProfileCollectionsUI.java,v 1.8 2006/05/11 15:05:09 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.Vector;

import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.ui.PlatformUI;

public class TraceProfileCollectionsUI extends AbstractChangeable
	implements SelectionListener, ModifyListener {
	
	private Label _limitInvocLabel;
	private Label _limitTimeLabel;
	private Text _limitTime;
	private Text _limitInvocations;
	private Button _limitTimeBtn;
	private Button _limitInvocBtn;
	
	private Composite _monitoringGroup;
	private Button _autoMonitoring;
	private Label _monitoringDesp;
	
	private WizardPage _wizardPage;
	private PreferencePage _preferencePage;
	private boolean isAttach = false;
	
	public TraceProfileCollectionsUI(boolean attach) {
		super();
		isAttach = attach;
		
	}
	
	public TraceProfileCollectionsUI() {
		super();
		
	}

	public Composite createControl(Composite parent) {
		GridLayout layout;

		Composite result = new Composite(parent, SWT.NONE);

		layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 2;

		GridData data = GridUtil.createFill();
		data.heightHint = 300;
		data.widthHint = 400;

		result.setLayout(layout);
		result.setLayoutData(data);

		addLimitTraceGroup(result);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitInvocLabel, UIPlugin
				.getPluginId()
				+ ".tlmt0001");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitInvocations, UIPlugin
				.getPluginId()
				+ ".tlmt0001");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitInvocBtn, UIPlugin
				.getPluginId()
				+ ".tlmt0001");

		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitTimeLabel, UIPlugin
				.getPluginId()
				+ ".tlmt0002");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitTime, UIPlugin
				.getPluginId()
				+ ".tlmt0002");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(_limitTimeBtn, UIPlugin
				.getPluginId()
				+ ".tlmt0002");		
		if (!isAttach)
		{
			addMonitorOption(result);
			_monitoringGroup.setVisible(true);
		}
		

		return result;
	}

	private void addLimitTraceGroup(Composite parent) {
		Group grp = new Group(parent, SWT.NULL);
		GridData data = GridUtil.createHorizontalFill();
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		grp.setLayout(layout);
		grp.setLayoutData(data);
		grp.setText(UIPlugin.getResourceString(TraceMessages.LIMIT_DTMT));

		_limitInvocBtn = new Button(grp, SWT.CHECK);
		_limitInvocBtn.setText(UIPlugin.getResourceString(TraceMessages.INV_LO));
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		_limitInvocBtn.setLayoutData(data);

		_limitInvocLabel = new Label(grp, SWT.NULL);
		_limitInvocLabel.setText(UIPlugin.getResourceString(TraceMessages.INV_LNB));
		data = new GridData();
		data.horizontalIndent = 20;
		_limitInvocLabel.setLayoutData(data);
		_limitInvocations = new Text(grp, SWT.BORDER);
		_limitInvocations.setLayoutData(GridUtil.createHorizontalFill());

		_limitTimeBtn = new Button(grp, SWT.CHECK);
		_limitTimeBtn.setText(UIPlugin.getResourceString(TraceMessages.TML_OPT));
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		_limitTimeBtn.setLayoutData(data);

		_limitTimeLabel = new Label(grp, SWT.NULL);
		_limitTimeLabel.setText(UIPlugin.getResourceString(TraceMessages.TML_NB));
		data = new GridData();
		data.horizontalIndent = 20;
		_limitTimeLabel.setLayoutData(data);
		_limitTime = new Text(grp, SWT.BORDER);
		_limitTime.setLayoutData(GridUtil.createHorizontalFill());

		_limitInvocBtn.addSelectionListener(this);
		_limitTimeBtn.addSelectionListener(this);

		_limitInvocations.addModifyListener(this);
		_limitTime.addModifyListener(this);
		
	}

	private void addMonitorOption(Composite parent) {
		_monitoringGroup = new Composite(parent, SWT.NULL);
		GridData data = GridUtil.createHorizontalFill();
		//data.horizontalSpan = 2;
		_monitoringGroup.setLayoutData(data);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		_monitoringGroup.setLayout(layout);
	
		_autoMonitoring = new Button(_monitoringGroup, SWT.CHECK);
		_autoMonitoring.setText(
			UIPlugin.getResourceString(TraceMessages.AMONC));
		_autoMonitoring.addSelectionListener(this);
		_monitoringDesp = new Label(_monitoringGroup, SWT.NULL);
		_monitoringDesp.setText(
			UIPlugin.getResourceString(TraceMessages.AMONL));
		data = new GridData();
		data.horizontalIndent = 20;
		_monitoringDesp.setLayoutData(data);
		_monitoringDesp.setVisible(false);
	}

	public void enable(boolean enable) {
		_limitInvocations.setEnabled(enable);
		_limitInvocBtn.setEnabled(enable);
		_limitInvocLabel.setEnabled(enable);
		_limitTime.setEnabled(enable);
		_limitTimeBtn.setEnabled(enable);
		_limitTimeLabel.setEnabled(enable);
		if (!isAttach) _autoMonitoring.setEnabled(enable);
	}

	public void enableButtons() {

		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		setLimitInvocations(
			store.getBoolean(TraceConstants.LIMIT_TRACE_INVOC_OPTION));
		setLimitTime(store.getBoolean(TraceConstants.LIMIT_TRACE_TIME_OPTION));
		setLimitInvocNb(store.getString(TraceConstants.LIMIT_INVOC_NB));
		setLimitTimeNb(store.getString(TraceConstants.LIMIT_TIME_NB));
		if (!isAttach) setAutoMonitoring(store.getBoolean(TraceConstants.PROF_AUTO_MONITOR));
		
	}

	
	/**
	 * Return the options as a set of AgentConfigurationEntry entries.
	 * 
	 * @deprecated use getProfilingOptions instead
	 */
	public Vector getOptions() 
	{
		return getGeneralOptions (AgentConfigurationEntry.class);
	}

	
	/**
	 * Return the options as a set of ProfilingAttribute entries.
	 *
	 * @return A set of profiling options.
	 */
	public Vector getProfilingOptions() 
	{
		return getGeneralOptions (ProfilingAttribute.class);
	}
	
	
	private Vector getGeneralOptions(Class optionType)
	{
		Vector options = new Vector();

		//time,imvocations limited events
		if (getLimitInvocations()) 
		{
			if (getLimitTime()) 
			{
				options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_MODE, Options.OPTION_VALUE_SECONDSANDINVOCATIONS));
				options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_SECONDS, String.valueOf(getLimitTimeNb())));
			} 
			else 
				options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_MODE, Options.OPTION_VALUE_SECONDSANDINVOCATIONS));

			options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_INVOCATIONS, String.valueOf(getLimitInvocNb())));
		} 
		else if (getLimitTime()) 
		{
			options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_MODE, Options.OPTION_VALUE_SECONDS));
			options.addElement(createOption (optionType, LauncherConstants.SET_OPTION_PREFIX, Options.OPTION_BURST_SECONDS, String.valueOf(getLimitTimeNb())));
		}
		
		return options;
	}
	
	private Object createOption(Class optionType, String type, String name, String value)
	{
		if (optionType == AgentConfigurationEntry.class)
		{
			AgentConfigurationEntry ace = new AgentConfigurationEntry();
			ace.setType(type);
			ace.setName(name);
			ace.setValue(value);
			return ace;
		}
		else if (optionType == ProfilingAttribute.class)
		{
			ProfilingAttribute pa = new ProfilingAttribute(type + "_" + name, value);
			return pa;
		}
		
		return null;
	}

	public void modifyText(ModifyEvent event)
	{   
		if (event.getSource()==_limitInvocations
			|| event.getSource() == _limitTime)
		{
			checkNumbersAndPrintErrorMsg();
		}
	}
	
	private boolean checkValidTextInput()
	{
		boolean status = true;
		
		status = status && isValuePositiveInteger(_limitInvocBtn.getSelection(), _limitInvocations.getText().trim());
		status = status && isValuePositiveInteger(_limitTimeBtn.getSelection(), _limitTime.getText().trim());

		return status;
	}

	
	private boolean isValuePositiveInteger(boolean condition, String value)
	{
		if (condition)
			return LauncherUtility.isInputPositiveInteger(value);
			
		return true;
	}
	
	
	
	public boolean getLimitInvocations() {
		return _limitInvocBtn.getSelection();
	}

	public int getLimitInvocNb() {
		String text = _limitInvocations.getText().trim();

		try {
			int value = Integer.parseInt(text);
			if (value>0) return value;
			else return  UIPlugin.getDefault().getPreferenceStore().getDefaultInt(TraceConstants.LIMIT_INVOC_NB);
		} catch (Exception exc) {
			return UIPlugin.getDefault().getPreferenceStore().getDefaultInt(TraceConstants.LIMIT_INVOC_NB);
		}
	}

	public boolean getLimitTime() {
		return _limitTimeBtn.getSelection();
	}

	public int getLimitTimeNb() {
		String text = _limitTime.getText().trim();

		try {
			int value = Integer.parseInt(text);
			if (value>0) return value;
			else return UIPlugin.getDefault().getPreferenceStore().getDefaultInt(TraceConstants.LIMIT_TIME_NB);
		} catch (Exception exc) {
			return UIPlugin.getDefault().getPreferenceStore().getDefaultInt(TraceConstants.LIMIT_TIME_NB);
		}
	}

	public void setLimitInvocations(boolean setValue) {
		_limitInvocBtn.setSelection(setValue);

		if (!_limitInvocBtn.isEnabled())
			return;

		_limitInvocations.setEnabled(setValue);
		_limitInvocLabel.setEnabled(setValue);
	}

	public void setLimitInvocNb(String value) {
		_limitInvocations.setText(value);
	}

	public void setLimitTime(boolean setValue) {
		_limitTimeBtn.setSelection(setValue);

		if (!_limitTimeBtn.isEnabled())
			return;

		_limitTime.setEnabled(setValue);
		_limitTimeLabel.setEnabled(setValue);
	}

	public void setLimitTimeNb(String value) {
		_limitTime.setText(value);
	}

	public void widgetDefaultSelected(SelectionEvent event) {
	}

	public void widgetSelected(SelectionEvent event) {
		if (event.widget == _limitInvocBtn) {
			_limitInvocations.setEnabled(_limitInvocBtn.getSelection());
			_limitInvocLabel.setEnabled(_limitInvocBtn.getSelection());
		} else if (event.widget == _limitTimeBtn) {
			_limitTime.setEnabled(_limitTimeBtn.getSelection());
			_limitTimeLabel.setEnabled(_limitTimeBtn.getSelection());
		}
		changed();
	}
	
	public void checkNumbersAndPrintErrorMsg()
	{
		if (_wizardPage!=null)
		{
			if (!checkValidTextInput())
			{
				_wizardPage.setErrorMessage(UIPlugin.getResourceString(TraceMessages.INV_CVAL));
				_wizardPage.setPageComplete(false);
			}
			else
			{
				_wizardPage.setErrorMessage(null);
				_wizardPage.setPageComplete(true);
			}
		}
		else if (_preferencePage!=null)
		{
			if (!checkValidTextInput())
			{
				_preferencePage.setErrorMessage(UIPlugin.getResourceString(TraceMessages.INV_CVAL));
				_preferencePage.setValid(false);
			}
			else
			{
				_preferencePage.setErrorMessage(null);
				_preferencePage.setValid(true);
			}
		}
		changed();
	}


	public void setWizardPage(WizardPage page) {
		_wizardPage = page;
	}

	public void setPreferencePage(PreferencePage page) {
		_preferencePage = page;
	}

	public void enableMonitoringSlectionGroup(boolean visible) {
		if (!isAttach) _monitoringGroup.setVisible(visible);
	}

	public void enableMonitoringDesp(boolean visible) {
		if (!isAttach) _monitoringDesp.setVisible(visible);
	}

	public boolean getAutoMonitoring() {
		if (!isAttach) return _autoMonitoring.getSelection();
		else return false;
		
	}

	public void setAutoMonitoring(boolean select) {
		if (!isAttach) _autoMonitoring.setSelection(select);
	}

	public void storeValues() {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		store.setValue(TraceConstants.PROF_AUTO_MONITOR, getAutoMonitoring());
	}

	public void setDefaultOptions()
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
	
		setLimitInvocations(store.getBoolean(TraceConstants.LIMIT_TRACE_INVOC_OPTION));
		setLimitInvocNb(String.valueOf(store.getInt(TraceConstants.LIMIT_INVOC_NB)));
		setLimitTime(store.getBoolean(TraceConstants.LIMIT_TRACE_TIME_OPTION));
		setLimitTimeNb(String.valueOf(store.getInt(TraceConstants.LIMIT_TIME_NB)));
		setAutoMonitoring(store.getBoolean(TraceConstants.PROF_AUTO_MONITOR));
	}
	
	
	/**
	 * Returns an error message if there is an error with the limitations; 
	 * otherwise null
	 * 
	 * @return null or an error message
	 */
	public String validate ()
	{
		if (!isValuePositiveInteger(_limitInvocBtn.getSelection(), _limitInvocations.getText().trim()))
			return TraceMessages.ERROR_LAUNCH_CONF_LIMIT_INVOC;
		
		if (!isValuePositiveInteger(_limitTimeBtn.getSelection(), _limitTime.getText().trim()))
			return TraceMessages.ERROR_LAUNCH_CONF_LIMIT_TIME;
		
		return null;
	}

}
